/*  
 *  ZX-PM - Power Manager for ZX Spectrum. Controls ATX Power Supply.
 *          Version: 1.0
 *
 *          Designed for ATtiny13
 *
 *  2007-05-15 - initial release
 *
 *
 *  Copyright (C) 2007 Alexander V. Melnikoff
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <avr/interrupt.h>
#include <avr/io.h>
#include <util/delay.h>

// pin configuration
#define ZX_PM_PWRLED_PORT		PORTB
#define ZX_PM_PWRLED_PIN		PB0
#define ZX_PM_PWRLED_DDR		DDRB
#define ZX_PM_PWRLED_MSK		(1 << ZX_PM_PWRLED_PIN)

#define ZX_PM_PWRSW_PORT		PINB
#define ZX_PM_PWRSW_PIN			PINB2
#define ZX_PM_PWRSW_DDR			DDRB
#define ZX_PM_PWRSW_MSK			(1 << ZX_PM_PWRSW_PIN)

#define ZX_PM_PSON_PORT			PORTB
#define ZX_PM_PSON_PIN			PB3
#define ZX_PM_PSON_DDR			DDRB
#define ZX_PM_PSON_MSK			(1 << ZX_PM_PSON_PIN)

#define ZX_PM_PWROK_PORT		PINB
#define ZX_PM_PWROK_PIN			PINB4
#define ZX_PM_PWROK_DDR			DDRB
#define ZX_PM_PWROK_MSK			(1 << ZX_PM_PWROK_PIN)

#define ZX_PM_BLINK_DELAY		1000

unsigned int second_count = 0;

int main(void)
{
	// initialize port pins
	ZX_PM_PWRLED_DDR |=  ZX_PM_PWRLED_MSK; // power led port
	ZX_PM_PWRSW_DDR  &= ~ZX_PM_PWRSW_MSK;  // power switch port
	ZX_PM_PSON_DDR   |=  ZX_PM_PSON_MSK;   // atx ps_on signal port
	ZX_PM_PWROK_DDR  &= ~ZX_PM_PWROK_MSK;  // atx pwr_ok signal port

	ZX_PM_PSON_PORT   |=  ZX_PM_PSON_MSK;
	ZX_PM_PWRLED_PORT &= ~ZX_PM_PWRLED_MSK;


	// initialize Pin Change Interrupts (PCMSK)
	PCMSK |= (1 << PCINT2);
	GIMSK |= (1 << PCIE);

	// initialize INT0 interrupt (MCUCR)
	// log 0 of PWR_EVT requests an interrupt
	GIMSK |= (1 << INT0);

	// initialize timer/counter for blincking power led
	// TCCR0A, TIMSK0
	TCCR0B |= (1 << CS01);
	TCNT0   = 0;

	TIMSK0 |= (1 << TOIE0);

	second_count = ZX_PM_BLINK_DELAY;

	sei();

	// interrupts will do the rest
	while(1);

	// not reached in regular world :-)
	return 0;
}

// INT0 interrupt event handler
ISR(INT0_vect)
{
	ZX_PM_PSON_PORT ^= ZX_PM_PSON_MSK;
}

// PCINT0 interrupt event handler
ISR(PCINT0_vect)
{
	unsigned char i;

	// when button pressed, port pin is logick 0.
	if( !(ZX_PM_PWRSW_PORT & ZX_PM_PWRSW_MSK) )
	{
		unsigned int delay_seconds;

		// select delay time
		if( !(ZX_PM_PSON_PORT & ZX_PM_PSON_MSK) )
			delay_seconds = 40; // wait 4 sec before off
		else
			delay_seconds =  1; // wait 0.1 sec before on

		// wait for power off or power on
		for(i = 0; i < delay_seconds; i++)
			_delay_ms(100); // 0.1 second delay

		// if power switch button is still pushed, we need
		// to switch power supply off/on
		if( !(ZX_PM_PWRSW_PORT & ZX_PM_PWRSW_MSK) )
		{
			// check power supply state, i.e. its on and off state
			// PSON in low
			if( !(ZX_PM_PSON_PORT & ZX_PM_PSON_MSK) )
			{
				// switch power off, set bit to 1 and LED bit to 0
				ZX_PM_PSON_PORT   |= ZX_PM_PSON_MSK;
				ZX_PM_PWRLED_PORT &= ~ZX_PM_PWRLED_MSK;
			}
			else
			{
				// switch power on, set bit to 0 and LED bit to 1
				ZX_PM_PSON_PORT   &= ~ZX_PM_PSON_MSK;
				ZX_PM_PWRLED_PORT |= ZX_PM_PWRLED_MSK;
			}

			// wait for a second to let user releae the button
			for(i = 0; i < 10; i++)
				_delay_ms(100); // 0.1 second delay
		}
	}

	// we do not need Pin Change Interrupt for power switch button to
	// be stacked while we were in the interrupt handler. reset PCIF 
	// flag in the GIFR register.
	GIFR &= ~(1 << PCIF);
}

ISR(TIM0_OVF_vect)
{
int i;
	// if power supply is off (bit set to 1), we need to blink power led
	if(ZX_PM_PSON_PORT & ZX_PM_PSON_MSK)
	{
		if(--second_count == 0)
		{
			second_count = ZX_PM_BLINK_DELAY;

			// if LED is on, we need to turn it off
			if(ZX_PM_PWRLED_PORT & ZX_PM_PWRLED_MSK)
				ZX_PM_PWRLED_PORT &= ~ZX_PM_PWRLED_MSK;

			ZX_PM_PWRLED_PORT ^= ZX_PM_PWRLED_MSK;

			_delay_ms(100);

			ZX_PM_PWRLED_PORT ^= ZX_PM_PWRLED_MSK;
		}

	}
	else
	{
		ZX_PM_PWRLED_PORT |= ZX_PM_PWRLED_MSK;
	}
}

