/*
 * demo5.c
 *
 * Copyright (c) 1996 by Paul W. Lee
 *
 * Permission is granted to anyone to use this software for any purpose
 * on any computer system, and to redistribute it freely, with the
 * following restrictions:
 * 1) No charge may be made other than reasonable charges for reproduction.
 * 2) Modified versions must be clearly marked as such.
 * 3) The authors are not responsible for any harmful consequences
 *    of using this software, even if they result from defects in it.
 *
 *
 * This program will load in the Atari 800 character set and display
 * it on the Genesis as in demo1.c.  However, this version will use
 * DMA, fade in of the character set, play a sample, and then scroll
 * the text down the screen.
 *
 * Note: Added a VBL routine.  Also, we only scrolled plane A, that
 *       is the reason we get the dual character set.
 *
 */

#include "genesis.h"

/*
 * Function prototypes
 */
void init_gfx();
void dma_vram_copy(source, dest, len);
void show_tiles();
void fade_in(palette);
void wait_sync();
void play_sample(addr, len);
void scroll_down();

/*
 *  External Vars.
 */
extern ulong tiledata[];
extern uchar SOUND[];
extern uchar SOUND_END[];

/*
 *  Color palette.
 *
 *  Do not try to change these values in
 *  the code.  This array should be
 *  treated like constants.
 *
 */

uint col_pal[16] = {
    0,
    0xeee,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0    
};

/*
 *  Variables...
 */
ulong vtimer;   /* VBL timer */

/*
 *  Main program
 */
void main()
{
    register uint i;

    /* Initialize Genesis GFX registers */
    init_gfx();

    /* Load the character set */
    dma_vram_copy(tiledata, 0, 128 * 32);
    
    /* Show the character set */
    show_tiles();

    /* Fade in colors */
    fade_in(col_pal);

    /* Play 'SEGA' digitized sound */
    play_sample(SOUND, SOUND_END-SOUND);
    
    /* Wait for sample to finish */
    for (i = 0; i < 120; i++)
        wait_sync();

    /* Scroll text down the screen */
    scroll_down();

    /* LOOP FOREVER */
    while (1)
    {
        asm("nop");
    }
}

/*
 * Initilize the GFX.
 */
void init_gfx()
{
    register uint *pw;

    pw = (uint *) GFXCNTL;

    *pw = 0x8016;   /* reg. 80 - Enable HBL */
    *pw = 0x8174;   /* reg. 81 - Enable VBL */
    *pw = 0x8238;   /* reg. 82 - Plane A = $E000 */
    *pw = 0x8338;   /* reg. 83 -    "        "   */
    *pw = 0x8407;   /* reg. 84 - Plane B = $E000 */
    *pw = 0x857e;
    *pw = 0x8600;
    *pw = 0x8700;
    *pw = 0x8801;
    *pw = 0x8901;
    *pw = 0x8a01;
    *pw = 0x8b00;
    *pw = 0x8c08;
    *pw = 0x8d08;
    *pw = 0x8f02;
    *pw = 0x9000;
    *pw = 0x9100;
    *pw = 0x92ff;
}

/*
 * Show the tiles by writing the tile
 * number (1-128) to the GFX memory.
 */
void show_tiles()
{
    register ulong *pl;
    register uint *pw;
    register uint i;

    pl = (ulong *) GFXCNTL;
    *pl = GFX_WRITE_ADDR(0xe000);

    pw = (uint *) GFXDATA;
    for (i = 0; i < 128; i++)
        *pw = 0x8000 + i;
}

/*
 * Use the DMA to copy the character
 * set data to the GFX RAM.
 */
void dma_vram_copy(source, dest, len)
ulong source;
uint dest;
uint len;
{
    register uint *pw;
    register ulong *pl;

    /* Halt the Z80 for DMA */
    pw = (uint *) Z80_HALT;
    *pw = 0x100;

    /* Point to GFX register */
    pw = (uint *) GFXCNTL;

    /* Setup DMA length */
    len >>= 1;
    *pw = 0x9300 + (len & 0xff);
    *pw = 0x9400 + ((len >> 8) & 0xff);

    /* Setup DMA address */
    source >>= 1;
    *pw = 0x9500L + (source & 0xff);

    source >>= 8;
    *pw = 0x9600L + (source & 0xff);

    source >>= 8;
    *pw = 0x9700L + (source & 0xff);

    /* Enable DMA */
    pl = (ulong *) GFXCNTL;
    *pl = GFX_DMA_ADDR((long) dest);

    /* Enable Z80 */
    pw = (uint *) Z80_HALT;
    *pw = 0;
}

/*
 * Fade in a given palette.
 */
void fade_in(palette)
uint *palette;
{
    uint tmp_pal[16];
    register ulong *pl;
    register uint *pw;
    register uint *pal_ptr;
    register uint i, j, c;

    /* Initialize a temporary palette */
    for (i = 0; i < 16; i++)
        tmp_pal[i] = 0;

    /* Fade in loop */
    for (i = 0; i < 16; i++)
    {
        /* Pointer to palette we want */
        pal_ptr = palette;
    
        /* Setup GFX to save colors */
        pl = (ulong *) GFXCNTL;
        *pl = GFX_COLOR_WRITE_ADDR(0);

        /* Setup GFX data register */
        pw = (uint *) GFXDATA;
        
        /* Increment one color B,G,R at a time */
        for (j = 0; j < 16; j++)
        {
            /* increment BLUE till reaching target color */
            c = tmp_pal[j] & 0xf00;
            if (c < (*pal_ptr & 0xf00))
                tmp_pal[j] += 0x100;
            /* check GREEN */
            c = tmp_pal[j] & 0xf0;
            if (c < (*pal_ptr & 0xf0))
                tmp_pal[j] += 0x10;
            /* check RED */
            c = tmp_pal[j] & 0xf;
            if (c < (*pal_ptr++ & 0xf))
                tmp_pal[j] += 0x1;

            /* Store color */
            *pw = tmp_pal[j];
        }

        /* Wait a sec... */
        for (j = 0; j < 4; j++)
            wait_sync();
    }
}

/*
 *  Wait for VBL.
 */
void wait_sync()
{
    register ulong x;

    x = vtimer;
    while (x == vtimer);
}

/*
 *  Play sample
 */
void play_sample(addr, len)
ulong addr;
ulong len;
{
    register uint *pw;
    register uchar *pb;

    /* Halt the Z80 for DMA */
    pw = (uint *) Z80_HALT;
    *pw = 0x100;

    /* Point to Z80 RAM */
    pb = (uchar *) 0xa00039;

    /* Flag to begin playing */
    *pb++ = 1;

    /* Store address of sample */
    *pb++ = addr & 0xff;
    *pb++ = (addr >> 8) & 0xff;
    *pb++ = (addr >> 16) & 0xff;

    /* Store length of sample */
    *pb++ = len & 0xff;
    *pb++ = (len >> 8) & 0xff;
    *pb++ = (len >> 16) & 0xff;

    /* Enable Z80 and begin playing */
    pw = (uint *) Z80_HALT;
    *pw = 0;

}

/*
 * Scroll the text down the screen
 */
void scroll_down()
{
    register uint i;
    register uint *pw;
    register ulong *pl;

    /* Setup GFX pointers */
    pw = (uint *) GFXDATA;
    pl = (ulong *) GFXCNTL;

    for (i = 0; i < 180; i++)
    {
         /* GFX scroll plane A down */
         *pl = GFX_VERT_SCROLL(0);
         *pw = -i;

         /* Wait a VBL*/
         wait_sync();
    }

    for (i = 0; i < 180; i++)
    {
         /* GFX scroll plane B down */
         *pl = GFX_VERT_SCROLL(2);
         *pw = -i;

         /* Wait a VBL*/
         wait_sync();
    }

}

