
#include "z80.h"


#define XMASK 0300
#define YMASK 070
#define ZMASK 7
#define PMASK 060
#define QMASK 010

#define YPART(c) (((c) & YMASK) >> 3)
#define ZPART(c) ((c) & ZMASK)
#define PPART(c) (((c) & PMASK) >> 4)

#define ZF_BIT 6
#define HF_BIT 4
#define PF_BIT 2
#define NF_BIT 1

#define PF_LOG4(n) ((1u << ((n) & 0xf)) & 0x9669u)
#define PF_LOG(n) ((!PF_LOG4((n)) == !PF_LOG4((n) >> 4)) << PF_BIT)
#define PF_INC(n) (((n) == 0x80) << PF_BIT)
#define PF_DEC(n) (((n) == 0x7f) << PF_BIT)
#define PF_ARI(r, a, b) (((((r) ^ (a) ^ (b)) >> 6) ^ (((r) ^ (a) ^ (b)) >> 5)) & Z80_PF)  /* !!! refine */
#define HF_INC(n) ((((n) & 0xf) == 0x0) << HF_BIT)
#define HF_DEC(n) ((((n) & 0xf) == 0xf) << HF_BIT)
#define HF_ARI(r, a, b) (((r) ^ (a) ^ (b)) & Z80_HF)
#define ZF_ARI(n) (!(n) << ZF_BIT)

#define CC(i) (!(Z80_AF & cc_tab[(i) / 2]) ^ ((i) & 1))

#define ADD16(a, b) (((a) + (b)) & LIB_UZX_DATA_MASK16)
#define SUB16(a, b) (((a) - (b)) & LIB_UZX_DATA_MASK16)

#define INC16(w) ADD16((w), 1)
#define DEC16(w) SUB16((w), 1)

#define CONT(a) ((void) (z80_cont_handler((a), Z80_ticks), Z80_ticks++))
#define CONT2(a) ((void) (CONT((a)), CONT((a))))
#define CONT3(a) ((void) (CONT2((a)), CONT((a))))
#define CONT4(a) ((void) (CONT2((a)), CONT2((a))))
#define CONT5(a) ((void) (CONT4((a)), CONT((a))))
#define FETCH(a) (t1 = z80_fetch_handler(Z80_PC, Z80_ticks), Z80_PC = INC16(Z80_PC), INC_R, t1)
#define READ(a) (z80_read_handler((a), Z80_ticks))
#define WRITE(a, b) ((void) (z80_write_handler((a), (b), Z80_ticks)))
#define INPUT(a) (z80_input_handler((a), Z80_ticks))
#define OUTPUT(a, b) ((void) (z80_output_handler((a), (b), Z80_ticks)))

#define READ_PAIR(a) (t2 = READ((a)), LIB_UZX_DATA_PAIR(READ(INC16((a))), t2))
#define WRITE_PAIR(a, b) ((void) (WRITE((a), LIB_UZX_DATA_LOW((b))), WRITE(INC16((a)), LIB_UZX_DATA_HIGH((b)))))

#define IMM16 (t3 = READ_PAIR(Z80_PC), Z80_PC = ADD16(Z80_PC, 2), t3)
#define IMM8 (t3 = READ(Z80_PC), Z80_PC = INC16(Z80_PC), t3)

#define DISP(c) ((pi == 2 || (c)) ? 0 : \
    (t4 = READ(Z80_PC), CONT5(Z80_PC), Z80_PC = INC16(Z80_PC), t4))

#define PUSH(w) ((void) (Z80_SP = DEC16(Z80_SP), WRITE(Z80_SP, LIB_UZX_DATA_HIGH((w))), \
    Z80_SP = DEC16(Z80_SP), WRITE(Z80_SP, LIB_UZX_DATA_LOW((w)))))
#define POP (t3 = READ(Z80_SP), Z80_SP = INC16(Z80_SP), \
    t3 = LIB_UZX_DATA_PAIR(READ(Z80_SP), t3), Z80_SP = INC16(Z80_SP), t3)

#define INC_R ((void) (Z80_R7++, z80_R_change_handler(Z80_R, Z80_ticks)))

unsigned z80_prefix, z80_regs[5], z80_alts[4];
unsigned Z80_AF, Z80_SP, Z80_PC, Z80_I, Z80_R8, Z80_R7, z80_MEMPTR;
unsigned Z80_IFF1, Z80_IFF2, Z80_IM, Z80_HALT, Z80_INT_disabled;
unsigned long Z80_ticks, xZ80_INT_tick;

static const unsigned cc_tab[4] = { Z80_ZF, Z80_CF, Z80_PF, Z80_SF };


void z80_init(void)
{
    Z80_PC = 0x0000, Z80_SP = 0xffff;
    Z80_I = Z80_R8 = Z80_R7 = z80_MEMPTR = 0;  /* !!! does MEMPTR really initialize with 0x0000? */
    Z80_IFF1 = Z80_IFF2 = Z80_IM = Z80_HALT = Z80_INT_disabled = 0;
    xZ80_INT_tick = Z80_ticks = 0;
    z80_prefix = 2;  /* HL */
}

static void swap(unsigned a[], unsigned b[], unsigned size)
{
    unsigned i, t;
    for(i = 0; i < size; i++) t = a[i], a[i] = b[i], b[i] = t;
}

static unsigned disp(unsigned a, unsigned d)
{
    assert(a < 0x10000);
    assert(d < 0x100);

    return (!(d & LIB_UZX_DATA_SIGN8) ? (a + d) :
        (a - ((d ^ LIB_UZX_DATA_MASK8) + 1))) & LIB_UZX_DATA_MASK16;
}

static unsigned char get_r(unsigned i, unsigned pi, unsigned d, unsigned t)
{
    unsigned r;

    assert(i < 8);
    assert(t == 0 || t == 1);

    if(i < 4) return (i & 1) ? LIB_UZX_DATA_LOW(z80_regs[i / 2]) : LIB_UZX_DATA_HIGH(z80_regs[i / 2]);
    if(i < 6) return (i & 1) ? LIB_UZX_DATA_LOW(z80_regs[pi]) : LIB_UZX_DATA_HIGH(z80_regs[pi]);
    if(i == 6) return (d = disp(z80_regs[pi], d), r = READ(d),
        (pi == 2 || (z80_MEMPTR = d, 1)), (t == 0 || (CONT(d), 1)), r);
    return LIB_UZX_DATA_HIGH(Z80_AF);
}

static void put_r(unsigned i, unsigned pi, unsigned d, unsigned n)
{
    unsigned r;

    assert(i < 8);
    assert(n < 0x100);

    if(i < 4) z80_regs[i / 2] = (i & 1) ? ((z80_regs[i / 2] & LIB_UZX_DATA_HIGH_MASK) | n) :
        ((z80_regs[i / 2] & LIB_UZX_DATA_LOW_MASK) | (n << 8));
    else if(i < 6) z80_regs[pi] = (i & 1) ? ((z80_regs[pi] & LIB_UZX_DATA_HIGH_MASK) | n) :
        ((z80_regs[pi] & LIB_UZX_DATA_LOW_MASK) | (n << 8));
    else if(i == 6) r = disp(z80_regs[pi], d),
        (pi == 2 || (z80_MEMPTR = r, 1)), WRITE(r, n);
    else Z80_AF = LIB_UZX_DATA_PAIR(n, LIB_UZX_DATA_LOW(Z80_AF));
}

static unsigned get_rp(unsigned i, unsigned pi)
{
    assert(i < 4);
    assert(pi == 2 || pi == 3 || pi == 4);  /* HL, IX, or IY */
    if(i == 2) return z80_regs[pi];
    return i == 3 ? Z80_SP : z80_regs[i];
}

static void put_rp(unsigned i, unsigned pi, unsigned n)
{
    assert(i < 4 && n < 0x10000);
    assert(pi == 2 || pi == 3 || pi == 4);  /* HL, IX, or IY */
    if(i == 2) z80_regs[pi] = n;
    else if(i < 3) z80_regs[i] = n;
    else Z80_SP = n;
}

static void do_alu(unsigned alu, unsigned n)
{
    unsigned long r, s;  /* !!! long */

    assert(n < 0x100);

    switch(alu)  {
    case 0:  /* ADD */
        r = LIB_UZX_DATA_HIGH(Z80_AF), s = (r + n) & LIB_UZX_DATA_MASK8;
        Z80_AF = LIB_UZX_DATA_PAIR(s, (s & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(s) | HF_ARI(s, r, n) | PF_ARI(r + n /* !!! long */, r, n) | /* Z80_CF */ (s < r));
        break;
    case 1:  /* ADC */
        r = LIB_UZX_DATA_HIGH(Z80_AF), s = (r + n + (Z80_AF & Z80_CF)) & LIB_UZX_DATA_MASK8;
        Z80_AF = LIB_UZX_DATA_PAIR(s, (s & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(s) | HF_ARI(s, r, n) | PF_ARI(r + n + (Z80_AF & Z80_CF) /* !!! long */, r, n) | /* Z80_CF */ (s < r || ((Z80_AF & Z80_CF) && n == 0xff)));
        break;
    case 2:  /* SUB */
        r = LIB_UZX_DATA_HIGH(Z80_AF), s = (r - n) & LIB_UZX_DATA_MASK8;
        Z80_AF = LIB_UZX_DATA_PAIR(s, (s & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(s) | HF_ARI(s, r, n) | PF_ARI(r - n /* !!! long */, r, n) | /* Z80_CF */ (s > r) | Z80_NF);
        break;
    case 3:  /* SBC */
        r = LIB_UZX_DATA_HIGH(Z80_AF), s = (r - n - (Z80_AF & Z80_CF)) & LIB_UZX_DATA_MASK8;
        Z80_AF = LIB_UZX_DATA_PAIR(s, (s & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(s) | HF_ARI(s, r, n) | PF_ARI(r - n - (Z80_AF & Z80_CF) /* !!! long */, r, n) | /* Z80_CF */ (s > r || ((Z80_AF & Z80_CF) && n == 0xff)) | Z80_NF);
        break;
    case 4:  /* AND */
        s = LIB_UZX_DATA_HIGH(Z80_AF) & n;
        Z80_AF = LIB_UZX_DATA_PAIR(s, (s & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(s) | PF_LOG(s) | Z80_HF);
        break;
    case 5:  /* XOR */
        s = LIB_UZX_DATA_HIGH(Z80_AF) ^ n;
        Z80_AF = LIB_UZX_DATA_PAIR(s, (s & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(s) | PF_LOG(s));
        break;
    case 6:  /* OR */
        s = LIB_UZX_DATA_HIGH(Z80_AF) | n;
        Z80_AF = LIB_UZX_DATA_PAIR(s, (s & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(s) | PF_LOG(s));
        break;
    case 7:  /* CP */
        r = LIB_UZX_DATA_HIGH(Z80_AF), s = (r - n) & LIB_UZX_DATA_MASK8;
        Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (s & Z80_SF) | ZF_ARI(s) | (n & (Z80_YF | Z80_XF)) | HF_ARI(s, r, n) | PF_ARI(r - n /* !!! long */, r, n) | /* Z80_CF */ (s > r) | Z80_NF;
        break;
    default:
        assert(0);
    }
}

static unsigned do_rot(unsigned rot, unsigned n)
{
    unsigned r;

    assert(rot < 8 && n < 0x100);

    r = 0;
    switch(rot)  {
    case 0:  /* RLC */
        return (r = ((n << 1) | (n >> 7)) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF | Z80_CF)) | ZF_ARI(r) | PF_LOG(r), r);
    case 1:  /* RRC */
        return (r = ((n >> 1) | (n << 7)) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r) | (n & Z80_CF), r);
    case 2:  /* RL */
        return (r = ((n << 1) | (Z80_AF & Z80_CF)) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r) | /* Z80_CF */ ((n & 0x80) >> 7), r);
    case 3:  /* RR */
        return (r = ((n >> 1) | (Z80_AF << 7)) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r) | (n & Z80_CF), r);
    case 4:  /* SLA */
        return (r = (n << 1) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r) | /* Z80_CF */ ((n & 0x80) >> 7), r);
    case 5:  /* SRA */
        return (r = (n >> 1) | (n & 0x80),
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r) | (n & Z80_CF), r);
    case 6:  /* SLL */
        return (r = ((n << 1) | 1) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r) | /* Z80_CF */ ((n & 0x80) >> 7), r);
    }
    /* SRL */
    return (r = (n >> 1),
        Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r) | (n & Z80_CF), r);
}

static unsigned do_cb(unsigned pi)
{
    unsigned c, y, z, r, s, d;
    unsigned t1, t3;

    if(pi == 2)  {
        d = 0;
        c = FETCH();
    } else  {
        d = z80_MEMPTR = disp(z80_regs[pi], IMM8);
        c = READ(Z80_PC), CONT2(Z80_PC), Z80_PC = INC16(Z80_PC);
    }

    last_fetched_opcode = ((last_fetched_opcode << 8) | c);

    y = YPART(c);
    z = ZPART(c);

    switch(c & XMASK)  {
    case 0000:
        /*  rot[y] r[z]:
            rot r                f(4)      f(4)
            rot (HL)             f(4)      f(4) r(4) w(3)
            rot (i+d)       f(4) f(4) r(3) f(5) r(4) w(3) */
        if(pi == 2) return (put_r(z, 2, 0, do_rot(y, get_r(z, 2, 0, 1))), Z80_PC);
        r = READ(d), CONT(d), r = do_rot(y, r), WRITE(d, r);
        if(z != 6) put_r(z, 2, 0, r);
        return Z80_PC;
    case 0100:
        /*  BIT y, r[z]:
            BIT b, r             f(4)      f(4)
            BIT b, (HL)          f(4)      f(4) r(4)
            BIT b, (i+d)    f(4) f(4) r(3) f(5) r(4) */
        if(pi == 2)  {
            r = get_r(z, 2, 0, 1);
            Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_CF)) | Z80_HF |
                ((s = r & (1u << y)) ? (s & Z80_SF) : (Z80_ZF | Z80_PF));
            if(z == 6) r = LIB_UZX_DATA_HIGH(z80_MEMPTR);
            Z80_AF |= (r & (Z80_XF | Z80_YF));
        } else  {
            r = READ(d), CONT(d);
            Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_CF)) | Z80_HF |
                ((s = r & (1u << y)) ? (s & Z80_SF) :
                (Z80_ZF | Z80_PF)) | (LIB_UZX_DATA_HIGH(d) & (Z80_XF | Z80_YF));
        }
        return Z80_PC;
    case 0200:
        /*  RES y, r[z]:
            RES b, r             f(4)      f(4)
            RES b, (HL)          f(4)      f(4) r(4) w(3)
            RES b, (i+d)    f(4) f(4) r(3) f(5) r(4) w(3) */
        if(pi == 2)  {
            put_r(z, pi, 0, get_r(z, pi, 0, 1) & ~(1u << y));
        } else  {
            r = (READ(d) & ~(1u << y)), CONT(d), WRITE(d, r);
            if(z != 6) put_r(z, 2, 0, r);
        }
        return Z80_PC;
    }
    /*  SET y, r[z]:
        SET b, r             f(4)      f(4)
        SET b, (HL)          f(4)      f(4) r(4) w(3)
        SET b, (i+d)    f(4) f(4) r(3) f(5) r(4) w(3) */
    if(pi == 2)  {
        put_r(z, pi, 0, get_r(z, pi, 0, 1) | (1u << y));
    } else  {
        r = (READ(d) | (1u << y)), CONT(d), WRITE(d, r);
        if(z != 6) put_r(z, 2, 0, r);
    }
    return Z80_PC;
}

static unsigned do_ed(void)
{
    unsigned c, y, p, r, s, w;
    unsigned t1, t2, t3, of, cf, pf;

    c = FETCH();
    last_fetched_opcode = (last_fetched_opcode << 8) | c;

    y = YPART(c);
    p = PPART(c);

    r = 0;
    switch(c & (XMASK | ZMASK))  {
    case 0100:
        /*  IN r[y], (C)/IN (C): f(4) f(4) i(4) */
        r = INPUT(Z80_BC), z80_MEMPTR = INC16(Z80_BC);
        Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_CF)) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r);
        if(y != 6) put_r(y, 2, 0, r);
        return Z80_PC;
    case 0101:  /* OUT (C), r[y]/OUT (C), 0: f(4) f(4) o(4) */
        r = (y == 6) ? 0 : get_r(y, 2, 0, 0), OUTPUT(Z80_BC, r);
        return (z80_MEMPTR = INC16(Z80_BC), Z80_PC);
    case 0102:
        /* note: ignore P part here */
        if(c & QMASK)  {
            /* ADC HL, rp[p]: f(4) f(4) e(4) e(3) */
            CONT4(Z80_IR), CONT3(Z80_IR);
            cf = (Z80_AF & Z80_CF), w = get_rp(p, 2), r = ADD16(w, cf);
            of = (cf && r == 0), s = ADD16(Z80_HL, r);
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | ((s >> 8) & (Z80_SF | Z80_YF | Z80_XF)) |
                ZF_ARI(s) | HF_ARI(s >> 8, Z80_HL >> 8, w >> 8) |
                (PF_ARI(((unsigned long) Z80_HL + r) >> 8, Z80_HL >> 8, w >> 8) ^ (of << PF_BIT)) |
                /* Z80_CF */ (s < Z80_HL || of);
            z80_MEMPTR = INC16(Z80_HL), Z80_HL = s;
        } else  {
            /* SBC HL, rp[p]: f(4) f(4) e(4) e(3) */
            CONT4(Z80_IR), CONT3(Z80_IR);
            cf = (Z80_AF & Z80_CF), w = get_rp(p, 2), r = ADD16(w, cf);
            of = (cf && r == 0), s = SUB16(Z80_HL, r);
            Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | ((s >> 8) & (Z80_SF | Z80_YF | Z80_XF)) |
                ZF_ARI(s) | HF_ARI(s >> 8, Z80_HL >> 8, w >> 8) |
                (PF_ARI(((unsigned long) Z80_HL - r) >> 8, Z80_HL >> 8, w >> 8) ^ (of << PF_BIT)) |
                /* Z80_CF */ (s > Z80_HL || of) | Z80_NF;
            z80_MEMPTR = INC16(Z80_HL), Z80_HL = s;
        }
        return Z80_PC;
    case 0103:
        /* note: ignore P part here */
        if(c & QMASK)  {
            /* LD rp[p], (nn): f(4) f(4) r(3) r(3) r(3) r(3) */
            r = IMM16, z80_MEMPTR = INC16(r), put_rp(p, 2, READ_PAIR(r));
        } else  {
            /* LD (nn), rp[p]: f(4) f(4) r(3) r(3) w(3) w(3) */
            r = IMM16, z80_MEMPTR = INC16(r), s = get_rp(p, 2), WRITE_PAIR(r, s);
        }
        return Z80_PC;
    case 0104:  /* NEG: f(4) f(4) */
        return (r = LIB_UZX_DATA_HIGH(Z80_AF), Z80_AF &= LIB_UZX_DATA_LOW_MASK, do_alu(2, r), Z80_PC);
    case 0105:  /* RETI/RETN: f(4) f(4) r(3) r(3) */
        if(y != 1) Z80_IFF1 = Z80_IFF2;
        return (z80_MEMPTR = Z80_PC = POP, Z80_PC);
    case 0106:  /* IM im[y]: f(4) f(4) */
        return (y &= 3, Z80_IM = y < 2 ? 0 : (y - 1) /* !!! correct? */, Z80_PC);
    case 0200:  /* LDI, LDD, LDIR, LDDR: f(4) f(4) r(3) w(5) + e(5) */
        if(y < 4) return Z80_PC;
        r = READ(Z80_HL), w = Z80_DE, WRITE(w, r), CONT2(w);
        Z80_BC = DEC16(Z80_BC), r += LIB_UZX_DATA_HIGH(Z80_AF);
        Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_SF | Z80_ZF | Z80_CF)) |
            ((r << 4) & Z80_YF) | (r & Z80_XF) | ((Z80_BC != 0) << PF_BIT);
        if(y & 1) Z80_HL = DEC16(Z80_HL), Z80_DE = DEC16(Z80_DE);
        else Z80_HL = INC16(Z80_HL), Z80_DE = INC16(Z80_DE);
        if((y & 2) && Z80_BC != 0) return (CONT5(w),
            Z80_PC = SUB16(Z80_PC, 2), z80_MEMPTR = INC16(Z80_PC));
        return Z80_PC;
    case 0201:  /* CPI, CPD, CPIR, CPDR: f(4) f(4) r(3) e(5) + e(5) */
        if(y < 4) return Z80_PC;
        s = Z80_AF, w = Z80_HL, do_alu(7, r = READ(w)), CONT5(w);
        r = LIB_UZX_DATA_HIGH(Z80_AF) - r - !!(Z80_AF & Z80_HF);
        Z80_BC = DEC16(Z80_BC);
        Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_SF | Z80_ZF | Z80_HF)) |
            ((r << 4) & Z80_YF) | (r & Z80_XF) | ((Z80_BC != 0) << PF_BIT) | Z80_NF | (s & Z80_CF);
        if(y & 1) Z80_HL = DEC16(Z80_HL), z80_MEMPTR = DEC16(z80_MEMPTR);
        else Z80_HL = INC16(Z80_HL), z80_MEMPTR = INC16(z80_MEMPTR);
        if((y & 2) && !(Z80_AF & Z80_ZF) && Z80_BC != 0)
            return (CONT5(w),  Z80_PC = SUB16(Z80_PC, 2), z80_MEMPTR = INC16(Z80_PC));
        return Z80_PC;
    case 0202:  /* INI, IND, INIR, INDR: f(4) f(5) i(4) w(3) + e(5) */
        if(y < 4) return Z80_PC;
        CONT(Z80_IR), r = INPUT(Z80_BC), w = Z80_HL, WRITE(w, r);
        if(y & 1) Z80_HL = DEC16(Z80_HL), z80_MEMPTR = DEC16(Z80_BC);
        else Z80_HL = INC16(Z80_HL), z80_MEMPTR = INC16(Z80_BC);
        Z80_BC = SUB16(Z80_BC, 0x0100), s = LIB_UZX_DATA_HIGH(Z80_BC);
        cf = LIB_UZX_DATA_LOW(z80_MEMPTR) + r, pf = ((cf & 7) ^ s);
        Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (s & (Z80_SF | Z80_YF | Z80_XF)) |
            ZF_ARI(s) | ((r & 0x80) >> (7 - NF_BIT)) | PF_LOG(pf) |
            ((cf < 0x100) ? 0 : (Z80_HF | Z80_CF));
        if((y & 2) && s != 0) return (CONT5(w), Z80_PC = SUB16(Z80_PC, 2));
        return Z80_PC;
    case 0203:  /* OUTI, OUTD, OTIR, OTDR: f(4) f(5) r(3) o(4) + e(5) */
        if(y < 4) return Z80_PC;
        CONT(Z80_IR), r = READ(Z80_HL), w = Z80_BC, OUTPUT(w, r);
        Z80_BC = SUB16(Z80_BC, 0x0100), s = LIB_UZX_DATA_HIGH(Z80_BC);
        if(y & 1) Z80_HL = DEC16(Z80_HL), z80_MEMPTR = DEC16(Z80_BC);
        else Z80_HL = INC16(Z80_HL), z80_MEMPTR = INC16(Z80_BC);
        cf = LIB_UZX_DATA_LOW(Z80_HL) + r, pf = ((cf & 7) ^ s);
        Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (s & (Z80_SF | Z80_YF | Z80_XF)) |
            ZF_ARI(s) | ((r & 0x80) >> (7 - NF_BIT)) | PF_LOG(pf) |
            ((cf < 0x100) ? 0 : (Z80_HF | Z80_CF));
        if((y & 2) && s != 0) return (CONT5(w), Z80_PC = SUB16(Z80_PC, 2));
        return Z80_PC;
    case 0204:
    case 0205:
    case 0206:
    case 0207:
        /* invalid instruction */
        return Z80_PC;
    }
    switch(c)  {
    case 0x47:  /* LD I, A: f(4) f(5) */
        /* !!! does I change before or after the contention? */
        return (CONT(Z80_IR), Z80_I = LIB_UZX_DATA_HIGH(Z80_AF),
            z80_I_change_handler(Z80_I, Z80_ticks), Z80_PC);
    case 0x4f:  /* LD R, A: f(4) f(5) */
        /* note: use incremented R value here */
        return (CONT(Z80_IR), Z80_R8 = Z80_R7 = LIB_UZX_DATA_HIGH(Z80_AF),
            z80_R_change_handler(Z80_R, Z80_ticks), Z80_PC);
    case 0x57:  /* LD A, I: f(4) f(5) */
        /* emulate INT sampling when the instruction is interrupted */
        CONT(Z80_IR);
        return (Z80_AF = LIB_UZX_DATA_PAIR(Z80_I, (Z80_AF & Z80_CF) |
            (Z80_I & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(Z80_I) | (r << PF_BIT)), Z80_PC);
    case 0x5f:  /* LD A, R: f(4) f(5) */
        /* emulate INT sampling when the instruction is interrupted */
        CONT(Z80_IR);
        /* note: assign to R after it is incremeneted */
        return (r = Z80_R,
            Z80_AF = LIB_UZX_DATA_PAIR(r, (Z80_AF & Z80_CF) |
                (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | (r << PF_BIT)), Z80_PC);
    case 0x67:  /* RRD: f(4) f(4) r(3) e(4) w(3) */
        return (r = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | READ(Z80_HL), z80_MEMPTR = INC16(Z80_HL),
            CONT4(Z80_HL), r = (r & 0xf000) | ((r & 0xf) << 8) | ((r & 0x0ff0) >> 4),
            WRITE(Z80_HL, r & LIB_UZX_DATA_MASK8), r >>= 8,
            Z80_AF = LIB_UZX_DATA_PAIR(r, (Z80_AF & Z80_CF) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r)),
            Z80_PC);
    case 0x6f:  /* RLD: f(4) f(4) r(3) e(4) w(3) */
        return (r = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | READ(Z80_HL), z80_MEMPTR = INC16(Z80_HL),
            CONT4(Z80_HL), r = (r & 0xf000) | ((r & 0xff) << 4) | ((r & 0x0f00) >> 8),
            WRITE(Z80_HL, r & LIB_UZX_DATA_MASK8), r >>= 8,
            Z80_AF = LIB_UZX_DATA_PAIR(r, (Z80_AF & Z80_CF) | (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | PF_LOG(r)),
            Z80_PC);
    case 0x77:  /* invalid instructions */
    case 0x7f:
        return Z80_PC;
    }
    /* handle invalid instructions */
    switch(c & XMASK)  {
    case 0000:
    case 0300:
        return Z80_PC;
    }
    fprintf(stderr, "%04X %02X (0x%02x)\n", Z80_PC, c, c & (XMASK | ZMASK));  /* !!! */
    assert(0);
    return Z80_PC;
}

static unsigned z80_int(void)
{
    unsigned t2;

    Z80_IFF1 = Z80_IFF2 = 0;
    if(Z80_HALT)  {
        Z80_HALT = 0, Z80_PC = INC16(Z80_PC);
    }
    INC_R, Z80_ticks += 7;
    switch(Z80_IM)  {
    case 0:  /* !!! */
    case 1:
        /* ack(7) w(3) w(3) */
        PUSH(Z80_PC), z80_MEMPTR = Z80_PC = 0x0038;
        return 1;
    case 2:
        /* ack(7) w(3) w(3) r(3) r(3) */
        /* !!! Spectrum-specific */
        PUSH(Z80_PC),  z80_MEMPTR = Z80_PC = READ_PAIR(LIB_UZX_DATA_PAIR(Z80_I, 0xff));
        return 1;
    default:
        assert(0);  /* !!! */
    }
    return 0;
}

unsigned z80_step(unsigned long tc)
{
    unsigned c, y, z, p, r, s, d, pi;
    unsigned long t0 = Z80_ticks;
    unsigned t1, t2, t3, t4;

next:
    if((Z80_ticks - t0) <= tc)  {

    if(!Z80_INT_disabled && Z80_IFF1 && (Z80_ticks - 0) % ticks_per_frame < 32)  {  /* !!! constants */
        z80_int();
    }
    Z80_INT_disabled = 0;

    pi = z80_prefix;
    z80_prefix = 2;  /* HL */

    assert(Z80_PC < 0x10000);  /* !!! */
    assert(Z80_AF < 0x10000);  /* !!! */
    assert(Z80_HL < 0x10000);  /* !!! */
    assert(Z80_DE < 0x10000);  /* !!! */
    assert(Z80_BC < 0x10000);  /* !!! */
    assert(Z80_IX < 0x10000);  /* !!! */
    assert(Z80_IY < 0x10000);  /* !!! */
    assert(Z80_SP < 0x10000);  /* !!! */

    c = FETCH();

    y = YPART(c);
    z = ZPART(c);
    p = PPART(c);

    switch(c & XMASK)  {
    case 0100:
        /*  LD r[y], r[z] or HALT (replaces LD (HL), (HL)):
            LD r, r'             f(4)
            LD r, (HL)           f(4)           r(3)
            LD r, (i+d)     f(4) f(4) r(3) e(5) r(3)
            LD (HL), r           f(4)           w(3)
            LD (i+d), r     f(4) f(4) r(3) e(5) w(3)
            HALT                 f(4) */
        if(y == 6 && z == 6) Z80_HALT = 1, Z80_PC = DEC16(Z80_PC);
        else if(y == 6) d = DISP(0), put_r(y, pi, d, get_r(z, 2, d, 0));
        else if(z == 6) d = DISP(0), put_r(y, 2, d, get_r(z, pi, d, 0));
        else put_r(y, pi, 0, get_r(z, pi, 0, 0));
        goto next;
    case 0200:
        /*  alu[y] r[z]:
            alu r            f(4)
            alu (HL)         f(4)           r(3)
            alu (i+d)   f(4) f(4) r(3) e(5) r(3) */
        d = DISP(z != 6), do_alu(y, get_r(z, pi, d, 0));
        goto next;
    }
    switch(c & (XMASK | ZMASK))  {
    case 0004:
        /*  INC r[y]:
            INC r            f(4)
            INC (HL)         f(4)           r(4) w(3)
            INC (i+d)   f(4) f(4) r(3) e(5) r(4) w(3) */
        d = DISP(y != 6), put_r(y, pi, d, r = (get_r(y, pi, d, 1) + 1) & LIB_UZX_DATA_MASK8),
            Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_CF)) |
                (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | HF_INC(r) | PF_INC(r);
        goto next;
    case 0005:
        /*  DEC r[y]:
            DEC r            f(4)
            DEC (HL)         f(4)           r(4) w(3)
            DEC (i+d)   f(4) f(4) r(3) e(5) r(4) w(3) */
        d = DISP(y != 6), r = (get_r(y, pi, d, 1) - 1) & LIB_UZX_DATA_MASK8, put_r(y, pi, d, r);
        Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_CF)) |
            (r & (Z80_SF | Z80_YF | Z80_XF)) | ZF_ARI(r) | HF_DEC(r) | PF_DEC(r) | Z80_NF;
        goto next;
    case 0006:
        /*  LD r[y], n:
            LD r, n              f(4)      r(3)
            LD (HL), n           f(4)      r(3) w(3)
            LD (i+d), n     f(4) f(4) r(3) r(5) w(3) */
        d = IMM8;
        if(pi == 2 || y != 6) put_r(y, pi, 0, d);
        else r = READ(Z80_PC), CONT2(Z80_PC), Z80_PC = INC16(Z80_PC), put_r(y, pi, d, r);
        goto next;
    case 0300:  /* RET cc[y]: f(5) + r(3) r(3) */
        CONT(Z80_IR);
        if(CC(y)) z80_MEMPTR = Z80_PC = POP;
        goto next;
    case 0302:  /* JP cc[y], nn: f(4) r(3) r(3) */
        z80_MEMPTR = IMM16;
        if(CC(y)) Z80_PC = z80_MEMPTR;
        goto next;
    case 0304:  /* CALL cc[y], nn: f(4) r(3) r(3) / f(4) r(3) r(4) w(3) w(3) */
        z80_MEMPTR = IMM16;
        if(CC(y)) CONT(DEC16(Z80_PC)), PUSH(Z80_PC), Z80_PC = z80_MEMPTR;
        goto next;
    case 0306:  /* alu[y] n: f(4) r(3) */
        do_alu(y, IMM8);
        goto next;
    case 0307:  /* RST y*8: f(5) w(3) w(3) */
        CONT(Z80_IR), PUSH(Z80_PC), z80_MEMPTR = Z80_PC = y * 8;
        goto next;
    }
    switch(c & (XMASK | ZMASK | (YMASK - 030)))  {
    case 040:  /* JR cc[y-4], d: f(4) r(3) + e(5) */
        d = IMM8;
        if(CC((c & (YMASK - 040)) >> 3))
            CONT5(DEC16(Z80_PC)), z80_MEMPTR = Z80_PC = disp(Z80_PC, d);
        goto next;
    }
    switch(c & (XMASK | ZMASK | QMASK))  {
    case 0001:
        /*  LD rp[p], nn:
            LD rr, nn        f(4) r(3) r(3)
            LD i, nn    f(4) f(4) r(3) r(3) */
        put_rp(p, pi, IMM16);
        goto next;
    case 0003:
        /*  INC rp[p]:
            INC rr           f(6)
            INC i       f(4) f(6) */
        CONT2(Z80_IR), put_rp(p, pi, (get_rp(p, pi) + 1) & LIB_UZX_DATA_MASK16);
        goto next;
    case 0011:
        /*  ADD HL, rp[p]:
            ADD HL, rr           f(4) e(4) e(3)
            ADD i, rr       f(4) f(4) e(4) e(3) */
        CONT4(Z80_IR), CONT3(Z80_IR), r = get_rp(p, pi), s = (z80_regs[pi] + r) & LIB_UZX_DATA_MASK16,
            Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_SF | Z80_ZF | Z80_PF)) |
                ((s >> 8) & (Z80_YF | Z80_XF)) | HF_ARI(s >> 8, z80_regs[pi] >> 8, r >> 8) |
                /* Z80_CF */ (s < z80_regs[pi]);
        z80_MEMPTR = INC16(z80_regs[pi]), z80_regs[pi] = s;
        goto next;
    case 0013:
        /*  DEC rp[p]:
            DEC rr           f(6)
            DEC i       f(4) f(6) */
        CONT2(Z80_IR), put_rp(p, pi, (get_rp(p, pi) - 1) & LIB_UZX_DATA_MASK16);
        goto next;
    case 0301:
        /*  POP rp2[p]:
            POP rr           f(4) r(3) r(3)
            POP i       f(4) f(4) r(3) r(3) */
        if(p == 2) z80_regs[pi] = POP;
        else if(p == 3) Z80_AF = POP;
        else z80_regs[p] = POP;
        goto next;
    case 0305:
        /*  PUSH rp2[p]:
            PUSH rr          f(5) w(3) w(3)
            PUSH i      f(4) f(5) w(3) w(3) */
        CONT(Z80_IR);
        if(p == 2) PUSH(z80_regs[pi]);
        else if(p == 3) PUSH(Z80_AF);
        else PUSH(z80_regs[p]);
        goto next;
    }
    switch(c & (XMASK | ZMASK | QMASK | (PMASK - 1)))  {
    case 0002:  /* LD (BC/DE), A: f(4) w(3) */
        r = z80_regs[(c & PMASK) >> 4];
        WRITE(r, LIB_UZX_DATA_HIGH(Z80_AF));
        z80_MEMPTR = LIB_UZX_DATA_PAIR(LIB_UZX_DATA_HIGH(Z80_AF), (r + 1) & 0xff);
        goto next;
    case 0012:  /* LD A, (BC/DE): f(4) r(3) */
        r = z80_regs[(c & PMASK) >> 4];
        Z80_AF = (Z80_AF & LIB_UZX_DATA_LOW_MASK) | (READ(r) << 8);
        z80_MEMPTR = INC16(r);
        goto next;
    }
    switch(c)  {
    case 0x00:  /* NOP: f(4) */
        /* do nothing */
        goto next;
    case 0x07:  /* RLCA: f(4) */
        r = LIB_UZX_DATA_HIGH(Z80_AF), r = ((r << 1) | (r >> 7)) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & (Z80_SF | Z80_ZF | Z80_PF)) | (r << 8) |
            (r & (Z80_YF | Z80_XF)) | (r & Z80_CF);
        goto next;
    case 0x08:  /* EX AF, AF': f(4) */
        swap(&Z80_AF, &z80_alts[3], 1);
        goto next;
    case 0x0f:  /* RRCA: f(4) */
        r = LIB_UZX_DATA_HIGH(Z80_AF), r = ((r >> 1) | (r << 7)) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & (Z80_SF | Z80_ZF | Z80_PF)) | (r << 8) |
                (r & (Z80_YF | Z80_XF)) | /* Z80_CF */ (r >> 7);
        goto next;
    case 0x10:  /* DJNZ: f(5) r(3) + e(5) */
        CONT(Z80_IR), r = IMM8, Z80_BC = SUB16(Z80_BC, 0x0100);
        if(LIB_UZX_DATA_HIGH(Z80_BC) != 0)
            CONT5(DEC16(Z80_PC)), z80_MEMPTR = Z80_PC = disp(Z80_PC, r);
        goto next;
    case 0x17:  /* RLA: f(4) */
        r = ((LIB_UZX_DATA_HIGH(Z80_AF) << 1) | (Z80_AF & Z80_CF)) & LIB_UZX_DATA_MASK8,
            Z80_AF = (Z80_AF & (Z80_SF | Z80_ZF | Z80_PF)) | (r << 8) |
                (r & (Z80_YF | Z80_XF)) | /* Z80_CF */ (Z80_AF >> 15);
        goto next;
    case 0x18:  /* JR e: f(4) r(3) e(5) */
        d = IMM8, CONT5(DEC16(Z80_PC)), z80_MEMPTR = Z80_PC = disp(Z80_PC, d);
        goto next;
    case 0x1f:  /* RRA: f(4) */
        r = (LIB_UZX_DATA_HIGH(Z80_AF) >> 1) | ((Z80_AF & Z80_CF) << 7),
            Z80_AF = (Z80_AF & (Z80_SF | Z80_ZF | Z80_PF)) | (r << 8) |
            (r & (Z80_YF | Z80_XF)) | /* Z80_CF */ ((Z80_AF & 0x100) >> 8);
        goto next;
    case 0x22:
        /*  LD (nn), HL:
            LD (nn), HL          f(4) r(3) r(3) w(3) w(3)
            LD (nn), i      f(4) f(4) r(3) r(3) w(3) w(3) */
        r = IMM16, z80_MEMPTR = INC16(r), WRITE_PAIR(r, z80_regs[pi]);
        goto next;
    case 0x27:  /* DAA: f(4) */
        d = 0x0000;
        if((Z80_AF & Z80_CF) || Z80_AF >= 0x9a00) d |= 0x6000, Z80_AF |= Z80_CF;
    	if((Z80_AF & Z80_HF) || ((Z80_AF & 0x0f00) >= 0x0a00)) d |= 0x0600;

        if(!(Z80_AF & Z80_NF)) Z80_AF = ((Z80_AF + d) & 0xff00) |
            (((Z80_AF & 0x0f00) >= 0x0a00) << HF_BIT) | (Z80_AF & Z80_CF);
        else Z80_AF = ((Z80_AF - d) & 0xff00) |
            (((Z80_AF & Z80_HF) && (Z80_AF & 0x0f00) <= 0x0500) << HF_BIT) | Z80_NF | (Z80_AF & Z80_CF);

        r = (Z80_AF >> 8);
        Z80_AF |= PF_LOG(r) | (!r << ZF_BIT) | (r & (Z80_SF | Z80_XF | Z80_YF));
        goto next;
    case 0x2a:
        /*  LD HL, (nn):
            LD HL, (nn)          f(4) r(3) r(3) r(3) r(3)
            LD i, (nn)      f(4) f(4) r(3) r(3) r(3) r(3) */
        r = IMM16, z80_MEMPTR = INC16(r), z80_regs[pi] = READ_PAIR(r);
        goto next;
    case 0x2f:  /* CPL: f(4) */
        r = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) ^ LIB_UZX_DATA_HIGH_MASK,
        Z80_AF = (Z80_AF & (Z80_SF | Z80_ZF | Z80_PF | Z80_CF)) |
            r | ((r >> 8) & (Z80_YF | Z80_XF)) | Z80_HF | Z80_NF;
        goto next;
    case 0x32:  /* LD (nn), A: f(4) r(3) r(3) w(3) */
        r = IMM16, z80_MEMPTR = LIB_UZX_DATA_PAIR(LIB_UZX_DATA_HIGH(Z80_AF), (r + 1) & 0xff);
        WRITE(r, LIB_UZX_DATA_HIGH(Z80_AF));
        goto next;
    case 0x37:  /* SCF: f(4) */
        Z80_AF = (Z80_AF & (LIB_UZX_DATA_HIGH_MASK | Z80_SF | Z80_ZF | Z80_PF)) |
            (LIB_UZX_DATA_HIGH(Z80_AF) & (Z80_YF | Z80_XF)) | Z80_CF;
        goto next;
    case 0x3a:  /* LD A, (nn): f(4) r(3) r(3) r(3) */
        r = IMM16, z80_MEMPTR = INC16(r), Z80_AF = LIB_UZX_DATA_PAIR(READ(r), LIB_UZX_DATA_LOW(Z80_AF));
        goto next;
    case 0x3f:  /* CCF: f(4) */
        Z80_AF = (Z80_AF & LIB_UZX_DATA_HIGH_MASK) | (Z80_AF & (Z80_SF | Z80_ZF | Z80_PF)) |
            (LIB_UZX_DATA_HIGH(Z80_AF) & (Z80_YF | Z80_XF)) | ((Z80_AF & Z80_CF) << HF_BIT) |
            /* Z80_CF */ !(Z80_AF & Z80_CF);
        goto next;
    case 0xc3:  /* JP nn: f(4) r(3) r(3) */
        z80_MEMPTR = IMM16, Z80_PC = z80_MEMPTR;
        goto next;
    case 0xc9:  /* RET: f(4) r(3) r(3) */
        z80_MEMPTR = Z80_PC = POP;
        goto next;
    case 0xcb:  /* CB prefix */
        do_cb(pi);
        goto next;
    case 0xcd:  /* CALL nn: f(4) r(3) r(4) w(3) w(3) */
        z80_MEMPTR = IMM16, CONT(DEC16(Z80_PC)), PUSH(Z80_PC), Z80_PC = z80_MEMPTR;
        goto next;
    case 0xd3:  /* OUT (n), A: f(4) r(3) o(4) */
        r = LIB_UZX_DATA_HIGH(Z80_AF), s = IMM8;
        z80_MEMPTR = LIB_UZX_DATA_PAIR(LIB_UZX_DATA_HIGH(Z80_AF), (s + 1) & 0xff);
        OUTPUT(LIB_UZX_DATA_PAIR(r, s), r);
        goto next;
    case 0xd9:  /* EXX: f(4) */
        swap(z80_regs, z80_alts, 3);
        goto next;
    case 0xdb:  /* IN A, (n): f(4) r(3) i(4) */
        r = LIB_UZX_DATA_PAIR(LIB_UZX_DATA_HIGH(Z80_AF), IMM8), z80_MEMPTR = INC16(r);
        Z80_AF = (Z80_AF & LIB_UZX_DATA_LOW_MASK) | (INPUT(r) << 8);
        goto next;
    case 0xdd:  /* DD prefix */
        z80_prefix = 3;  /* IX */
        Z80_INT_disabled = 1;
        goto next;
    case 0xe3:
        /*  EX (SP), HL:
            EX (SP), HL          f(4) r(3) r(4) w(3) w(5)
            EX (SP), i      f(4) f(4) r(3) r(4) w(3) w(5) */
        r = POP, CONT(DEC16(Z80_SP)), swap(&r, &z80_regs[pi], 1);
        z80_MEMPTR = z80_regs[pi], PUSH(r), CONT2(Z80_SP);
        goto next;
    case 0xe9:
        /*  JP HL:
            JP HL       f(4)
            JP i        f(4) f(4) */
        Z80_PC = z80_regs[pi];
        goto next;
    case 0xeb:  /* EX DE, HL: f(4) */
        swap(&Z80_DE, &Z80_HL, 1);
        goto next;
    case 0xed:  /* ED prefix */
        do_ed();
        goto next;
    case 0xf3:  /* DI: f(4) */
        Z80_IFF1 = Z80_IFF2 = 0;
        goto next;
    case 0xf9:
        /*  LD SP, HL:
            LD SP, HL        f(6)
            LD SP, i    f(4) f(6) */
        CONT2(Z80_IR), Z80_SP = z80_regs[pi];
        goto next;
    case 0xfb:  /* EI: f(4) */
        Z80_IFF1 = Z80_IFF2 = 1;
        Z80_INT_disabled = 1;
        goto next;
    case 0xfd:  /* FD prefix */
        z80_prefix = 4;  /* IY */
        Z80_INT_disabled = 1;
        goto next;
    default:
        fprintf(stderr, "%04X %02X\n", Z80_PC, c);  /* !!! */
        assert(0);
    }
    }

    return Z80_PC;
}
