sub error {
	my ($file, $line, $msg) = @_;
	die "$file, line $line: $msg\n";
}

sub syntax_error {
	error @_, "syntax_error";
}

sub map_spectrum_key {
	my ($key_name, $kakl, $in_file, $line) = @_;
	return 0xFF if $key_name eq '';	
	defined $kakl->{$key_name}
		or error $in_file, $line, "undefined Spectrum key - $key_name";
	return $kakl->{$key_name};
}

sub write_keymap {
	my ($label, @map) = @_;
	print OUT "$label:\n";
	for	my $e (sort { $a->{'scancode'} <=> $b->{'scancode'} } @map) {
		printf OUT "\t.db 0x%02X, 0x%02X",
			$e->{'scancode'},
			$e->{'em'} << 4 | $e->{'em_upper'} << 5 | $e->{'numpad'} << 6;
		printf OUT ", 0b%08b, 0b%08b", $e->{'kakl_upper1'}, $e->{'kakl_upper2'};
		printf OUT ", 0b%08b, 0b%08b", $e->{'kakl1'}, $e->{'kakl2'};
		print OUT "\t; $e->{'pc_keys'} -> $e->{'spec_keys'}";
		print OUT " / $e->{'spec_keys_upper'}" if $e->{'spec_keys_upper'};
		print OUT "\n";
	}
	print OUT "\t.db 0xFF, 0xFF\n\n";
#	print OUT "${label}_end:\n";
#	print OUT '.equ ', uc($label), "_SIZE = ${label}_end - $label\n\n";
}

if (@ARGV != 1 && @ARGV != 2) {
	print "Usage: perl makemap.pl spectrum-plus.txt [ spectrum-plus.inc ]\n";
	exit 1;
}

my $in_file = $ARGV[0];
my $map_name = $in_file;
$map_name =~ s/\.[^\.]*?$//;
my $out_file;
if (@ARGV > 1) {
	$out_file = $ARGV[1];
} else {
	$out_file = "$map_name.inc";
}

# load ZX-Spectrum key to KA/KL value mappings
open IN, "<ka-kl.txt" or die;
scalar <IN>;	# skip table header
my $line = 2;
my %kakl;
while (!eof IN) {
	$_ = <IN>;
	/^(.+)\t(\d+)\t(\d+)\n?$/
		or syntax_error "ka-kl.txt", $line;
	$kakl{$1} = (($2 - 8) << 5) | (~(1 << $3) & 0x1F);
	$line++;
}
close IN;

# load PC key to scancode mappings
open IN, "<scancodes.txt" or die;
scalar <IN>;
$line = 2;
my %scancode;
while (!eof IN) {
	$_ = <IN>;
	/^(.+)\t([0-9A-Fa-f]{2}(\s*,\s*[0-9A-Fa-f]{2})*)\n?$/
		or syntax_error "scancodes.txt", $line;
	$scancode{$1} = [map hex, split(/\s*,\s*/, $2)];
	$line++;
}
close IN;

# load PC key to ZX-Spectrum key mappings
open IN, "<$in_file" or die;
scalar <IN>;	#skip table header
$line = 2;
my @map_std;
my @map_E0;
while (!eof IN) {
	$_ = <IN>;
	if (/^(.+?)\t((EM,)?(.+?)(\+(.+?))?)(\t((EM,)?(.+?)(\+(.+?))?))?\n?$/) {
		my $pc_key = $1;
		defined($scancode{$pc_key})
			or error $in_file, $line, "undefined PC key $1\n";
		my @scancode = @{$scancode{$pc_key}};
		my $entry = {
			#'key_type' => 0,
			'em' => ($3 eq '' ? 0 : 1),
			'em_upper' => ($9 eq '' ? 0 : 1),
			'kakl1' => map_spectrum_key($4, \%kakl, $in_file, $line),
			'kakl2' => map_spectrum_key($6, \%kakl, $in_file, $line),
			'kakl_upper1' => map_spectrum_key($10, \%kakl, $in_file, $line),
			'kakl_upper2' => map_spectrum_key($12, \%kakl, $in_file, $line),
			'pc_keys' => $pc_key,
			'spec_keys' => $2,
			'spec_keys_upper' => $8
		};
		$entry->{'numpad'} = $pc_key =~ /^KP (\d|\.)/ ? 1 : 0;
		
		if (@scancode == 1) {
			$entry->{'scancode'} = $scancode[0];
			push @map_std, $entry;
		} elsif (@scancode == 2 && $scancode[0] == 0xE0) {
			$entry->{'scancode'} = $scancode[1];
			push @map_E0, $entry;
		} else {
			error $in_file, $line, 'scancode ('.join(', ', @scancode).') is not supported';
		}
	}
	$line++;
}
close IN;

open OUT, ">$out_file" or die;
write_keymap "${map_name}_std", @map_std;
write_keymap "${map_name}_E0", @map_E0;
close OUT;
