; Spectrum Keyboard Controller (SKC), rev 1a
; (c) m.d., 2009-2010

.include "m88def.inc"

; ALT_PINOUT = 0 - KA sequence 8-14-12-11-10-13-9-15
; ALT_PINOUT = 1 - KA sequence 11-10-9-12-13-8-14-15 (Pentagon 1024SL v1.xx)
.equ ALT_PINOUT = 0

; delay (ms) in 'Extended mode' emulation
.equ AUTOTYPE_DELAY = 40
; delay (ms) in key combo emulation
.equ COMBO_DELAY	= 20

; enable watchdog timer
.equ USE_WATCHDOG = 1

; controller CPU frequency in Hz
.equ F_CPU = 20000000

.equ nEWAIT = PC5
.equ nPFE	= PD2

.equ KCLK	= PB3
.equ KDATA	= PB4

.equ nRSTBTN = PB1
.equ nEMAG	 = PB2

; row registers
.def row8 = r0
.def row9 = r1
.def row10 = r2
.def row11 = r3
.def row12 = r4
.def row13 = r5
.def row14 = r6
.def row15 = r7

; this one is always zero
.def _0	= r9

; registers used in INT0 handler
.def ssreg = r8
.def kl = r17

; registers used in main program
.def tmp = r16
.def i = r18
.def kbd_data = r19
.def parity = r21
.def state_l = r22
.def state_h = r23

.def kakl2 = r10
.def kakl1 = r24

.def flags = r20
.equ F_SHIFT_DOWN	= 0
.equ F_RSHIFT		= 1
.equ F_NUMLOCK_DOWN	= 2
.equ F_NUMLOCK		= 3
.equ F_SCROLL_DOWN	= 4
.equ F_SCROLL		= 5
.equ F_EXT_MODE		= 6

.def keymap_std_l	= r12
.def keymap_std_h	= r13
.def keymap_E0_l	= r14
.def keymap_E0_h	= r15

.cseg

; toggle scancode parser state
.macro set_state
	ldi state_h,high(state_@0)
	ldi state_l,low(state_@0)
.endm

.macro delay_us
	ldi xh,high(@0)
	ldi xl,low(@0)
	rcall delay_20c
.endm

.macro delay_ms
	ldi xh,high(@0 * F_CPU / (1024 * 1000))
	ldi xl,low(@0 * F_CPU / (1024 * 1000))
	rcall delay_1024c
.endm

.macro wait_clk_low
wait:
	sbic PINB,KCLK
	rjmp wait
.endm

.macro wait_clk_high
wait:
	sbis PINB,KCLK
	rjmp wait
.endm

.macro set_lpm_addr
	ldi zh,high(@0 << 1)
	ldi zl,low(@0 << 1)
.endm

.macro select
	set_lpm_addr @0
select:
	lpm tmp,Z
	tst tmp
	breq default
	cp tmp,@1
	breq jump
	adiw zh:zl,4
	rjmp select
jump:
	adiw zh:zl,2
	lpm xl,Z+
	lpm xh,Z
	movw zh:zl,xh:xl
	ijmp
default:
.endm

.macro use_keymap_std
	movw zh:zl,keymap_std_h:keymap_std_l
.endm

.macro use_keymap_E0
	movw zh:zl,keymap_E0_h:keymap_E0_l
.endm

.macro set_keymap_addr
	ldi tmp,high(@1 << 1)
	mov @0_h,tmp
	ldi tmp,low(@1 << 1)
	mov @0_l,tmp
.endm

.macro wdt_reset
.if USE_WATCHDOG
	wdr
.endif
.endm

.org 0
	rjmp reset

;-------------------------------------------------------------------------------
; INT0 handler is activated when Z80 performs port 0xFE read

.org INT0addr
.if ALT_PINOUT
	in ssreg,SREG
	sbis PIND,PD0
	and kl,row11
	sbis PIND,PD1
	and kl,row10
	sbis PIND,PD3
	and kl,row9
	sbis PIND,PD4
	and kl,row12
	sbis PIND,PD5
	and kl,row13
	sbis PIND,PD6
	and kl,row8
	sbis PIND,PD7
	and kl,row14
	sbis PINB,PB0
	and kl,row15
.else
	in ssreg,SREG
	sbis PIND,PD0
	and kl,row8
	sbis PIND,PD1
	and kl,row14
	sbis PIND,PD3
	and kl,row12
	sbis PIND,PD4
	and kl,row11
	sbis PIND,PD5
	and kl,row10
	sbis PIND,PD6
	and kl,row13
	sbis PIND,PD7
	and kl,row9
	sbis PINB,PB0
	and kl,row15
.endif
; output KL value and release /WAIT
	out PORTC,kl
; wait until Z80 complete I/O
wait_io:
	sbis PIND,nPFE
	rjmp wait_io
	ldi tmp,~(1<<nEWAIT)
	out PORTC,tmp
	ser kl
	out SREG,ssreg
	reti

; resend request routine for `kbd_read'
read_error:
	ldi kbd_data,0xFE
	rcall kbd_write
	brne exit_read

;-------------------------------------------------------------------------------
; read a byte from PC keyboard
;
; out: ZF = 1 on success, kbd_data = read byte; ZF = 0 on failure

kbd_read:
	clr parity
	cbi DDRB,KCLK
wait_start:
	wdt_reset
	sbic PINB,KDATA
	rjmp wait_start
	wait_clk_low
	wait_clk_high
	ldi i,8
read_bit:
	wait_clk_low
	clc
	sbic PINB,KDATA
	sec
	ror kbd_data
	eor parity,kbd_data
	wait_clk_high
	dec i
	brne read_bit
	wait_clk_low
	sbic PINB,KDATA
	com parity
	wait_clk_high
	wait_clk_low
	clz
	sbis PINB,KDATA
	rjmp exit_read
	wait_clk_high
	sbi DDRB,KCLK
	sbrs parity,7
	rjmp read_error
	sez
exit_read:
	ret

;-------------------------------------------------------
; send a byte to keyboard
;
; in: kbd_data = byte to send
; out: ZF = 1 - success, ZF = 0 - failure

kbd_write:
	clr parity
	delay_us 100
	sbi DDRB,KDATA
	cbi DDRB,KCLK
	wait_clk_low
	wait_clk_high
	ldi i,8
write_bit:
	wait_clk_low
	sbrs kbd_data,0
	sbi DDRB,KDATA
	sbrc kbd_data,0
	cbi DDRB,KDATA
	eor parity,kbd_data
	lsr kbd_data
	wait_clk_high
	dec i
	brne write_bit
	wait_clk_low
	sbrs parity,0
	cbi DDRB,KDATA
	sbrc parity,0
	sbi DDRB,KDATA
	wait_clk_high
	wait_clk_low
	cbi DDRB,KDATA
	wait_clk_high
	wait_clk_low
	sez
	sbic PINB,KDATA
	clz
	wait_clk_high
	ret

;-------------------------------------------------------------------------------
; send a command to keyboard
;
; in: tmp = command
; out: ZF = 1 - success, ZF = 0 - failure

kbd_write_cmd:
	mov kbd_data,tmp
	rcall kbd_write
	brne exit_write_cmd
	rcall kbd_read
	brne exit_write_cmd
; check for RESEND reply
	cpi kbd_data,0xFE
	breq kbd_write_cmd
; check for ACK reply
	cpi kbd_data,0xFA
exit_write_cmd:
	ret

;-------------------------------------------------------------------------------
; convert scancode to 2 KA/KL value (Spectrum key)
;
; in: kbd_data = scancode
; out: kakl1, kakl2 = Spectrum key combo

map_key:
	lpm tmp,z+
	cpi tmp,0xFF
	breq no_entry
	cp tmp,kbd_data
	breq parse_entry
	adiw zh:zl,5
	rjmp map_key
no_entry:
	ser kakl1
	mov kakl2,kakl1
	ret
parse_entry:
; check entry flags
	lpm tmp,z+
	sbrc tmp,MAP_NUMPAD
	rjmp map_numpad_key
; check if uppercase mapping is present
	lpm kakl1,z
	cpi kakl1,0xFF
	breq skip_upper
; check if upper case is needed
	sbrs flags,F_SHIFT_DOWN
	rjmp fetch_lower

	sbrc tmp,MAP_EM_UPPER
	sbr flags,1<<F_EXT_MODE

; fetch second key of uppercase combo
	adiw zh:zl,1
	lpm kakl2,z
; suspend SHIFT mapping
	push kakl1
	push kakl2
	ldi kbd_data,0x12
	sbrc flags,F_RSHIFT
	ldi kbd_data,0x59
	use_keymap_std
	rcall map_key
	rcall spec_release
	pop kakl2
	pop kakl1
	ret

; process numpad numeric keys
map_numpad_key:
; NUMLOCK inverts case for numpad keys
	sbrc flags,F_NUMLOCK
	rjmp skip_upper
; fetch uppercase mapping for numpad key
	lpm kakl1,z+
	lpm kakl2,z
	ret

; process lowercase mapping
skip_upper:
	sbrs flags,F_SHIFT_DOWN
	rjmp fetch_lower
; resume SHIFT mapping
	push tmp
	push zh
	push zl
	cbr flags,1<<F_SHIFT_DOWN
	ldi kbd_data,0x12
	sbrc flags,F_RSHIFT
	ldi kbd_data,0x59
	use_keymap_std
	rcall map_key
	rcall spec_press
	sbr flags,1<<F_SHIFT_DOWN
	pop zl
	pop zh
	pop tmp
fetch_lower:
	sbrc tmp,MAP_EM_LOWER
	sbr flags,1<<F_EXT_MODE
	adiw zh:zl,2
	lpm kakl1,z+
	lpm kakl2,z
	ret

;-------------------------------------------------------------------------------
; apply Spectrum key combo press
;
; in: kakl1, kakl2 = Spectrum key combo

spec_press:
	sbrs flags,F_EXT_MODE
	rjmp extended
	cbr flags,1<<F_EXT_MODE
; hit EM key
	push kakl1
	push kakl2
	ldi kakl1,0b00011110
	mov kakl2,kakl1
	ldi kakl1,0b11111101
	rcall spec_press
	delay_ms AUTOTYPE_DELAY
	ldi kakl1,0b00011110
	mov kakl2,kakl1
	ldi kakl1,0b11111101
	rcall spec_release
	pop kakl2
	pop kakl1
extended:
	rcall press_single
	mov kakl1,kakl2
	breq press_single
	cpi kakl1,0xFF
	breq exit_press
; insert delay bitween key presses in combo
	delay_ms COMBO_DELAY
press_single:
	cpi kakl1,0xFF
	breq exit_press
	clr xh
	mov xl,kakl1
	swap xl
	lsr xl
	andi xl,0x07
	ori kakl1,0xE0
	ld tmp,x
	and tmp,kakl1
	st x,tmp
;attach controller on any keypress
	cbi PORTC,nEWAIT
	sbi EIMSK,INT0
	clz
exit_press:
	ret

;-------------------------------------------------------------------------------
; apply Spectrum key combo release
;
; in: kakl1, kakl2 = Spectrum key combo

spec_release:
	sbrs flags,F_EXT_MODE
	rjmp no_ext_release
	delay_ms AUTOTYPE_DELAY
	cbr flags,1<<F_EXT_MODE
no_ext_release:
; swap kakl1 and kakl2 - release key combo in reverse order
	mov tmp,kakl2
	mov kakl2,kakl1
	mov kakl1,tmp
	rcall release_single
	mov kakl1,kakl2
	breq release_single
	cpi kakl1,0xFF
	breq exit_release
; insert delay bitween key releases in combo
	delay_ms COMBO_DELAY
release_single:
	cpi kakl1,0xFF
	breq exit_release
	clr xh
	mov xl,kakl1
	swap xl
	lsr xl
	andi xl,0x07
	ori kakl1,0xE0
	com kakl1
	ld tmp,x
	or tmp,kakl1
	st x,tmp
; check if any Spectrum keys are being pressed
	ser tmp
	and tmp,row8
	and tmp,row9
	and tmp,row10
	and tmp,row11
	and tmp,row12
	and tmp,row13
	and tmp,row14
	and tmp,row15
	cpi tmp,0xFF
	brne exit_release
;detach controller if there are no pressed keys
	cbi EIMSK,INT0
	sbi PORTC,nEWAIT
	clz
exit_release:
	ret

; (X * 20) cycles delay
delay_20c:
	rcall delay_8c	; 8 c
	rcall delay_8c	; 8 c 
	sbiw xh:xl,1	; 2 c
	brne delay_20c	; 2 c / 1 c
delay_8c:
	nop				; 1 c
	ret

; (X * 1024) cycles delay
delay_1024c:
	sts OCR1AH,xh
	sts OCR1AL,xl
	sts TCNT1H,_0
	sts TCNT1L,_0
	ldi tmp,1<<OCF1A
	out TIFR1,tmp
wait:
	wdt_reset
	sbis TIFR1,OCF1A
	rjmp wait
	ret

update_leds:
	ldi tmp,0xED
	rcall kbd_write_cmd
	brne exit_update_leds
	clr tmp
	sbrc flags,F_NUMLOCK
	sbr tmp,2
	sbrc flags,F_SCROLL
	sbr tmp,1
	rcall kbd_write_cmd
exit_update_leds:
	ret

;-------------------------------------------------------------------------------
; device reset handler

reset:
.if USE_WATCHDOG
	in flags,MCUSR
	clr _0
	out MCUSR,_0
	wdr
	ldi tmp,1<<WDE
	sts WDTCSR,tmp
	andi flags,0xF
halt:
	breq halt
.endif

	ldi tmp,0x3F
	out PORTC,tmp
	out DDRC,tmp

	ldi tmp,high(RAMEND)
	out SPH,tmp
	ldi tmp,low(RAMEND)
	out SPL,tmp

	ldi tmp,1<<SE
	out SMCR,tmp

	ldi tmp,~(1<<PD2)
	out PORTD,tmp
	ldi tmp,1<<PB0|1<<nEMAG
	out PORTB,tmp
	sbi DDRB,nEMAG

	ser kl
	mov row8,kl
	mov row9,kl
	mov row10,kl
	mov row11,kl
	mov row12,kl
	mov row13,kl
	mov row14,kl
	mov row15,kl

	ldi tmp,1<<WGM12|1<<CS12|1<<CS10
	sts TCCR1B,tmp

	ldi flags,1<<F_NUMLOCK
	set_keymap_addr keymap_std,unreal_std
	set_keymap_addr keymap_E0,unreal_E0

	delay_ms 250

;inhibit data transmission from keyboard
	sbi DDRB,KCLK

; reset keyboard
	ldi tmp,0xFF
	rcall kbd_write_cmd
	brne init_failure
	rcall kbd_read
	brne init_failure
	cpi kbd_data,0xAA
	brne init_failure

; scancode set #2
	ldi tmp,0xF0
	rcall kbd_write_cmd
	brne init_failure
	ldi tmp,2
	rcall kbd_write_cmd
	brne init_failure

; run LED blinking sequence
	ldi tmp,0xED
	rcall kbd_write_cmd
	brne init_failure
	ldi tmp,1
	rcall kbd_write_cmd
	brne init_failure
	delay_ms 100
	ldi tmp,0xED
	rcall kbd_write_cmd
	brne init_failure
	ldi tmp,4
	rcall kbd_write_cmd
	brne init_failure
	delay_ms 100

; set final LEDs state
	rcall update_leds
	brne init_failure

	set_state idle
	sei

;-------------------------------------------------------------------------------
; scancode parser state machine

select_state:
	rcall kbd_read
	movw zh:zl,state_h:state_l
	ijmp

init_failure:
	rjmp init_failure

;-------------------------------------------------------------------------------
; initial state

press_jumps:
	.dw 0xF0, set_release
	.dw 0xE0, set_ext_code
	.dw 0x12, press_lshift
	.dw 0x59, press_rshift
	.dw 0x77, press_numlock
	.dw 0x07, press_reset
	.dw 0x78, press_magic
	.dw 0x7E, press_scroll
	.dw 0

state_idle:
	select press_jumps,kbd_data

press_std:
	use_keymap_std
	rcall map_key
	rcall spec_press
	rjmp select_state

set_release:
	set_state release
	rjmp select_state
set_ext_code:
	set_state ext_code
	rjmp select_state

press_numlock:
	sbrc flags,F_NUMLOCK_DOWN
	rjmp select_state
	sbr flags,1<<F_NUMLOCK_DOWN
	ldi tmp,1<<F_NUMLOCK
	eor flags,tmp
	rcall update_leds
	rjmp select_state

press_lshift:
	cbr flags,1<<F_RSHIFT
	sbr flags,1<<F_SHIFT_DOWN
	rjmp press_std

press_rshift:
	sbr flags,1<<F_SHIFT_DOWN|1<<F_RSHIFT
	rjmp press_std

press_reset:
	sbi DDRB,nRSTBTN
	rjmp select_state

press_magic:
	cbi PORTB,nEMAG
	rjmp select_state

press_scroll:
	sbrc flags,F_SCROLL_DOWN
	rjmp select_state
	sbr flags,1<<F_SCROLL_DOWN
	ldi tmp,1<<F_SCROLL
	eor flags,tmp
	rcall update_leds
	sbrc flags,F_SCROLL
	rjmp select_advanced

; select UnrealSpeccy keymap
	set_keymap_addr keymap_std,unreal_std
	set_keymap_addr keymap_E0,unreal_E0
	rjmp select_state

; select 'advanced' keymap
select_advanced:
	set_keymap_addr keymap_std,advanced_std
	set_keymap_addr keymap_E0,advanced_E0	
	rjmp select_state	

set_idle:
	set_state idle
	rjmp select_state

;-------------------------------------------------------------------------------
; process standard key release

release_jumps:
	.dw 0x12, release_shift
	.dw 0x59, release_shift
	.dw 0x77, release_numlock
	.dw 0x07, release_reset
	.dw 0x78, release_magic
	.dw 0x7E, release_scroll
	.dw 0

state_release:
	select release_jumps,kbd_data

release_std:
	use_keymap_std
	rcall map_key
	rcall spec_release
	rjmp set_idle

release_shift:
	cbr flags,1<<F_SHIFT_DOWN|1<<F_RSHIFT
	rjmp release_std

release_numlock:
	cbr flags,1<<F_NUMLOCK_DOWN
	rjmp set_idle

release_reset:
	cbi DDRB,nRSTBTN
	rjmp set_idle

release_magic:
	sbi PORTB,nEMAG
	rjmp set_idle

release_scroll:
	cbr flags,1<<F_SCROLL_DOWN
	rjmp set_idle

;-------------------------------------------------------------------------------
; precess extended key press

state_ext_code:
	cpi kbd_data,0xF0
	breq set_ext_release
	use_keymap_E0
	rcall map_key
	rcall spec_press
	rjmp set_idle

set_ext_release:
	set_state ext_release
	rjmp select_state

;-------------------------------------------------------------------------------
; precess extended key release

state_ext_release:
	use_keymap_E0
	rcall map_key
	rcall spec_release
	rjmp set_idle

;-------------------------------------------------------------------------------
; PC to ZX-Spectrum key mappings

.equ MAP_EM_LOWER	= 4
.equ MAP_EM_UPPER	= 5
.equ MAP_NUMPAD		= 6

.include "advanced.inc"
.include "unreal.inc"
