/* vim: set syntax=c foldmethod=marker: */

#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <stdio.h>
#include <stdarg.h>

#include <errno.h>


// 訠   訡
#define HEAP_ERR 0xde00

#define HEAP_MAGIC 0xdea0
#define HEAP_BUSY 1

struct hblock {
	// ; heap block structure: SIZE = 6 !
	unsigned magic; //+0,+1  HEAP_MAGIC, bit0=busy flag
	unsigned size; //; +2,+3  full size of block (including this structure)
	char data[sizeof(int)]; //; +4...  data... FULLSIZE-6
			// NOTE: DIRTY HACK
	// ; FULLSIZE-2,FULLSIZE-1 0-fullsize of block
	// sizeof == 6
};

struct heap {
		 // heapdata structure: SIZE = 8
	unsigned magic;  // +0,+1  HEAP_MAGIC
	void *top;    // +2,+3  heap top address
	void *bottom; // +4,+5  heap bottom address
	struct hblock *cur;    // +6,+7  current heap pointer
	unsigned free;  // TODO.
	
};

#define HEAP_MINALLOC sizeof(struct hblock)

void fatal(char *f, int l);
#ifndef __FUNCTION__
#define __FUNCTION__ "f_unknown()"
#endif
#define FATAL() fatal(__FUNCTION__, __LINE__)

#define IS_HEAP_VALID(hptr) ((hptr)->magic == HEAP_MAGIC)

#define IS_BLOCK_VALID_BUSY(bptr) ((bptr)->magic == (HEAP_MAGIC|HEAP_BUSY))
#define IS_BLOCK_VALID_FREE(bptr) ((bptr)->magic == HEAP_MAGIC)
#define IS_BLOCK_VALID(bptr) (((bptr)->magic&~HEAP_BUSY) == HEAP_MAGIC)

#define IS_BLOCK_FREE(bptr) (((bptr)->magic&HEAP_BUSY) == 0)
#define IS_BLOCK_BUSY(bptr) (!IS_BLOCK_FREE(bptr))
#define BLOCK_SET_BUSY(bptr) ((bptr)->magic=(HEAP_MAGIC|HEAP_BUSY))
#define BLOCK_SET_FREE(bptr) ((bptr)->magic=HEAP_MAGIC)

#define BLOCK_DATA_SIZE(bptr) ((bptr)->size - sizeof(struct hblock))
#define BLOCK_SET_SIZE(bptr, top) ( (bptr)->size = ((char*)top-(char*)bptr) )
#define BLOCK_PREV_SIZE(bptr) (((int*)(bptr))[-1])
#define BLOCK_NEXT(bptr) ( (struct hblock*)((char*)(bptr) + (bptr)->size) )
#define BLOCK_PREV(bptr) \
	 ( (struct hblock*)((char*)(bptr) - BLOCK_PREV_SIZE(bptr)) )

#define BLOCK_FROM_DATA(addr) \
	( (struct hblock *)((char *)(addr) - (unsigned)(((struct hblock*)0)->data)) )
#define BLOCK_DATA(bptr) ( (void*)((bptr)->data) )
#define IS_BLOCK_IN_HEAP(hptr, bptr) \
	(((void*)(bptr) >= (hptr)->bottom) && ((char*)(bptr) <= ((char*)((hptr)->top) - sizeof(struct hblock))))


/*  xref:
	CUT_TAIL:
	MERGE_PREV:
	MERGE_NEXT:


*/	

#if 1
struct hblock *heap_search_free (struct heap *h, int size);
#else
struct hblock *heap_search_free (struct heap *h, int size)
{
struct hblock *b, *nextb;
b=h->cur;
do {
	struct hblock *n;
	if (!IS_BLOCK_VALID(b)) FATAL();

	n=BLOCK_NEXT(b);
	if ((void*)n >= h->top) {
		if ((void*)n != h->top) FATAL();
		nextb = h->bottom;
	} else {
		if (!IS_BLOCK_VALID(n)) FATAL();
		nextb = n;
	}
	
	if (IS_BLOCK_FREE(b) && (BLOCK_DATA_SIZE(b) >= size) ) {

		h->cur = nextb;
		BLOCK_SET_BUSY(b);

		/* cut tail if possible:
		b = *current block, size = needed size,
		n = *next block */
		//CUT_TAIL:
		if (BLOCK_DATA_SIZE(b) - size >= HEAP_MINALLOC) {
			struct hblock *t;
			BLOCK_SET_SIZE(b, (char*)b+size+sizeof(struct hblock));
			t=BLOCK_NEXT(b);
			BLOCK_PREV_SIZE(t)=b->size;
			if (((void*)n != h->top) && (IS_BLOCK_FREE(n))) {
				// this is not last (upper) block in heap
				// next free: merge tail with next
				struct hblock *nn;
				nn=BLOCK_NEXT(n); 
				if ((void*)nn >= h->top) {
					if ((void*)nn != h->top) FATAL();
				} else {
					if (!IS_BLOCK_VALID_BUSY(nn)) FATAL();
				}
				n->magic = 0;  // NOTE: n destroyed
				n = nn;		// top of tail
				h->cur = t;	// avoid joining current
			}

			BLOCK_SET_FREE(t);
			BLOCK_SET_SIZE(t, n);
			BLOCK_PREV_SIZE(n) = t->size;
		}

		return b;	
	}

	b=nextb;
	
} while (b != h->cur);

return NULL;
}
#endif

/* {{{   =============================================== */

void heap_init (struct heap *heap, void *bottom, void *top)
{
struct hblock *b;
unsigned size;
	if ((top <= bottom) 
		|| ( size=(char*)top-(char*)bottom, 
			size < sizeof(struct hblock) )
	) FATAL();

	heap->magic=HEAP_MAGIC;
	heap->bottom=bottom;
	heap->top=top;

	b=bottom;
	heap->cur=b;
	BLOCK_SET_FREE(b);
	BLOCK_SET_SIZE(b, top);
	BLOCK_PREV_SIZE(top) = b->size;
}

/* 樠  }}} */


/* {{{   ================================================= */

void *heap_alloc (struct heap *h, int size)
{
struct hblock *b;
	if (!IS_HEAP_VALID(h)) FATAL();
	if (size == 0) FATAL();
	b=heap_search_free(h, size);
	if (b==NULL) return NULL;
		else return BLOCK_DATA(b);
}

/* 뤥  }}} */

// TODO: check PREV,NEXT usage

/* {{{   ============================================== */

void heap_free (struct heap *h, void *addr)
{
struct hblock *b, *n;
	if (!IS_HEAP_VALID(h)) FATAL();
	
	b=BLOCK_FROM_DATA(addr);
	if (!IS_BLOCK_IN_HEAP(h, b)) FATAL();
	if (!IS_BLOCK_VALID_BUSY(b)) FATAL();
	
	n=BLOCK_NEXT(b);
	
	/* merge with previos if possible DON'T SET SIZE
	b = *current block
	*/
	//MERGE_PREV:
	if (b != h->bottom) {  // is lowest block?
		struct hblock *p;
		p=BLOCK_PREV(b);
		if (!IS_BLOCK_VALID(p)) FATAL();
		if (IS_BLOCK_FREE(p)) {
			// merge previous with current
			b->magic=0;  // NOTE: b destroyed
			b = p;  // ptr start of new free block
		}
	}

	/* merge with next free if possible
	b = *current block bottom, n = *next block
	*/
	//MERGE_NEXT:
	if ((void*)n >= h->top) {
		// this is last (upper) block in heap
		if ((void*)n != h->top) FATAL();
		
	} else {
		// this is not last (upper) block in heap
		if (!IS_BLOCK_VALID(n)) FATAL();
		
		if (IS_BLOCK_FREE(n)) {
			// merge with next
			struct hblock *nn;
			nn=BLOCK_NEXT(n); 
			if ((void*)nn >= h->top) {
				if ((void*)nn != h->top) FATAL();
			} else {
				if (!IS_BLOCK_VALID_BUSY(nn)) FATAL();
			}
			n->magic=0;  // NOTE: n destroyed

			n = nn;	// ptr to end of new free block 
		}
	}

	// here: b=bottom, n=top, all ptrs are valid
	BLOCK_SET_FREE(b);
	BLOCK_SET_SIZE(b, n);
	BLOCK_PREV_SIZE(n) = b->size;

	h->cur = b; // avoid joining of current...
}

/* ᢮  }}} */

/* {{{    ========================================= */

void *heap_realloc (struct heap *h, void *addr, int size)
{
struct hblock *b, *n;
if (!IS_HEAP_VALID(h)) FATAL();

b=BLOCK_FROM_DATA(addr);
if (!IS_BLOCK_IN_HEAP(h, b)) FATAL();
if (!IS_BLOCK_VALID_BUSY(b)) FATAL();

if (size == 0) FATAL();

if (BLOCK_DATA_SIZE(b) < size) {
// ---- increase block size -----
n=BLOCK_NEXT(b); 
if ((void*)n < h->top) {
	// this is NOT last (upper) block in heap
	if(!IS_BLOCK_VALID(n)) FATAL();
	// try merge with next
	if ( IS_BLOCK_FREE(n) &&
		(BLOCK_DATA_SIZE(b) + BLOCK_DATA_SIZE(n) >= size)
	) {
		// merge with next
		struct hblock *nn;
		n->magic = 0;		// NOTE: n destroyed
		nn=BLOCK_NEXT(n); 
		if ((void*)nn >= h->top) {
			if ((void*)nn != h->top) FATAL();
			h->cur = h->bottom; // avoid joining of current
		} else {
			if (!IS_BLOCK_VALID_BUSY(nn)) FATAL();
			h->cur = nn;
		}
		// create new block from tail if possible
		if (BLOCK_DATA_SIZE(b) - size >= HEAP_MINALLOC) {
			struct hblock *t;
			BLOCK_SET_SIZE(b, (char*)b+size+sizeof(struct hblock));
			t=BLOCK_NEXT(b);	// tail start
			BLOCK_PREV_SIZE(t)=b->size;
			// new block in tail
			BLOCK_SET_FREE(t);
			BLOCK_SET_SIZE(t, nn);
			BLOCK_PREV_SIZE(nn) = t->size;
			
		} else {
			// block(b) occupies whole next(n), no tail.
			BLOCK_SET_SIZE(b, nn);
			BLOCK_PREV_SIZE(nn) = b->size;
		}
		
		return BLOCK_DATA(b); 	// block merged with next

	} // can't merge with next -> re-allocate
	
} else {
	// this is last (upper) block in heap
	if ((void*)n != h->top) FATAL();
}

{ // reallocate block with size, and memcpy
	struct hblock *t;
	t=heap_search_free(h, size);
	if (t==NULL) return NULL;    // error FIXME?
	memcpy(BLOCK_DATA(t), BLOCK_DATA(b), BLOCK_DATA_SIZE(b));
	
	// ---- free old block -------
	// merge with previos if possible
	if (b != h->bottom) {  // is lowest block?
		struct hblock *p;
		p=BLOCK_PREV(b);
		if (!IS_BLOCK_VALID(p)) FATAL();
		if (IS_BLOCK_FREE(p)) {
			// merge previous with current
			b->magic=0;  // NOTE: b destroyed
			b = p;  // ptr start of new free block
		}
	}
	// merge with next if possible
	if ((void*)n < h->top) {
		// this is not last (upper) block in heap
		if (!IS_BLOCK_VALID(n)) FATAL();
		if (IS_BLOCK_FREE(n)) {
			// merge with next
			struct hblock *nn;
			nn=BLOCK_NEXT(n); 
			if ((void*)nn >= h->top) {
				if ((void*)nn != h->top) FATAL();
			} else {
				if (!IS_BLOCK_VALID_BUSY(nn)) FATAL();
			}
			n->magic=0;  // NOTE: n destroyed
			n = nn;	// ptr to end of new free block 
		}
	}
	// here: b=bottom, n=top, all ptrs are valid
	BLOCK_SET_FREE(b);
	BLOCK_SET_SIZE(b, n);
	BLOCK_PREV_SIZE(n) = b->size;
	h->cur = b; // avoid joining of current...
	// --- end of freeing block ---

	return BLOCK_DATA(t);
} 
	

} else {
// --- decrease block size or leave unchanged ---
if (BLOCK_DATA_SIZE(b) - size >= HEAP_MINALLOC) {
	struct hblock *t;
	n=BLOCK_NEXT(b);
	
	BLOCK_SET_SIZE(b, (char*)b+size+sizeof(struct hblock));
	t=BLOCK_NEXT(b);	// tail
	BLOCK_PREV_SIZE(t)=b->size;
	
	if ((void*)n >= h->top) {
		if ((void*)n != h->top) FATAL();

	} else {
		if (!IS_BLOCK_VALID(n)) FATAL();
		if (IS_BLOCK_FREE(n)) {
			// next free: merge tail with next
			struct hblock *nn;
			nn=BLOCK_NEXT(n); 
			if ((void*)nn >= h->top) {
				if ((void*)nn != h->top) FATAL();
			} else {
				if (!IS_BLOCK_VALID_BUSY(nn)) FATAL();
			}
			n->magic=0;		// NOTE: nn destroyed
			n = nn;		// new top of tail
			h->cur = t;  // avoid joining of current
		}
	}
	
	BLOCK_SET_FREE(t);
	BLOCK_SET_SIZE(t, n);
	BLOCK_PREV_SIZE(n) = t->size;

}
return BLOCK_DATA(b);

}
//return 0; // make compiler happy
}

/*  ࠧ  }}} */


/* {{{ Testing */

#define HEAPSIZE 32768
//#define TESTCOUNT 10000
#define POINTERS 1024
#define TESTMAX 4096


/* {{{ ᯥ⪠ ﭨ  */

void print_heap(struct heap *h)
{
struct hblock *b;
unsigned total = 0;
unsigned blocks = 0;
putchar('\n');
printf("HEAP MAGIC = 0x%x\n", h->magic);
printf("current = 0x%x\n", (unsigned)h->cur);
printf("bottom = 0x%x\n", (unsigned)h->bottom);
printf("top = 0x%x\n", (unsigned)h->top);
putchar('\n');
b=h->bottom;
do {
	printf("BLOCK 0x%x, MAGIC = 0x%x ", (unsigned)b, b->magic);
	if (b->magic==HEAP_MAGIC) printf("(free)");
	else if (b->magic==(HEAP_MAGIC|HEAP_BUSY)) printf("(busy)");
	else {
		puts("(ERROR!)");
		break;
	}
	printf(" size = %u (%u)\n", b->size, ((unsigned*)((char*)b+b->size))[-1]);
	if (b->size > HEAPSIZE) {
		printf("(HEAP INCONSISTENCY!)\n");
		break;
	}
	total += b->size;
	blocks++;
	b=(struct hblock*)((char*)b+(b->size));
} while ((void*)b < h->top);
if ((void*)b != h->top) {
	printf("ERROR: BLOCK (0x%x) != TOP (0x%x) !\n", 
		(unsigned)b, (unsigned)(h->top));
} else {
	printf("END OF HEAP. %u bytes, %u blocks. OK.\n", total, blocks);
}

}

/* }}} */


struct heap *Heap;

//void fatal(char *f, ...)
void fatal(char *f, int l)
{
//va_list args;
//	va_start(args, f);
//	vfprintf(stderr, f, args);
	fflush(stdout);
	fprintf(stderr, "\n--- FATAL ERROR IN %s (line %d) %s\n\n", f, l, "");
	print_heap(Heap);
	abort();
//	va_end(args);
}

int int_flag = 0;
nm
void int_sigint(int signo)
{
	int_flag=1;
}

#define FDEBUG 0

int main()
{
struct heap heap;
void *memory; 
void **pointers;
int x;

signal(SIGINT, int_sigint);

pointers=calloc(POINTERS, sizeof(void*));
if (pointers==NULL) {
	fprintf(stderr, "%s\n", strerror(errno));
}

memory=malloc(HEAPSIZE);
if (memory==NULL) {
	fprintf(stderr, "%s\n", strerror(errno));
	exit(1);
}

heap_init(&heap, memory, (char*)memory+HEAPSIZE);
Heap=&heap;
if (FDEBUG) print_heap(Heap);

for (x=0; ; x++) {
	
int i = rand() % POINTERS;
switch (rand() % 4) {

case 0:
	do {
		if (pointers[i]!=NULL) {
			printf("free[%d] 0x%x\n", i, (unsigned)(pointers[i]));
			if (*(unsigned char*)(pointers[i]) != ((unsigned)(pointers[i]) & 0xff)) {
				fprintf(stderr, "--- DATA CORRUPTION 0x%x, 0x%x <> 0x%x\n",
						(unsigned)(pointers[i]),
						*(unsigned*)(pointers[i]), 
						(unsigned)(pointers[i])
				);
				FATAL();
			}
			heap_free(&heap, pointers[i]);
			pointers[i]=NULL;
			break;
		}
	} while (++i < POINTERS);
	break;

case 1:
	do {
		if (pointers[i]==NULL) {
			int size;
			size=rand()%TESTMAX + 1;
			printf("alloc[%d] %d = ", i, size);
			pointers[i]=heap_alloc(&heap, size);
			if (pointers[i]==NULL) printf("NULL\n");
			else {
				printf("0x%x\n", (unsigned)(pointers[i]));
				memset(pointers[i], (unsigned)(pointers[i]) & 0xff, size);
			}
			break;
		}
	} while (++i < POINTERS);
	break;

case 2:
	do {
		if (pointers[i]!=NULL) {
			int newsize;
			void *p;
			newsize=rand()%TESTMAX + 1;
			printf("realloc[%d] %d (0x%x) = ", i, (unsigned)(pointers[i]), newsize);
			p=heap_realloc(&heap, pointers[i], newsize);
			if (p==NULL) printf("NULL\n");
			else {
				printf("0x%x\n", (unsigned)p);
				pointers[i]=p;
				memset(pointers[i], (unsigned)(pointers[i]) & 0xff, newsize);
			}
			break;
		}
	} while (++i < POINTERS);
	break;

}
	
if (FDEBUG) print_heap(Heap);
if (int_flag) break;
}

if (int_flag) fprintf(stderr, "\n--- ABORTED!\n");
print_heap(Heap);

free(memory);
free(pointers);
printf("ok\n");
return 0;
}

/* testing }}} */



