/*
   YM-2149F emulator for Unreal Speccy project
   created under public domain license by SMT, jan.2006
*/


#ifndef _SNDCHIP_H_INCLUDED
#define _SNDCHIP_H_INCLUDED

#include "sndrender.h"

const unsigned SNDR_DEFAULT_AY_RATE = 1774400; // original ZX-Spectrum soundchip clock fq

struct AYOUT;
struct SNDCHIP_VOLTAB;
struct SNDCHIP_PANTAB;

extern const char * const ay_chips[];

#ifdef __GNUC__
#pragma pack(push,1)
#else
#pragma pack(push)
#pragma pack(1) // envelope period (envT, R13-R14) has invalid word alignment
#endif
struct AYREGS
{
   unsigned short fA, fB, fC;
   unsigned char noise, mix;
   unsigned char vA, vB, vC;
   unsigned short envT;
   unsigned char env;
   unsigned char portA, portB;
};
#pragma pack(pop)

class SNDCHIP : public SNDRENDER
{

 public:

   enum CHIP_TYPE { CHIP_AY, CHIP_YM, CHIP_MAX };
   static const char *get_chipname(CHIP_TYPE i) { return ay_chips[i]; }

   void set_chip(CHIP_TYPE type) { chiptype = type; }
   void set_timings(unsigned system_clock_rate, unsigned chip_clock_rate, unsigned sample_rate);
   void set_volumes(unsigned global_vol, SNDCHIP_VOLTAB &voltab, SNDCHIP_PANTAB &stereo);

   void reset(unsigned timestamp = 0); // call with default parameter, when context outside start_frame/end_frame block

   // 'render' is a function that converts array of register writes into PCM-buffer
   unsigned render(AYOUT *src, unsigned srclen, unsigned clk_ticks, bufptr_t dst);

   // set of functions that fills buffer in emulation progress
   void start_frame(bufptr_t dst);
   void select(unsigned char nreg);
   void write(unsigned timestamp, unsigned char val);
   unsigned char read();
   unsigned end_frame(unsigned clk_ticks);

   SNDCHIP();

   // for monitoring, chip can't report this values
   unsigned char get_activereg() { return activereg; }
   unsigned char get_r13_reloaded() { return r13_reloaded; }
   unsigned char get_reg(unsigned nreg) { return reg[nreg]; }

 private:

   unsigned t, ta, tb, tc, tn, te, env, denv;
   unsigned bitA, bitB, bitC, bitN, ns;
   unsigned bit0, bit1, bit2, bit3, bit4, bit5;
   unsigned ea, eb, ec, va, vb, vc;
   unsigned fa, fb, fc, fn, fe;
   unsigned mult_const;

   unsigned char activereg, r13_reloaded;

   unsigned vols[6][32];
   CHIP_TYPE chiptype;

   union {
      unsigned char reg[16];
      struct AYREGS r;
   };

   __int64 passed_chip_ticks, passed_clk_ticks;
   void flush(unsigned chiptick);
};

struct AYOUT
{
   unsigned timestamp; // in 'external' ticks
   unsigned char reg_num;
   unsigned char reg_value;
   unsigned char res1, res2; // padding
};

// output volumes (#0000-#FFFF) for given envelope state or R8-R10 value
// AY chip has only 16 different volume values, so v[0]=v[1], v[2]=v[3], ...
struct SNDCHIP_VOLTAB
{
   unsigned v[32];
};

// generator's channel panning, % (0-100)
struct SNDCHIP_PANTAB
{
   unsigned raw[6];
   // structured as 'struct { unsigned left, right; } chan[3]';
};

// used as parameters to SNDCHIP::set_volumes(),
// if application don't want to override defaults
extern SNDCHIP_VOLTAB SNDR_VOL_AY;
extern SNDCHIP_VOLTAB SNDR_VOL_YM;
extern SNDCHIP_PANTAB SNDR_PAN_MONO;
extern SNDCHIP_PANTAB SNDR_PAN_ABC;
extern SNDCHIP_PANTAB SNDR_PAN_ACB;
extern SNDCHIP_PANTAB SNDR_PAN_BAC;

#endif // _SNDCHIP_H_INCLUDED
