#include <string.h>
#include "zemu.h"
#include "exceptions.h"
#include "bin.h"
#include "lib_wd1793/unreal_wd1793.h"
#include "dirwork.h"
#include "ftbos_font.h"
#include "font.h"
#include "dialog.h"

#define OS_LINUX
const unsigned SND_FQ = 44100; //48000;
#include "wavplay/wavplay.cpp"

SNDSAMPLE soundBuffer[SOUND_BUFFER_SIZE];
int soundBufferSize;
volatile int audioCnt;

Z80EX_CONTEXT *cpu;
unsigned long long clk, currClk;
s_Params params;
SDL_Surface *screen, *scrSurf[2];
int PITCH;
bool drawFrame;
int frames;
C_Config config;
C_Font font;
bool disableSound = false;
unsigned frameSamples; //[boo] sound frame size

int colors[0x10] = {
	DRGB(  0,   0,   0),
	DRGB(  0,   0, 128),
	DRGB(128,   0,   0),
	DRGB(128,   0, 128),
	DRGB(  0, 128,   0),
	DRGB(  0, 128, 128),
	DRGB(128, 128,   0),
	DRGB(128, 128, 128),
	DRGB(  0,   0,   0),
	DRGB(  0,   0, 255),
	DRGB(255,   0,   0),
	DRGB(255,   0, 255),
	DRGB(  0, 255,   0),
	DRGB(  0, 255, 255),
	DRGB(255, 255,   0),
	DRGB(255, 255, 255)
};

//--------------------------------------------------------------------------------------------------------------

bool (* hnd_z80read[MAX_HANDLERS])(Z80EX_WORD, bool, Z80EX_BYTE&);
bool (* hnd_z80write[MAX_HANDLERS])(Z80EX_WORD, Z80EX_BYTE);
bool (* hnd_z80input[MAX_HANDLERS])(Z80EX_WORD, Z80EX_BYTE&);
bool (* hnd_z80output[MAX_HANDLERS])(Z80EX_WORD, Z80EX_BYTE);
void (* hnd_frameStart[MAX_HANDLERS])(void);
void (* hnd_afterFrameRender[MAX_HANDLERS])(void);
int types_sdl[MAX_HANDLERS];
bool (* hnd_sdl[MAX_HANDLERS])(SDL_Event&);
void (* hnd_reset[MAX_HANDLERS])(void);

int cnt_z80read = 0;
int cnt_z80write = 0;
int cnt_z80input = 0;
int cnt_z80output = 0;
int cnt_frameStart = 0;
int cnt_afterFrameRender = 0;
int cnt_sdl = 0;
int cnt_reset = 0;

void AttachZ80ReadHandler(bool (* func)(Z80EX_WORD, bool, Z80EX_BYTE&))
{
	if (cnt_z80read >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80read[cnt_z80read++] = func;
}

void AttachZ80WriteHandler(bool (* func)(Z80EX_WORD, Z80EX_BYTE))
{
	if (cnt_z80write >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80write[cnt_z80write++] = func;
}

void AttachZ80InputHandler(bool (* func)(Z80EX_WORD, Z80EX_BYTE&))
{
	if (cnt_z80input >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80input[cnt_z80input++] = func;
}

void AttachZ80OutputHandler(bool (* func)(Z80EX_WORD, Z80EX_BYTE))
{
	if (cnt_z80output >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80output[cnt_z80output++] = func;
}

void AttachFrameStartHandler(void (* func)(void))
{
	if (cnt_frameStart >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_frameStart[cnt_frameStart++] = func;
}

void AttachAfterFrameRenderHandler(void (* func)(void))
{
	if (cnt_afterFrameRender >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_afterFrameRender[cnt_afterFrameRender++] = func;
}

void AttachSDLHandler(int eventType, bool (* func)(SDL_Event&))
{
	if (cnt_sdl >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	types_sdl[cnt_sdl] = eventType;
	hnd_sdl[cnt_sdl++] = func;
}

void AttachResetHandler(void (* func)(void))
{
	if (cnt_reset >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_reset[cnt_reset++] = func;
}

//--------------------------------------------------------------------------------------------------------------

#include "devices/device.h"
#include "devices/border/border.h"
#include "devices/keyboard/keyboard.h"
#include "devices/trdos/trdos.h"
#include "devices/mmanager/mmanager.h"
#include "devices/ay/ay.h"

C_Border dev_border;
C_Keyboard dev_keyboard;
C_TrDos dev_trdos;
C_MemoryManager dev_mman;
C_Ay dev_ay;

C_Device* devs[] =
{
	&dev_border,
	&dev_keyboard,
	&dev_trdos,
	&dev_mman,
	&dev_ay,
	NULL
};

#include "snap_z80.h"

//--------------------------------------------------------------------------------------------------------------

int messageTimeout = 0;
char message[0x100];

void OutputText(char *str)
{
	int x, y;

	x = (WIDTH - font.StrLenPx(str)) / 2;
	y = HEIGHT - font.Height() - 4;

	font.PrintString(x, y, str);
}

void ShowMessage(void)
{
	if (messageTimeout <= 0) return;
	messageTimeout--;
	OutputText(message);
}

void SetMessage(char *str)
{
	strcpy(message, str);
	messageTimeout = 50;
}

//--------------------------------------------------------------------------------------------------------------

void ResetSequence(void);

void TryNLoadFile(char *fname, int drive)
{
	if (!stricmp(C_DirWork::ExtractExt(fname), "z80")) {
		if (!load_z80_snap(fname, cpu, dev_mman, dev_border)) StrikeMessage("Error loading snapshot");
	} else {
		wd1793_load_dimage(fname, drive);
	}
}

void TryNLoadFile(char *fname)
{
	TryNLoadFile(fname, 0);
}

//--------------------------------------------------------------------------------------------------------------

void Action_Reset(void) {
	ResetSequence();
}

void Action_ResetTrDos(void)
{
	ResetSequence();
	dev_mman.OnOutputByte(0x7FFD, 0x10);
	dev_trdos.trdos = true;
}

void Action_MaxSpeed(void) {
	params.maxSpeed = !params.maxSpeed;
}

void Action_QuickLoad(void) {
	if (!load_z80_snap("snap.z80", cpu, dev_mman, dev_border)) StrikeMessage("Error loading snapshot");
}

void Action_QuickSave(void) {
	save_z80_snap("snap.z80", cpu, dev_mman, dev_border);
}

void Action_AntiFlicker(void)
{
	params.antiFlicker = !params.antiFlicker;
	if (params.antiFlicker) SetMessage("AntiFlicker ON"); else SetMessage("AntiFlicker OFF");
}

void Action_LoadFile(void) {
	FileDialog();
}

s_Action cfgActions[] =
{
	{"reset",		Action_Reset},
	{"reset_trdos",		Action_ResetTrDos},
	{"max_speed",		Action_MaxSpeed},
	{"quick_load",		Action_QuickLoad},
	{"quick_save",		Action_QuickSave},
	{"anti_flicker",	Action_AntiFlicker},
	{"load_file",		Action_LoadFile},
	{"",			NULL}
};

//--------------------------------------------------------------------------------------------------------------

Z80EX_BYTE ReadByte(Z80EX_CONTEXT *cpu, Z80EX_WORD addr, int m1_state, void *userData)
{
	int i;
	Z80EX_BYTE retval;

	for (i = 0; i < cnt_z80read; i++) {
		if (hnd_z80read[i](addr, m1_state, retval)) return retval;
	}

	return 0xFF;
}

void WriteByte(Z80EX_CONTEXT *cpu, Z80EX_WORD addr, Z80EX_BYTE value, void *userData)
{
	int i;
	for (i = 0; i < cnt_z80write; i++) {
		if (hnd_z80write[i](addr, value)) return;
	}
}

Z80EX_BYTE InputByte(Z80EX_CONTEXT *cpu, Z80EX_WORD port, void *userData)
{
	int i;
	Z80EX_BYTE retval;

	for (i = 0; i < cnt_z80input; i++) {
		if (hnd_z80input[i](port, retval)) return retval;
	}

	return 0xFF;
}

void OutputByte(Z80EX_CONTEXT *cpu, Z80EX_WORD port, Z80EX_BYTE value, void *userData)
{
	int i;
	for (i = 0; i < cnt_z80output; i++) {
		if (hnd_z80output[i](port, value)) return;
	}
}

Z80EX_BYTE ReadIntVec(Z80EX_CONTEXT *cpu, void *userData)
{
	return 0xFF;
}

//--------------------------------------------------------------------------------------------------------------

void InitSurfaces(void)
{
	SDL_PixelFormat *fmt = screen->format;
	scrSurf[0] = SDL_CreateRGBSurface(SDL_SWSURFACE, WIDTH, HEIGHT, fmt->BitsPerPixel, fmt->Rmask, fmt->Gmask, fmt->Bmask, 0);
	scrSurf[1] = SDL_CreateRGBSurface(SDL_SWSURFACE, WIDTH, HEIGHT, fmt->BitsPerPixel, fmt->Rmask, fmt->Gmask, fmt->Bmask, 0);
}

void InitAll(void)
{
	int i;
	for (i = 0; devs[i]; i++) devs[i]->Init();

	InitSurfaces();
	font.Init(ftbos_font_data);
	FileDialogInit();

	cpu = z80ex_create(
			ReadByte, NULL,
			WriteByte, NULL,
			InputByte, NULL,
			OutputByte, NULL,
			ReadIntVec, NULL
			);
}

#define MAX_FRAME_TACTS 71680
#define INT_BEGIN (MAX_FRAME_TACTS-3606)	// this is right INT, but UNREAL by KSA don't work
// #define INT_BEGIN (MAX_FRAME_TACTS-3682)	// this INT is specially for UNREAL by KSA (demo work in MaxSpeed mode)

bool attributesHack = false;

// processor_speed    = 3546900
// ay_speed           = 1773400

// left_border        = 36
// horizontal_screen  = 128
// right_border       = 28
// horizontal_retrace = 32

// top_border         = 64
// vertical_screen    = 192
// bottom_border      = 48
// vertical_retrace   = 16

// top_left_pixel     = 17988

void AntiFlicker(void)
{
	int i, j;
	BYTE *s1, *s2, *sr;
	BYTE *s1w, *s2w, *srw;

	if (SDL_MUSTLOCK(screen)) {if (SDL_LockSurface(screen) < 0) return;}
	if (SDL_MUSTLOCK(scrSurf[0])) {if (SDL_LockSurface(scrSurf[0]) < 0) return;}
	if (SDL_MUSTLOCK(scrSurf[1])) {if (SDL_LockSurface(scrSurf[1]) < 0) return;}

	sr = (BYTE *)screen->pixels;
	s1 = (BYTE *)scrSurf[0]->pixels;
	s2 = (BYTE *)scrSurf[1]->pixels;

	for (i = 0; i < HEIGHT; i++)
	{
		srw = sr;
		s1w = s1;
		s2w = s2;

		for (j = 0; j < WIDTH; j++)
		{
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
		}

		sr += screen->pitch;
		s1 += scrSurf[0]->pitch;
		s2 += scrSurf[1]->pitch;
	}

	if (SDL_MUSTLOCK(scrSurf[1])) SDL_UnlockSurface(scrSurf[1]);
	if (SDL_MUSTLOCK(scrSurf[0])) SDL_UnlockSurface(scrSurf[0]);
	if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
}

void Render_Pentagon(void)
{
	SDL_Surface *surf;
	unsigned long cmdClk, clMod, lastClk, rnClk;
	int line, scrLine, pos, cl;
	int zxLine, zxScreen, ci, cp, bt;
	int *scr, j, how;
	bool intr;
	static int sn = 0;

	if (params.antiFlicker)
	{
		surf = scrSurf[sn];
		sn = 1 - sn;
	} else surf = screen;

	if (SDL_MUSTLOCK(surf)) {if (SDL_LockSurface(surf) < 0) return;}

	currClk = 0;
	lastClk = 0;
	intr = true;

	while (currClk < MAX_FRAME_TACTS)
	{
		cmdClk = z80ex_step(cpu);
		currClk += cmdClk;
		clk += cmdClk;

		if (params.maxSpeed && z80ex_doing_halt(cpu))
		{
			cmdClk = (MAX_FRAME_TACTS - currClk - 1) / 4 + 1;
			currClk += cmdClk * 4;
			clk += cmdClk * 4;
			break;
		}

		if (drawFrame)
		{
			for (rnClk = lastClk; rnClk < currClk;)
			{
				line = rnClk / 224;

				if (line>=40 && line<280)	// in 320x240 screen (vertical)
				{
					scrLine = line - 40;
					clMod = rnClk % 224;

					if (clMod>=(20-5) && clMod<(180-5))	// in 320x240 screen (horizontal)
					{
						if (line<64 || line>=256 || clMod<(36-5) || clMod>=(164-5))	// border
						{
							cl = colors[dev_border.portFB & 7];
							pos = (clMod - (20-5)) * 2;
							scr = (int *)surf->pixels + PITCH*scrLine + pos;
							*(scr++) = cl;
							*(scr) = cl;
							rnClk++;
						}
						else
						{
							pos = (clMod - (36-5)) / 4;
							zxLine = line - 64;
							scr = (int *)surf->pixels + PITCH*scrLine + 32 + pos*8;

							zxScreen = (dev_mman.port7FFD & 8) ? RAM_BANK7 : RAM_BANK5;
							bt = dev_mman.ram[ zxScreen + ((zxLine & 0xC0) << 5) + ((zxLine & 7) << 8) + ((zxLine & 0x38) << 2) + pos ];
							cl = dev_mman.ram[ zxScreen + 0x1800 + ((zxLine & 0xF8) << 2) + pos ];

							if ((frames&32) && (cl&128))
							{
								cp = colors[((cl & 64) >> 3) | (cl & 7)];
								ci = colors[((cl & 64) >> 3) | ((cl >> 3) & 7)];
							}
							else
							{
								ci = colors[((cl & 64) >> 3) | (cl & 7)];
								cp = colors[((cl & 64) >> 3) | ((cl >> 3) & 7)];
							}

							if (attributesHack)
							{
								ci = DRGB(0,0,0);
								cp = DRGB(255,255,255);
							}

							*(scr++) = (bt & 128 ? ci : cp);
							*(scr++) = (bt &  64 ? ci : cp);
							*(scr++) = (bt &  32 ? ci : cp);
							*(scr++) = (bt &  16 ? ci : cp);
							*(scr++) = (bt &   8 ? ci : cp);
							*(scr++) = (bt &   4 ? ci : cp);
							*(scr++) = (bt &   2 ? ci : cp);
							*(scr)   = (bt &   1 ? ci : cp);

							if (clMod <= 160) rnClk += 4;
							else rnClk++;
						}
					} else rnClk++;
				} else rnClk++;
			}
			lastClk = rnClk;
		}

		if (intr && currClk>=INT_BEGIN)
		{
			cmdClk = z80ex_int(cpu);
			currClk += cmdClk;
			clk += cmdClk;
			intr = false;
		}
	}

	// it can't be happend, but...
	if (intr) clk += z80ex_int(cpu);

	if (SDL_MUSTLOCK(surf)) SDL_UnlockSurface(surf);
	if (params.antiFlicker && drawFrame) AntiFlicker();
}

void DrawInd(int x, int y, int b, int c)
{
	int i, j;

	for (i = 0; i < 8; i++) ((int *)screen->pixels)[y*PITCH + x+i] = b;
	for (i = 1; i < 7; i++)
	{
		((int *)screen->pixels)[(y+i)*PITCH + x] = b;
		for (j = 1; j < 7; j++) ((int *)screen->pixels)[(y+i)*PITCH + x+j] = c;
		((int *)screen->pixels)[(y+i)*PITCH + x+7] = b;
	}
	for (i = 0; i < 8; i++) ((int *)screen->pixels)[(y+7)*PITCH + x+i] = b;
}

void DrawIndicators(void)
{
	if (dev_trdos.wd_flag) DrawInd(5, 5, DRGB(4, 8, 64), DRGB(8, 16, 128));
	if (params.maxSpeed) DrawInd(15, 5, DRGB(0, 64, 0), DRGB(0, 200, 0));
}

void ResetSequence(void)
{
	int i;
	z80ex_reset(cpu);
	for (i = 0; i < cnt_reset; i++) hnd_reset[i]();
}

void Process(void)
{
	int key;
	SDL_Event event;
	int i, j, *s, *p, st;
	unsigned int btick;
	int frameSkip = 0;

	clk = 0;
	frames = 0;
	params.maxSpeed = false;

	static unsigned soundPtr; //[boo] - pointer in ring buffer
	soundPtr=SOUND_BUFFER_SIZE-frameSamples-1; //[boo] - give to sound renderers a odd	

	btick = SDL_GetTicks() + FRAME_WAIT_MS;

	for (;;)
	{
		if (params.sdl_sound && params.timing_snd && params.sound && !params.maxSpeed && soundBufferSize>AUDIO_BUFFER_SIZE)
		{
			int ayCntX;
			do
			{
				SDL_Delay(1);
				ayCntX = dev_ay.ayCnt;
				if (ayCntX < audioCnt) ayCntX += soundBufferSize;
			} while ((ayCntX-audioCnt) > AUDIO_BUFFER_SIZE);
		}

		for (i = 0; i < cnt_frameStart; i++) hnd_frameStart[i]();

		if (params.maxSpeed)
		{
			if (frameSkip > 0) {
				frameSkip--;
				drawFrame = false;
			} else {
				frameSkip = MAX_SPEED_FRAME_SKIP;
				drawFrame = true;
			}
		} else drawFrame = true;

		dev_trdos.wd_flag = false;
		Render_Pentagon();
		frames++;

		if (drawFrame)
		{
			DrawIndicators();
			ShowMessage();
			SDL_UpdateRect(screen, 0, 0, WIDTH, HEIGHT);
		}

		if (!params.timing_snd || !params.sdl_sound)
		{
			if (!params.maxSpeed && SDL_GetTicks()<btick) SDL_Delay(btick-SDL_GetTicks());
			btick = SDL_GetTicks() + FRAME_WAIT_MS;
		}

		for (i = 0; i < cnt_afterFrameRender; i++) hnd_afterFrameRender[i]();

		//[boo] -- play sound frame
		if(!params.sdl_sound)
		{
			wav_play(soundBuffer, soundPtr, frameSamples, SOUND_BUFFER_SIZE);
			soundPtr+=frameSamples;
			if (soundPtr >= SOUND_BUFFER_SIZE) soundPtr -= SOUND_BUFFER_SIZE;
		}			
		
		while (SDL_PollEvent(&event))
		{
			if (event.type == SDL_QUIT) exit(0);

			if (event.type == SDL_KEYUP)
			{
				key = event.key.keysym.sym;
				if (key == SDLK_ESCAPE) return;
				if (key == SDLK_F6) attributesHack = !attributesHack;
			}

			for (i = 0; i < cnt_sdl; i++) {
				if (types_sdl[i] == event.type) {
					if (hnd_sdl[i](event)) break;
				}
			}
		}
	}
}

SDL_AudioSpec *hardware_spec;

void AudioCallback(void *userdata, Uint8 *stream, int len)
{
	if (disableSound || params.maxSpeed)
	{
		while (len--) *(stream++) = 0;
	}
	else
	{
		for (len /= 4; len--;)
		{
		
			*((WORD *)stream) = soundBuffer[audioCnt].ch.left;
			stream += 2;
			*((WORD *)stream) = soundBuffer[audioCnt].ch.right;
			stream += 2;
			
			soundBuffer[audioCnt].ch.left=0;
			soundBuffer[audioCnt].ch.right=0;

			audioCnt++;
			if(audioCnt >= SOUND_BUFFER_SIZE) audioCnt = 0;
		}
	}
}

void InitAudio(void)
{
	SDL_AudioSpec *desired, *obtained;

	//[boo]
	memset(soundBuffer,0,SOUND_BUFFER_SIZE);	
	//[boo] - samples per frame
	frameSamples = ((unsigned long)71680/*frame*/*44100/*sound freq*/) / 3500000 /*Z80 clock*/;
	
	if (params.sdl_sound)
	{
		desired = (SDL_AudioSpec *)malloc(sizeof(SDL_AudioSpec));
		obtained = (SDL_AudioSpec *)malloc(sizeof(SDL_AudioSpec));

		desired->freq = 44100;
		desired->format = AUDIO_S16LSB;
		desired->channels = 2;
		desired->samples = /*AUDIO_HW_BUFFER*/frameSamples; //[boo]
		desired->callback = AudioCallback;

		if (SDL_OpenAudio(desired, obtained) < 0) StrikeError("Couldn't open audio: %s\n", SDL_GetError());

		free(desired);
		hardware_spec = obtained;

		audioCnt = SOUND_BUFFER_SIZE-frameSamples-1; //[boo]
		
		SDL_PauseAudio(0);
	}
	else
	{
		wav_start(frameSamples*2*2 /*[boo] sound frame size in bytes*/);
	}
}

void FreeAudio(void)
{
	if (params.sdl_sound)
	{
		SDL_CloseAudio();
		free(hardware_spec);
	}
	else
	{
		wav_stop();
	}
}

void FreeAll(void)
{
	int i;
	for (i = 0; devs[i]; i++) devs[i]->Close();

	SDL_FreeSurface(scrSurf[0]);
	SDL_FreeSurface(scrSurf[1]);

	z80ex_destroy(cpu);
	if (params.sound) FreeAudio();
}

void ParseCmdLine(int argc, char *argv[])
{
	if (argc != 2) return;
	TryNLoadFile(argv[1]);
}

void OutputLogo(void)
{
	printf("                                       \n");
	printf("   $ww,.                               \n");
	printf("    `^$$$ww,.                          \n");
	printf("       `^$$$$$$                        \n");
	printf("         ,$$7'                         \n");
	printf("       _j$$'   __    __  _ _           \n");
	printf("     ,j$$7      /__ (-_ | ) ) (_|      \n");
	printf("    $$$$$$w.                           \n");
	printf("     `^^T$$$w,             rst'o6      \n");
	printf("           `^T$$                       \n");
	printf("                                       \n");
	printf(" restorer [ restorer_fct(^at^)tut.by ] \n");
	printf(" boo_boo  [   boo_boo(^at^)inbox.ru  ] \n");
	printf("                                       \n");
	printf(" with help of SMT                      \n");
	printf("                                       \n");
}

int main(int argc, char *argv[])
{
	int spec;
	char *str;
	bool wd1793delay;

	OutputLogo();

	/* [boo] >> */
	char cfg_fname[MAX_PATH] = "";	// [rst] path to home directory can be more than 200. I think than 4096 is enougth ^_~
	strcat(cfg_fname, getenv("HOME"));
	strcat(cfg_fname, "/.zemu/config.xml");
	/* << [boo] */

	try
	{
		if (C_File::FileExists(cfg_fname)) config.Load(cfg_fname);	/* [boo] -- firstly look in user home directory */
		else config.Load("config.xml");

		if (!config.GetBool("root/SoundEnable", &params.sound)) params.sound = false;
		if (!config.GetBool("root/FullScreen", &params.fullscreen)) params.fullscreen = false;
		if (!config.GetBool("root/AntiFlicker", &params.antiFlicker)) params.antiFlicker = false;
		if (!config.GetBool("root/SoundCardTiming", &params.timing_snd)) params.timing_snd = false;
		if (!config.GetBool("root/UseSdlSound", &params.sdl_sound)) params.sdl_sound = false;

		if (config.GetString("root/Drives/A", &str)) wd1793_load_dimage(str, 0);
		if (config.GetString("root/Drives/B", &str)) wd1793_load_dimage(str, 1);
		if (config.GetString("root/Drives/C", &str)) wd1793_load_dimage(str, 2);
		if (config.GetString("root/Drives/D", &str)) wd1793_load_dimage(str, 3);
		if (config.GetBool("root/Drives/NoDelay", &wd1793delay)) wd1793_set_nodelay(wd1793delay);

		spec = SDL_INIT_VIDEO;
		if (params.sound && params.sdl_sound) spec |= SDL_INIT_AUDIO;
		if (SDL_Init(spec) < 0) StrikeError("Unable to init SDL: %s\n", SDL_GetError());

		atexit(SDL_Quit);

		spec = SDL_SWSURFACE;
		if (params.fullscreen) spec |= SDL_FULLSCREEN;
		screen = SDL_SetVideoMode(WIDTH, HEIGHT, 32, spec);

		if (screen == NULL) StrikeError("Unable to set requested video mode: %s\n", SDL_GetError());
		PITCH = screen->pitch / 4;

		SDL_WM_SetCaption("ZEmu", "ZEmu");
		SDL_ShowCursor(SDL_DISABLE);

		atexit(FreeAll);

		InitAll();
		ResetSequence();
		if (argc != 1) ParseCmdLine(argc, argv);
		if (params.sound) InitAudio();
		Process();
	}
	catch (C_E &e)
	{
		StrikeError("Error %d: %s (%s)", e.exc, e.Descr(), e.param);
	}

	return 0;
}
