#include <sys/types.h>
#include <sys/ioctl.h>
#include <err.h>
#include <pcap.h>
#include <poll.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#ifndef PROTOCOL
#define PROTOCOL	"ether proto 0x757e"
#endif

#ifndef HZ
#define HZ		50
#endif


#ifndef __OpenBSD__
long long strtonum(const char *, long long, long long, const char **);
#endif


static char *interface;
static int snaplen = 50;
static int hz = HZ;
static int debug;
static int log;


struct time_packet {
	u_int8_t	dst[6];
	u_int8_t	src[6];
	u_int16_t	proto;
	u_int16_t	date;
	u_int16_t	time[2];
};

static void
usage(void)
{
	extern char *__progname;

	(void)fprintf(stderr, "usage: %s [-dl] [-i interface] [-t hz]\n",
	    __progname);
	_exit(1);
}

static void
send_time(pcap_t *p, const struct time_packet *tp)
{
	struct time_packet stp;
	struct tm *tm;
	time_t t;
	u_int32_t s;
	u_int16_t y, m, d;

	bzero(&stp, sizeof(stp));
	bcopy(tp->src, stp.dst, sizeof(stp.dst));
	stp.proto = tp->proto;

	t = time(NULL);
	tm = localtime(&t);
	y = tm->tm_year - 72;
	m = tm->tm_mon + 1;
	d = tm->tm_mday;
	s = (tm->tm_hour * 3600 + tm->tm_min * 60 + tm->tm_sec) * hz;
	stp.date = (y & 037) | ((y & 0140) << 9) | (m << 10) | (d << 5);
	stp.time[0] = s >> 16;
	stp.time[1] = s & 0177777;

	(void)pcap_inject(p, &stp, sizeof(stp));
}

static const char *
ether_addr(const u_int8_t *addr)
{
	static char ether[18];

	(void)snprintf(ether, sizeof(ether), "%02x:%02x:%02x:%02x:%02x:%02x",
	    addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);

	return (ether);
}

int
main(int argc, char *const argv[])
{
	char errbuf[PCAP_ERRBUF_SIZE];
	struct bpf_program bprog;
	struct pollfd pfd;
	const char *errstr;
	const struct time_packet *tp;
	long long val;
	pcap_t *p;
	int ch;

	while ((ch = getopt(argc, argv, "di:lt:")) != -1)
		switch (ch) {
		case 'd':
			debug++;
			break;
		case 'i':
			interface = optarg;
			break;
		case 'l':
			log++;
			break;
		case 't':
			val = strtonum(optarg, 50, 60, &errstr);
			if (errstr != NULL || (val != 50 && val != 60))
				errx(1, "Invalid clock frequence: %s", optarg);
			hz = (int)val;
			break;
		default:
			usage();
			/*NOTREACHED*/
		}
	argv += optind;
	argc -= optind;

	if (interface == NULL && (interface = pcap_lookupdev(errbuf)) == NULL)
		errx(1, "%s", errbuf);

	if ((p = pcap_open_live(interface, snaplen, 0, 0, errbuf)) == NULL)
		errx(1, "%s", errbuf);

	if (pcap_compile(p, &bprog, PROTOCOL, 1, 0) < 0)
		errx(1, "%s", pcap_geterr(p));

	if (pcap_setfilter(p, &bprog) < 0)
		errx(1, "%s", pcap_geterr(p));

	pcap_freecode(&bprog);
	pfd.fd = pcap_fileno(p);

#ifdef __OpenBSD__
	ch = 1;
	if (ioctl(pfd.fd, BIOCIMMEDIATE, &ch) < 0)
		err(1, "BIOCIMMEDIATE");

	ch = BPF_DIRECTION_OUT;
	if (ioctl(pfd.fd, BIOCSDIRFILT, &ch) < 0)
		err(1, "BIOCSDIRFILT");
#endif

	if (!debug) {
		if (daemon(0, 0) < 0)
			err(1, NULL);

		openlog("ethtimed", LOG_PID, LOG_DAEMON);
	} else
		openlog("ethtimed", LOG_PERROR | LOG_PID, LOG_DAEMON);

	pfd.events = POLLIN;
	while (poll(&pfd, 1, -1) > 0) {
		struct pcap_pkthdr h;

		if (pfd.revents & (POLLERR | POLLHUP | POLLNVAL)) {
			syslog(LOG_ERR, "Shit happens...");
			return (1);
		}

		if ((tp = (const struct time_packet *)pcap_next(p, &h))
		    != NULL && h.caplen >= sizeof(*tp)) {
			if (debug || log)
				syslog(debug ? LOG_DEBUG : LOG_INFO,
				    "request from %s", ether_addr(tp->src));

			for (ch = 0; ch < 4; ch++) {
				send_time(p, tp);
				(void)usleep(300000);
			}
		}
	}

	syslog(LOG_ERR, "poll: %m");

	return (1);
}
