/*       ͻ
             TITLE:  RTL8019AS ethernet routines for 8051 and      
                     Keil C51 v7.00 port of Adam Dunkels' uIP v0.9 
          FILENAME:  etherdev.c                                    
          REVISION:  VER 0.0                                       
          REV.DATE:  21-01-05                                      
           ARCHIVE:                                                
            AUTHOR:  Copyright (c) 2005, Murray R. Van Luyn.       
            Refer to National Semiconductor DP8390 App Note 874.   
         ͼ        */

/*  Ŀ
      THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS   
      OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED   
      WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE  
      ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY     
      DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL  
      DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE   
      GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS       
      INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,        
      WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING           
      NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS  
      SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.        
      */

#define _NOINC_
#define _NOAPP_

#include "uip-base.h"
#include "uip-arp.h"
#include "etherdev.h"
#include "timer.h"

#include <string.h>

extern struct uip_eth_addr uip_ethaddr;

/*  ͻ
                                Private defines.                         
    ͼ  */

/* Packet transmit & receive buffer configuration */
#define ETH_TX_PAGE_START  0x40    /* 0x4000 Tx buffer is  6 * 256 = 1536 bytes */
#define ETH_RX_PAGE_START  0x46    /* 0x4600 Rx buffer is 26 * 256 = 6656 bytes */
#define ETH_RX_PAGE_STOP   0x60    /* 0x6000 */

#define ETH_MIN_PACKET_LEN 0x3C

/*  ͻ
                              Private Macro Defines                      
    ͼ  */

#define ETHERDEV_REG_WRITE(reg, wr_data) *reg=wr_data
#define ETHERDEV_REG_READ(reg) *reg
#define ETHERDEV_REG_PAGE(page) ETHERDEV_REG_WRITE(CR,(page << 6)|RD2|STP)

/*  ͻ
                                  Global Variables                       
    ͼ  */
struct timer etherdev_timer;

/*  ͻ
                         Private Function Implementation                 
    ͼ  */

void delay_ticks(int ticks)
{
    timer_set(&etherdev_timer, ticks);
    while (!timer_expired(&etherdev_timer)) continue;
}

/* mass copy to RTL8019AS  ix+6       ix+8  TODO: rewrite with LDIR */
void etherdev_hwsend(BYTE* ptr, register u16_t len)
{
    for(; len; len--)
      ETHERDEV_REG_WRITE(RDMA, *ptr++);
}

/* mass copy from RTL8019  ix+6       ix+8  TODO: rewrite with LDIR */
void etherdev_hwreceive(BYTE* ptr, register u16_t len)
{
    for(; len; len--) {
      *ptr=ETHERDEV_REG_READ(RDMA);
      ++ptr;
    }
}

/*  ͻ
                         Public Function Implementation                  
    ͼ  */

/*  ͻ
                                                                         
                                etherdev_init()                          
                                                                         
      Returns: 1 on success, 0 on failure.                               
      Refer to National Semiconductor DP8390 App Note 874, July 1993.    
                                                                         
    ͼ  */
BOOL etherdev_init(void)
{
    /* RTL8019AS hardware reset by pulse RSTDRV pin. */
    ETHERDEV_REG_WRITE(CR, ETHERDEV_RESET);
    delay_ticks(2);		/* delay at least 20ms */
    ETHERDEV_REG_PAGE(0);
    delay_ticks(3);		/* delay at least 40ms */
    ETHERDEV_REG_PAGE(0);

    /* Check if RTL8019AS fully reset. */
    if(!(ETHERDEV_REG_READ(ISR) & RST))
    {
        return 0;
    }

    /* Select RTL8019AS register page 3. */
    ETHERDEV_REG_PAGE(3);

    /* Temporarily disable config register write protection. */
    ETHERDEV_REG_WRITE(_9346CR, EEM1 | EEM0);

    /* Disable boot ROM & select 10BaseT with TP/CX auto-detect. */
    ETHERDEV_REG_WRITE(CONFIG2, BSELB);

    /* Select half-duplex, awake, power-up & LED_TX/ LED_RX/ LED_LNK behaviour. */
    ETHERDEV_REG_WRITE(CONFIG3, LEDS0);
    delay_ticks(11);		/* delay at least 200ms */

    /* Re-enable config register write protection. */
    ETHERDEV_REG_WRITE(_9346CR, 0x00);

    /* Stop RTL8019AS, select page 0 and abort DMA operation. */
    ETHERDEV_REG_PAGE(0);
    delay_ticks(1);		/* delay 20ms */

    /* Initialise data configuration register.  */
    /* FIFO threshold 8 bytes, no loopback, don't use auto send packet. */
    ETHERDEV_REG_WRITE(DCR, FT1 | LS);

    /* Reset remote byte count registers. */
    ETHERDEV_REG_WRITE(RBCR0, 0x00);
    ETHERDEV_REG_WRITE(RBCR1, 0x00);

    /* Receive configuration register to monitor mode. */
    ETHERDEV_REG_WRITE(RCR, MON);

    /* Initialise transmit configuration register to loopback internally. */
    ETHERDEV_REG_WRITE(TCR, LB0);

    /* Clear interrupt status register bits by writing 1 to each. */
    ETHERDEV_REG_WRITE(ISR, 0xFF);

    /* Mask all interrupts in mask register. */
    ETHERDEV_REG_WRITE(IMR, 0x00);

    /* Set transmit page start. */
    ETHERDEV_REG_WRITE(TPSR, ETH_TX_PAGE_START);

    /* Set receive buffer page start. */
    ETHERDEV_REG_WRITE(PSTART, ETH_RX_PAGE_START);

    /* Initialise last receive buffer read pointer. */
    ETHERDEV_REG_WRITE(BNRY, ETH_RX_PAGE_START);

    /* Set receive buffer page stop. */
    ETHERDEV_REG_WRITE(PSTOP, ETH_RX_PAGE_STOP);

    /* Select RTL8019AS register page 1. */
    ETHERDEV_REG_PAGE(1);
    delay_ticks(1);		/* delay 20ms */

    /* Initialise current packet receive buffer page pointer */
    ETHERDEV_REG_WRITE(CURR, ETH_RX_PAGE_START);

    /* Set physical address. TODO: LDIR instead of memcpy */
    if (uip_ethaddr.addr[1]) {
      memcpy(PAR0, uip_ethaddr.addr, 6);
/*
      ETHERDEV_REG_WRITE(PAR0, uip_ethaddr.addr[0]);
      ETHERDEV_REG_WRITE(PAR1, uip_ethaddr.addr[1]);
      ETHERDEV_REG_WRITE(PAR2, uip_ethaddr.addr[2]);
      ETHERDEV_REG_WRITE(PAR3, uip_ethaddr.addr[3]);
      ETHERDEV_REG_WRITE(PAR4, uip_ethaddr.addr[4]);
      ETHERDEV_REG_WRITE(PAR5, uip_ethaddr.addr[5]);
*/
    }
    else {
      memcpy(PAR0, uip_ethaddr.addr, 6);
/*
      uip_ethaddr.addr[0]=ETHERDEV_REG_READ(PAR0);
      uip_ethaddr.addr[1]=ETHERDEV_REG_READ(PAR1);
      uip_ethaddr.addr[2]=ETHERDEV_REG_READ(PAR2);
      uip_ethaddr.addr[3]=ETHERDEV_REG_READ(PAR3);
      uip_ethaddr.addr[4]=ETHERDEV_REG_READ(PAR4);
      uip_ethaddr.addr[5]=ETHERDEV_REG_READ(PAR5);
*/
    }

    /* Select RTL8019AS register page 0 and abort DMA operation. */
    ETHERDEV_REG_PAGE(0);

    ETHERDEV_REG_WRITE(DCR, FT1 | LS);

    /* Restart RTL8019AS. */
    ETHERDEV_REG_WRITE(CR, RD2 | STA);

    /* Receive configuration register to accept broadcast packets. */
    ETHERDEV_REG_WRITE(RCR, AB);

    /* Clear interrupt status register bits by writing 1 to each. */
    ETHERDEV_REG_WRITE(ISR, 0xFF);

    /* Mask all interrupts in mask register. */
    ETHERDEV_REG_WRITE(IMR, 0x00);

    /* Initialise transmit configuration register for normal operation. */
    ETHERDEV_REG_WRITE(TCR, 0x00);

    /* Start RTL8019AS. */
    ETHERDEV_REG_WRITE(CR, RD2 | STA);

    return 1;
}


/*  ͻ
                                                                         
                               etherdev_send()                           
                                                                         
     Send the packet in the uip_buf and uip_appdata buffers using the    
     RTL8019AS ethernet card.                                            
                                                                         
    ͼ  */
void etherdev_send(void)
{
    /* Setup for DMA transfer from uip_buf & uip_appdata buffers to RTL8019AS. */

    /* Select RTL8019AS register page 0 and abort DMA operation. */
    ETHERDEV_REG_WRITE(CR, RD2 | STA);

    /* Wait until pending transmit operation completes. */
    while(ETHERDEV_REG_READ(CR) & TXP) continue;

    /* Clear remote DMA complete interrupt status register bit. */
    ETHERDEV_REG_WRITE(ISR, RDC);

    /* Set remote DMA start address registers to indicate where to load packet. */
    ETHERDEV_REG_WRITE(RSAR0, 0x00);
    ETHERDEV_REG_WRITE(RSAR1, ETH_TX_PAGE_START);

    /* Set remote DMA byte count registers to indicate length of packet load. */
    ETHERDEV_REG_WRITE(RBCR0, (unsigned char)(uip_len & 0xFF));
    ETHERDEV_REG_WRITE(RBCR1, (unsigned char)(uip_len >> 8));

    /* Initiate DMA transfer of uip_buf & uip_appdata buffers to RTL8019AS. */
    ETHERDEV_REG_WRITE(CR, RD1 | STA);

    /* DMA transfer packet from uip_buf & uip_appdata to RTL8019AS local */
    /* transmit buffer memory. */

    if(uip_len <= UIP_CONF_LLH_LEN + UIP_TCPIP_HLEN) {
      etherdev_hwsend(&uip_buf[0], uip_len);
    } else {
      etherdev_hwsend(&uip_buf[0], UIP_CONF_LLH_LEN + UIP_TCPIP_HLEN);
      etherdev_hwsend(uip_appdata, uip_len - UIP_TCPIP_HLEN - UIP_CONF_LLH_LEN);
    }

    /* Wait until remote DMA operation complete. */
    while(!(ETHERDEV_REG_READ(ISR) & RDC)) continue;

    /* Abort/ complete DMA operation. */
    ETHERDEV_REG_WRITE(CR, RD2 | STA);

    /* Set transmit page start to indicate packet start. */
    ETHERDEV_REG_WRITE(TPSR, ETH_TX_PAGE_START);

    /* Ethernet packets must be > 60 bytes, otherwise are rejected as runts. */
    if(uip_len < ETH_MIN_PACKET_LEN)
    {
        uip_len = ETH_MIN_PACKET_LEN;
    }

    /* Set transmit byte count registers to indicate packet length. */
    ETHERDEV_REG_WRITE(TBCR0, (unsigned char)(uip_len & 0xFF));
    ETHERDEV_REG_WRITE(TBCR1, (unsigned char)(uip_len >> 8));

    /* Issue command for RTL8019AS to transmit packet from it's local buffer. */
    ETHERDEV_REG_WRITE(CR, RD2 | TXP | STA);

    return;
}


/*  ͻ
                                                                         
                               etherdev_poll()                           
                                                                         
     Poll the RTL8019AS ethernet device for an available packet.         
                                                                         
    ͼ  */
static unsigned int etherdev_poll(void)
{
    register WORD len = 0;
    BOOL retransmit;

    /* Check if there is a packet in the rx buffer. */
    if(ETHERDEV_REG_READ(ISR) & PRX)
    {
        /* Check if the rx buffer has overflowed. */
        if(ETHERDEV_REG_READ(ISR) & OVW)
        {
            retransmit = 0;

            /* If the receive buffer ring has overflowed we dump the whole */
            /* thing and start over. There is no way of knowing whether the */
            /* data it contains is uncorrupted, or will cause us grief.*/

            /* Stop RTL8019AS and abort DMA operation.*/
            ETHERDEV_REG_WRITE(CR, RD2 | STP);

            /* Reset remote byte count registers.*/
            ETHERDEV_REG_WRITE(RBCR0, 0x00);
            ETHERDEV_REG_WRITE(RBCR1, 0x00);

            /* Wait for controller to halt after any current tx completes.*/
            while(!(ETHERDEV_REG_READ(ISR) & RST)) continue;

            /* Check whether currently transmitting a packet.*/
            if(ETHERDEV_REG_READ(CR) & TXP)
            {
                /* If neither a successful transmission nor a tx abort error */
                /* has occured, then flag current tx packet for resend. */
                if(!((ETHERDEV_REG_READ(ISR) & PTX)
                                          || (ETHERDEV_REG_READ(ISR) & TXE)))
                {
                    retransmit = 1;
                }
            }

            /* Set transmit configuration register to loopback internally. */
            ETHERDEV_REG_WRITE(TCR, LB0);

            /* Restart the RTL8019AS. */
            ETHERDEV_REG_WRITE(CR, RD2 | STA);

            /* Re-initialise last receive buffer read pointer. */
            ETHERDEV_REG_WRITE(BNRY, ETH_RX_PAGE_START);

            /* Select RTL8019AS register page 1. */
            ETHERDEV_REG_PAGE(1);

            /* Re-initialise current packet receive buffer page pointer. */
            ETHERDEV_REG_WRITE(CURR, ETH_RX_PAGE_START);

            /* Select RTL8019AS register page 0. */
            ETHERDEV_REG_PAGE(0);

            /* Clear rx buffer overflow & packet received interrupt flags. */
            ETHERDEV_REG_WRITE(ISR, PRX | OVW);

            /* Re-itialise transmit configuration reg for normal operation. */
            ETHERDEV_REG_WRITE(TCR, 0x00);
        
            if(retransmit)
            {
                /* Retransmit packet in RTL8019AS local tx buffer. */
                ETHERDEV_REG_WRITE(CR, RD2 | TXP | STA);
            }
        }
        else /* Rx buffer has not overflowed, so read a packet into uip_buf. */
        {
            unsigned char next_rx_packet;
            unsigned char current;

            /* Retrieve packet header. (status, next_ptr, length_l, length_h) /

            /* Clear remote DMA complete interrupt status register bit. */
            ETHERDEV_REG_WRITE(ISR, RDC);

            /* Set remote DMA start address registers to packet header. */
            ETHERDEV_REG_WRITE(RSAR0, 0x00);
            ETHERDEV_REG_WRITE(RSAR1, ETHERDEV_REG_READ(BNRY));

            /* Set remote DMA byte count registers to packet header length. */
            ETHERDEV_REG_WRITE(RBCR0, 0x04);
            ETHERDEV_REG_WRITE(RBCR1, 0x00);

            /* Initiate DMA transfer of packet header. */
            ETHERDEV_REG_WRITE(CR, RD0 | STA);

            /* Drop packet status. We don't use it. */
            ETHERDEV_REG_READ(RDMA);

            /* Save next packet pointer. */
            next_rx_packet = ETHERDEV_REG_READ(RDMA);

            /* Retrieve packet data length and subtract CRC bytes. */
            len =  ETHERDEV_REG_READ(RDMA);
            len += ETHERDEV_REG_READ(RDMA) << 8;
            len -= 4;

            /* Wait until remote DMA operation completes. */
            while(!(ETHERDEV_REG_READ(ISR) & RDC)) continue;

            /* Abort/ complete DMA operation. */
            ETHERDEV_REG_WRITE(CR, RD2 | STA);


            /* Retrieve packet data. */

            /* Check if incoming packet will fit into rx buffer. */
            if(len <= UIP_CONF_BUFFER_SIZE)
            {
                /* Clear remote DMA complete interrupt status register bit. */
                ETHERDEV_REG_WRITE(ISR, RDC);

                /* Set remote DMA start address registers to packet data. */
                ETHERDEV_REG_WRITE(RSAR0, 0x04);
                ETHERDEV_REG_WRITE(RSAR1, ETHERDEV_REG_READ(BNRY));

                /* Set remote DMA byte count registers to packet data length. */
                ETHERDEV_REG_WRITE(RBCR0, (unsigned char)(len & 0xFF));
                ETHERDEV_REG_WRITE(RBCR1, (unsigned char)(len >> 8));

                /* Initiate DMA transfer of packet data. */
                ETHERDEV_REG_WRITE(CR, RD0 | STA);

                /* Read packet data directly into uip_buf. */
		etherdev_hwreceive(&uip_buf[0], len);

                /* Wait until remote DMA operation complete. */
                while(!(ETHERDEV_REG_READ(ISR) & RDC)) continue;

                /* Abort/ complete DMA operation. */
                ETHERDEV_REG_WRITE(CR, RD2 | STA);

            }
            else
            {
                /* Incoming packet too big, so dump it. */
                len = 0;
            }

            /* Advance boundary pointer to next packet start. */
            ETHERDEV_REG_WRITE(BNRY, next_rx_packet);

            /* Select RTL8019AS register page 1. */
            ETHERDEV_REG_PAGE(1);

            /* Retrieve current receive buffer page */
            current = ETHERDEV_REG_READ(CURR);

            /* Select RTL8019AS register page 0. */
            ETHERDEV_REG_PAGE(0);

            /* Check if last packet has been removed from rx buffer. */
            if(next_rx_packet == current)
            {
                /* Clear packet received interrupt flag. */
                ETHERDEV_REG_WRITE(ISR, PRX);
            }
        }
    }

    return len;
}


/*  ͻ
                                                                         
                               etherdev_read()                           
                                                                         
     This function will read an entire IP packet into the uip_buf.       
     If it must wait for more than 0.5 seconds, it will return with      
     the return value 0. Otherwise, when a full packet has been read     
     into the uip_buf buffer, the length of the packet is returned.      
                                                                         
    ͼ  */
WORD etherdev_read(void)
{    
    register WORD bytes_read;

    timer_set(&etherdev_timer, CLOCK_SECOND / 5);

    /* tick_count threshold should be 12 for 0.5 sec bail-out
       One second (24) worked better for me, but socket recycling
       is then slower. I set UIP_TIME_WAIT_TIMEOUT 60 in uipopt.h
       to counter this. Retransmission timing etc. is affected also. */

    while ((!(bytes_read = etherdev_poll())) &&
           (!timer_expired(&etherdev_timer))) continue;

    return bytes_read;
}

