/*
 * Copyright (c) 2001, Adam Dunkels.
 * All rights reserved. 
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met: 
 * 1. Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright 
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.  
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.  
 *
 * This file is part of the uIP TCP/IP stack.
 *
 * $Id: uip_arch.c,v 1.1 2004/05/09 00:24:47 Louis Exp $
 *
 */

#define _NOINC_

#include "uip-arch.h"

/*-----------------------------------------------------------------------------------*/
#ifdef UIP_ARCH_ADD32
void
uip_add32(u8_t *op32, register u16_t op16)
{
  
  uip_acc32[3] = op32[3] + (op16 & 0xff);
  uip_acc32[2] = op32[2] + (op16 >> 8);
  uip_acc32[1] = op32[1];
  uip_acc32[0] = op32[0];
  
  if(uip_acc32[2] < (op16 >> 8)) {
    ++uip_acc32[1];    
    if(uip_acc32[1] == 0) {
      ++uip_acc32[0];
    }
  }
  
  if(uip_acc32[3] < (op16 & 0xff)) {
    ++uip_acc32[2];  
    if(uip_acc32[2] == 0) {
      ++uip_acc32[1];    
      if(uip_acc32[1] == 0) {
	++uip_acc32[0];
      }
    }
  }
}
#endif /* UIP_ARCH_ADD32 */

#ifdef UIP_ARCH_CHKSUM

#define BUF ((uip_tcpip_hdr *)&uip_buf[UIP_LLH_LEN])
#define IP_PROTO_TCP    6

/*-----------------------------------------------------------------------------------*/
u16_t
uip_chksum(u16_t *sdata, u16_t len)
{
  register u16_t acc;
  
  for(acc = 0; len > 1; len -= 2) {
    acc += *sdata;
    if(acc < *sdata) {
      /* Overflow, so we add the carry to acc (i.e., increase by
         one). */
      ++acc;
    }
    ++sdata;
  }

  /* add up any odd byte */
  if(len == 1) {
    acc += htons(((u16_t)(*(u8_t *)sdata)) << 8);
    if(acc < htons(((u16_t)(*(u8_t *)sdata)) << 8)) {
      ++acc;
    }
  }

  return acc;
}
/*-----------------------------------------------------------------------------------*/
u16_t
uip_ipchksum(void)
{
  return uip_chksum((u16_t *)&uip_buf[UIP_LLH_LEN], 20);
}
/*-----------------------------------------------------------------------------------*/
u16_t
uip_tcpchksum(void)
{
  u16_t hsum;
  register u16_t sum;
  
  /* Compute the checksum of the TCP header. */
  hsum = uip_chksum((u16_t *)&uip_buf[20 + UIP_LLH_LEN], 20);

  /* Compute the checksum of the data in the TCP packet and add it to
     the TCP header checksum. */
  sum = uip_chksum((u16_t *)uip_appdata,
		   (u16_t)(((((u16_t)(BUF->len[0]) << 8) + BUF->len[1]) - 40)));

  if((sum += hsum) < hsum) {
    ++sum;
  }
  
  if((sum += BUF->srcipaddr[0]) < BUF->srcipaddr[0]) {
    ++sum;
  }
  if((sum += BUF->srcipaddr[1]) < BUF->srcipaddr[1]) {
    ++sum;
  }
  if((sum += BUF->destipaddr[0]) < BUF->destipaddr[0]) {
    ++sum;
  }
  if((sum += BUF->destipaddr[1]) < BUF->destipaddr[1]) {
    ++sum;
  }
  if((sum += (u16_t)htons((u16_t)IP_PROTO_TCP)) < (u16_t)htons((u16_t)IP_PROTO_TCP)) {
    ++sum;
  }

  hsum = (u16_t)htons((((u16_t)(BUF->len[0]) << 8) + BUF->len[1]) - 20);
  
  if((sum += hsum) < hsum) {
    ++sum;
  }
  
  return sum;
}
/*-----------------------------------------------------------------------------------*/
#endif /* UIP_ARCH_CHKSUM */

#ifndef UIP_PRINTF

static	char	*kbufptr;
static	int	kbuflen;

char	knstorech(char ch)
{
        if (--kbuflen>0) {
	  *kbufptr=ch;
	  ++kbufptr;
	  return ch;
	}
	return 0;		/* return 0 if some errors */
}

char	kstorech(char ch)
{
	*kbufptr=ch;
	++kbufptr;
	return ch;		
}

char	kprintch(char ch)
{
   	if (ch=='\n')
	  kputch(0x0d);  /*CR*/
	kputch(ch);
	return ch;		
}

/* number-to-string coversion routines */

static 
void __itoa (unsigned value, char *strP, uchar radix) {
	char buf[34];
	register char *_di = strP, *_si = buf;
	uchar len;

	do {
		*_si++ = (char)(value % radix);
		value /= radix;
	} while (value != 0);
	/* The value has now been reduced to zero and the
	 * digits are in the buffer.
	 */
	/* The digits in the buffer must now be copied in reverse order into
	 *  the target string, translating to ASCII as they are moved.
	 */
	len = (uchar)(_si-buf);
	while (len != 0) {
		--len;
		radix = (uchar)*--_si;
		*_di++ = (char)((uchar)radix + (((uchar)radix < 10) ? '0' : 'A'-10));
	}
	/* terminate the output string with a zero. */
	*_di ='\0';
}

static
void __ltoa (long value, char *strP, uchar radix) {
	char buf[34];
	register char *_di = strP, *_si = buf;
	uchar len;

	do {
		*_si++ = (char)(value % radix);
		value /= radix;
	} while (value != 0);
	/* The value has now been reduced to zero and the
	 * digits are in the buffer.
	 */
	/* The digits in the buffer must now be copied in reverse order into
	 *  the target string, translating to ASCII as they are moved.
	 */
	len = (uchar)(_si-buf);
	while (len != 0) {
		--len;
		radix = (uchar)*--_si;
		*_di++ = (char)((uchar)radix + (((uchar)radix < 10) ? '0' : 'A'-10));
	}
	/* terminate the output string with a zero. */
	*_di ='\0';
}
 
/*
 */
char *itoa(value, strP, radix)
	register int value;
	char *strP;
	int radix;
{
	char *p = strP;

	if (radix == 0) {
		if (value < 0) {
			*p++ = '-';
			value = -value;
		}
		radix = 10;
	}
	__itoa((unsigned)value,p,(unsigned)radix);
	return strP;
}

/*
 */
char *ltoa(value, strP, radix)
	long value;
	char *strP;
	int radix;
{
	register char *p = strP;

	if (radix == 0) {
		if (value < 0) {
			*p++ = '-';
			value = -value;
		}
		radix = 10;
	}
	__ltoa(value,p,(unsigned)radix);
	return strP;
}

/* Short version of (s)printf */

int kprint(pr_callback OnChar, char* fmt, uchar **arg)
{
	register int base;
	uchar l, w, c, pad, s[12], *p, *q;
	int len = 0;

	if (! fmt)
		return -1;	/* return EOF if some errors while processing */

	while ((w = (uchar)*fmt++) != 0) {
		if (w != '%') {
			if (!OnChar(w)) return -1;
			continue;
		}
		pad = (uchar)(*fmt == '0' ? '0' : ' ');
		w = 0;
		while ((*fmt >= '0') && (*fmt <= '9')) {
			w = w * 10;
			w += (uchar)(*fmt - '0');
			++fmt;
		}
		s[1] = 0;
		p = s;
		len = 0x7FFF;
		switch (c = (uchar)*fmt++) {
		case 'c':
			s[0] = *(uchar *)arg++;
			break;
		case 'l': base = 0; 
                          if ((uchar)*fmt=='u') {
				base=10; fmt++;
			  }
			  ltoa(*(long *)arg++, (char *)s, base);
                          arg++;
                          break;
		case 'd': base = 0;	goto prt;
		case 'o': base = 8;	goto prt;
		case 'b': base = 2;	goto prt;
		case 'u': base = 10;	goto prt;
		case 'p':
			w = 4;
			pad = '0';
		case 'x':
			base = 16;
prt:			itoa(*(int *)arg++, (char *)s, base);
			break;
		case 'n':
			p = *arg++;
			len = *(int *)arg++;
			break;
		case 's':
			p = *arg++;
			break;
		default:
			s[0] = c;
			break;
		}
		if (w) {
			l = 0;
			q = p;
			while (--len != 0 && *q++ != 0)
				++l;
			w -= l;
			while ((int)w-- > 0)
				if (!OnChar(pad)) return -1;
		}
		while (*p)
		  if (!OnChar(*(p++))) return -1;
	}
	return 0;	/* return EOF if some errors while processing */
}

int kprintf(fmt)
	char* fmt;
{
	return kprint(kprintch, fmt, (uchar **)&fmt + 1);
}

int ksprintf(buf, fmt)
	char* buf;
	char* fmt;
{
	if (buf) {
          kbufptr=buf;
	  kbuflen=kprint(kstorech, fmt, (uchar **)&fmt + 1);
	  *kbufptr='\0';
	  return (kbuflen==-1) ? -1 : (int)(kbufptr - buf);
        }
	else
	  return -1;	/* return EOF if some errors while processing */
}
 
int ksnprintf(buf, bufsize, fmt)
	char* buf;
	int bufsize;
	char* fmt;
{
	if (buf) {
          kbufptr=buf;
	  kbuflen=bufsize;
	  bufsize=kprint(knstorech, fmt, (uchar **)&fmt + 1);
	  *kbufptr='\0';
	  return (bufsize==-1) ? -1 : (int)(kbufptr - buf);
        }
	else
	  return -1;	/* return EOF if some errors while processing */
}

#endif


