// sanxion.cpp

// UnrealSpeccy sources used for sampler algorithm
// Melody alghorithm reversed by KolAnVi

#include <stdio.h>
#include <malloc.h>
#include <memory.h>

//*****
char sFileName[]="sanxion.wav";

//*****
#define errNone 0

#define appErrorClass 0x80000000

#define appErrMemoOverflow appErrorClass|1
#define appErrMemAlloc appErrorClass|2
#define appErrInvalidParam appErrorClass|3
#define appErrBufferOverflow    appErrorClass|4
#define appErrUninitialized appErrorClass|5
#define appErrInternalError appErrorClass|6
#define appErrOpenFile appErrorClass|7

//*****
typedef unsigned long UInt32;
typedef unsigned short UInt16;
typedef short Int16;
typedef unsigned long Err;
typedef char Boolean;

#define false 0
#define true 1

//*****
typedef struct pcmwaveformat_tag {
   UInt16 wFormatTag;  /* format type */
   UInt16 nChannels;   /* number of channels (i.e. mono, stereo...) */
   UInt32 nSamplesPerSec;  /* sample rate */
   UInt32 nAvgBytesPerSec; /* for buffer estimation */
   UInt16 nBlockAlign; /* block size of data */
   UInt16 wBitsPerSample;  /* number of bits per sample of mono data */
} PCMWAVEFORMAT;

#define WAVE_FORMAT_PCM 1
#define SAMPLE_RATE 44100
       
//*****
typedef struct {
   FILE *f;
   UInt32 sndDataSize;
   UInt16 nChannels;
   UInt32 nSamples;
   UInt16 nBits;
} SoundFile;


//***** MusicSampler

#define TICK_BIT 6
#define TICK 64 // (1<<TICK_BIT)
#define TICKMASK (0xFFFFFFFFU-TICK+1)

#define MULT_C 14

#define SNDBUFLEN 16384
#define BEEPER_VOL 4000
#define BEEPER1_VOL 8000
 
//*****
typedef struct {
   UInt32 framefq; // Frame Hz 
   UInt32 samplerate; // Sound Hz 
   UInt32 frame; // (Z80) t-states per frame

   UInt32 dssamples; // samples/frame
   UInt32 dspiece; // size of soundbuffer/frame
   UInt32 dsticks; // sound ticks/frame
   UInt32 mult_const1; 

//   UInt32 cpu_t; // Z80 t-states count in current frame (less than frame)
   Boolean spkr_state1;
   UInt16 spkr_dig1; // =(port_FE&0x10)?sound_beeper:0;
   Boolean spkr_state2;
   UInt16 spkr_dig2;

   UInt32 digtick;
   Int16 *sndbuf;
   UInt32 endtick;
} Sampler;


//***** MusicMachine
#define NUMNOTES 656
#define NOTETICKS 4608
#define NOTESTEP 1854
 
#define OUTWAITS 48
 
//*****
typedef struct {
   Boolean ch1State;
   Boolean ch2State;
   Boolean ch2Remain;
   UInt16 ch1Count;
   UInt16 ch2Count;
   UInt16 ch1Note;
   UInt16 ch2Note;

   UInt16 noteState; 
   UInt16 musicState; 
    
   UInt32 remainT;   // remain t-states at begin of the frame

   Sampler *pSmp;

   Boolean fMusicEnd;
} MusicMachine;

//*****
Err SoundFileCreate(SoundFile *pSndFile,char *filename,
   UInt16 nChannels,UInt32 nSamples,UInt16 nBits,UInt16 fileType);
Err SoundFileWriteBlock(SoundFile *pSndFile,void *pSndBlock,UInt32 sndBlockSize);
Err SoundFileClose(SoundFile *pSndFile);

void SamplerInit(Sampler *psmp,UInt32 SampleRate);
void SamplerNewFrame(Sampler *psmp);
void *SamplerGetData(Sampler *psmp,UInt32 *pSize);
void Sampler1Out(Sampler *psmp,Boolean state,UInt32 curT);
void Sampler2Out(Sampler *psmp,Boolean state,UInt32 curT);
void SamplerFlush(Sampler *psmp,UInt32 curT);
void SamplerSetEnd(Sampler *psmp,UInt32 curT);
void SamplerClose(Sampler *psmp);

void MusicMachineInit(Sampler *psmp);
void MusicMachineNextNote();
Boolean MusicMachineFrame(UInt32 frameT);

//***** Globals
MusicMachine gMM;

int gSingleChannelF=0;

//*****
// MusicData.c

unsigned char data1[657]={ 
 7,  7,  7,  7,  7,  7,  7,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
19, 19, 19, 19, 19, 19, 19, 19, 
19, 19, 19, 19, 19, 19, 19, 19, 
 7,  7,  7,  7,  7,  7,  7,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
19, 19, 19, 19, 19, 19, 19, 19, 
19, 19, 19, 19, 19, 19, 19, 19, 
 7,  7,  7,  7,  7,  7,  7,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
19, 19, 14, 41, 17, 41, 12, 14, 
41, 10, 41,  5,  7, 41, 10, 12, 
 9,  9, 12, 41,  5, 41,  9, 12, 
22, 22, 21, 41, 17, 41, 12, 12, 
10, 10, 12, 41, 22, 41, 12, 22, 
24, 22, 12, 41, 22, 41, 17, 19, 
41, 19, 19, 17, 12, 41, 14,  7, 
 7,  7,  7, 41, 41, 41, 41, 41, 
17, 17, 12, 41, 10, 41,  9, 10, 
41, 10, 41, 12,  9, 41, 12, 17, 
22, 22, 21, 41, 17, 41, 12, 10, 
41, 10, 10, 12,  9, 41, 10, 12, 
10, 10, 12, 41, 22, 41, 12, 22, 
24, 22, 12, 41, 22, 41, 17, 19, 
41, 41, 19, 19, 17, 12, 14,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
19, 19, 14, 14, 17, 17, 12, 14, 
14, 10, 10,  5,  7,  7, 10, 12, 
 9,  9, 12, 12,  5,  5,  9, 12, 
22, 22, 21, 21, 17, 17, 12, 12, 
10, 10, 12, 12, 22, 22, 12, 22, 
24, 22, 12, 12, 22, 22, 17, 19, 
19, 19, 19, 17, 12, 12, 14,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
17, 17, 12, 12, 10, 10,  9, 10, 
10, 10, 10, 12,  9,  9, 12, 17, 
22, 22, 21, 21, 17, 17, 12, 10, 
10, 10, 10, 12,  9,  9, 10, 12, 
10, 10, 12, 12, 22, 22, 12, 22, 
24, 22, 12, 12, 22, 22, 17, 19, 
19, 19, 19, 17, 12, 12, 14,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
 7,  7,  7,  7,  7,  7,  7,  7, 
26, 26, 26, 26, 26, 26, 26, 26, 
26, 26, 26, 26, 26, 26, 26, 26, 
29, 29, 29, 29, 29, 29, 29, 29, 
29, 29, 29, 29, 26, 26, 29, 29, 
29, 29, 29, 29, 31, 31, 31, 31, 
31, 31, 31, 31, 31, 31, 31, 31, 
34, 34, 34, 34, 34, 34, 34, 34, 
34, 34, 34, 34, 31, 34, 31, 34, 
36, 36, 36, 36, 36, 36, 36, 36, 
36, 36, 36, 36, 36, 36, 36, 36, 
36, 36, 36, 36, 36, 36, 36, 36, 
36, 36, 36, 36, 31, 34, 36, 36, 
38, 38, 38, 38, 38, 38, 38, 38, 
38, 38, 38, 38, 38, 38, 38, 38, 
38, 38, 38, 38, 38, 38, 38, 38, 
38, 38, 38, 38, 38, 38, 38, 38, 
26, 26, 26, 26, 26, 26, 26, 26, 
26, 26, 26, 26, 26, 26, 26, 26, 
26, 26, 26, 26, 26, 26, 26, 26, 
26, 26, 26, 26, 26, 26, 26, 26, 
31, 31, 26, 26, 29, 29, 24, 26, 
26, 22, 22, 17, 19, 19, 22, 24, 
21, 21, 24, 24, 17, 17, 21, 24, 
34, 34, 33, 33, 29, 29, 24, 24, 
22, 22, 24, 24, 34, 34, 24, 34, 
36, 34, 24, 24, 34, 34, 29, 31, 
31, 31, 31, 29, 24, 24, 26, 19, 
19, 19, 19, 19, 19, 19, 19, 19, 
29, 29, 24, 24, 22, 22, 21, 22, 
22, 22, 22, 24, 21, 21, 24, 29, 
34, 34, 33, 33, 29, 29, 24, 22, 
22, 22, 22, 24, 21, 21, 22, 24, 
22, 22, 24, 24, 34, 34, 24, 34, 
36, 34, 24, 24, 34, 34, 29, 31, 
31, 31, 31, 29, 24, 24, 26, 19, 
19, 19, 19, 19, 19, 19, 19, 19, 
0x40 //pDataEnd1 
};
 
unsigned char data2[657]={ 
251,251,251,251,251,251,251,251, 
251,251,251,251,251,251,251,251, 
  7,  7,  7,  7,  7,  7,  7,  7, 
  7,  7,  7,  7,  7,  7,  7,  7, 
251,251,251,251,251,251,251,251, 
251,251,251,251,251,251,251,251, 
  7,  7,  7,  7,  7,  7,  7,  7, 
  7,  7,  7,  7,  7,  7,  7,  7, 
  7,  7,  7,  7,  7,  7,  7,  7, 
  7,  7,  7,  7,  7,  7,  7,  7, 
  7,  8,  9, 10, 11, 12, 13, 14, 
 15, 16, 17, 18, 19, 19,251,251, 
251,251,  7,251,251,  7,251,  7, 
251,  7,  7,251,  7,251,251,  7, 
249,249,  5,249,249,  5,249,  5, 
249,249,  5,249,249,  5,249,  5, 
244,244,  0,244,244,  0,244,  0, 
244,  0,  0,244,244,  0,244,  0, 
251,251,  7,251,251,  7,251,  7, 
251,251,251,251,  7,251,251,  7, 
249,249,  5,249,249,  5,249,  5, 
249,249,  5,249,249,  5,249,  5, 
254,254, 10,254,254,254,254, 10, 
254,254, 10,254,254, 10,254, 10, 
244,244,  0,244,244,  0,  0,244, 
244,  0,244,  0,  0,  0,  0,244, 
251,  7,  7,251,251,  7,251,  7, 
251,  7,251,251,  7,251,251,  7, 
251,  7,251,251,  7,251,251,  7, 
251,  7,251,251,  7,251,251,  7, 
249,  5,249,249,  5,249,249,  5, 
249,  5,249,249,  5,249,249,  5, 
244,  0,  0,244,244,  0,244,  0, 
244,  0,244,244,  0,244,244,  0, 
251,  7,251,251,  7,251,251,  7, 
251,  7,251,251,  7,251,251,  7, 
249,  5,249,249,  5,249,249,  5, 
249,  5,249,249,  5,249,249,  5, 
254, 10,254,254, 10,254,254, 10, 
254, 10,254,254, 10,254,254, 10, 
244,  0,244,244,  0,244,244,  0, 
244,  0,244,244,  0,244,244,  0, 
251,  7,251,251,  7,251,251,  7, 
251,  7,251,251,  7,251,251,  7, 
251,251,251,251,251,251,251,251, 
251,251,251,251,251,251,251,251, 
254, 10,254,254, 10,254,254, 10, 
254, 10,254,254, 10,254,254, 10, 
254,254, 10,254,254, 10,254, 10, 
254, 10,254,254, 10,254, 10,254, 
247,  3,247,247,  3,247,  3,247, 
247,  3,247,247,  3,247,247,  3, 
247,  3,247,  3,  3,247,247,  3, 
247,  3,247,247,  3,247,  3,247, 
244,  0,244,244,  0,244,  0,244, 
244,  0,244,244,  0,244,244,  0, 
244,  0,244,  0,  0,244,  0,244, 
244,  0,244,244,  0,244,244,  0, 
246,  2,246,  2,  2,246,246,  2, 
246,  2,246,246,  2,246,246,  2, 
246,246,  2,246,246,  2,246,  2, 
246,  2,  2,246,  2,246,246,  2, 
  2,  2,  2,  2,  2,  2,  2,  2, 
246,246,246,246,246,246,246,246, 
  2,  2,  2,  2,  2,  2,  2,  2, 
246,246,246,246,246,246,246,246, 
251,  7,251,251,  7,251,251,  7, 
251,  7,251,251,  7,251,251,  7, 
249,  5,249,249,  5,249,249,  5, 
249,  5,249,249,  5,249,249,  5, 
244,  0,244,244,  0,244,244,  0, 
244,  0,244,244,  0,244,244,  0, 
251,  7,251,251,  7,251,  7,251, 
251,  7,251,251,  7,251,251,  7, 
249,249,  5,249,249,  5,249,  5, 
249,  5,249,249,  5,249,249,  5, 
254, 10,254,254, 10,254,254, 10, 
254, 10,254,254, 10,254,254, 10, 
244,  0,244,244,  0,244,244,  0, 
244,  0,244,244,  0,244,244,  0, 
251,  7,251,251,  7,251,  7,251, 
251,251,  7,251,251,  7,251,  7, 
0x40 // pDataEnd2 
};
 
unsigned char freqtbl[54]={ 
255,240,227,215,203,192,180,171, 
161,151,144,136,128,121,114,108, 
102, 96, 91, 86, 81, 76, 72, 68, 
 64, 61, 57, 54, 51, 48, 45, 43, 
 40, 38, 36, 34, 32, 30, 28, 27, 
 25, 24, 23, 21, 20, 19, 18, 17, 
 16, 15, 14, 13, 12,  1 
};


//***** MusicSampler
void SamplerInit(Sampler *psmp,UInt32 SampleRate)
{ 
   if (!psmp) return;
   psmp->framefq=50; 
   psmp->samplerate=SampleRate; 
   psmp->frame=71680; 
    
   psmp->dssamples=psmp->samplerate/psmp->framefq; //=882
   psmp->dspiece=psmp->dssamples*2; //=1764
   psmp->dsticks=psmp->dssamples<<TICK_BIT; //=56448 
    
   psmp->mult_const1=(psmp->dsticks<<MULT_C)/psmp->frame; //=12902 
   psmp->spkr_state1=false;
   psmp->spkr_dig1=0;
   psmp->spkr_state2=false;
   psmp->spkr_dig2=0;

   psmp->sndbuf=(Int16 *)malloc(SNDBUFLEN);
   if (!psmp->sndbuf) return;
   memset(psmp->sndbuf,0,psmp->dspiece);
}

//*****
void SamplerNewFrame(Sampler *psmp)
{
   if (!psmp) return;
   memset(psmp->sndbuf,0,psmp->dspiece);
   psmp->digtick=0;
   psmp->endtick=psmp->dsticks;
}

//*****
void *SamplerGetData(Sampler *psmp,UInt32 *pSize)
{
   if (!psmp) return 0;
   if (pSize) *pSize=(psmp->endtick>>TICK_BIT)*sizeof(psmp->sndbuf[0]);
   return psmp->sndbuf;
}

//*****
void Sampler1Out(Sampler *psmp,Boolean state,UInt32 curT)
{
   if (psmp->spkr_state1 != state) {
      SamplerFlush(psmp,curT);
      psmp->spkr_state1=state;
      if (gSingleChannelF) psmp->spkr_dig1=state? BEEPER1_VOL:0;
      else psmp->spkr_dig1=state? BEEPER_VOL:0;
   }
}

//*****
void Sampler2Out(Sampler *psmp,Boolean state,UInt32 curT)
{
   if (gSingleChannelF) { Sampler1Out(psmp,state,curT); return; }
   if (psmp->spkr_state2 != state) {
      SamplerFlush(psmp,curT);
      psmp->spkr_state2=state;
      psmp->spkr_dig2=state? BEEPER_VOL:0;
   }
}
 
//*****
void SamplerFlush(Sampler *psmp,UInt32 curT)
{ 
   UInt32 digtick,endtick;
   UInt16 p,scale;
   UInt32 add;
   Int16 *sndbuf;

   sndbuf=psmp->sndbuf;
   endtick=(curT*psmp->mult_const1)>> MULT_C;   //(cpu.t*dsticks)/conf.frame
   digtick=psmp->digtick;
   p=(UInt16)(digtick >> TICK_BIT);
   if ((digtick ^ endtick) & ~(TICK-1)) {
      scale=(UInt16)(TICK-(digtick & (TICK-1)));
      add=((UInt32)(psmp->spkr_dig1+psmp->spkr_dig2)*scale)>>TICK_BIT;
      sndbuf[p]+=(Int16)add;
      p++;
      digtick+=scale; 
      while (endtick-digtick>=TICK) { 
         sndbuf[p]+=psmp->spkr_dig1+psmp->spkr_dig2;
         p++;
         digtick+=TICK; 
      }
   } 
   scale=(UInt16)(endtick-digtick);
   add=((UInt32)(psmp->spkr_dig1+psmp->spkr_dig2)*scale)>>TICK_BIT;
   sndbuf[p]+=(Int16)add;
   psmp->digtick=endtick;
}

//*****
void SamplerSetEnd(Sampler *psmp,UInt32 curT)
{
   psmp->endtick=((curT*psmp->mult_const1)>>MULT_C)+TICK;
}

//*****
void SamplerClose(Sampler *psmp)
{
   if (!psmp) return;
   if (psmp->sndbuf) free(psmp->sndbuf);
}

//***** MusicMachine
void MusicMachineInit(Sampler *psmp)
{
   if (!psmp) return;

   gMM.pSmp=psmp;
   gMM.fMusicEnd=false;

   gMM.musicState=0; // =127;
   gMM.ch2Remain=false;
   gMM.remainT=0;

   MusicMachineNextNote();
} 

//*****
void MusicMachineNextNote()
{
   gMM.noteState=0; 
   gMM.ch1State=false;
   gMM.ch2State=false;
   gMM.ch1Count=1; 
   gMM.ch2Count=1;
   gMM.ch1Note=freqtbl[(data1[gMM.musicState]+12)&0xFF]; 
   gMM.ch2Note=freqtbl[(data2[gMM.musicState]+12)&0xFF]; 
}
 
//*****
Boolean MusicMachineFrame(UInt32 frameT) 
{ 
   UInt32 curT; 

   curT=gMM.remainT;
   if (curT>=frameT) {
      SamplerFlush(gMM.pSmp,frameT);
      gMM.remainT-=frameT;
      return true;
   }

   if (gMM.ch2Remain!=0) {
      Sampler2Out(gMM.pSmp,gMM.ch2State,curT);
      gMM.ch2Count--;
      if (gMM.ch2Count==0) {
         gMM.ch2Count=gMM.ch2Note;
         gMM.ch2State=!gMM.ch2State;
      }
      curT+=OUTWAITS; 
      gMM.ch2Remain=false; 
      gMM.noteState++; 
   } 
    
   while (curT<frameT) {
      if (gMM.noteState>=NOTETICKS) {
         gMM.musicState++;
         if (gMM.musicState>=NUMNOTES) gMM.musicState=0;

         MusicMachineNextNote();

         if (gMM.musicState==0) {
            gMM.fMusicEnd=true;
            SamplerSetEnd(gMM.pSmp,curT);
         } 
         curT+=NOTESTEP;
         if (curT>=frameT) {
            gMM.remainT=curT-frameT;
            break;
         }
      } 
      if (frameT-curT<OUTWAITS) break;
      Sampler1Out(gMM.pSmp,gMM.ch1State,curT);
      gMM.ch1Count--; 
      if (gMM.ch1Count==0) {
         gMM.ch1Count=gMM.ch1Note;
         gMM.ch1State=!gMM.ch1State;
      }
      curT+=OUTWAITS;

      if (frameT-curT<OUTWAITS) {
         gMM.ch2Remain=true;
         break;
      }
      Sampler2Out(gMM.pSmp,gMM.ch2State,curT);
      gMM.ch2Count--;
      if (gMM.ch2Count==0) {
         gMM.ch2Count=gMM.ch2Note;
         gMM.ch2State=!gMM.ch2State;
      }
      curT+=OUTWAITS;
      gMM.noteState++;
   }
   SamplerFlush(gMM.pSmp,frameT);
   if (frameT>curT) gMM.remainT=OUTWAITS+curT-frameT;
   else if (frameT==curT) gMM.remainT=0;

   return gMM.fMusicEnd;
}


//***** Main Module
void MakeMusic(SoundFile *pSndFile)
{
   Sampler smp;
   Boolean fend;
   void *p;
   UInt32 size;

   SamplerInit(&smp,SAMPLE_RATE);
   MusicMachineInit(&smp);
   do {
      SamplerNewFrame(&smp);
      fend=MusicMachineFrame(smp.frame);
      p=SamplerGetData(&smp,&size);
      SoundFileWriteBlock(pSndFile,p,size);
   } while (!fend);

   SamplerClose(&smp);
}

//*****
Err SoundFileCreate(SoundFile *pSndFile,char *filename,
   UInt16 nChannels,UInt32 nSamples,UInt16 nBits,UInt16 fileType)
{
   PCMWAVEFORMAT wf;
   UInt32 tmp;
   FILE *f;
   char dataData[8]="data   ";

   if (!pSndFile) return appErrInvalidParam;

   f=fopen(filename,"wb+");
   if (f==0) return appErrOpenFile;

   wf.wFormatTag=WAVE_FORMAT_PCM;
   wf.nChannels=nChannels;   // 1
   wf.nSamplesPerSec=nSamples;   // 44100
   wf.nBlockAlign=nBits/8*nChannels;
   wf.nAvgBytesPerSec=nSamples*wf.nBlockAlign;
   wf.wBitsPerSample=nBits;   // 16

   fseek(f,0,0);
   fwrite("RIFF    WAVEfmt ",1,16,f);
   tmp=sizeof(PCMWAVEFORMAT);
   fwrite(&tmp,1,4,f);
   fwrite(&wf,1,tmp,f);
   dataData[4]=0; //Make online compiller happy
   dataData[5]=0;
   dataData[6]=0;
   fwrite(dataData,1,8,f);

   pSndFile->f=f;
   pSndFile->sndDataSize=0;
   pSndFile->nChannels=nChannels;
   pSndFile->nSamples=nSamples;
   pSndFile->nBits=nBits;

   return errNone;
}

//*****
Err SoundFileWriteBlock(SoundFile *pSndFile,void *pSndBlock,UInt32 sndBlockSize)
{
   if (!pSndFile) return appErrInvalidParam;
   if (pSndFile->f==0) return appErrUninitialized;

   fwrite(pSndBlock,1,sndBlockSize,pSndFile->f);
   pSndFile->sndDataSize+=sndBlockSize;
   return errNone;
}

//*****
Err SoundFileClose(SoundFile *pSndFile)
{
   UInt32 tmp;

   if (!pSndFile) return appErrInvalidParam;
   if (pSndFile->f==0) return appErrUninitialized;

   fseek(pSndFile->f,4,0);
   tmp=pSndFile->sndDataSize+sizeof(PCMWAVEFORMAT)+20;
   fwrite(&tmp,1,4,pSndFile->f);
   fseek(pSndFile->f,sizeof(PCMWAVEFORMAT)+24,0);
   tmp=pSndFile->sndDataSize;
   fwrite(&tmp,1,4,pSndFile->f);
   fclose(pSndFile->f);
   pSndFile->f=0;
   return errNone;
}

//*****
int main(int argc,char **argv)
{
   SoundFile sndFile;
   Err err;

   if (argc>1 && argv[1][0]=='1') gSingleChannelF=1;
   err=SoundFileCreate(&sndFile,sFileName,1,SAMPLE_RATE,16,0);
   if (err) {
      printf("SoundFileCreate: %x",err);
      return -1;
   }
   MakeMusic(&sndFile);

   SoundFileClose(&sndFile);
   return 0;
}
