require 'rubygems'
require 'midilib'
require 'generator'

# midibeep2.rb: Convert a Standard MIDI (.mid) file into ZX Assembly Listing
# designed to play the file via the Spectrum BEEP Rom routines
# Original Concept by Matthew Westcott 2009 (C) CopyRight
# This Edited variation by Karl McNeil 2010 May 27th
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
# 
# Contact details: 
# For the Original File: 
# <matthew@west.co.tt>
# Matthew Westcott, 14 Daisy Hill Drive, Adlington, Chorley, Lancs PR6 9NE UNITED KINGDOM
#
# For this version:
# <KgMcNeil@Yahoo.Com>


FNAME = if ARGV[0] then
           ARGV[0]
           else
           Dir["*.{mid,midi}"][0]
           end

##########################################################
### Outputing Assembly Listing (First Part) Then Notes ###
##########################################################
puts "
ORG 25000

;; BEEPER assembly listing... May 27/2010 By Karl McNeil

LD HL,NOTES

read_notes_loop:

LD A,(HL)

INC A ; If duration byte is 0, then EXIT...
DEC A ; (Zero is the end marker for the note data)
RET Z

LD B,A
INC HL

LD C,(HL)
INC HL

CheckKey:
        xor a
        in a, ($fe)
        cpl
        and %00011111
        RET nz ; EXIT routine if key is pressed

PUSH HL
CALL BEEPIT ; BC is set, so now Play note...
POP HL

Jp read_notes_loop

BEEPIT:
; Input BC = B=Duration, C=Pitch
; Output Action: Beeps note using the ROM beeper routine...

; Our duration will be a multiple of our mini-unit
; (a mini-unit is actually 5/255)...
; To convert a Basic BEEP duration into our assembled value,
; Assembled Duration = INT(Basic duration in Sec / (5/255) )
; Limitations: Our Duration can thus go no higher than 5 seconds (value 255)

; Our pitch will be the same as BASIC but with 60 added
; This avoids messing with negative numbers while storing data

PUSH BC

 ;SET-UP Basic Unit of duration (5/255)
RST $28 ; use the floating point calculator
DEFB $A2; stk_half
DEFB $A4; stk_ten
DEFB $04; multiple ; Now we have 5 on the stack
DEFB $38 ; end-calc

LD A,255;(units are 5/255)
CALL $2D28; Push A onto Calc Stack via Rom routine
RST $28 ; use the floating point calculator
DEFB $05; division
DEFB $38 ; end-calc
; Fraction of (5/255) is now be on CalcStack
; now to push Our duration Value and multiple
POP BC
PUSH BC

LD A,B
CALL $2D28; Push A onto Calc Stack via Rom routine: Duraton (B)
RST $28 ; use the floating point calculator
DEFB $04; multiple
DEFB $38 ; end-calc

POP BC

LD A,C
CALL $2D28; Push A onto Calc Stack via Rom routine: Pitch (C)
LD A,60
CALL $2D28; Push A onto Calc Stack via Rom routine: Pitch (C)
RST $28 ; use the floating point calculator
DEFB $03; subtract
DEFB $38 ; end-calc

; Currect duration & pitch value now on calc stack and ready

CALL $03F8; entry point for BEEP
RET

NOTES:
"

##########################################################
##########################################################

MIN_NOTE_LENGTH = 19607.843 # Minimum number of microseconds each note must be played for
#MIN_NOTE_LENGTH = 10_000 # Works better with Rachmaninov. :-)
NOTES_PER_LINE = 4 # There are 2 bytes per note.
byte = 0

# Create a new, empty sequence.
seq = MIDI::Sequence.new()

# Utility class to merge several Enumerables (each of which emit comparable
# objects in order) into one ordered Enumerable. Used to merge all MIDI tracks
# into a single stream
class IteratorMerger
	include Enumerable
	
	def initialize
		@streams = []
	end
	
	def add(enumerable)
		# convert enumerable object to an iterator responding to end?, current and next
		@streams << Generator.new(enumerable)
	end
	
	def each
		until @streams.all?{|stream| stream.end?}
			# while there are still some objects in the stream,
			# pick the stream whose next object is first in order
			next_stream = @streams.reject{|stream| stream.end?}.min{|a,b|
				a.current <=> b.current
			}
			yield next_stream.next
		end
	end
end

# Fiddle the ordering of MIDI event objects so that lower notes come first,
# which means that when we come to play them they'll fan upwards
class MIDI::Event
	def <=>(other)
		this_event_comparator = [
			self.time_from_start, (self.is_a?(MIDI::NoteEvent) ? self.note : -1)]
		other_event_comparator = [
			other.time_from_start, (other.is_a?(MIDI::NoteEvent) ? other.note : -1)]
		this_event_comparator <=> other_event_comparator
	end
end

File.open(FNAME, 'rb') { | file |
	# Create a stream of all MIDI events from all tracks
	event_stream = IteratorMerger.new
	seq.read(file) { | track, num_tracks, i |
		# puts "Loaded track #{i} of #{num_tracks}"
		next unless track
		event_stream.add(track)
	}
	
	# Keeping track of the time at which the last tempo change event occurred,
	# and the new tempo, will allow us to calculate an exact microsecond time
	# for each subsequent event.
	last_tempo_event_microsecond_time = 0
	default_bpm = MIDI::Sequence::DEFAULT_TEMPO
	default_microseconds_per_beat = MIDI::Tempo.bpm_to_mpq(default_bpm)
	last_tempo_event = MIDI::Tempo.new(default_microseconds_per_beat)
	
	last_note_on_event = nil
	last_note_on_microsecond_time = 0
	last_note_off_event = nil
	last_note_off_microsecond_time = 0
	
	line_number = 0 # tracks the BASIC line number to emit
		
	overshoot = 0 # number of microseconds we've played longer than we should have,
	# to allow excessively short notes to be heard
	
	# Function to emit a BEEP statement for a note whose start time and pitch
	# are given by last_note_on_event and last_note_on_microsecond_time, and
	# end time is passed as end_microsecond_time.
	# This is called on encountering the next 'note on' event (at which point
	# we know how long the previous note should last), and also on the final
	# 'note off' event of the stream.
	add_beep = lambda { |end_microsecond_time|
		real_note_duration = end_microsecond_time - last_note_on_microsecond_time
		# Reduce by overshoot if necessary, to compensate for previous notes
		# that were played for longer than the real duration (due to MIN_NOTE_LENGTH)
		# 
		# Playing a note of duration target_duration will get us back to the correct time
		# (aside from the fact that this might be negative...)
		target_duration = real_note_duration - overshoot

		# Extend actual duration to at least MIN_NOTE_LENGTH
		actual_duration = [target_duration, MIN_NOTE_LENGTH].max
		overshoot = actual_duration - target_duration
		# translate MIDI note number to BEEP pitch: middle C is 48 in MIDI, 0 in BEEP
		pitch = last_note_on_event.note - 48

# Now my modifications...
		pitch +=60
		actual_duration = ((actual_duration / 1_000_000.0)/0.019607843).to_i
		byte +=2


#ERROR TESTING FOR NOTES ABOVE 5 SECOND RANGE GOES HERE: if byte == 16 then actual_duration = 512 end


		repeat_note = (actual_duration /255)
		remainder = (actual_duration % 255)
		
	#add an exception, where the amount is exactly 255...
		if remainder == 0 then
			remainder = 255
			repeat_note -= 1
			end

	begin 
		if repeat_note == 0 then
			actual_duration = remainder
			else
			actual_duration = 255
		end
		
		       if (line_number==NOTES_PER_LINE) then
                   line_number = 0
                   end

		line_number==0?
		               print("\nDEFB "):
                               print(",")

		print "#{actual_duration},#{pitch}"
		
		line_number += 1
		repeat_note -= 1

	end until repeat_note < 0
	}
	
	event_stream.each do |event|
		# Calculate absolute microsecond time of the event
		delta_from_last_tempo_event = event.time_from_start - last_tempo_event.time_from_start
		current_microseconds_per_beat = last_tempo_event.tempo
		
		#beats_since_last_tempo_event = delta_from_last_tempo_event / seq.ppqn
		#microseconds_since_last_tempo_event = beats_since_last_tempo_event * current_microseconds_per_beat
		# -> refactored to avoid floating point division:
		microseconds_since_last_tempo_event = delta_from_last_tempo_event * current_microseconds_per_beat / seq.ppqn
		
		current_microsecond_time = last_tempo_event_microsecond_time + microseconds_since_last_tempo_event
		
		case event
			when MIDI::Tempo
				# Keep track of tempo changes so that we can calculate subsequent microsecond timings
				last_tempo_event = event
				last_tempo_event_microsecond_time = current_microsecond_time
			when MIDI::NoteOnEvent
				if last_note_on_event
					# insert a BEEP for the previous note, now we know how long it should be
					add_beep.call(current_microsecond_time)
				end
				last_note_on_event = event
				last_note_on_microsecond_time = current_microsecond_time
			when MIDI::NoteOffEvent
				# keep track of the last note off event, so that we can time the last note
				# of the track by it
				last_note_off_event = event
				last_note_off_microsecond_time = current_microsecond_time
		end
		
	end
	
	# add a beep for the final note
	if (last_note_on_event and last_note_off_event)
		add_beep.call(last_note_off_microsecond_time)
	end
}
puts ",0\n\nEND_OF_PROG:"
puts "; All done..."

