-- ZX Spectrum for Altera DE1
--
-- Copyright (c) 2009-2011 Mike Stirling
--
-- All rights reserved
--
-- Redistribution and use in source and synthezised forms, with or without
-- modification, are permitted provided that the following conditions are met:
--
-- * Redistributions of source code must retain the above copyright notice,
--   this list of conditions and the following disclaimer.
--
-- * Redistributions in synthesized form must reproduce the above copyright
--   notice, this list of conditions and the following disclaimer in the
--   documentation and/or other materials provided with the distribution.
--
-- * Neither the name of the author nor the names of other contributors may
--   be used to endorse or promote products derived from this software without
--   specific prior written agreement from the author.
--
-- * License is granted for non-commercial use only.  A fee may not be charged
--   for redistributions as source code or in synthesized/hardware form without 
--   specific prior written agreement from the author.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
-- THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
-- PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.
--
-- Sinclair ZX Spectrum
--
-- Terasic DE1 top-level
-- (C) 2011 Mike Stirling
--
-- DE2-115 Modification & some adjusments by Anton Vasilenok 2012:
--
-- Switches:
-- SW0/SW1 - Initial data for zxmmc
-- SW2/SW3 - CPU Speed selectors (00 - 1x, 10 - 2x, 01- 4x, 11 - 8x)
-- SW17 - Run/Stop CPU 
-- SW16 - Select ROM (00000h,10000h) (for memory test ROM)
-- SW15/SW14/SW13 - Enable/disable channels of YM2149
-- SW12 - turn on/off - LPF for YM2149
--
-- Keys:
-- KEY0 - Reset
-- KEY3 - NMI (haven't tested yet)


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

-- Generic top-level entity for Altera DE1 board
entity spectrum_de2_115 is
	
port (

	CLOCK_50	:	in	std_logic;
	--EXT_CLOCK	:	in	std_logic;
	
	-- Switches
	SW			:	in	std_logic_vector(17 downto 0);
	-- Buttons
	KEY			:	in	std_logic_vector(3 downto 0);
	
	-- 7 segment displays
	HEX0		:	out	std_logic_vector(6 downto 0);
	HEX1		:	out	std_logic_vector(6 downto 0);
	HEX2		:	out	std_logic_vector(6 downto 0);
	HEX3		:	out	std_logic_vector(6 downto 0);
	HEX4		:	out	std_logic_vector(6 downto 0);
	HEX5		:	out	std_logic_vector(6 downto 0);	
	HEX6		:	out	std_logic_vector(6 downto 0);
	HEX7		:	out	std_logic_vector(6 downto 0);	
	
	-- Red LEDs
	LEDR		:	out	std_logic_vector(17 downto 0);
	-- Green LEDs
	LEDG		:	out	std_logic_vector(8 downto 0);
	
	-- VGA
	VGA_R		:	out	std_logic_vector(7 downto 0);
	VGA_G		:	out	std_logic_vector(7 downto 0);
	VGA_B		:	out	std_logic_vector(7 downto 0);
	VGA_HS		:	out	std_logic;
	VGA_VS		:	out	std_logic;
	VGA_CLK		:	out	std_logic;
	
	-- Serial
--	UART_RXD	:	in	std_logic;
--	UART_TXD	:	out	std_logic;
	
	-- PS/2 Keyboard
	PS2_CLK		:	inout	std_logic;
	PS2_DAT		:	inout	std_logic;
	
	-- I2C
	I2C_SCLK	:	inout	std_logic;
	I2C_SDAT	:	inout	std_logic;
	
	-- Audio
	AUD_XCK		:	out		std_logic;
	AUD_BCLK	:	out		std_logic;
	AUD_ADCLRCK	:	out		std_logic;
	AUD_ADCDAT	:	in		std_logic;
	AUD_DACLRCK	:	out		std_logic;
	AUD_DACDAT	:	out		std_logic;
	
	-- SRAM
	SRAM_ADDR	:	out		std_logic_vector(19 downto 0);
	SRAM_DQ		:	inout	std_logic_vector(15 downto 0);
	SRAM_CE_N	:	out		std_logic;
	SRAM_OE_N	:	out		std_logic;
	SRAM_WE_N	:	out		std_logic;
	SRAM_UB_N	:	out		std_logic;
	SRAM_LB_N	:	out		std_logic;

	-- SDRAM
--	DRAM_ADDR	:	out		std_logic_vector(11 downto 0);
--	DRAM_DQ		:	inout	std_logic_vector(15 downto 0);
--	DRAM_BA_0	:	in		std_logic;
--	DRAM_BA_1	:	in		std_logic;
--	DRAM_CAS_N	:	in		std_logic;
--	DRAM_CKE	:	in		std_logic;
--	DRAM_CLK	:	in		std_logic;
--	DRAM_CS_N	:	in		std_logic;
--	DRAM_LDQM	:	in		std_logic;
--	DRAM_RAS_N	:	in		std_logic;
--	DRAM_UDQM	:	in		std_logic;
--	DRAM_WE_N	:	in		std_logic;

	-- Flash
	FL_ADDR		:	out		std_logic_vector(22 downto 0);
	FL_DQ		:	inout	std_logic_vector(7 downto 0);
	FL_RST_N	:	out		std_logic;
	FL_OE_N		:	out		std_logic;
	FL_WE_N		:	out		std_logic;
	FL_CE_N		:	out		std_logic;
	
	-- SD card (SPI mode)
--	SD_nCS		:	out		std_logic;
--	SD_MOSI		:	out		std_logic;
--	SD_SCLK		:	out		std_logic;
--	SD_MISO		:	in			std_logic

   SD_DAT	:	inout	std_logic_vector(3 downto 0); --	nCS	I	PP	Card Select (Neg True)
	SD_CMD	:	out	std_logic; --	2	DI		I	PP	Data In [MOSI]
	SD_CLK	:	out	std_logic --	5	CLK	I	PP	Clock [SCLK]


	);
end entity;

architecture rtl of spectrum_de2_115 is

--------------------------------
-- PLL
-- 50 MHz input
-- 28 MHz master clock output
-- 24 MHz audio clock output+
--------------------------------

component pll_main IS
	PORT
	(
		areset	: IN STD_LOGIC  := '0';
		inclk0	: IN STD_LOGIC  := '0';
		c0			: OUT STD_LOGIC ;
		c1			: OUT STD_LOGIC ;
		locked	: OUT STD_LOGIC 
	);
end component;

component zxram is
generic
(
   addr_lines : natural := 21
);
port (
	CLK				:	in std_logic;
	CLKEN_CPU		:	in std_logic;
	ADDR_CPU			:  in std_logic_vector(addr_lines-1 downto 0);
	DI_CPU			:  in std_logic_vector(7 downto 0);
	WR_N				:  in std_logic;
	DO_CPU			:  out std_logic_vector(7 downto 0);
	CLKEN_VIDEO		:	in std_logic;
	ADDR_VIDEO		:  in std_logic_vector(addr_lines-1 downto 0);
	DO_VIDEO			:  out std_logic_vector(7 downto 0);
	
	--- SRAM interface
	SRAM_ADDR	:	out		std_logic_vector(19 downto 0);
	SRAM_DQ		:	inout		std_logic_vector(15 downto 0);
	SRAM_CE_N	:	out		std_logic;
	SRAM_OE_N	:	out		std_logic;
	SRAM_WE_N	:	out		std_logic;
	SRAM_UB_N	:	out		std_logic;
	SRAM_LB_N	:	out		std_logic	
	);
end component;

-----------------------------
-- SEG 7 driver
-----------------------------

component seg7 is
port (
	D			: in std_logic_vector(3 downto 0);
	Q			: out std_logic_vector(6 downto 0)
);
end component;


-------------------
-- MMC interface
-------------------

component zxmmc is
port (
	CLOCK		:	in	std_logic;
	nRESET		:	in	std_logic;
	CLKEN		:	in	std_logic;
	
	-- Bus interface
	ENABLE		:	in	std_logic;
	-- 0 - W  - Card chip selects (active low)
	-- 1 - RW - SPI tx/rx data register
	-- 2 - Not used
	-- 3 - RW - Paging control register
	RS			:	in	std_logic_vector(1 downto 0);
	nWR			:	in	std_logic;
	DI			:	in	std_logic_vector(7 downto 0);
	DO			:	out	std_logic_vector(7 downto 0);
	
	-- SD card interface
	SD_CS0		:	out	std_logic;
	SD_CS1		:	out	std_logic;
	SD_CLK		:	out	std_logic;
	SD_MOSI		:	out	std_logic;
	SD_MISO		:	in	std_logic;
	
	-- Paging control for external RAM/ROM banks
	EXT_WR_EN	:	out	std_logic; -- Enable writes to external RAM/ROM
	EXT_RD_EN	:	out	std_logic; -- Enable reads from external RAM/ROM (overlay internal ROM)
	--EXT_ROM_nRAM	:	out	std_logic; -- Select external ROM or RAM banks
	EXT_BANK	:	out	std_logic_vector(5 downto 0); -- Selected bank number
	
	-- DIP switches (reset values for corresponding bits above)
	INIT_RD_EN	:	in	std_logic;
	INIT_ROM_nRAM	:	in	std_logic;
	
	DebugPagePort		: out std_logic_vector(7 downto 0)
	);
end component;


-------------------
-- Clock enables
-------------------

component clocks is
port (
	-- 28 MHz master clock
	CLK				:	in std_logic;
	-- Master reset
	nRESET			:	in std_logic;
	
	-- 1.75 MHz clock enable for sound
	CLKEN_PSG		:	out	std_logic;
	-- 3.5 MHz clock enable (1 in 8)
	CLKEN_CPU		:	out std_logic;
	CLKEN_CPU_IO   :  out std_logic;	
	-- 14 MHz clock enable (out of phase with CPU)
	CLKEN_VID		:	out std_logic;
	SpeedSW			:  in  std_logic_vector(1 downto 0);
	RunCPU			:  in  std_logic
	);
end component;


---------
-- CPU
---------

component T80se is
	generic(
		Mode : integer := 0;    -- 0 => Z80, 1 => Fast Z80, 2 => 8080, 3 => GB
		T2Write : integer := 0;  -- 0 => WR_n active in T3, /=0 => WR_n active in T2
		IOWait : integer := 1   -- 0 => Single cycle I/O, 1 => Std I/O cycle
	);
	port(
		RESET_n         : in  std_logic; -- �����^
		CLK_n           : in  std_logic; -- 
		CLKEN           : in  std_logic;
		WAIT_n          : in  std_logic; -- ������ ��������
		INT_n           : in  std_logic; -- ����������
		NMI_n           : in  std_logic; -- ����������
		BUSRQ_n         : in  std_logic;
		M1_n            : out std_logic;
		MREQ_n          : out std_logic;
		IORQ_n          : out std_logic;
		RD_n            : out std_logic;
		WR_n            : out std_logic;
		RFSH_n          : out std_logic;
		HALT_n          : out std_logic;
		BUSAK_n         : out std_logic;
		A               : out std_logic_vector(15 downto 0);
		DI              : in  std_logic_vector(7 downto 0);
		DO              : out std_logic_vector(7 downto 0)
	);
end component;

--------------
-- ULA port
--------------

component ula_port is
port (
	CLK		:	in	std_logic;
	nRESET	:	in	std_logic;
	
	-- CPU interface with separate read/write buses
	D_IN	:	in	std_logic_vector(7 downto 0);
	D_OUT	:	out	std_logic_vector(7 downto 0);
	ENABLE	:	in	std_logic;
	nWR		:	in	std_logic;
	
	BORDER_OUT	:	out	std_logic_vector(2 downto 0);
	EAR_OUT		:	out	std_logic;
	MIC_OUT		:	out std_logic;
	
	KEYB_IN		:	in 	std_logic_vector(4 downto 0);
	EAR_IN		:	in	std_logic	
	);
end component;

---------------
-- ULA video
---------------

component video is
port(
	-- Master clock (28 MHz)
	CLK			:	in std_logic;
	-- Video domain clock enable (14 MHz)
	CLKEN		:	in std_logic;
	-- Master reset
	nRESET 		: 	in std_logic;

	-- Mode
	VGA			:	in std_logic;

	-- Memory interface
	VID_A		:	out	std_logic_vector(12 downto 0);
	VID_D_IN	:	in	std_logic_vector(7 downto 0);
	nVID_RD	:	out	std_logic;
	nWAIT		:	out	std_logic;
	
	-- IO interface
	BORDER_IN	:	in	std_logic_vector(2 downto 0);

	-- Video outputs
	R			:	out	std_logic_vector(3 downto 0);
	G			:	out	std_logic_vector(3 downto 0);
	B			:	out	std_logic_vector(3 downto 0);
	nVSYNC		:	out std_logic;
	nHSYNC		:	out std_logic;
	nCSYNC		:	out	std_logic;
	nHCSYNC		:	out std_logic;
	IS_BORDER	: 	out std_logic;
	IS_VALID	:	out std_logic;
	
	-- Clock outputs, might be useful
	PIXCLK		:	out std_logic;
	FLASHCLK	: 	out std_logic;
	
	-- Interrupt to CPU (asserted for 32 T-states, 64 ticks)
	nIRQ		:	out	std_logic
);
end component;

--------------
-- Keyboard
--------------

component keyboard is
port (
	CLK			:	in	std_logic;
	nRESET		:	in	std_logic;

	-- PS/2 interface
	PS2_CLK		:	in	std_logic;
	PS2_DATA	:	in	std_logic;
	
	-- CPU address bus (row)
	A			:	in	std_logic_vector(15 downto 0);
	-- Column outputs to ULA
	KEYB		:	out	std_logic_vector(4 downto 0)
	);
end component;

-----------
-- Sound
-----------

component YM2149 is
  port (
  -- data bus
  I_DA                : in  std_logic_vector(7 downto 0);
  O_DA                : out std_logic_vector(7 downto 0);
  O_DA_OE_L           : out std_logic;
  -- control
  I_A9_L              : in  std_logic;
  I_A8                : in  std_logic;
  I_BDIR              : in  std_logic;
  I_BC2               : in  std_logic;
  I_BC1               : in  std_logic;
  I_SEL_L             : in  std_logic;

  O_AUDIO             : out std_logic_vector(7 downto 0);
  -- port a
  I_IOA               : in  std_logic_vector(7 downto 0);
  O_IOA               : out std_logic_vector(7 downto 0);
  O_IOA_OE_L          : out std_logic;
  -- port b
  I_IOB               : in  std_logic_vector(7 downto 0);
  O_IOB               : out std_logic_vector(7 downto 0);
  O_IOB_OE_L          : out std_logic;
  --
  ENA                 : in  std_logic;
  RESET_L             : in  std_logic;
  CLK                 : in  std_logic;
  I_ChannelEnable		 : in std_logic_vector(2 downto 0);
  LPF_Enable			 : in std_logic
  );
end component;

component i2s_intf is
generic(
	mclk_rate : positive := 12000000;
	sample_rate : positive := 8000;
	preamble : positive := 1; -- I2S
	word_length : positive := 16
	);

port (
	-- 2x MCLK in (e.g. 24 MHz for WM8731 USB mode)
	CLK			:	in	std_logic;
	nRESET		:	in	std_logic;
	
	-- Parallel IO
	PCM_INL		:	out	std_logic_vector(word_length - 1 downto 0);
	PCM_INR		:	out	std_logic_vector(word_length - 1 downto 0);
	PCM_OUTL	:	in	std_logic_vector(word_length - 1 downto 0);
	PCM_OUTR	:	in	std_logic_vector(word_length - 1 downto 0);
	
	-- Codec interface (right justified mode)
	-- MCLK is generated at half of the CLK input
	I2S_MCLK	:	out	std_logic;
	-- LRCLK is equal to the sample rate and is synchronous to
	-- MCLK.  It must be related to MCLK by the oversampling ratio
	-- given in the codec datasheet.
	I2S_LRCLK	:	out	std_logic;
	
	-- Data is shifted out on the falling edge of BCLK, sampled
	-- on the rising edge.  The bit rate is determined such that
	-- it is fast enough to fit preamble + word_length bits into
	-- each LRCLK half cycle.  The last cycle of each word may be 
	-- stretched to fit to LRCLK.  This is OK at least for the 
	-- WM8731 codec.
	-- The first falling edge of each timeslot is always synchronised
	-- with the LRCLK edge.
	I2S_BCLK	:	out	std_logic;
	-- Output bitstream
	I2S_DOUT	:	out	std_logic;
	-- Input bitstream
	I2S_DIN		:	in	std_logic
	);
end component;

component i2c_loader is
generic (
	-- Address of slave to be loaded
	device_address : integer := 16#1a#;
	-- Number of retries to allow before stopping
	num_retries : integer := 0;
	-- Length of clock divider in bits.  Resulting bus frequency is
	-- CLK/2^(log2_divider + 2)
	log2_divider : integer := 6
);

port (
	CLK			:	in	std_logic;
	nRESET		:	in	std_logic;
	
	I2C_SCL		:	inout	std_logic;
	I2C_SDA		:	inout	std_logic;
	
	IS_DONE		:	out std_logic;
	IS_ERROR	:	out	std_logic
	);
end component;

-------------
-- Signals
-------------

-- Master clock - 28 MHz
signal pll_reset		:	std_logic;
signal pll_locked		:	std_logic;
signal clock			:	std_logic;
signal audio_clock		:	std_logic;
signal reset_n			:	std_logic;
signal reset_n_video	:	std_logic;

-- Clock control
signal psg_clken		:	std_logic;
signal cpu_clken		:	std_logic;
signal vid_clken		:	std_logic;
signal cpu_io_clken  :	std_logic;

-- Address decoding
signal ula_select		:	std_logic; -- all even IO addresses
signal Bank00_select		:	std_logic; -- 0x0000-0x3FFF
signal BankXX_select		:	std_logic; -- 0x4000-0xFFFF
signal psg_select		:	std_logic;

-- SRAM bus high/low byte mux
signal sram_di			:	std_logic_vector(7 downto 0);
signal sram_wr_n		:  std_logic;


-- CPU signals
signal cpu_wait_n	:	std_logic;
signal cpu_irq_n	:	std_logic;
signal cpu_nmi_n	:	std_logic;
signal cpu_busreq_n	:	std_logic;
signal cpu_m1_n		:	std_logic;
signal cpu_mreq_n	:	std_logic;
signal cpu_ioreq_n	:	std_logic;
signal cpu_rd_n		:	std_logic;
signal cpu_wr_n		:	std_logic;
signal cpu_rfsh_n	:	std_logic;
signal cpu_halt_n	:	std_logic;
signal cpu_busack_n	:	std_logic;
signal cpu_a		:	std_logic_vector(15 downto 0);
signal cpu_di		:	std_logic_vector(7 downto 0);
signal cpu_do		:	std_logic_vector(7 downto 0);

-- ULA port signals
signal ula_do		:	std_logic_vector(7 downto 0);
signal ula_border	:	std_logic_vector(2 downto 0);
signal ula_ear_out	:	std_logic;
signal ula_mic_out	:	std_logic;
signal ula_ear_in	:	std_logic;
signal ula_rom_sel	:	std_logic;
signal ula_shadow_vid	:	std_logic;
signal ula_ram_page	:	std_logic_vector(2 downto 0);

-- ULA video signals
signal vid_a		:	std_logic_vector(12 downto 0);
signal vid_di		:	std_logic_vector(7 downto 0);
signal vid_rd_n		:	std_logic;
signal vid_wait_n	:	std_logic;
signal vid_r_out	:	std_logic_vector(3 downto 0);
signal vid_g_out	:	std_logic_vector(3 downto 0);
signal vid_b_out	:	std_logic_vector(3 downto 0);
signal vid_vsync_n	:	std_logic;
signal vid_hsync_n	:	std_logic;
signal vid_csync_n	:	std_logic;
signal vid_hcsync_n	:	std_logic;
signal vid_is_border	:	std_logic;
signal vid_is_valid	:	std_logic;
signal vid_pixclk	:	std_logic;
signal vid_flashclk	:	std_logic;
signal vid_irq_n	:	std_logic;

-- Keyboard
signal keyb			:	std_logic_vector(4 downto 0);

-- Sound (PSG default values for systems that don't have it)
signal psg_do		:	std_logic_vector(7 downto 0) := "11111111";
signal psg_bdir		:	std_logic;
signal psg_bc1		:	std_logic;
signal psg_aout		:	std_logic_vector(7 downto 0) := "00000000";
signal pcm_lrclk	:	std_logic;
signal pcm_outl		:	std_logic_vector(15 downto 0);
signal pcm_outr		:	std_logic_vector(15 downto 0);
signal pcm_inl		:	std_logic_vector(15 downto 0);
signal pcm_inr		:	std_logic_vector(15 downto 0);

-- ZX RAM
signal zxram_do_cpu     : std_logic_vector(7 downto 0);
signal zxram_addr_video : std_logic_vector(15 downto 0);



-- 128K paging register (with default values for systems that don't have it)
signal zx48mode				: std_logic := '0';    
-- signal rom_page			 	: std_logic_vector(1 downto 0) := "00";
-- signal rom_page_disabled	: std_logic := '0';
-- signal video_page				: std_logic := '0'; -- 0 - Bank 5 / 1 - Bank 7
-- signal ram_page				:	std_logic_vector(2 downto 0) := "000";
signal current_ram_page		:	std_logic_vector(6 downto 0);

signal zx3plus_special_mode_enable : std_logic := '0';
-- signal zx3plus_special_mode_scheme : std_logic_vector(1 downto 0) := "00";

signal port7FFD_select 	: std_logic;
signal port1FFD_select 	: std_logic;

-- New Bank Selector
-- Definitions:
-- Banks - are the 16kB parts of addressed by CPU adresses
--         There are 4 banks: Bank00 Bank01 Bank10 Bank11
-- Pages - are the 16kB memory blocks
-- for 2MB of RAM: 128 Pages by 16kB
signal BankROM   : std_logic_vector(1 downto 0) := "00";
signal Bank00    : std_logic_vector(6 downto 0) := "0000000"; -- undef
signal Bank01    : std_logic_vector(6 downto 0) := "0000101"; -- Bank 5
signal Bank10    : std_logic_vector(6 downto 0) := "0000010"; -- Bank 2
signal Bank11    : std_logic_vector(6 downto 0) := "0000000"; -- Bank 0
signal BankVideo : std_logic_vector(6 downto 0) := "0000101"; -- Bank 5
signal Bank00enable : std_logic := '0'; -- Enabling Bank00 instead of ROM
--signal Bank00wp     : std_logic := '1'; -- Write protection for Bank00

-- ZXMMC
signal zxmmc_select		:	std_logic; -- A4-A0 set
signal zxmmc_do			:	std_logic_vector(7 downto 0);

-- ZXMMC+ external ROM/RAM interface (for ResiDOS)
signal zxmmc_wr_en		:	std_logic;
signal zxmmc_rd_en		:	std_logic;
--signal zxmmc_rom_nram	:	std_logic;
signal zxmmc_bank			:	std_logic_vector(5 downto 0);

signal zxmmc_sclk	:	std_logic;
signal zxmmc_mosi	:	std_logic;
signal zxmmc_miso	:	std_logic;
signal zxmmc_cs0	:	std_logic;

signal port7ffd_data : std_logic_vector(7 downto 0);
signal port1ffd_data : std_logic_vector(7 downto 0);

---- for my debugging...
signal DebugPagePort		: std_logic_vector(7 downto 0);

--signal debugMem      : std_logic_vector(7 downto 0);

begin
	-- 28 MHz master clock
	pll: pll_main port map (
		pll_reset,
		CLOCK_50,
		clock,
		audio_clock,
		pll_locked
		);
	
	ram1: zxram port map (
  	   clock,cpu_clken,
		current_ram_page & cpu_a(13 downto 0), 
		cpu_do, sram_wr_n, --cpu_wr_n, 
		zxram_do_cpu,
		vid_clken,
		BankVideo & "0" & vid_a,
		vid_di, --zxram_do_video
		
		SRAM_ADDR,
		SRAM_DQ,
		SRAM_CE_N,
		SRAM_OE_N,
		SRAM_WE_N,
		SRAM_UB_N,
		SRAM_LB_N
		);
		
	current_ram_page <= Bank00 when cpu_a(15 downto 14)="00" else 
							  Bank01 when cpu_a(15 downto 14)="01" else 
							  Bank10 when cpu_a(15 downto 14)="10" else 
							  Bank11;
			
--	current_ram_page <= ram_page when (zx3plus_special_mode_enable = '0' and zxram_addr_cpu(15 downto 14) = "11") else
--			zxram_addr_cpu(14) & zxram_addr_cpu(15 downto 14) when zx3plus_special_mode_enable = '0' else
--			"0" & zxram_addr_cpu(15 downto 14) when zx3plus_special_mode_enable = '1' and zx3plus_special_mode_scheme = "00" else
--			"1" & zxram_addr_cpu(15 downto 14) when zx3plus_special_mode_enable = '1' and zx3plus_special_mode_scheme = "01" else
--			(not(zxram_addr_cpu(15) and zxram_addr_cpu(14))) & zxram_addr_cpu(15 downto 14) when zx3plus_special_mode_enable = '1' and zx3plus_special_mode_scheme = "10" else
--			(not(zxram_addr_cpu(15) and zxram_addr_cpu(14))) & (zxram_addr_cpu(15) or zxram_addr_cpu(14)) & zxram_addr_cpu(14);
--
--	current_ram_page2 <= "0000" & current_ram_page when (Bank00_select='0' or zx3plus_special_mode_enable = '1') else
--							   "01" & zxmmc_bank(4 downto 0);
		
	-- Clock enable logic
	clken: clocks port map (
		clock,
		reset_n_video,
		psg_clken,
		cpu_clken,
		cpu_io_clken,
		vid_clken,
		SW(3)&SW(2),
		SW(17)
		);
		
	-- CPU
	cpu: T80se port map (
		reset_n,
		clock, cpu_clken,
		cpu_wait_n,
		cpu_irq_n, 
		KEY(3) and cpu_nmi_n,
		cpu_busreq_n, cpu_m1_n,
		cpu_mreq_n, cpu_ioreq_n,
		cpu_rd_n, cpu_wr_n,
		cpu_rfsh_n, cpu_halt_n, cpu_busack_n,
		cpu_a, cpu_di, cpu_do
		);
	
   LEDG(0) <= not cpu_halt_n;		
	LEDG(1) <= not cpu_irq_n;
	LEDG(2) <= not cpu_nmi_n;
	LEDG(3) <= not cpu_rfsh_n;			
		
		
	-- VSYNC interrupt routed to CPU
	cpu_irq_n <= vid_irq_n;
	-- Unused CPU input signals
	cpu_wait_n <= '1';
	cpu_nmi_n <= '1';
	cpu_busreq_n <= '1';
		
	-- Keyboard
	kb:	keyboard port map (
		clock, reset_n,
		PS2_CLK, PS2_DAT,
		cpu_a, keyb
		);
		
	-- ULA port
	ula: ula_port port map (
		clock, reset_n,
		cpu_do, ula_do,
		ula_select, cpu_wr_n,
		ula_border,
		ula_ear_out, ula_mic_out,
		keyb,
		ula_ear_in
		);
		
	-- ULA video
	vid: video port map (
		clock, vid_clken, reset_n_video,
		'1', -- SW(7),
		vid_a, vid_di, vid_rd_n, vid_wait_n,
		ula_border,
		vid_r_out, vid_g_out, vid_b_out,
		vid_vsync_n, vid_hsync_n,
		vid_csync_n, vid_hcsync_n,
		vid_is_border, vid_is_valid,
		vid_pixclk, vid_flashclk,
		vid_irq_n
		);
		
	
	i2s: i2s_intf port map (
		audio_clock, reset_n,
		pcm_inl, pcm_inr,
		pcm_outl, pcm_outr,
		AUD_XCK, pcm_lrclk,
		AUD_BCLK, AUD_DACDAT, AUD_ADCDAT
		);
	AUD_DACLRCK <= pcm_lrclk;
	AUD_ADCLRCK <= pcm_lrclk;
	
	i2c: i2c_loader 
		generic map (
			log2_divider => 7
		)
		port map (
			clock, reset_n,
			I2C_SCLK, I2C_SDAT,
			LEDR(1), -- IS_DONE
			LEDR(0) -- IS_ERROR
		);

	psg : YM2149 port map (
		cpu_do, psg_do, open,
		'0', -- /A9 pulled down internally
		'1', -- A8 pulled up on Spectrum
		psg_bdir,
		'1', -- BC2 pulled up on Spectrum
		psg_bc1,
		'1', -- /SEL is high for AY-3-8912 compatibility
		psg_aout,
		(others => '0'), open, open, -- port A unused (keypad and serial on Spectrum 128K)
		(others => '0'), open, open, -- port B unused (non-existent on AY-3-8912)
		psg_clken,
		reset_n,
		clock,
		SW(15)&SW(14)&SW(13),
		SW(12)
		);

	psg_bdir <= psg_select and cpu_rd_n;
	psg_bc1 <= psg_select and cpu_a(14);

	-- ZXMMC interface
	mmc: zxmmc port map (
		clock, reset_n, cpu_clken,
		zxmmc_select, cpu_a(6 downto 5), -- A6/A5 selects register
		cpu_wr_n, cpu_do, zxmmc_do,
		zxmmc_cs0, open,
		zxmmc_sclk, zxmmc_mosi, zxmmc_miso,
		zxmmc_wr_en, zxmmc_rd_en, -- zxmmc_rom_nram,
		zxmmc_bank,
		SW(1), SW(0),
		DebugPagePort
		);
	SD_DAT(3) <= zxmmc_cs0;
	SD_CLK <= zxmmc_sclk;
	SD_CMD <= zxmmc_mosi;
	zxmmc_miso <= SD_DAT(0);
	SD_DAT(0) <= 'Z';


		
-- SD_DAT(3)	:	out	std_logic; --	nCS	I	PP	Card Select (Neg True)
--	SD_CMD		:	out	std_logic; --	2	DI		I	PP	Data In [MOSI]
--	SD_CLK		:	out	std_logic; --	5	CLK	I	PP	Clock [SCLK]
-- SD_DAT(0)	:	in		std_logic; --7	DO		O	PP	Data Out [MISO]

		
--Pin	Name	I/O	Logic	Description
--SD_DAT(3)		1	nCS	I	PP	Card Select (Neg True)
--SD_CMD			2	DI		I	PP	Data In [MOSI]
--GND				3	VSS	S	S	Ground
--VCC				4	VDD	S	S	Power
--SD_CLK			5	CLK	I	PP	Clock [SCLK]
--GND				6	VSS	S	S	Ground
--SD_DAT(0)		7	DO		O	PP	Data Out [MISO]
--SD_DAT(1)		8	nIRQ	O	Interrupt (SDIO Cards)
	
	
	-- Asynchronous reset
	-- PLL is reset by external reset switch
	pll_reset <= '0';
	
	-- System is reset by external reset switch or PLL being out of lock
	reset_n 			<= pll_locked and KEY(0);
	reset_n_video  <= pll_locked;
	
	-- Address decoding.  Z80 has separate IO and memory address space
	-- IO ports (nominal addresses - incompletely decoded):
	-- 0xXXFE R/W = ULA
	-- 0x7FFD W   = 128K paging register
	-- 0xFFFD W   = 128K AY-3-8912 register select
	-- 0xFFFD R   = 128K AY-3-8912 register read
	-- 0xBFFD W   = 128K AY-3-8912 register write
	-- 0x1FFD W   = +3 paging and control register
	-- 0x2FFD R   = +3 FDC status register
	-- 0x3FFD R/W = +3 FDC data register
	-- 0xXXXF R/W = ZXMMC interface
	-- FIXME: Revisit this - could be neater
	
	ula_select <= (not cpu_ioreq_n) and not cpu_a(0); -- all even IO addresses
	psg_select <= (not cpu_ioreq_n) and cpu_a(0) and cpu_a(15) and not cpu_a(1);
	
	
	--	Port 0x7ffd behaves in the almost exactly the same way as on the 128K/+2, with two exceptions:
	--	
	--	Bit 4 is now the low bit of the ROM selection.
	--	The partial decoding used is now slightly different: the hardware will respond only to those port addresses with bit 1 reset, bit 14 set and bit 15 reset (as opposed to just bits 1 and 15 reset on the 128K/+2).
	--	The extra paging features of the +2A/+3 are controlled by port 0x1ffd (again, partial decoding applies here: the hardware will respond to all port addresses with bit 1 reset, bit 12 set and bits 13, 14 and 15 reset). This port is also write-only, and its last value should be saved at 0x5b67 (23399).
	--	
	--	Port 0x1ffd responds as follows:
	--	
	--	  Bit 0: Paging mode. 0=normal, 1=special
	--	  Bit 1: In normal mode, ignored.
	--	  Bit 2: In normal mode, high bit of ROM selection. The four ROMs are:
	--				 ROM 0: 128k editor, menu system and self-test program
	--				 ROM 1: 128k syntax checker
	--				 ROM 2: +3DOS
	--				 ROM 3: 48 BASIC
	--	  Bit 3: Disk motor; 1=on, 0=off
	--	  Bit 4: Printer port strobe.	
	
	
	port7FFD_select <= (not cpu_ioreq_n) and cpu_a(0) and cpu_a(14) and not (cpu_a(15) or cpu_a(1));
	port1FFD_select <= (not cpu_ioreq_n) and cpu_a(0) and cpu_a(12) and not (cpu_a(15) or cpu_a(14) or cpu_a(13) or cpu_a(1));	

	zxmmc_select    <= (not cpu_ioreq_n) and cpu_a(4) and cpu_a(3) and cpu_a(2) and cpu_a(1) and cpu_a(0);	
	
--signal zx48mode				: std_logic := '0';    
--signal rom_page			 	: std_logic_vector(1 downto 0) := "00";
--signal rom_page_disabled	: std_logic := '0';
--signal video_page				: std_logic := '0'; -- 0 - Bank 5 / 1 - Bank 7
--signal ram_page			:	std_logic_vector(2 downto 0) := "000";
--signal zx3plus_special_mode_enable : std_logic := '0';
--signal zx3plus_special_mode_scheme : std_logic_vector(1 downto 0) := "00";


	process(clock,reset_n)
		begin
			if reset_n = '0' then
				zx48mode <= '0';
				
				BankROM <= "00";
--				Bank00  <= "0000000"; -- undef
--				Bank01  <= "0000101"; -- Bank 5
--				Bank10  <= "0000010"; -- Bank 2
--				Bank11  <= "0000000"; -- Bank 0
				BankVideo <=  "0000101"; -- Bank 5
				
				port7ffd_data <= "00000000";
				port1ffd_data <= "00000000";
				zx3plus_special_mode_enable <= '0';
				
--				Bank00enable <= '0'; -- Enabling Bank00 instead of ROM
--				Bank00wp     <= '1'; -- Write protection for Bank00				
				

			elsif rising_edge(clock) then
			
				if port7FFD_select = '1' and zx48mode = '0' and cpu_wr_n = '0' then
					port7ffd_data <= cpu_do;
					zx48mode    <= cpu_do(5);
					BankROM(0)  <= cpu_do(4);
					BankVideo   <= "00001" & cpu_do(3) & "1"; -- 5 or 7 page
				end if;
				
				if port1FFD_select = '1' and zx48mode = '0' and cpu_wr_n = '0' then
					port1ffd_data <= cpu_do;
					zx3plus_special_mode_enable <= cpu_do(0);
					BankROM(1)  <= cpu_do(2); -- if not special mode
				end if;
				
				if zxmmc_rd_en = '1' then 
					Bank00 <= "1" & zxmmc_bank;
				elsif port1ffd_data(2 downto 1) = "00" and zx3plus_special_mode_enable = '1' then 
					Bank00 <= "0000000";
				else 
					Bank00 <= "0000100";
				end if;
				
				if zx3plus_special_mode_enable = '0' then
					Bank01 <= "0000101";
				elsif port1ffd_data(2 downto 1) = "00" then
					Bank01 <= "0000001";
				elsif port1ffd_data(2 downto 1) = "11" then
					Bank01 <= "0000111";
				else
					Bank01 <= "0000101";
				end if;
				
				if zx3plus_special_mode_enable = '0' then
					Bank10 <= "0000010" ;
				elsif port1ffd_data(2 downto 1) = "00" then
					Bank10 <= "0000010";
				else
					Bank10 <= "0000110";
				end if;
					
				if zx3plus_special_mode_enable = '0' then
					Bank11 <= "0000" & port7ffd_data(2 downto 0);
				elsif port1ffd_data(2 downto 1) = "01" then
					Bank11 <= "0000111";
				else
					Bank11 <= "0000011";
				end if;
				
				
				
			end if;
		end process;	
	

--	Bank00 <= "1" & zxmmc_bank when zxmmc_rd_en = '1'     else
--				 "0000000"        when port1ffd_data(2 downto 1) = "00" and zx3plus_special_mode_enable = '1' else
--				 "0000100";
--				 
--	Bank01 <= "0000101"        when "0000101"; else
--				 "0000001"        when port1ffd_data(2 downto 1) = "00"  else -- special mode
--				 "0000111"        when port1ffd_data(2 downto 1) = "11"  else -- special mode
--				 "0000101";
--				 
--	Bank10 <= "0000010"        when zx3plus_special_mode_enable = '0' else
--				 "0000010"        when port1ffd_data(2 downto 1) = "00"  else -- special mode
--				 "0000110";
--				
--	Bank11 <= "0000" & port7ffd_data(2 downto 0) when zx3plus_special_mode_enable = '0' else
--				 "0000111"        when port1ffd_data(2 downto 1) = "01"  else -- special mode
--				 "0000011";				
	

				 
	
	
	Bank00_select <= not (cpu_mreq_n or cpu_a(15) or cpu_a(14));
--	-- RAM is enabled for any memory request when ROM isn't enabled
	BankXX_select <= not (cpu_mreq_n or Bank00_select);

	Bank00enable <= zx3plus_special_mode_enable or zxmmc_rd_en;
	
	
--		ext_ram_write := (rom_enable and zxmmc_wr_en and not zxmmc_rom_nram) and not cpu_wr_n;
--		int_ram_write := ram_enable and not cpu_wr_n;
--		-sram_write := int_ram_write or ext_ram_write;	
	
	sram_wr_n  <= cpu_wr_n or (not (BankXX_select or (Bank00_select and zx3plus_special_mode_enable) or (Bank00_select and zxmmc_wr_en)));
	
	-- CPU data bus mux
	cpu_di <=
		-- System RAM
		zxram_do_cpu when BankXX_select = '1' else
		zxram_do_cpu when Bank00_select = '1' and Bank00enable  = '1' else
		
		FL_DQ        when Bank00_select = '1' else
		-- IO ports
		ula_do       when ula_select = '1'    else
		psg_do       when psg_select = '1'    else
		zxmmc_do     when zxmmc_select = '1'  else
		vid_di       when cpu_a(0) = '1' and cpu_ioreq_n = '0' else -- port 0ffh :-)
		-- Idle bus
		(others => '1');
	
	-- ROMs are in external flash starting at 0x20000
	-- (lower addresses contain the BBC ROMs)
	FL_RST_N <= reset_n;
	FL_CE_N <= '0';
	FL_OE_N <= '0';
	FL_WE_N <= '1';
	
	
	FL_ADDR <= "000000" & SW(16) & BankROM & cpu_a(13 downto 0);

	sg4_do: seg7 port map (port7ffd_data(3 downto 0), HEX4);
	sg5_do: seg7 port map (port7ffd_data(7 downto 4), HEX5);
	sg6_dq: seg7 port map (port1ffd_data(3 downto 0), HEX6);
	sg7_dq: seg7 port map (port1ffd_data(7 downto 4), HEX7);


	sg0_dq: seg7 port map (DebugPagePort(3 downto 0), HEX3);
	sg1_dq: seg7 port map (DebugPagePort(7 downto 4), HEX2);

	sg0_a: seg7 port map (cpu_a(3 downto 0), HEX0);
	sg1_a: seg7 port map (cpu_a(7 downto 4), HEX1);
--	sg2_a: seg7 port map (cpu_a(11 downto 8), HEX2);
--	sg3_a: seg7 port map (cpu_a(15 downto 12), HEX3);
	


	-- Connect audio to PCM interface
--	pcm_outl <= psg_aout(7) & (psg_aout(6) or (ula_ear_out xor ula_mic_out)) & psg_aout(5 downto 0) & "00000000";
--	pcm_outr <= psg_aout(7) & (psg_aout(6) or (ula_ear_out xor ula_mic_out)) & psg_aout(5 downto 0) & "00000000";

	pcm_outl <= ula_ear_out & psg_aout(7 downto 0) & "0000000";
	pcm_outr <= ula_ear_out & psg_aout(7 downto 0) & "0000000";
	
	-- Hysteresis for EAR input (should help reliability)
	process(clock)
	variable in_val : integer;
	begin
		in_val := to_integer(signed(pcm_inl));
		
		if rising_edge(clock) then
			if in_val < -15 then
				ula_ear_in <= '0';
			elsif in_val > 15 then
				ula_ear_in <= '1';
			end if;
		end if;
	end process;
	
	-- Connect ULA to video output
	VGA_R(7 downto 4) <= vid_r_out (3 downto 0);
	VGA_R(3 downto 0) <= "0000";
	VGA_G(7 downto 4) <= vid_g_out (3 downto 0);
	VGA_G(3 downto 0) <= "0000";
	VGA_B(7 downto 4) <= vid_b_out (3 downto 0);	
	VGA_B(3 downto 0) <= "0000";
	VGA_HS <= vid_hcsync_n;
	VGA_VS <= vid_vsync_n;
	VGA_CLK <= vid_pixclk;	
	
end architecture;
