#include <windows.h>
#include "..\\dtr.hpp"

// detector functions
bool search_with_table( const byte* data_ptr,  const byte* table_ptr )
{
  while( *table_ptr != 0xff )
  {
    if( *table_ptr == 0xfe )
    {
      ++table_ptr;
      data_ptr += *table_ptr++;
      continue;
    }

    if( *table_ptr == 0xfd )
    {
      ++table_ptr;
      data_ptr += 2;
    continue;
    }
    if( *data_ptr++ != *table_ptr++ ) return false;
  }
  return true;
}

bool is_pt1( const byte* data, int size )
{
  byte pt1_player[] = { 0x21,0xFD,0xC3,0xFD,0xCD,0xFD,0xC3,0xFF };
  if( search_with_table(data, pt1_player) ) return true;

  byte delay         = data[0];
  byte music_length  = data[1];
  byte loop_position = data[2];

  word pattern_table_offset = *(word*)(data+0x43);
  word position_list_offset = 0x64;

  if(delay < 3 || delay > 0x0f) return false;

  if(music_length < loop_position) return false;

  if(pattern_table_offset != position_list_offset + music_length) return false;

  if( data[pattern_table_offset - 1] != 0xff ) return false;

  return true;
}

bool is_pt2( const byte* data, int size )
{
  byte pt20_player[] = { 0x21,0xFD,0xC3,0xFD,0xC3,0xFE,0xFF,0xFE,0xFF,0xFE,53,0xF3,0xE5,0x22,0xFD,0xE5,0x7E,0x32,0xFF };
  if( search_with_table(data, pt20_player) ) return true;

  byte pt21_player[] = { 0x21,0xFD,0xC3,0xFD,0xC3,0xFE,0x0E,0x32,0x2E,0x31,0xFF };
  if( search_with_table(data, pt21_player) ) return true;

  byte pt24_player[] = { 0x21,0xFD,0x18,3,0xC3,0xFD,0xF3,0xE5,0x7E,0x32,0xFD,0x32,0xFD,0x23,0x23,0x7E,0x23,0xFF };
  if( search_with_table(data, pt24_player) ) return true;

  byte delay         = data[0];
  byte music_length  = data[1];
  byte loop_position = data[2];

  word pattern_table_offset = *(word*)(data+0x63);
  word position_list_offset = 0x84;

  if(delay < 3 || delay > 0x0f) return false;

  if(music_length < loop_position) return false;

  if(pattern_table_offset != position_list_offset + music_length) return false;

  if( data[pattern_table_offset - 1] != 0xff ) return false;

  return true;
}

bool is_pt3( const byte* data, int size )
{
  byte pt3x_player[] = { 0x21,0xFD,0x18,0xFE,0x01,0xC3,0xFD,0xC3,0xFD,0xFE,33,0xF3,0x22,0xFD,0x22,0xFD,0x22,0xFD,0x22,0xFD,0x01,0x64,0x00,0x09,0xFF };
  if( search_with_table(data, pt3x_player) ) return true;

  byte pt35x_player[] = { 0x21,0xFD,0x18,0xFE,0x01,0xC3,0xFD,0xC3,0xFD,0xFE,35,0xF3,0xED,0x73,0xFD,0x22,0xFD,0x22,0xFD,0x22,0xFD,0x22,0xFD,0x01,0x64,0x00,0x09,0xFF };
  if( search_with_table(data, pt35x_player) ) return true;

  byte pt36x_player[] = { 0x21,0xFD,0x18,0xFE,0x01,0xC3,0xFD,0x18,0xFE,0x01,0xF3,0xED,0x73,0xFD,0x22,0xFF };
  if( search_with_table(data, pt36x_player) ) return true;

  byte delay         = data[0x64];
  byte music_length  = data[101];

  word pattern_table_offset = *(word*)(data+103);
  word position_list_offset = 0xC9;

  if(delay < 1 || delay >= 0x40) return false;
  if( music_length == 0 ) return false;

  const byte* position_list = data + position_list_offset;
  word i = 0;
  for( ; i < music_length; ++i )
  {
    //if( position_list[i] == 0xff )  break;
    if( position_list[i] > 0x30*3 ) return false;
    if( position_list[i] % 3 ) return false;
  }
  if( position_list[i] != 0xff ) return false;

  return true;
}

bool is_asm1xx( const byte* data, int size )
{
  byte asm200_player[] = { 0x18,0x5C,0x18,0x51,0x18,0x3F,0xFE,63,0x16,0x0D,0x01,0xFD,0xED,0x51,0x06,0xBF,0xED,0x71,0xFF };
  if( search_with_table(data, asm200_player) ) return true;

  byte asm112_player[] = { 0xFE,0x0B,0x18,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xCD,0x52,0x00,0xFD,0xFD,0x71,0xC6,0xFF };
  if( search_with_table(data, asm112_player) ) return true;

  byte asm112_2_player[] = { 0xFE,0x0B,0x18,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xCD,0x52,0,0xFD,0xFD,0xDF,0x04,0xFF };
  if( search_with_table(data, asm112_2_player) ) return true;

  byte asm111_player[] = { 0xFE,0x0B,0x18,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xCD,0x52,0x00,0xFD,0xFD,0x17,0xC6,0xFF };
  if( search_with_table(data, asm111_player) ) return true;

  byte asm111_2_player[] = { 0xFE,0x0B,0x18,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xCD,0x52,0x00,0xFD,0xFD,0x8D,0x04,0xFF };
  if( search_with_table(data, asm111_2_player) ) return true;

  byte asm110_player[] = { 0xFE,0x0B,0xC3,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xAF,0x21,0xFE,0x1C,0xF9,0xC5,0xFF };
  if( search_with_table(data, asm110_player) ) return true;

  byte asm103_player[] = { 0xFE,0x0B,0xC3,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xAF,0x21,0xFE,0x1C,0xCC,0xC4,0xFF };
  if( search_with_table(data, asm103_player) ) return true;

  if(data[0] < 2 || data[0] >= 0x33) return false;
  if(data[1] >= 100) return false;
  if(data[8] == 0 || data[8] > 100) return false;

  word size1 = *(word*)(data+2);
  if(size1 < 9) return false;
  word size2 = *(word*)(data+4);
  if(size2 < 0x18) return false;
  word size3 = *(word*)(data+6);
  if(size3 < 0x8B) return false;

  if(size2 - 1 > size) return false;
  if(data[size2-1] != 0xFF) return false;

  if(size1 >= size2) return false;
  if(size2 >= size3) return false;

  for(int i = 0; i < data[8]; ++i) if(data[9+i] >= 0x20) return false;

  return true;
}

bool is_asm012( const byte* data, int size )
{
  byte asm0xx_player[] = { 0xFE,0x0B,0xC3,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xAF,0x21,0xFE,0x1C,0x78,0xC4,0xFF };
  if( search_with_table(data, asm0xx_player) ) return true;

  byte asm012_player[] = { 0xFE,0x0B,0xC3,0xFD,0xC3,0xFD,0xC3,0xFE,0x41,0xAF,0x21,0xFE,0x1C,0xA7,0xC4,0xFF };
  if( search_with_table(data, asm012_player) ) return true;

  if(data[0] < 2 || data[0] >= 0x33) return false;
  if(data[7] >= 0x64) return false;

  WORD size1 = *(WORD*)(data+1);
  if(size1 < 8) return false;
  WORD size2 = *(WORD*)(data+3);
  if(size2 < 0x18) return false;
  WORD size3 = *(WORD*)(data+5);
  if(size3 < 0x87) return false;

  if(size2 - 1 > size) return false;
  if(data[size2-1] != 0xFF) return false;

  if(size1 >= size2) return false;
  if(size2 >= size3) return false;

  if(8 + data[7] != size1) return false;

  return true;
}

bool is_sqt( const byte* data, int size )
{
  byte sqt_player[] = { 0x3E,0xFE,0x01,0x32,0xFD,0x32,0xFD,0x32,0xFD,0x01,0x01,0x01,0xCD,0xFD,0x2A,0xFD,0xDD,0x21,0xFD,0xCD,0xFD,0xCD,0xFD,0xCD,0xFD,0x11,0xFD,0xCD,0xFD,0x1E,0x00,0x14,0x7A,0xFD,0x20,0xF5,0xC9,0x21,0xFF };
  if( search_with_table(data, sqt_player) ) return true;

  if(data[3] < 0x40) return false;

  WORD size1 = *(WORD*)(data+0);
  WORD size2 = *(WORD*)(data+2);
  WORD size3 = *(WORD*)(data+6);
  WORD size4 = *(WORD*)(data+8);
  WORD size5 = *(WORD*)(data+10);

  WORD bc  = size2 - 10;
  WORD sq1 = size1 + bc;
  WORD hl  = size1 + size2 - 10;
  if(hl < size4) return false;
  hl -= size4;
  if(hl%7) return false;
  if(hl/7 == 0 || hl/7 >= 121) return false;
  if(sq1 < size5) return false;
  if(size4 > size5) return false;
  if(size3+2 < size2) return false;

  return true;
}

bool is_stp( const byte* data, int size )
{
  byte stp_player[] = { 0x21,0xFD,0xC3,0xFD,0xC3,0xFD,0xED,0xFE,0x03,0xC3,0xFF };
  if( search_with_table(data, stp_player) ) return true;

  byte stp20_player[] = { 0xC3,0xFD,0xC3,0xFD,0xFE,0x48-0x06,0xF3,0x21,0xFD,0x22,0xFD,0x3E,0xFE,1,0x32,0xFD,0x32,0xFD,0x32,0xFD,0x7E,0x23,0x32,0xFD,0xCD,0xFD,0x7E,0xFF };
  if( search_with_table(data, stp20_player) ) return true;

  byte delay = data[0];

  if(delay < 3 || delay > 0x0f) return false;

  word position_list_offset = *(word*)(data+1);
  word ornament_list_offset = *(word*)(data+3);
  word pattern_list_offset  = *(word*)(data+5);
  word samples_list_offset  = *(word*)(data+7);

  if(position_list_offset <= 0x3c) return false;
  if(ornament_list_offset <= 0x40) return false;
  if(pattern_list_offset  <= 0x46) return false;
  if(samples_list_offset  <= 0x66) return false;

  if(position_list_offset >= ornament_list_offset) return false;
  if(ornament_list_offset >= pattern_list_offset ) return false;

  if(samples_list_offset - pattern_list_offset != 0x20) return false;

  byte music_length  = data[position_list_offset];
  byte loop_position = data[position_list_offset+1];
  if(music_length < loop_position) return false;
  for(byte i = 0; i < music_length; ++i)
   if( data[position_list_offset+2+2*i] % 6 ) return false;

  return true;
}

bool is_str( const byte* data, int size )
{
  byte str30_player[] = { 0xFE,0x0B,0x18,0xFD,0x18,0xFD,0x18,0xFE,0x41,0xAF,0x06,0xBF,0xC3,0xFF };
  if( search_with_table(data, str30_player) ) return true;

  byte str00_player[] = { 0x21,0xFD,0xC3,0xFD,0xC3,0xFE,0x03,0x7E,0x32,0xFF };
  if( search_with_table(data, str00_player) ) return true;

  byte str_ksa_player[] = { 0x21,0xFD,0xC3,0xFD,0xC3,0xFD,0xC3,0xFF };
  if( search_with_table(data, str_ksa_player) ) return true;

  byte str_fls_player[] = { 0x21,0xFD,0xC3,0xFD,0x97,0x32,0xFD,0x32,0xFD,0x21,0xFF };
  if( search_with_table(data, str_fls_player) ) return true;

  byte delay = data[0];

  if(delay < 3 || delay > 0x0f) return false;

  word position_list_offset = *(word*)(data+1);
  word ornament_list_offset = *(word*)(data+3);
  word pattern_list_offset  = *(word*)(data+5);

  if(position_list_offset <= 0x1A) return false;
  if(ornament_list_offset <= 0x1D) return false;
  if(pattern_list_offset  <= 0x3E) return false;

  if(position_list_offset >= ornament_list_offset) return false;
  if(ornament_list_offset >= pattern_list_offset ) return false;

  if(pattern_list_offset >= size) return false;

  byte position_list_size = data[position_list_offset];
  if(ornament_list_offset != position_list_offset + position_list_size*2 + 3) return false;

  return true;
}

bool is_gtr( const byte* data, int size )
{
  byte gtr1xx_player[] = { 0x21,0xFD,0xC3,0xFD,0xC3,0xFD,0xFE,143,0x00,0xE1,0x00,0xD6,0x00,0xC8,0x00,0xBD,0x00,0xB2,0x00,0xA8,0x00,0x9F,0x00,0x96,0x00,0x8E,0xFF };
  if( search_with_table(data, gtr1xx_player) ) return true;

  byte  delay         = data[0];
  byte  music_length  = data[293];
  byte  loop_position = data[294];

  const byte* id      = data + 1;

  if( id[0] != 'G' || id[1] != 'T' || id[2] != 'R' ) return false;

  if(delay < 3 || delay > 0x0f) return false;
  if(music_length + 1 < loop_position ) return false;

  const byte* position_list = data + 293 + 2;
  for( byte i = 0;  i < music_length; ++i )
    if(position_list[i] % 6) return false;

  return true;
}

bool is_psc( const byte* data, int size )
{
  byte psc1xx_player[] = { 0x21,0xFD,0xC3,0xFD,0xC3,0xFE,0x19,0x22,0xFD,0xEB,0x21,0x4C,0x00,0x19,0x22,0xFD,0x22,0xFD,0x22,0xFD,0x2B,0x2B,0x7E,0x23,0x66,0x6F,0x19,0x22,0xFF };
  if( search_with_table(data, psc1xx_player) ) return true;

  byte  delay = data[73];
  if(delay < 3 || delay >= 32) return false;

  word pattern_list_offset  = *(word*)(data+71);
  if( pattern_list_offset >= size ) return false;

  const byte *p = data + pattern_list_offset;

  for( byte i = 0; i < 0xff; ++i )
  {
    if( *p == i || p[1] != 0xff )
    {
      if( p[1] > 65 || p[1] == 0 ) return false;
      p += 8;
    }
    else
    {
      if( *p > i ) return false;
    }
  }
  return true;
}

// end of detectors

typedef bool (*detector_t)(const byte* data, int file_size);

struct Pair
{
  detector_t detect;
  char*      title;
};

Pair detectors[] =
{
  {is_pt1, "[ProTracker v1.xx]"}, {is_pt2, "[ProTracker v2.xx]"}, {is_pt3, "[ProTracker v3.xx]"},
  {is_asm1xx, "[ASM v1.xx]"}, {is_asm012, "[ASM v0.12]"},
  {is_sqt, "[SQ-Tracker]"},
  {is_stp, "[Sound Tracker Pro]"}, {is_str, "[Sound Tracker]"},
  {is_gtr, "[Global Tracker]" }, {is_psc, "[Pro Sound Creator]"}
};

bool WINAPI detect(DetectorResult& result, const char*, int file_size, const byte* data)
{

  for( int i = 0; i < sizeof(detectors)/sizeof(detectors[0]); ++i )
  {
    if( detectors[i].detect(data + sizeof(HoHdr), file_size - sizeof(HoHdr)) )
    {
      result.action = do_nothing;
      lstrcpy(result.title,  detectors[i].title);
      lstrcpy(result.format, "tracker");
      return true;
    }
  }
  return false;
}
