#include "aychip.h"

const unsigned AYCHIP_MULT_C = 14;	// fixed point precision for 'system tick -> ay tick'

C_AyChip::C_AyChip()
{
	bitA = bitB = bitC = 0;
	SetChipType(C_AyChip::AYCHIP_YM);

	SetTimings(3500000, 1774400, 44100);
	SetVolumes(0x7FFF, AYCHIP_VOL_YM, AYCHIP_PAN_MONO);
	Reset();
}

void C_AyChip::StartFrame(s_waveSample *dst)
{
	renderer.StartFrame(dst);
}

unsigned C_AyChip::EndFrame(unsigned clkTicks)
{
	// adjusting 't' with whole history will fix accumulation of rounding errors
	__int64 endChipTick = ((passedClkTicks + clkTicks) * chipClockRate) / systemClockRate;

	Flush((unsigned)(endChipTick - passedChipTicks));
	unsigned res = renderer.EndFrame(t);

	passedClkTicks += clkTicks;
	passedChipTicks += t;
	t = 0;

	return res;
}

static const char *GetChipName(C_AyChip::AYCHIP_TYPE type)
{
	static char chipName[32];

	switch (type)
	{
		case C_AyChip::AYCHIP_AY:
			sprintf(chipName, "AY-3-8910");
			break;

		case C_AyChip::AYCHIP_YM:
			sprintf(chipName, "YM2149F");
			break;

		default:
			sprintf(chipName, "WTF?");
	}

	return chipName;
}

void C_AyChip::SetChipType(AYCHIP_TYPE type)
{
	chipType = type;
}

void C_AyChip::SetTimings(unsigned systemClockRate, unsigned chipClockRate, unsigned sampleRate)
{
	chipClockRate /= 8;

	this->systemClockRate = systemClockRate;
	this->chipClockRate = chipClockRate;

	multConst = (unsigned)(((__int64)chipClockRate << AYCHIP_MULT_C) / systemClockRate);
	renderer.SetTimings(chipClockRate, sampleRate);

	passedChipTicks = passedClkTicks = 0;
	t = 0;
	ns = 0xFFFF;

	ApplyRegs();
}

void C_AyChip::SetVolumes(unsigned globalVolume, const unsigned *volTab, const unsigned *panTab)
{
	for (int j = 0; j < 6; j++) {
		for (int i = 0; i < 32; i++) {
			vols[j][i] = (unsigned)(((__int64)globalVolume * volTab[i] * panTab[j]) / (65535*100*3));
		}
	}
}

void C_AyChip::Reset(unsigned timestamp)
{
	r_fA = r_fB = r_fC = 0;
	r_noise = r_mix = 0;
	r_vA = r_vB = r_vC = 0;
	r_envT = 0;
	r_env = 0;
	r_portA = r_portB = 0;

	for (int i = 0; i < 14; i++) regs[i] = 0;
	ApplyRegs(timestamp);
}

void C_AyChip::ApplyRegs(unsigned timestamp)
{
	for (BYTE r = 0; r < 0x10; r++)
	{
		Select(r);
		BYTE p = regs[r];

		// clr cached values
		Write(timestamp, p ^ 1);
		Write(timestamp, p);
	}
}

void C_AyChip::Select(BYTE reg)
{
	if (chipType == AYCHIP_AY) reg &= 15;
	activeReg = reg;
}

void C_AyChip::Write(unsigned timestamp, BYTE val)
{
	if (activeReg > 15) return;

	if (activeReg==1 || activeReg==3 || activeReg==5 || activeReg==15) val &= 15;
	if (activeReg==6 || activeReg==8 || activeReg==9 || activeReg==10) val &= 0x1F;

	if (activeReg!=13 && regs[activeReg]==val) return;
	regs[activeReg] = val;

	switch (activeReg)
	{
		case 0:  r_fA = (r_fA & 0xFF00) | val;                  break;
		case 1:  r_fA = (r_fA & 0x00FF) | ((WORD)val << 8);     break;
		case 2:  r_fB = (r_fB & 0xFF00) | val;                  break;
		case 3:  r_fB = (r_fB & 0x00FF) | ((WORD)val << 8);     break;
		case 4:  r_fC = (r_fC & 0xFF00) | val;                  break;
		case 5:  r_fC = (r_fC & 0x00FF) | ((WORD)val << 8);     break;
		case 6:  r_noise = val;                                 break;
		case 7:  r_mix = val;                                   break;
		case 8:  r_vA = val;                                    break;
		case 9:  r_vB = val;                                    break;
		case 10: r_vC = val;                                    break;
		case 11: r_envT = (r_envT & 0xFF00) | val;              break;
		case 12: r_envT = (r_envT & 0x00FF) | ((WORD)val << 8); break;
		case 13: r_env = val;                                   break;
		case 14: r_portA = val;                                 break;
		case 15: r_portB = val;                                 break;
	}

	if (timestamp) Flush((timestamp * multConst) >> AYCHIP_MULT_C);

	switch (activeReg)
	{
		case 7:
			bit0 = 0 - ((val>>0) & 1);
			bit1 = 0 - ((val>>1) & 1);
			bit2 = 0 - ((val>>2) & 1);
			bit3 = 0 - ((val>>3) & 1);
			bit4 = 0 - ((val>>4) & 1);
			bit5 = 0 - ((val>>5) & 1);
			break;

		case 8:
			ea = (val & 0x10)? -1 : 0;
			va = ((val & 0x0F)*2+1) & ~ea;
			break;

		case 9:
			eb = (val & 0x10)? -1 : 0;
			vb = ((val & 0x0F)*2+1) & ~eb;
			break;

		case 10:
			ec = (val & 0x10)? -1 : 0;
			vc = ((val & 0x0F)*2+1) & ~ec;
			break;

		case 13:
			te = 0;

			if (r_env & 4)
			{
				// attack
				env = 0;
				denv = 1;
			}
			else
			{
				// decay
				env = 31;
				denv = ~(unsigned)0;
			}
			break;
	}
}

BYTE C_AyChip::Read(void)
{
	if (activeReg > 15) return 0xFF;
	return regs[activeReg];
}

void C_AyChip::Flush(unsigned chiptick)
{
	while (t < chiptick)
	{
		t++;

		if (++ta >= r_fA) {ta = 0; bitA ^= -1;}
		if (++tb >= r_fB) {tb = 0; bitB ^= -1;}
		if (++tc >= r_fC) {tc = 0; bitC ^= -1;}

		if (++tn >= ((unsigned)r_noise + (unsigned)r_noise))
		{
			tn = 0;
			ns = (ns*2+1) ^ (((ns>>16)^(ns>>13)) & 1);
			bitN = 0 - ((ns >> 16) & 1);
		}

		if (++te >= r_envT)
		{
			te = 0;
			env += denv;

			if (env & ~31)
			{
				unsigned mask = (1 << r_env);

				if (mask & ((1<<0)|(1<<1)|(1<<2)|(1<<3)|(1<<4)|(1<<5)|(1<<6)|(1<<7)|(1<<9)|(1<<15)))
				{
					env = 0;
					denv = 0;
				}
				else
				if (mask & ((1<<8)|(1<<12))) env &= 31;
				else
				if (mask & ((1<<10)|(1<<14))) {
					denv = -(int)denv, env = env + denv;
				}
				else
				{
					env = 31;
					denv = 0; //11,13
				}
			}
		}

		unsigned en, mixL, mixR;

		en = ((ea & env) | va) & ((bitA | bit0) & (bitN | bit3));
		mixL = vols[0][en];
		mixR = vols[1][en];

		en = ((eb & env) | vb) & ((bitB | bit1) & (bitN | bit4));
		mixL += vols[2][en];
		mixR += vols[3][en];

		en = ((ec & env) | vc) & ((bitC | bit2) & (bitN | bit5));
		mixL += vols[4][en];
		mixR += vols[5][en];

		renderer.Update(t, mixL, mixR);
	}
}
