#include "defines.h"
#include "wavplay.h"
#include "aychip.h"
#include "lib_z80ex/z80ex.h"
#include "bin.h"
#include "file.h"
#include "exceptions.h"

C_WavePlayer wp;

Z80EX_CONTEXT *cpu;
Z80EX_BYTE mem[0x10000];
C_AyChip ayChip;

#define MAX_AY_BUFFER_SIZE 2048
#define MAX_SOUND_BUFFER_SIZE (MAX_AY_BUFFER_SIZE * 4)
s_waveSample aySoundBuffer[MAX_SOUND_BUFFER_SIZE];
int aySamples;

unsigned long long clk, devClk, lastDevClk, devClkCounter; 
BYTE lastInstruction;

Z80EX_BYTE ReadByte(Z80EX_CONTEXT *cpu, Z80EX_WORD addr, int m1_state, void *userData)
{
	if (m1_state) lastInstruction = mem[addr];
	return mem[addr];
}

void WriteByte(Z80EX_CONTEXT *cpu, Z80EX_WORD addr, Z80EX_BYTE value, void *userData)
{
	if (addr >= 0x4000) mem[addr] = value;
}

Z80EX_BYTE InputByte(Z80EX_CONTEXT *cpu, Z80EX_WORD port, void *userData)
{
	if ((port & MAKEWORDX(BIN(0x11000000),BIN(0x00000010))) == MAKEWORDX(BIN(0x11000000),BIN(0x00000000)))	// 0xFFFD
	{
		return ayChip.Read();
	}
 
	return 0xFF;
}

void OutputByte(Z80EX_CONTEXT *cpu, Z80EX_WORD port, Z80EX_BYTE value, void *userData)
{
	if ((port & MAKEWORDX(BIN(0x11000000),BIN(0x00000010))) == MAKEWORDX(BIN(0x11000000),BIN(0x00000000)))	// 0xFFFD
	{
		ayChip.Select(value);
		return;
	}

	if ((port & MAKEWORDX(BIN(0x11000000),BIN(0x00000010))) == MAKEWORDX(BIN(0x10000000),BIN(0x00000000)))	// 0xFFFD
	{
		ayChip.Write((unsigned)devClk, value);
		return;
	}
}

Z80EX_BYTE ReadIntVec(Z80EX_CONTEXT *cpu, void *userData)
{
	return 0xFF;
}

int Z80Tick(void)
{
	int clk, p;
	char cmd[0x100];

	clk = z80ex_step(cpu);

	// LD B,B
	if (lastInstruction != 0x40) return clk;

	WORD addr = z80ex_get_reg(cpu, regPC);
	if (addr >= 0x4000) return clk;

	p = 0;
	while (p<255 && mem[addr]) cmd[p++] = mem[addr++];
	if (!p) return clk;

	cmd[p] = 0;
	addr++;

	if (!strcmp(cmd, "%A") || !strcmp(cmd, "/A") || !strcmp(cmd, "*A"))
	{
		WORD r_af = z80ex_get_reg(cpu, regAF);
		BYTE r_a = (BYTE)(r_af >> 8);
		BYTE op = mem[addr++];

		if (cmd[0] == '%') { if (op) r_a %= op; }
		else
		if (cmd[0] == '/') { if (op) r_a /= op; }
		else
		if (cmd[0] == '*') { r_a *= op; }

		r_af = ((WORD)r_a << 8) | (r_af & 0x00FF);
		z80ex_set_reg(cpu, regAF, r_af);
	}
	else
	if (!strcmp(cmd, "%HL") || !strcmp(cmd, "/HL") || !strcmp(cmd, "*HL"))
	{
		WORD r_hl = z80ex_get_reg(cpu, regHL);
		WORD op = ((WORD)mem[addr+1] << 8) | (WORD)mem[addr];
		addr += 2;

		if (cmd[0] == '%') { if (op) r_hl %= op; }
		else
		if (cmd[0] == '/') { if (op) r_hl /= op; }
		else
		if (cmd[0] == '*') { r_hl *= op; }

		z80ex_set_reg(cpu, regHL, r_hl);
	}
	else
	if (!strcmp(cmd, "DUMP"))
	{
		int count = mem[addr++];
		WORD from = ((WORD)mem[addr+1] << 8) | (WORD)mem[addr];
		addr += 2;

		for (int i = 0; i < count; i++)
		{
			int j;
			printf("%04X: ", from);
			for (j = 0; j < 8; j++) printf("%02X ", mem[from + j]);
			for (j = 0; j < 8; j++)
			{
				if (mem[from+j]>=' ' && mem[from+j]<=127) printf("%c", mem[from+j]);
				else printf(".");
			}
			from += 8;
			printf("\n");
		}
	}
	else
	if (cmd[0]=='D' && p>1)
	{
		char *str = NULL;

		if (cmd[1] == ':') str = &cmd[2];
		else
		if (!strcmp(cmd, "D.DE"))
		{
			WORD r_de = z80ex_get_reg(cpu, regDE);

			p = 0;
			while (p<255 && mem[r_de]) cmd[p++] = mem[r_de++];
			cmd[p] = 0;

			str = cmd;
		}
		else
		if (!strcmp(cmd, "D=A"))
		{
			WORD r_af = z80ex_get_reg(cpu, regAF);
			BYTE r_a = (BYTE)(r_af >> 8);
			sprintf(cmd, "A = #%02X = %d", r_a, r_a);
			str = cmd;
		}
		else
		if (!strcmp(cmd, "D=HL"))
		{
			WORD r_hl = z80ex_get_reg(cpu, regHL);
			sprintf(cmd, "HL = #%04X = %d", r_hl, r_hl);
			str = cmd;
		}
		else
		if (!strcmp(cmd, "D=DE"))
		{
			WORD r_de = z80ex_get_reg(cpu, regDE);
			sprintf(cmd, "DE = #%04X = %d", r_de, r_de);
			str = cmd;
		}
		else
		if (!strcmp(cmd, "D=BC"))
		{
			WORD r_bc = z80ex_get_reg(cpu, regBC);
			sprintf(cmd, "BC = #%04X = %d", r_bc, r_bc);
			str = cmd;
		}
		else
		if (!strcmp(cmd, "D=(HL)"))
		{
			WORD r_hl = z80ex_get_reg(cpu, regHL);
			sprintf(cmd, "(HL) = #%02X = %d", mem[r_hl], mem[r_hl]);
			str = cmd;
		}

		if (str) printf("%s\n", str);
	}
	else
	if (p>2 && cmd[0]=='L' && cmd[1]==':')
	{
		char *fname = &cmd[2];
		WORD loadTo = ((WORD)mem[addr+1] << 8) | (WORD)mem[addr];
		addr += 2;

		char fbuf[0x200];
		sprintf(fbuf, "plugins/%s.zx", fname);

		C_File fl;
		long fsize = fl.FileSize(fbuf);

		if ((long)loadTo + fsize < 0x10000)
		{
			fl.Read(fbuf);
			fl.ReadBlock(&mem[loadTo], fsize);
			fl.Close();
		}
	}

	z80ex_set_reg(cpu, regPC, addr);
	return clk;
}

#define MAX_FRAME_TACTS 71680
#define INT_LENGTH 32

void ProcessFrame(void)
{
	unsigned long cmdClk;

	ayChip.StartFrame(&aySoundBuffer[aySamples]);

	while (clk < MAX_FRAME_TACTS)
	{
		cmdClk = Z80Tick();
		devClkCounter += cmdClk;
		clk += cmdClk;
		devClk = clk;

		if (z80ex_doing_halt(cpu))
		{
			cmdClk = (unsigned long)(MAX_FRAME_TACTS - clk - 1) / 4 + 1;
			cmdClk *= 4;

			devClkCounter += cmdClk;
			clk += cmdClk;
			devClk = clk;

			break;
		}
	}

	cmdClk = z80ex_int(cpu);
	devClkCounter += cmdClk;
	clk += cmdClk;
	devClk = clk;

	lastDevClk = devClk;
	clk -= MAX_FRAME_TACTS;
	devClk = clk;

	aySamples += ayChip.EndFrame((unsigned)lastDevClk);
}

void Setup(void)
{
	aySamples = 0;
	devClkCounter = 0;
	clk = 0;
	devClk = 0;
	lastDevClk = 0; 
}

void Reset(void)
{
	z80ex_reset(cpu);
	ayChip.Reset();

	z80ex_set_reg(cpu, regSP, 0x7000);
	z80ex_set_reg(cpu, regIM, 1);
	z80ex_set_reg(cpu, regIFF1, 1);
	z80ex_set_reg(cpu, regIFF2, 1);
}

void PlayMelody(WORD musicSize)
{
	int ch;

	Reset();
	mem[0x6000] = musicSize % 0x100;
	mem[0x6001] = musicSize / 0x100;

	for (;;)
	{
		if (kbhit())
		{
			ch = getch();
			if (ch == 27) break;
		}

		do
		{
			ProcessFrame();
		} while (aySamples < MAX_SOUND_BUFFER_SIZE-MAX_AY_BUFFER_SIZE);

		wp.Play(aySoundBuffer, aySamples);
		aySamples = 0;

		Sleep(10);
	}
}

#define MUSIC_NAME "data/mfx.sqt"

int main(int argc, char *argv[])
{
	C_File fl;

	try
	{
		cpu = z80ex_create(
				ReadByte, NULL,
				WriteByte, NULL,
				InputByte, NULL,
				OutputByte, NULL,
				ReadIntVec, NULL
			);

		Setup();

		fl.Read("plugins/main.zx");
		fl.ReadBlock(mem, fl.FileSize("plugins/main.zx"));
		fl.Close();

		fl.Read(MUSIC_NAME);
		fl.ReadBlock(&mem[0x8000], fl.FileSize(MUSIC_NAME));
		fl.Close();

		PlayMelody((WORD)fl.FileSize(MUSIC_NAME));
	}
	catch (C_E ex)
	{
		printf("\n%s %d (%s)\n", ex.Descr(), ex.exc, ex.param);
	}
}
