﻿// (c) 2013 Eltaron
using System;
using System.Windows.Forms;
using Microsoft.DirectX.DirectInput;
using ZXMAK2.Interfaces;
using System.Collections.Generic;


namespace ZXMAK2.MDX
{
    public class DirectJoystick : IHostJoystick, IDisposable
    {
        #region Fields

        private static List<Device> m_attachedJoysticks = new List<Device>();
        private static Dictionary<Guid, IJoystickState> m_states = new Dictionary<Guid, IJoystickState>();
        static Form m_form;

        #endregion Fields
        
        #region Properties


        #endregion Properties

        #region Public

        public DirectJoystick(Form form)
        {
            m_form = form;
        }

        public void Dispose()
        {
            lock (m_attachedJoysticks)
            {
                var attachedJoysticksCopy = new List<Device>(m_attachedJoysticks); // To avoid 'Collection changed during enumeration' exception
                foreach (var joystick in attachedJoysticksCopy)
                    ReleaseJoystick(joystick);

                m_attachedJoysticks.Clear();
                m_attachedJoysticks = null;
            }
        }

        #endregion Public

        #region Private

        private static void ReleaseJoystick(Device joystick)
        {
            try
            {
                m_attachedJoysticks.Remove(joystick);
                var instanceGuid = joystick.DeviceInformation.InstanceGuid;

                joystick.Unacquire();
                joystick.Dispose();

                if (m_states.ContainsKey(instanceGuid))
                    m_states.Remove(instanceGuid);

            }
            catch (Exception ex)
            {
                LogAgent.Error(ex);
            }
        }

        private class StateWrapper : IJoystickState
        {
            #region Static

            private static StateWrapper s_empty = new StateWrapper(false, false, false, false, false);

            public static StateWrapper Empty
            {
                get { return s_empty; }
            }

            #endregion


            #region Public

            public StateWrapper(
                bool isLeft,
                bool isRight,
                bool isUp,
                bool isDown,
                bool isFire)
            {
                IsLeft = isLeft;
                IsRight = isRight;
                IsUp = isUp;
                IsDown = isDown;
                IsFire = isFire;
            }

            public StateWrapper()
                : this(false, false, false, false, false)
            {
            }

            public bool IsLeft { get; private set; }
            public bool IsRight { get; private set; }
            public bool IsUp { get; private set; }
            public bool IsDown { get; private set; }
            public bool IsFire { get; private set; }

            #endregion
        }

        #endregion Private

        #region IHostJoystick

        public static List<HostJoystickInfo> GetHostJoysticks()
        {
            var result = new List<HostJoystickInfo>();

            var gameControllerList = Manager.GetDevices(
                DeviceClass.GameControl,
                EnumDevicesFlags.AttachedOnly);
            for (int i = 0; i < gameControllerList.Count; i++)
            {
                gameControllerList.MoveNext();
                var deviceInstance = (DeviceInstance)gameControllerList.Current;
                result.Add(new HostJoystickInfo() { Name = deviceInstance.ProductName, Id = deviceInstance.InstanceGuid });
            }

            return result;
        }

        public void Scan()
        {
            try
            {
                // Downto loop for easy remove of disconnected devices
                for (var i = m_attachedJoysticks.Count - 1; i >= 0; i--)
                {
                    var joystick = m_attachedJoysticks[i];

                    // axisTolerance check is needed because of little fluctuation of axis values even when nothing is pressed.
                    int axisTolerance = 0x1000; // Should this be taken from joystick device somehow?
                    ushort center = 0x7FFF;

                    try
                    {
                        joystick.Poll();
                        var diState = joystick.CurrentJoystickState;

                        var isUp = diState.Y > center && diState.Y - center > axisTolerance;
                        var isDown = diState.Y < center && center - diState.Y > axisTolerance;
                        var isLeft = diState.X > center && diState.X - center > axisTolerance;
                        var isRight = diState.X < center && center - diState.X > axisTolerance;
                        var isFire = false;

                        var buttons = diState.GetButtons();
                        foreach (var button in buttons)
                        {
                            // fire = any key pressed
                            isFire |= (button & 0x80) != 0;
                        }

                        m_states[joystick.DeviceInformation.InstanceGuid] = new StateWrapper(
                                                                                    isLeft,
                                                                                    isRight,
                                                                                    isUp,
                                                                                    isDown,
                                                                                    isFire);
                    }
                    catch (InputLostException)
                    {
                        ReleaseJoystick(joystick);
                    }
                }
            }
            catch (Exception ex)
            {
                LogAgent.Error(ex);
            }
        }

        private static Device GetJoystickByInstanceId(Guid joystickInstance)
        {
            var gameControllerList = Manager.GetDevices(
                DeviceClass.GameControl,
                EnumDevicesFlags.AttachedOnly);
            for (int i = 0; i < gameControllerList.Count; i++)
            {
                gameControllerList.MoveNext();
                var deviceInstance = (DeviceInstance)gameControllerList.Current;
                if (deviceInstance.InstanceGuid == joystickInstance)
                    return new Device(deviceInstance.InstanceGuid);
            }
            return null;
        }

        public static void AttachJoystick(Guid joystickInstance)
        {
            lock (m_attachedJoysticks)
            {
                foreach (var joystick in m_attachedJoysticks)
                    if (joystick.DeviceInformation.InstanceGuid == joystickInstance)
                        return; // Already attached

                var joy = GetJoystickByInstanceId(joystickInstance);
                if (joy != null && m_attachedJoysticks.IndexOf(joy) < 0)
                {
                    m_attachedJoysticks.Add(joy);
                    joy.SetCooperativeLevel(m_form, CooperativeLevelFlags.Background | CooperativeLevelFlags.NonExclusive);
                    joy.SetDataFormat(DeviceDataFormat.Joystick);
                    joy.Acquire();
                }
            }
        }

        public static void DetachJoystick(Guid joystickInstance)
        {
            lock (m_attachedJoysticks)
            {
                List<Device> joysticksToDetach = new List<Device>();

                foreach (var joystick in m_attachedJoysticks)
                    if (joystick.DeviceInformation.InstanceGuid == joystickInstance)
                        joysticksToDetach.Add(joystick);

                foreach (var joystick in joysticksToDetach)
                    ReleaseJoystick(joystick);
            }
        }

        public IJoystickState GetJoystickState(Guid joystickInstance)
        {
            if (!m_states.ContainsKey(joystickInstance))
                return StateWrapper.Empty;
            else
                return m_states[joystickInstance];
        }

        #endregion
    }
}
