#include <stdio.h>
#include <string.h>
#include <math.h>

const char tzx_signature[9] = {'Z','X','T','a','p','e','!',0x1A,1};
#define BUFSIZE 4096

#define SAMPLE_RATE_CSW 44100
// Our data pulse duration is aligned to the sample rate and hence
// deviates by a few T-states from the nominal 875
#define PULSE_SAMS_CSW (SAMPLE_RATE_CSW/4000)

#pragma pack(push,1)
struct STANDARD_DATA_BLOCK_HEADER
{
    unsigned short pause;
    unsigned short datalen;
};

struct GENERALIZED_DATA_BLOCK_HEADER
{
    unsigned block_length;
    unsigned short pause;
    unsigned totp;
    unsigned char npp;
    unsigned char asp;
    unsigned totd;
    unsigned char npd;
    unsigned char asd;
};

struct SYMDEF_1
{
    unsigned char flag;
    unsigned short length;
};

struct SYMDEF_2
{
    unsigned char flag;
    unsigned short length[2];
};

struct PRLE
{
    unsigned char symbol;
    unsigned short length;
};

struct CSW_1_01_FILE_HEADER
{
    char csw_signature[23];
    unsigned char rev_major;
    unsigned char rev_minor;
    unsigned short sample_rate;
    unsigned char compr_type;
    unsigned char flags;
    unsigned char reserved[3];
};

struct TZX_CSW_HEADER
{
    unsigned char block_type;
    unsigned block_length;
    unsigned short pause;
    unsigned char sample_rate_bytes[3];
    unsigned char compr_type;
    unsigned n_pulses;
};

struct TZX_FILE_HEADER
{
    char tzx_signature[8];
    unsigned char rev_major;
    unsigned char rev_minor;
};

#pragma pack(pop)

const struct SYMDEF_2 symdef_pilot_sync[2] = {{0,{1750,0}},{0,{875,875}}};
const struct SYMDEF_1 symdef_data[2] = {{2,875},{3,875}};

const struct PRLE pilot_sync_long[2] = {0,10000}; // 5s leader for flags 00-7F ("header")
const struct PRLE pilot_sync_short[2] = {1,4000}; // 2s leader for flags 80-FF ("data")

const unsigned char coding_tbl_5b6b[32] =
{
    0x9D, 0x75, 0xB5, 0xC6, 0xD5, 0xA6, 0x66, 0xE0, 0xE5, 0x96, 0x56, 0xD2, 0x36, 0xB2, 0x72, 0x5D,
    0x6D, 0x8E, 0x4E, 0xCA, 0x2E, 0xAA, 0x6A, 0xE9, 0xCD, 0x9A, 0x5A, 0xD9, 0x3A, 0xB9, 0x79, 0xAD
};

const unsigned char coding_tbl_3b4b_p[8] =
{
    0xB3, 0x98, 0x58, 0xC2, 0xD3, 0xA8, 0x68, 0xE3
};

const unsigned char coding_tbl_3b4b_a[8] =
{
    0xB3, 0x98, 0x58, 0xC2, 0xD3, 0xA8, 0x68, 0x73
};

static void print_write_error(FILE* ft)
{
    if(ferror(ft))
    {
        perror("Output file write error");
    }
    else
    {
        fprintf(stderr,"Output file write error\n");
    }
}

static void print_read_error(FILE* fs)
{
    if(ferror(fs))
    {
        perror("Input file read error");
    }
    else if(feof(fs))
    {
        fprintf(stderr,"Unexpected end of input file\n");
    }
    else
    {
        fprintf(stderr,"Input file_read_error\n");
    }
}

static void print_seek_error(const char* s)
{
    fprintf(stderr,"%s file seek error\n",s);
}

static int encode_byte(int c, int* rd)
{
    int d6,d4,tmp;
    int p7a7_flag = 0; // Flag to use the Primary or Alternative coding for D.x7
    // Encode the 5 MSBs
    d6 = coding_tbl_5b6b[c>>3];
    // Check if the codeword affects RD or requires optional inversion
    if(d6&2)
    {
        // parity-neutral codeword - check P7/A7 condition
        tmp = ~d6;
        if(*rd)
            tmp = ~tmp; // invert if current RD=+1;
        if(!(tmp&0x0C))
        {
            // If two last data bits, inverted and RD-inverted, are zeros, use the A7 code
            p7a7_flag = 1;
        }
    }
    else
    {
        // not a parity-neutral codeword or D.07, update RD and use optional inversion
        if(*rd)
            d6 = ~d6; // invert if current RD=+1
        // After the inversion, new RD can be taken from bit 0 which tells whether to update it or not
        *rd = d6&1;
    }
    d6 = (d6>>2) & 0x3F; // Extract data bits only

    // Encode the 3 LSBs, depending on P7/A7 flag
    if(p7a7_flag)
        d4 = coding_tbl_3b4b_a[c&7]; // Alternate encoding for D.x7
    else
        d4 = coding_tbl_3b4b_p[c&7]; // Primary encoding for D.x7
    if(d4&2)
    {
        // Not a parity-neutral codeword or D.x3, update RD and use optional inversion
        if(*rd)
            d4 = ~d4; // invert if current RD=+1
        // After the inversion, new RD can be taken from bit 0 which tells whether to update it or not
        *rd = d4&1;
    }
    d4 = (d4>>4) & 0x0F; // Extract data bits only

    return (d6<<4) | d4; // Return combined codeword
}

static int write_bits_buf(FILE* ft, int d, int* bits_buf)
{
    int bb = *bits_buf;
    do
    {
        if(putc((d>>(bb-8))&0xFF,ft)==EOF)
            return 1; // write error
        bb-=8;
    } while(bb>=8);
    *bits_buf = bb;
    return 0;
}

static void update_fletcher(unsigned* fletcher_1, unsigned* fletcher_2, unsigned char c)
{
    *fletcher_1 += c;
    *fletcher_1 = (*fletcher_1 + (*fletcher_1>>8))&0xFF;
    *fletcher_2 += *fletcher_1;
    *fletcher_2 = (*fletcher_2 + (*fletcher_2>>8))&0xFF;
}



enum input_type
{
    itBin = 0,
    itTzx
};

enum output_type
{
    otCsw = 0,
    otTzxgen,
    otTzxcsw
};



static int write_csw_header(FILE* ft)
{
    struct CSW_1_01_FILE_HEADER hdr;

    memset(&hdr,0,sizeof(struct CSW_1_01_FILE_HEADER));
    strcpy(hdr.csw_signature,"Compressed Square Wave\x1A");
    hdr.rev_major = 1;
    hdr.rev_minor = 1;
    hdr.sample_rate = SAMPLE_RATE_CSW;
    hdr.compr_type = 1;
    hdr.flags = 0;
    if(fwrite(&hdr,sizeof(struct CSW_1_01_FILE_HEADER),1,ft)!=1)
    {
        print_write_error(ft);
        return 1;
    }
    return 0;
}

static int write_csw_leader_sync(FILE* ft, unsigned* n_pulses, int leader_duration)
{
    int i;

    // Write the 2s leader (0011 bit pattern)
    for(i=0; i<leader_duration*SAMPLE_RATE_CSW/(2*PULSE_SAMS_CSW); i++)
    {
        if(putc(2*PULSE_SAMS_CSW,ft) == EOF)
            goto return_write_error;
        (*n_pulses)++;
    }
    // Write one sync pulse. The other will be added at the start of data
    if(putc(PULSE_SAMS_CSW,ft) == EOF)
        goto return_write_error;
    (*n_pulses)++;

    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
}

static int write_csw_data_word(FILE* ft, int d, unsigned char* lastbits, unsigned* n_pulses)
{
    int i;
    unsigned char curbit;
    unsigned char lbs = *lastbits; // Cache locally for speed and source clarity
    unsigned char lbs2;
    int runlength;
    // Write the 10 bits of the data word
    for(i=0; i<10; i++)
    {
        curbit = (d>>9)&1;
        if(curbit != (lbs&1))
        {
            // Different bit - write the accumulated pulse length
            lbs2 = lbs;
            if(lbs&1)
                lbs2 = ~lbs2;
            // Find the first one in the bit sequence
            runlength = 0;
            do
            {
                lbs2 >>= 1;
                runlength++;
            } while(runlength<6 && !(lbs2&1));
            if(runlength==6)
            {
                printf("Internal encoding error - runlength>=6\n");
                return 3;
            }
            if(putc(runlength*PULSE_SAMS_CSW,ft) == EOF)
                goto return_write_error;
            (*n_pulses)++;
        }
        lbs = (lbs<<1) | curbit;
        d<<=1;
    }
    *lastbits = lbs; // Return back the current sequence of last bits

    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
}

static int finalize_csw(FILE* ft, unsigned char lastbits, unsigned* n_pulses)
{
    unsigned char lbs2;
    int runlength;
    // The last bit of a CSW stream is 0. If there was a train of previous '1's,
    // end it here.
    if(lastbits&1)
    {
        lbs2 = ~lastbits;
        // Find the first one in the bit sequence
        runlength = 0;
        do
        {
            lbs2 >>= 1;
            runlength++;
        } while(runlength<6 && !(lbs2&1));
        if(runlength==6)
        {
            printf("Internal encoding error - runlength>=6\n");
            return 3;
        }
        if(putc(runlength*PULSE_SAMS_CSW,ft) == EOF)
            goto return_write_error;
        (*n_pulses)++;
    }
    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
}

static int part_to_csw_data(FILE* fs, FILE* ft, int part_length, unsigned char flag, unsigned* n_pulses)
{
    int c;
    int retval;
    unsigned char lastbits = 2; // Last data bits (from the sync pulse) - "10"
    int rd = 0; // Running disparity - starts at -1
    unsigned fletcher_1 = 1; // Initial values of the two Fletcher's checksum bytes
    unsigned fletcher_2 = 1;
    int d; // current data word (10 bits)
    int leader_duration; // Number of seconds in the leader tone

    // Write the leader pulses
    if(flag&0x80)
        leader_duration = 2;
    else
        leader_duration = 5;
    retval = write_csw_leader_sync(ft,n_pulses,leader_duration);
    if(retval)
        return retval; // Error return

    // Write the flag byte
    update_fletcher(&fletcher_1,&fletcher_2,flag);
    d = encode_byte(flag,&rd);
    retval = write_csw_data_word(ft,d,&lastbits,n_pulses);
    if(retval)
        return retval; // Error return
    // Process data bytes
    while(part_length--)
    {
        if((c=getc(fs))==EOF)
            goto return_read_error;
        update_fletcher(&fletcher_1,&fletcher_2,c);
        d = encode_byte(c,&rd);
        retval = write_csw_data_word(ft,d,&lastbits,n_pulses);
        if(retval)
            return retval; // Error return
    }
    // Write the Fletcher's checksum
    d = encode_byte(fletcher_1,&rd);
    retval = write_csw_data_word(ft,d,&lastbits,n_pulses);
    if(retval)
        return retval; // Error return
    d = encode_byte(fletcher_2,&rd);
    retval = write_csw_data_word(ft,d,&lastbits,n_pulses);
    if(retval)
        return retval; // Error return
    // Finalize the CSW stream (the final state of the mic bit must be 0)
    retval = finalize_csw(ft,lastbits,n_pulses);
    if(retval)
        return retval; // Error return

    return 0;
// Error handling return path
return_read_error:
    print_read_error(fs);
    return 1;
}

static int part_to_tzx_gen(FILE* fs, FILE* ft, int part_length, unsigned char flag, unsigned short pause)
{
    struct GENERALIZED_DATA_BLOCK_HEADER gh;
    int c;
    int d = 0;
    int rd = 0; // Running disparity - starts at -1
    unsigned fletcher_1 = 1; // Initial values of the two Fletcher's checksum bytes
    unsigned fletcher_2 = 1;
    int bits_buf = 0;

    // Fill in the generalized header
    memset(&gh,0,sizeof(struct GENERALIZED_DATA_BLOCK_HEADER));
    gh.block_length = sizeof(struct GENERALIZED_DATA_BLOCK_HEADER)-4+2*sizeof(struct SYMDEF_2)+2*sizeof(struct PRLE)+2*sizeof(struct SYMDEF_1)+(int)ceil(((part_length+3)*10.0)/8.0);
    gh.pause = pause;
    gh.totp = 2;
    gh.npp = 2;
    gh.asp = 2;
    gh.totd = (part_length+3)*10; // Data length (10 encoded bits per byte) + 3 extra bytes for the flag and Fletcher's checksum
    gh.npd = 1;
    gh.asd = 2; // 2 possible data symbols, bits 0 and 1
    // Write the generalized data block tag
    if(putc(0x19,ft)==EOF)
        goto return_write_error;
    // Write the fixed part of the generalized header
    if(fwrite(&gh,sizeof(struct GENERALIZED_DATA_BLOCK_HEADER),1,ft)!=1)
        goto return_write_error;
    // Write the pilot/sync definition table
    if(fwrite(symdef_pilot_sync,sizeof(struct SYMDEF_2),2,ft)!=2)
        goto return_write_error;
    // Write the pilot/sync data stream, depending on the header/data flag value
    if(flag&0x80)
    {
        // Short leader
        if(fwrite(pilot_sync_short,sizeof(struct PRLE),2,ft)!=2)
            goto return_write_error;
    }
    else
    {
        // Long leader
        if(fwrite(pilot_sync_long,sizeof(struct PRLE),2,ft)!=2)
            goto return_write_error;
    }
    // Write the data symbols definition table
    if(fwrite(symdef_data,sizeof(struct SYMDEF_1),2,ft)!=2)
        goto return_write_error;

    // Write the flag
    update_fletcher(&fletcher_1,&fletcher_2,flag);
    d = d<<10 | encode_byte(flag,&rd);
    bits_buf+=10;
    if(write_bits_buf(ft,d,&bits_buf))
        goto return_write_error;
    // Convert data bytes
    while(part_length--)
    {
        if((c=getc(fs))==EOF)
            goto return_read_error;
        // Update the Fletcher's checksum
        update_fletcher(&fletcher_1,&fletcher_2,c);
        d = d<<10 | encode_byte(c,&rd);
        bits_buf+=10;
        if(write_bits_buf(ft,d,&bits_buf))
            goto return_write_error;
    }
    // Add 2-byte Fletcher's checksum
    d = d<<10 | encode_byte(fletcher_1,&rd);
    bits_buf+=10;
    if(write_bits_buf(ft,d,&bits_buf))
        goto return_write_error;
    d = d<<10 | encode_byte(fletcher_2,&rd);
    bits_buf+=10;
    if(write_bits_buf(ft,d,&bits_buf))
        goto return_write_error;
    if(bits_buf)
    {
        // Save the remaining bits, if any
        if(putc((d<<(8-bits_buf))&0xFF,ft)==EOF)
            goto return_write_error;
    }

    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
return_read_error:
    print_read_error(fs);
    return 1;
}

static int convert_file_part(FILE* fs, FILE* ft, int output_format, int part_length, unsigned char flag, unsigned short pause)
{
    int retval;
    unsigned n_pulses = 0; // Number of pulses counter
    long filepos_hdr;
    long filepos_next;
    struct TZX_CSW_HEADER tchdr;
    unsigned pause_sam; // Number of CSW samples in a pause

    if(output_format == otTzxcsw)
    {
        // Write a TZX/CSW data block header and remember its file position
        filepos_hdr = ftell(ft);
        memset(&tchdr,0,sizeof(struct TZX_CSW_HEADER));
        tchdr.block_type = 0x18;
        tchdr.compr_type = 1; // RLE compression
        tchdr.sample_rate_bytes[0] = SAMPLE_RATE_CSW & 0xFF;
        tchdr.sample_rate_bytes[1] = (SAMPLE_RATE_CSW>>8) & 0xFF;
        tchdr.sample_rate_bytes[2] = (SAMPLE_RATE_CSW>>16) & 0xFF;
        tchdr.pause = pause;
        if(fwrite(&tchdr,sizeof(struct TZX_CSW_HEADER),1,ft) != 1)
            goto return_write_error;
        // Write CSW data (we don't know its size yet)
        retval = part_to_csw_data(fs,ft,part_length,flag,&n_pulses);
        if(retval)
            return retval;
        // Remember the file position to restore it after updating the header
        filepos_next = ftell(ft);
        // Update the header
        if(fseek(ft,filepos_hdr,SEEK_SET))
            goto return_out_seek_error;
        tchdr.n_pulses = n_pulses;
        tchdr.block_length = n_pulses + sizeof(struct TZX_CSW_HEADER) - 5;
        if(fwrite(&tchdr,sizeof(struct TZX_CSW_HEADER),1,ft) != 1)
            goto return_write_error;
        // Seek the file back to where data writing has finished
        if(fseek(ft,filepos_next,SEEK_SET))
            goto return_out_seek_error;
    }
    else if(output_format == otTzxgen)
    {
        // Generalized Data block (0x19) output
        retval = part_to_tzx_gen(fs,ft,part_length,flag,pause);
        if(retval)
            return retval;
    }
    else
    {
        // CSW output
        retval = part_to_csw_data(fs,ft,part_length,flag,&n_pulses);
        if(retval)
            return retval; // Error return
        // Append a pause, if it was specified by the caller
        if(pause>0)
        {
            pause_sam = pause*SAMPLE_RATE_CSW/1000;
            if(putc(0,ft)==EOF)
                goto return_write_error;
            if(fwrite(&pause_sam,sizeof(unsigned),1,ft)!=1)
                goto return_write_error;
        }
    }

    return 0;
// Error handling return path
return_write_error:
    print_write_error(ft);
    return 1;
return_out_seek_error:
    print_seek_error("Output");
    return 1;
}

static int convert_from_bin(FILE* fs, FILE* ft, int output_format)
{
    int retval;
    long file_length;

    // Determine the input file length
    if(fseek(fs,0,SEEK_END))
    {
        print_seek_error("Input");
        return 1;
    }
    file_length = ftell(fs);
    rewind(fs);
    if(file_length > 65535)
    {
        fprintf(stderr,"Input file is too big, must be <65536 bytes\n");
        return 2;
    }

    retval = convert_file_part(fs,ft,output_format,(int)file_length,0xFF,0);
    if(retval)
        return retval; // Error return

    return 0;
}

// A function to convert a standard TZX data block
int convert_tzx_data_block(FILE* fs, FILE* ft, int output_format)
{
    struct STANDARD_DATA_BLOCK_HEADER sbh;
    int c;
    int retval;

    // Read block header
    if(fread(&sbh,sizeof(struct STANDARD_DATA_BLOCK_HEADER),1,fs)!=1)
        goto return_read_error;
    if(sbh.datalen < 2)
    {
        fprintf(stderr,"Data blocks of length <2 are not supported\n");
        return 1;
    }
    // Read the first data byte ("header/data flag")
    if((c=getc(fs))==EOF)
        goto return_read_error;
    // Convert the contiguous data stream by a common routine
    retval = convert_file_part(fs,ft,output_format,sbh.datalen-2,c,sbh.pause);
    if(retval)
        return retval;
    // Read and discard the last data byte (parity)
    if((c=getc(fs))==EOF)
        goto return_read_error;

    return 0;
// Error handling return path
return_read_error:
    print_read_error(fs);
    return 1;
}

static int convert_from_tzx(FILE* fs, FILE* ft, int output_format)
{
    char buf[BUFSIZE];
    int retval;
    int c;

    // Check input file signature
    if(fread(buf,1,10,fs)!=10)
    {
        goto return_read_error;
    }
    if(memcmp(buf,tzx_signature,9))
    {
        fprintf(stderr,"Input file format error\n");
        return 1;
    }
    
    // Process data blocks until the end of file
    while((c=getc(fs))!=EOF)
    {
        if(c==0x10)
        {
            // Standard data block - convert into 8b/10b
            retval = convert_tzx_data_block(fs,ft,output_format);
            if(retval)
                return retval; // Error encountered - break
        }
        else
        {
            // Unsupported block type encountered
            fprintf(stderr,"Unsupported block type 0x%02X. Only standard data blocks are supported\n",c);
            return 1;
        }
    }
    return 0;
// Error handling return path
return_read_error:
    print_read_error(fs);
    return 1;
}


static int write_tzx_file_header(FILE* ft)
{
    struct TZX_FILE_HEADER hdr;

    memset(&hdr,0,sizeof(struct TZX_FILE_HEADER));
    strcpy(hdr.tzx_signature,"ZXTape!\x1A");
    hdr.rev_major = 1;
    hdr.rev_minor = 20;
    if(fwrite(&hdr,sizeof(struct TZX_FILE_HEADER),1,ft)!=1)
    {
        print_write_error(ft);
        return 1;
    }
    return 0;
}

static void print_usage(void)
{
    printf("Usage: tzx_8b10b [-from input_type] [-to output_type] input_file output_file\n\n"
        "input_type can be one of the following:\n"
        " tzx (default): normal blocks (header/data) only are supported\n"
        " bin: raw data is encoded without any parsing\n"
        "  as data blocks (flag=0xFF)\n\n"
        "output_type can be one of the following:\n"
        " tzxgen: TZX Generalized data block (code 0x19) - unsupported by many emulators\n"
        " tzxcsw: TZX encapsulated CSW recording (code 0x18) - also rarely supported\n"
        " csw (default): CSW file\n\n");
}

int main(int argc, char* argv[])
{
    FILE *fs, *ft;
    int retval = 1;
    int input_format = itTzx; // Default input format is tzx
    int output_format = otCsw; // Default output format is CSW
    int cur_arg = 1; // Starting to parse argument 1
    int arg_ifn = 0; // Argument index for the input file name
    int arg_ofn = 0; // Argument index for the output file name

    printf("ZX Spectrum 8b/10b tape format encoder/converter V1.01\n"
        "(c) 2013, Michael Borisov\n\n");

    while(cur_arg<argc && (arg_ifn==0 || arg_ofn==0))
    {
        if(strcmp(argv[cur_arg],"-from")==0)
        {
            cur_arg++;
            if(cur_arg>=argc)
            {
                // A switch is given but nothing follows - syntax error
                print_usage();
                return 2;
            }
            if(strcmp(argv[cur_arg],"tzx")==0)
            {
                input_format = itTzx;
            }
            else if(strcmp(argv[cur_arg],"bin")==0)
            {
                input_format = itBin;
            }
            else
            {
                // Unknown input type - error
                print_usage();
                return 2;
            }
        }
        else if(strcmp(argv[cur_arg],"-to")==0)
        {
            cur_arg++;
            if(cur_arg>=argc)
            {
                // A switch is given but nothing follows - syntax error
                print_usage();
                return 2;
            }
            if(strcmp(argv[cur_arg],"csw")==0)
            {
                output_format = otCsw;
            }
            else if(strcmp(argv[cur_arg],"tzxgen")==0)
            {
                output_format = otTzxgen;
            }
            else if(strcmp(argv[cur_arg],"tzxcsw")==0)
            {
                output_format = otTzxcsw;
            }
            else
            {
                // Unknown output format - error
                print_usage();
                return 2;
            }
        }
        else if(arg_ifn==0)
        {
            arg_ifn = cur_arg;
        }
        else
        {
            arg_ofn = cur_arg;
        }
        cur_arg++;
    }
    if(!(arg_ifn && arg_ofn))
    {
        // Not enough input arguments
        print_usage();
        return 2;
    }

    printf("Converting \"%s\" -> \"%s\"\n",argv[arg_ifn],argv[arg_ofn]);

    fs = fopen(argv[arg_ifn],"rb");
    if(!fs)
    {
        perror("Error opening input file");
        return 1;
    }
    ft = fopen(argv[arg_ofn],"wb");
    if(!ft)
    {
        perror("Error opening output file");
        goto cleanup_fs;
    }

    // Write a CSW or TZX file header, depending on the chosen output format
    if(output_format == otTzxgen || output_format == otTzxcsw)
    {
        retval = write_tzx_file_header(ft);
    }
    else
    {
        retval = write_csw_header(ft);
    }
    if(retval)
        return retval; // Error return

    if(input_format == itTzx)
    {
        retval = convert_from_tzx(fs,ft,output_format);
        if(retval)
            goto cleanup_ft; // Error encountered - break
    }
    else
    {
        retval = convert_from_bin(fs,ft,output_format);
        if(retval)
            goto cleanup_ft; // Error encountered - break
    }

    printf("Complete.\n");
    retval = 0;

cleanup_ft:
    fclose(ft);
cleanup_fs:
    fclose(fs);
    return retval;
}