library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;

entity sdram is
port(
   CLK         : in     std_logic;
   RESET       : in     std_logic; 
   -- Controller Interface
   MEM_A       : in     std_logic_vector(31 downto 2);
   MEM_DI      : in     std_logic_vector(31 downto 0);
   MEM_DO      : out    std_logic_vector(31 downto 0);
   MEM_MASK    : in     std_logic_vector(3  downto 0);
   MEM_WR      : in     std_logic;
   MEM_REQ     : in     std_logic;
   MEM_BUSY    : out    std_logic;
   MEM_ACK     : out    std_logic;
   MEM_INIT    : out    std_logic;    
   -- Memory Interface
   SDRAM_CKE   : out    std_logic;
   SDRAM_CS    : out    std_logic;
   SDRAM_RAS   : out    std_logic;
   SDRAM_CAS   : out    std_logic;
   SDRAM_WE    : out    std_logic;
   SDRAM_DQM   : out    std_logic_vector(1 downto 0);
   SDRAM_BA    : out    std_logic_vector(1 downto 0);
   SDRAM_A     : out    std_logic_vector(12 downto 0);
   SDRAM_D     : inout  std_logic_vector(15 downto 0) );
end sdram;

architecture rtl of sdram is      

   -- SDRAM COMMANDS
   constant CMD_DESELECT   : std_logic_vector(3 downto 0) := "1111"; -- deselect
   constant CMD_PRECHARGE  : std_logic_vector(3 downto 0) := "0010"; -- precharge all
   constant CMD_REFRESH    : std_logic_vector(3 downto 0) := "0001"; -- refresh
   constant CMD_MODESET    : std_logic_vector(3 downto 0) := "0000"; -- mode regiser set
   constant CMD_NOP        : std_logic_vector(3 downto 0) := "0111"; -- no operation
   constant CMD_ACTIVATE   : std_logic_vector(3 downto 0) := "0011"; -- activate
   constant CMD_READ       : std_logic_vector(3 downto 0) := "0101"; -- read
   constant CMD_WRITE      : std_logic_vector(3 downto 0) := "0100"; -- write	  
   
   -- SDRAM MODE: CAS Latency = 2, Burst length = 2
   constant MODE           : std_logic_vector(12 downto 0) := "000" & '0' & "00" & "010" & "0" & "001";
   
   type STATES is (ST_INIT, ST_IDLE, ST_REFRESH, ST_READ, ST_WRITE);
   signal STATE : STATES := ST_INIT;                    
   
   signal SEQUENCE : unsigned(7 downto 0) := "00000000";
   signal SDRAM_CMD : std_logic_vector(3 downto 0) := CMD_NOP;

   signal RFSH_TICK : unsigned(10 downto 0);
   signal DO_MEM_REQ: std_logic;
   signal BUSY      : std_logic;

begin               

   MEM_BUSY <= BUSY or MEM_REQ or DO_MEM_REQ;
   
   p_main_statemachine : process(CLK)
   begin
      if rising_edge(CLK) then   
         
         if RESET = '1' then
            
            SDRAM_A <= (others => '1');
            SDRAM_D <= (others => 'Z');
            SDRAM_BA <= "11";
            SDRAM_DQM <= "11";
            SDRAM_CMD <= CMD_NOP;
            MEM_INIT <= '1';
            SEQUENCE <= X"00";
            RFSH_TICK <= (others => '0');
            DO_MEM_REQ <= '0';
            BUSY <= '0';
            MEM_ACK <= '0';
            
         else
         
            MEM_ACK <= '0';

            RFSH_TICK <= RFSH_TICK + 1;

            SDRAM_CMD <= CMD_NOP;
            SEQUENCE <= SEQUENCE + 1; 
            
            if MEM_REQ = '1' then
               DO_MEM_REQ <= '1';
               BUSY <= '1';
            end if;

            case STATE is         
            
               when ST_INIT =>
                  if SEQUENCE = X"01" then
                     SDRAM_BA <= "00";
                     SDRAM_CMD <= CMD_PRECHARGE;
                  elsif SEQUENCE = X"03" then
                     SDRAM_CMD <= CMD_REFRESH;
                  elsif SEQUENCE = X"08" then
                     SDRAM_CMD <= CMD_REFRESH;
                  elsif SEQUENCE = X"0E" then
                     SDRAM_CMD <= CMD_MODESET;  
                     SDRAM_A <= MODE;
                  elsif SEQUENCE = X"11" then
                     STATE <= ST_IDLE;
                     MEM_INIT <= '0';
                  end if;

               when ST_IDLE =>  

                  SEQUENCE <= X"00";        

--                  if RFSH_TICK > 1400 then
                  if RFSH_TICK > 1400 then
                     STATE <= ST_REFRESH;
                     RFSH_TICK <= (others => '0');
                  elsif DO_MEM_REQ = '1' or MEM_REQ = '1' then
                     BUSY <= '1';
                     DO_MEM_REQ <= '0';
                     SDRAM_BA <= MEM_A(24 downto 23);
                     SDRAM_A <= MEM_A(22 downto 10);
                     SDRAM_CMD <= CMD_ACTIVATE;
                     if MEM_WR = '0' then      
                        STATE <= ST_READ;
                     else
                        STATE <= ST_WRITE;
                     end if; 
                  end if;

               when ST_REFRESH =>
                  if SEQUENCE = X"00" then
                     SDRAM_CMD <= CMD_REFRESH;
                  elsif SEQUENCE = X"04" then
                     STATE <= ST_IDLE;
                  end if;

               when ST_READ =>  
                  if SEQUENCE = X"01" then
                     SDRAM_CMD <= CMD_READ;
                     SDRAM_DQM <= "00";
                     SDRAM_A <= "0010" & MEM_A(9 downto 2) & '0';
                  elsif SEQUENCE = X"04" then
                     MEM_DO(31 downto 16) <= SDRAM_D;
                  elsif SEQUENCE = X"05" then
                     MEM_DO(15 downto 0) <= SDRAM_D;     
                     BUSY <= '0';
                     MEM_ACK <= '1';
                     STATE <= ST_IDLE;
                  end if;
            
               when ST_WRITE =>     
                  if SEQUENCE = X"01" then
                     SDRAM_CMD <= CMD_WRITE;
                     SDRAM_DQM <= not MEM_MASK(3 downto 2);
                     SDRAM_A <= "0010" & MEM_A(9 downto 2) & '0';
                     SDRAM_D <= MEM_DI(31 downto 16);
                  elsif SEQUENCE = X"02" then
                     SDRAM_DQM <= not MEM_MASK(1 downto 0);
                     SDRAM_A <= "0010" & MEM_A(9 downto 2) & '1';
                     SDRAM_D <= MEM_DI(15 downto 0);
                  elsif SEQUENCE = X"03" then
                     SDRAM_D <= "ZZZZZZZZZZZZZZZZ";
                  elsif SEQUENCE = X"04" then
                     BUSY <= '0';
                     MEM_ACK <= '1';
                     STATE <= ST_IDLE;
                  end if;
            
               when OTHERS =>
                  STATE <= ST_IDLE;
            end case;
            
         end if;
      end if;
   end process;
   
   SDRAM_CKE <= '1'; 
   SDRAM_CS  <= SDRAM_CMD(3);
   SDRAM_RAS <= SDRAM_CMD(2);
   SDRAM_CAS <= SDRAM_CMD(1);
   SDRAM_WE  <= SDRAM_CMD(0);

end rtl;
