// --- [ intelli_mouse.c ] ---------------------------------------------------------
//
// liest eine Intelli-Maus aus, 
// gibt die Daten via RS232/SPI an den Host weiter
// Tabwidth = 3

// Anschluss Tastatur/Maus: clock an INT0, data an PD4 -> siehe kb.h
// notwendig ist ein Quarz fr den Betrieb der seriellen Schnittstelle
// Ausgabe mit 38400 Baud -> siehe rs232.h
// ----------------------------------------------------------------------------

#define EXTERN
#define MSG											// Meldungen whrend der Initialisierung ausgeben
														// sowie Daten ausgeben
														// deaktiviert wird nur ber SPI gesendet
//#define RAW										// gibt die puren Mausdaten ohne Auswertung aus

#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h> 
#include <util/delay.h>
#include <avr/pgmspace.h>   
#include "mouse.h"
#include "rs232.h"

//#define SLEEP									// den Sleep-Modus aktivieren

volatile uint8_t rx_count;						// Zhler fr eingehende Bits, bentigt in INT0_ISR
volatile uint8_t rx_error = 0;				// hier werden Fehlercodes abgelegt
uint8_t  ms_ctrl;									// hier wird das 1.Byte des 4-Byte-Packets abgelegt
int8_t   ms_x_move;								// hier wird das 2.Byte des 4-Byte-Packets abgelegt
int8_t   ms_y_move;								// hier wird das 3.Byte des 4-Byte-Packets abgelegt
int8_t   ms_z_move;								// hier wird das 4.Byte des 4-Byte-Packets abgelegt

void print_ms_data(void);
void print_buffer(void);
void get_ms_device_type(void);
void delay_10ms(uint8_t);
void beep(void);

extern void spi_master_init(void);
extern uint8_t spi_out(uint8_t);

// ----------------------------------------------------------------------------
int main(void)
{
uint8_t i = 0, j;

//TCCR0B = (1<<CS01 | 1<< CS00); // Prescale 64 (256 * 64 / 7372800 = 2.2ms)
TIMSK |= (1<<TOIE0);								// Timer0 Overflow enablen

DDRB |= (1<<PB3);									// OCR1A als Output fr den beeper

ACSR = 1<<ACD;										// Analog-Comparator disablen
rs232_init();										// serielle Schnittstelle initialisieren
spi_master_init();								// Portpin fr SPI konfigurieren

#ifdef MSG
put_c(13);
puts_p(PSTR("PS2-Mouse "));					// hello world
#endif

beep();												// der Signalton dauert so lange, bis die Interrupts 
														// freigegeben werden !
delay_10ms(100); 									// abwarten, bis die Maus sich selbst initialisiert hat
														// beim Einschalten der Stromversorgung dauert das etwas
														// alles, was die Maus sendet, wird hier noch nicht empfangen
PS2_init();              						// Initialize PS2-Schnittstelle
asm volatile("sei");             			// interrupt enable, jetzt wird der Signalton bald abgeschaltet !

														// nun ist der mc empfangsbereit

// RESET, die Maus meldet sich mit ID 0x00 (mit der Folge 0xFA, 0xAA, 0x00)
#ifdef MSG
put_c(13);
puts_p(PSTR("reset: "));
#endif

PS2_cmd(MS_RESET);								// jetzt sollte mit 0xFA, 0xAA, 0x00 geantwortet werden
														// wobei der Reset offensichtlich etwas lnger braucht ...
														// darum wird nachfolgend auf die Bytes gewartet
delay_10ms(100);									// und dann nachsehen, ob die 3 Byte die richtigen Werte haben

#ifdef MSG
if (PS2_is_byte_in_buffer()) 
	{
	i = PS2_get_byte_from_buffer();
	print_hexbyte(i);
	}
	
if (PS2_is_byte_in_buffer())
	{
	j = PS2_get_byte_from_buffer();
	print_hexbyte(j);
	i += j;
	}
	
if (PS2_is_byte_in_buffer())
	{
	j = PS2_get_byte_from_buffer();
	print_hexbyte(j);
	}	

if (i == (uint8_t)(0xFA + 0xAA)) puts_p(PSTR("ok")); else puts_p(PSTR("err"));
#endif

/*
// Initialisierung nur des scrolling wheel, die Maus meldet sich mit ID 0x03
#ifdef MSG
put_c(13);
puts_p(PSTR("enable scrolling wheel: "));
#endif

// die Folge SET_SAMPLE_RATE 200 100 80 ausgeben
PS2_cmd(MS_SET_SAMPLE_RATE);					// Datenrate einstellen
PS2_cmd(200);										// 
print_buffer();
PS2_cmd(MS_SET_SAMPLE_RATE);					// Datenrate einstellen
PS2_cmd(100);										// 
print_buffer();
PS2_cmd(MS_SET_SAMPLE_RATE);					// Datenrate einstellen
PS2_cmd(80);										// auf den kleinsten Wert 10 Samples / Sekunde
print_buffer();

#ifdef MSG
put_c(13);
puts_p(PSTR("get mouse id: "));				// schon mal den Text ausgeben
#endif

PS2_cmd(MS_GET_DEVICE_ID);						// command senden
print_buffer();									// die Antwort ausgeben
put_c(13);
*/


// Initialisierung des Intellimode, die Maus meldet sich nun mit ID 0x04

#ifdef MSG
put_c(13);
puts_p(PSTR("switch to intellimode: "));

#endif
// die Folge SET_SAMPLE_RATE 200 200 80 ausgeben
PS2_cmd(MS_SET_SAMPLE_RATE);					// Datenrate einstellen
PS2_cmd(200);										// erster Parameter 200
print_buffer();
PS2_cmd(MS_SET_SAMPLE_RATE);					// Datenrate einstellen
PS2_cmd(200);										// zweiter Paramter 200
print_buffer();
PS2_cmd(MS_SET_SAMPLE_RATE);					// Datenrate einstellen
PS2_cmd(80);										// dritter Parameter 80
print_buffer();


#ifdef MSG
put_c(13);
puts_p(PSTR("get mouse id: "));				// Meldung ausgeben
#endif

PS2_cmd(MS_GET_DEVICE_ID);						// command senden
print_buffer();									// die Antwort ausgeben

#ifdef MSG
put_c(13);
put_c(13);
puts_p(PSTR("set sample rate 10: ")); 
#endif
PS2_cmd(MS_SET_SAMPLE_RATE);					// Datenrate einstellen
PS2_cmd(10);										// auf den kleinsten Wert 10 Samples / Sekunde
print_buffer();

#ifdef MSG
put_c(13);
puts_p(PSTR("set resolution 0: "));
#endif

PS2_cmd(MS_SET_RESOLUTION);
PS2_cmd(0x00);										// 1count/mm
print_buffer();

#ifdef MSG
put_c(13);
puts_p(PSTR("set reporting on: "));
#endif
PS2_cmd(MS_ENABLE_DATA_REPORTING);			// Die Ausgabe der Daten aktivieren
print_buffer();

#ifdef MSG
put_c(13);
puts_p(PSTR("ready"));
put_c(13);
#endif
i = 0;												// Zhler fr die Vierergruppen, die nun immer folgen werden

while(1)
   {
	if (rx_error)									// wenn ein Fehler aufgetreten ist:
		{												// z.B. - Start- Paritts- oder Stop-Bit
		put_s(" Err: ");							// - timeout
		print_hexbyte(rx_error);				// - Fehler beim Senden
		rx_error = 0;								// Fehlerflag zurcksetzen
		//while (1);								// beenden - sofern gewnscht
		}
		
	if (PS2_is_byte_in_buffer())				// wenn Daten im Puffer sind
		{
		j = PS2_get_byte_from_buffer();		// j enthlt nun den SCAN-Code

#ifdef RAW
		print_hexbyte(j);							// hier werden die vier Maus-Bytes nacheinander ausgegeben
#endif

		if (i == 0) 								// und zur Auswertung an drei Variablen bergeben
			{
			ms_ctrl = j;
			i++;
			}

		else if (i == 1) 
			{
			ms_x_move = j;
			i++;
			}
		else if (i == 2)
			{
			ms_y_move = j;
			i++;
			}
		else //if (i == 3)
			{
			ms_z_move = j;
			i = 0;

#ifdef RAW
			put_c(13);								// neue Zeile
#else
			print_ms_data();						// die Mausdaten (im Klartext) auswerten und -geben
#endif
			}
		}

#ifdef SLEEP
   else 												// wenn nichts zu tun ist und kein Sendeflag gesetzt, 
														// dann in SLEEP-Modus gehen
		{												// zuerst sicherstellen, dass nicht noch Bytes bertragen werden
		if (UCSRA & (1<<UDRE))					// hier die serielle Schnittstelle testen
			{											// wenn USART Data Register empty, dann ist bertragung beendet
			MCUCR |= (1<<SE);						// Sleep enable, wegen XTAL ist nur der IDLE-Modus mglich:
														// reduziert die Stromaufnahme von ca. 10mA auf 7mA.
			asm volatile("sleep"); 				// sleep, Aufwecken durch INT0-Interrupt
			}
		}
#endif

	}
}


// ----------------------------------------------------------------------------
// Interpretiert die Mausdaten, Ergebnis steht in
//
// Byte 1: Bit.7 Cursor links
//         Bit.6 Cursor rechts
//         Bit.5 Cursor Down
//         Bit.4 Cursor up
//         Bit.3 0
//         Bit.2 Maus_M
//         Bit.1 Maus_R
//         Bit.0 Maus_L
// Byte 2: Bit.7 1			-> daran kann man Byte 2 von Byte 1 unterscheiden
//         Bit.6 1			-> denn Cursor links und Cursor rechts knnen nicht gleichzeitig gedrckt sein
//         Bit.5 Maus_5
//         Bit.4 Maus_4
//         Bit. 3...Bit0 z-movement scolling wheel  (-8 .. +7)
void print_ms_data(void)
{
uint8_t key_flag   = 0;											// als Flag zum Senden von key_status
uint8_t key_status = 0;											// enthlt den Status der Maus- und Cursortasten
static uint8_t last_byte1 = 0;
static uint8_t last_byte4 = 0;
static uint8_t i   = 0;

key_status = (ms_ctrl & 0x07);								// Button-Bits isolieren und kopieren

if ( !(key_status == last_byte1))							// wenn ein Button gedrckt, dann senden
	{																	// aber nur einmalig
	key_flag = TRUE;				
	last_byte1 = key_status;									// Mausbyte1 merken
	}

if ( !(ms_z_move == last_byte4))								// wenn in ms_z_move ein Bit gesetzt ist
	{																	// dann soll einmalig eine Ausgabe erfolgen
	key_flag = TRUE;
	last_byte4 = ms_z_move;										// letzten Status von ms_z_move merken
	}


// absolute Werte bilden
if (ms_ctrl & (1<<Y_SIGN)) ms_y_move = (-ms_y_move);	// Y-Negatives Vorzeichen
if (ms_ctrl & (1<<X_SIGN)) ms_x_move = (-ms_x_move);	// X-Negatives Vorzeichen

// Es ist eine Mindestbewegung erforderlich, damit die MAus nicht zu zappelig wird:
#define X_MINIMUM 	  	0x04									// Vektoren kleiner als LIMIT
#define Y_MINIMUM		 	0x04									// werden ignoriert

// Es gehen mehrere Parameter ein in die Agilitt der Maus:
// - die Auslesegeschwindigkeit (MS_SET_SAMPLE_RATE) -> 
// - die Auflsunge (MS_SET_RESOLUTION)
// - X/Y_MINIMUM
// - die nachfolgende Halbierung der Anzahl der Mausauswertungen

if ((++i) & 0x01)													// wertet nur jede 2. Mausbewegung aus
	{
	if (ms_x_move > X_MINIMUM)									// Bits in key_status setzen, dazu das Vorzeichenbit
		{																// aus ms_ctrl fr die Richtungsentscheidung nutzen
		key_flag = TRUE;
		if (ms_ctrl & (1<<X_SIGN)) key_status |= (1<<CURSOR_LEFT); else key_status |= (1<<CURSOR_RIGHT);
		}

	if (ms_y_move > Y_MINIMUM)									// Bits in key_status setzen, dazu das Vorzeichenbit
		{																// aus ms_ctrl fr die Richtungsentscheidung nutzen
		key_flag = TRUE;
		if (ms_ctrl & (1<<Y_SIGN)) key_status |= (1<<CURSOR_DOWN); else key_status |= (1<<CURSOR_UP);
		}
	}

if (key_flag && (key_status || ms_z_move))				// key_flag zeigt an, dass neue Daten vorliegen
	{																	// Ausgabe nur, wenn irgendein Bit gesetzt ist
																		// nicht beim Loslassen aller Tasten
#ifdef MSG
//	print_hexbyte(ms_x_move);									// zum debuggen: X-Vektor
//	print_hexbyte(ms_y_move);									// zum debuggen: Y-Vektor

	print_hexbyte(key_status);									// zum debuggen
	print_hexbyte(ms_z_move);
	put_c(13);
#endif

// nun kann key_status per SPI zur weiteren Auswertung weitergereicht werden !
	spi_out(key_status);
	spi_out(ms_z_move | (1<<7 | 1<<6));						// Bit.7 und Bit.6 setzen zur Erkennung von Byte 2 
	
	key_flag = FALSE;												// key_flag zurcksetzen
	beep();															// Timer1 starten
	}
}

// ----------------------------------------------------------------------------
// begrenzt den Zeitraum, in dem auf 11 eingehende Bits gewartet wird.
// Der Timer wird am Start einer Codesequenz zurckgesetzt.
// Wenn der Interrupt auslst, dann wird bit_count auf 11 gesetzt.
// Soll verhindern, dass die Bitzhlung aus dem Tritt kommt.
// Ein TimerOVF lt sich daran erkennen, dass TCCR0B auf 0 steht.

ISR(TIMER0_OVF_vect)
{
TCCR0B 	 = 0;										// Timer0 stoppen, 
														// anstelle eines Flags gleichzeitig Indikator fr TimerOVF
TCNT0  	 = 0;										// 
rx_count  = 11;									// Zhler zurcksetzen fr INT0
rx_error  = 0x10;									// Fehlercode fr TimerOVF speichern
}


// ---------------------------------------------------------------------------
void delay_10ms(uint8_t j)
{
uint8_t i;
for (i = 0; i < j; i++) _delay_ms(10); 	//maximal 262ms / 1 MHz
}

// ----------------------------------------------------------------------------
// Zum Beenden des Signaltons wird  
// der Timer/Counter1 Output Compare Match A Interrupt genutzt
void beep(void)
{
TCNT1H  = 0;
TCNT1L  = 0;										// Counter zurcksetzen
OCR1H   = 0;
OCR1L   = 100;										// Bestimmt die Tonfrequenz
EEDR    = 30;										// Dauer 3ms bei 4600 Hz
TIMSK  |= (1<<OCIE1A);							// Output Compare Match A enablen
TCCR1A |= (1<<COM1A0);							// toggle on Compare Match
TCCR1B |= (1<<WGM12 | 1<< CS11);				// CTC-Modus, Prescale = 8 (7372800/(8*100))*2 = 4608 Hz
														// die Frequenz muss auf den Summer abgestimmt werden
}

// ----------------------------------------------------------------------------
ISR(TIMER1_COMPA_vect)
// der Beepton hat eine Frequenz von 4600 Hz, es gibt also 9200 Interrupts pro Sekunde
// als Zhlvariable wird ein I/O Register benutzt (EEDR)
// wenn EEDR von der aufrufenden Routine mit 255 belegt wird, dann dauert der Ton
// 255/9200 = 72ms oder 45/9200 = 5ms
{
if (!(--EEDR)) TCCR1B = 0;						// Stoppt den Timer1 
}

// ----------------------------------------------------------------------------
// gibt alle von der Maus versandten Antworten aus, 
// entweder auf die seriellen Schnittstelle (wenn MSG definiert ist) oder ins Nirvana
void print_buffer(void)
{
while (PS2_is_byte_in_buffer()) 				// dann alle Bytes ausgeben
	{
#ifdef MSG
	print_hexbyte(PS2_get_byte_from_buffer());
#else
	PS2_clear_buffer();							// lscht eingegangene Daten
#endif
	}
}

// --- [ eof ] ----------------------------------------------------------------
