// --- [ mouse.c ] ---------------------------------------------------------------
// tabwidth 3

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include "mouse.h"
#include "rs232.h"

extern void delay_10ms(uint8_t);

// -- [ globale Variablen ] ---------------------------------------------------
extern volatile uint8_t rx_count;				// Zhler fr empfangene Bits
extern volatile uint8_t rx_error;				// Fehlermeldung

volatile uint8_t PS2_buffer[PS2_BUFFER];		// wird in INT0_ISR benutzt
volatile uint8_t PS2_head;							// wird in INT0_ISR benutzt
			uint8_t PS2_tail;

void 		wait40us(void);
uint8_t  put_byte(uint8_t);						// Routinen zum Senden
void 		wait_for_ack(void);

// ----------------------------------------------------------------------------
// Initialisiert den INT0 fr die CLOCK-Leitung des Keyboards
void PS2_init(void)
{
MCUCR = ( 1<<ISC01 );								// Falling edge of INT0
GIMSK = ( 1<<INT0  );        						// Enable INT0 interrupt

PS2_head  = 0;
PS2_tail  = 0;
rx_count = 11;
}

// ----------------------------------------------------------------------------
// lscht den Puffer logisch, indem die Pointer gleichgesetzt werden
// ist ntzlich, um die Quittungen der Maus, die nicht geprft werden sollen,
// mit wenig Aufwand zu entsorgen.
void PS2_clear_buffer(void)
{
while (TCCR0B);										// auf Ende einer laufenden bertragung warten
asm volatile ("cli");								// damit kein Interrupt
PS2_head  = PS2_tail;								// dazwischenfunkt
asm volatile ("sei");
}

// ----------------------------------------------------------------------------
// Testet, ob Daten im Keyboard-Buffer vorhanden sind
uint8_t PS2_is_byte_in_buffer(void)
{
return (!(PS2_head == PS2_tail));   			// Bytes vorhanden, wenn kb_head <> kb_tail
}

// ----------------------------------------------------------------------------
// holt ein Byte aus dem Keyboard-Buffer
uint8_t PS2_get_byte_from_buffer(void)
{
register uint8_t t;

t = PS2_buffer[PS2_tail++];						// erst das Byte holen, dann den Index incrementieren
PS2_tail &= PS2_MASK;
return(t);
}

// ----------------------------------------------------------------------------
// sendet ein Byte an das PS2-Gert
// liefert bei Erfolg TRUE zurck, die Antwort steht im PS2_buffer
// vor dem Aufruf muss geprft werden, ob noch Daten im Puffer sind,
// wenn ja, dann mssen sie vorher abgearbeitet werden

uint8_t PS2_cmd(uint8_t cmd)
{
while(TCCR0B);										// wenn der Timer luft, dann ist eine bertragung aktiv
if (!(put_byte(cmd))) return(FALSE);		// wenn beim bertragen des Kommandos ein Fehler auftritt

wait_for_ack();									// warten, bis das Gert antwortet oder der TImer0 berluft
delay_10ms(10);									// ohne eine Verzgerung antwortet die Maus mit 0xFE statt 0xFA !
														// offensichtlich drfen die Befehle nicht in zu schneller Folgen
														// bertragen werden 
if (!(rx_error)) return(TRUE);				// TimerOVF setzt rx_error = 0x10

return(FALSE);										// es liegt ein TimerOVF vor
}

// ------------------------------------------------------------------------------------------------
// wartet nach einem Kommando auf die Antwort des PS2-Gertes (in der Regel ACK).
// Der Timer0_OVF verhindert, dass sich das Programm in Endlosschleife aufhngt,
// wenn keine Antwort kommt: Der Timer beendet nach ca. 2.2ms
// an dieser STelle kann TCCR0B nicht zur Erkennung eines Timeouts dienen,
// da die INT0_ISR im Erfolgsfall TCCR0B bereits zurckgesetzt hat
// bleibt aber noch das Bit 0x10 in rx_error als INdikator !

void wait_for_ack(void)
{
TCNT0 = 0;											// fr eine Timeout Prfung den Timer starten
TCCR0B = PRESCALE64;								// Timer starten
														// warten, bis die Antwort eingegangen ist
while ((!(PS2_is_byte_in_buffer())) && (TCCR0B));

TCCR0B = 0;											// Timer stoppen
}

// ----------------------------------------------------------------------------
// Sendet ein Byte an das PS2-Gert
// die Flanken von clock werden durch pollen geprft
// Rckgabe - TRUE wenn kein timeout und das ACK-Bit des PS2-Gertes erkannt wird
//            FALSE bei timeout (Timer0_OVF) oder fehlendem ACK-Bit

uint8_t put_byte(uint8_t data)
{
uint8_t j;
uint8_t parity = 0;
uint8_t result = FALSE;

GIMSK &= ~(1<<INT0);										// INT0-Interrupts disablen
																// zuerst CLOCK fr mindestens 100us auf gnd ziehen
PORTDATA |= (1<<CLOCKPIN | 1<<DATAPIN);				// Clock und Data auf high
DDRDATA  |= (1<<CLOCKPIN | 1<<DATAPIN);				// Pins von INT0/INT1/Data als AUsgang schalten
PORTDATA &= ~(1<<CLOCKPIN );							// Clock-Pins auf low
wait40us();
wait40us();													// mindestens 100us warten
wait40us();

PORTDATA &= ~(1<<CLOCKPIN | 1<<DATAPIN);			// nun auch data auf gnd ziehen
DDRDATA  &= ~(1<<CLOCKPIN );							// clock als Input schalten

_delay_us(10);												// nicht entfernen, sonst gibts keine Antwort !!
																// damit ist das START-Bit abgeschickt
TCNT0  = 0;
TCCR0B = PRESCALE64; 									// Timer0 fr timeout starten

j = 11;														// wird als Bit-Zhler genutzt
do																// Schleife fr 11 fallende Flanken von clock
	{
	while ((PINDATA & (1<<CLOCKPIN)) && TCCR0B);	// warten, bis clock auf low gezogen wird

	if (TCCR0B == 0) break;								// Abbruch bei Timer0_OVF

	if (j > 3)												// 11 .. 4 -> 8 Flanken
		{
		if (data & 0x01)									// Bit.0 testen
			{													// wenn gesetzt
			PORTDATA |=(1<<DATAPIN); 					// auf high ziehen
			parity   ^= 0x01;								// Bit.0 togglen als Zhler fr Paritittsbits
			}

		else													// wenn Bit.0 nicht gesetzt
			{													// dann auf 0 ziehen
			PORTDATA &= ~(1<<DATAPIN);
			}

		data /= 2;											// Data nach rechts schieben
		}

	else if (j == 3)										// 9. Flanke -> Parittsbit senden
		{														// bei gesetztem P_Bit wird 0 gesendet !
		if (parity & 0x01)								// Bit.0 testen
			{													// wenn gesetzt
			PORTDATA &= ~(1<<DATAPIN); 				// auf low ziehen
			}
		else													// wenn Bit.0 nicht gesetzt
			{													// dann auf high ziehen
			PORTDATA |=(1<<DATAPIN);
			}
		}

	else if (j == 2)										// 10. Flanke -> Stop-Bit senden
		{
		PORTDATA |= (1<<DATAPIN); 						// auf high ziehen ( = STOP-Bit)
		}

	else if (j == 1)										// 11. Flanke -> ACK-Bit vom Keyboard prfen
		{
		DDRDATA &= ~(1<<CLOCKPIN | 1<<DATAPIN);	// clock und Data als Eingang
		_delay_us(10);										// etwas warten bis zur Abfrage des Pins

		if ( ( !(PINDATA & (1<<DATAPIN)) ) && TCCR0B) result = TRUE;

																// warten bis CLOCK und DATA high und kein Timerberlauf
		while ( ((!(PINDATA) & (1<<DATAPIN | 1<<CLOCKPIN)) == (1<<DATAPIN | 1<<CLOCKPIN)) && TCCR0B);
		if (TCCR0B == 0) result = FALSE;

		break;												// Abbruch mit result = FALSE oder TRUE
		}

	while ((!(PINDATA & (1<<CLOCKPIN))) && TCCR0B);	// warten bis Clock auf high geht
	if (TCCR0B == 0) break;								// Abbruch

	j--;

	} while (j);
																// Aufrumen:
DDRDATA &= ~(1<<CLOCKPIN | 1<<DATAPIN);			// clock und Data als Eingang
GIMSK |= (1<<INT0);										// INT0 wieder fr den Empfang von Daten setzen
rx_count = 11;												// Counter fr INT0
TCCR0B = 0;													// Timer0 stoppen
TCNT0  = 0;

return(result);
}


// ----------------------------------------------------------------------------
// maximum 768 us/ 1MHz-CPU-Takt
void wait40us(void)
{
_delay_us(40);												// Warteschleife, die bis 16 MHz
}

// ----------------------------------------------------------------------------
// wird bei fallender Flanke von INT0 = CLOCK aufgerufen
// an PINDATA wird das Bit abgegriffen und zu data addiert
// Geprft wird das Start-, das Paritts- und das Stop-Bit

ISR(INT0_vect)
{
register uint8_t j;
static uint8_t data = 0;
static uint8_t parity = 0;						// fr Error- und Parittsbits

j = rx_count;										// volatile in register laden

if (j == 11)										// das erste Bit
	{													// Timer starten
	TCCR0B = PRESCALE64;  						// Prescale 64 (256 * 64 / 7372800 = 2.2ms)
														// Data MUSS low sein, sonst Fehlerflag 0x80 setzen
														// wenn low, dann parity (= Fehlerflag) lschen
	if(PINDATA & (1<<DATAPIN)) parity = 0x80; else parity = 0x00;
	}

else if (j > 2) 									// Bit 3 to 10 is data
	{
   data /= 2;										// nach rechts schieben
   if (PINDATA & (1<<DATAPIN))
		{
		data   |= 0x80;							// von links eine 1 reinshiften
		parity ^= 0x01;							// Bit.0 togglen als Zhler fr Parittsbits
		}
	}

else if (j == 2)									// Parittsbit - ungerade Paritt
	{													// bei gerader ANzahl von Bits = 1
	if ( (parity & 0x01) == (PINDATA & (1<<DATAPIN)) ) parity |= 0x40;  // ggf. Fehlerflag 0x40 setzen
	}

else if (j == 1)									// Stop-Bit
	{													// MUSS high sein, sonst Fehlerflag 0x20 setzen
	if (!(PINDATA & (1<<DATAPIN))) parity |= 0x20;
	}

j--;													// Bitzhler decrementieren

if (!(j))											// auf 0 testen
	{
	if (parity & 0xF0) rx_error = parity;// Wenn ein Fehler vorliegt, dann diesen in rx_error ablegen
	PS2_buffer[PS2_head] = data;

	PS2_head++;										// Index incrementieren
	PS2_head &= PS2_MASK;

	j = 11;
	TCCR0B = 0;										// Counter stoppen
	TCNT0  = 0;
	data   = 0;
	}
rx_count = j;										// Register wieder in Variable ablegen
}

// --- [ eof ] ----------------------------------------------------------------



