//---------------------------------------------------------------------------
// This C-Builder version of Willem's Eprom software was
// constructed by Mike Coates (mike.coates@btinternet.com)
//---------------------------------------------------------------------------
// IO Port Access made possible (under NT) by
// DriverLINX (c) 1996 Scientific Software Tools, inc.
// C-Builder Component tDLPortIO (c) 1999 John Pappas 
//---------------------------------------------------------------------------
// Notes (anything I find out that may be useful later)
//
// ** Support for 27C1000 ? **
// To read a TC541001 (from a socket marked 27C1000)
// swop pins 2 (A16) and 24 (OE) over. Apply Vcc to Pin 31 (PGM)
//---------------------------------------------------------------------------
#include <vcl.h>
#include <stdio.h>
#include <string.h>
#pragma hdrstop
#include "uEprom.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma link "cgauges"
#pragma link "TDLPortIO"
#pragma resource "*.dfm"
TfrmEprom *frmEprom;
//---------------------------------------------------------------------------
__fastcall TfrmEprom::TfrmEprom(TComponent* Owner)
    : TForm(Owner)
{
}
//---------------------------------------------------------------------------

#define MAXROMSIZE 524288

#define EIN 128
#define outhi clkhi
#define outlo clklo
#define IN2 64
#define POW 4
#define RW  2
#define PP  1
#define PGM 8
#define ADR 2
#define CLK 1
#define CLA 4
#define CTW 160
#define CTR 161
#define BIC 0x20

#define einbit  (Inport(ipoort)&EIN)
#define inbit   (Inport(ipoort)&IN2)
#define powoff  (bits &= ~POW)
#define powon   (bits |= POW)
#define read    (bits &= ~RW)
#define write   (bits |= RW)
#define pgrhi   (bits &= ~PGM)
#define pgrlo   (bits |= PGM)
#define ppoff   (bits |= PP)
#define ppon    (bits &= ~PP)
#define adrlo   (outbit &= ~ADR)
#define adrhi   (outbit |= ADR)
#define clklo   (outbit &= ~CLK)
#define clkhi   (outbit |= CLK)
#define clalo   (outbit |=CLA)
#define clahi   (outbit &=~CLA)

#define assert  (Outport(spoort,bits))
#define asadres (Outport(poort,outbit))

#define LDCONF  0
#define LDPROG  2
#define RDPROG  4
#define INCADD  6
#define BEGPRG  8
#define LDDATA  3
#define RDDATA  5
#define ENDPROG 14

#define CP      16
#define PWRTE   8
#define WDTE    4
#define RC      3
#define HS      2
#define XT      1
#define LP      0

#define PSIZE   1024
#define DSIZE   64

#define A14 0x4000        //ARW: added makes bit toggling easier
#define A15 0x8000
#define A16 0x10000L
#define A17 0x20000L

#define TICKS(us)   ((unsigned) (1.193182*(us)+1))  /* us to ticks */
#define MAX(a,b)    ((a > b) ? a : b)

// Easier to recognise Functions

#define WE_High     {pgrhi;assert;}     // WE, code 8 = C17/S4 Pin 31 High
#define WE_Low      {pgrlo;assert;}     // WE, code 8 = C17/S4 Pin 31 Low
#define OE_High     {read;assert;}      // OE, code 2 = S6 high
#define OE_Low      {write;assert;}     // OE, code 2 = S6 low
#define VPP_On      {ppon;assert;}      // VPP, code 1 = C1 12v
#define VPP_Off     {ppoff;assert;}     // VPP, code 1 = C1 0v

//---------------------------------------------------------------------------
// Global Variables
//---------------------------------------------------------------------------

unsigned char bits,outbit;

int poort  = 0x378;     // Default to LPT1
int ipoort = 0x379;
int spoort = 0x37a;

int ep = 1;             // Express
int pin7 = 0;           // Low
int qe = 0;             // Fast Erase Off
int st = 0;             // Offset = 0
int burn = 30;
int fast=0;             //ARW: added fast intelligent mode 1ms pulses doubling till done
int e8751=0;            //ARW: 8751 type flag
int grot=0;             // Size of Loaded data
int ZIFImage=0;         // ZIF Image to Display

int eep=0,pe=0,epr64=1;
int C622,C50,C89,F84,PIC,C93;
int C66,C56,A46,B46;

int B56=0;              // Used to switch code blocks but always 0!
int zes=0;              // Used for a delay timing, but never set

int osc=1;int cp=0;int pt=0;int wd=0;

int BufferSize;
short int fuses;        // PIC Fuse Settings
unsigned char Buffer[MAXROMSIZE];
unsigned short int *picbuf = (unsigned short int *)Buffer;
int FuseUpdate = FALSE;
int check;
int cancel=FALSE;       // Cancel selected
int erase = 0;          // Program or Erase Flag

FILE *fp;
int image_highwater=0;
unsigned char inp_line[160];
unsigned bytes[80];

LARGE_INTEGER PerformanceFrequency;

TImage *DipSwitch[13];
char dip[32]="  |           ";

/* Table from CRC32 by Mark Adler */
unsigned int crc_table[256] =
{
  0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
  0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
  0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
  0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
  0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
  0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
  0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
  0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
  0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
  0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
  0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
  0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
  0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
  0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
  0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
  0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
  0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
  0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
  0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
  0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
  0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
  0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
  0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
  0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
  0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
  0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
  0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
  0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
  0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
  0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
  0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
  0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
  0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
  0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
  0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
  0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
  0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
  0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
  0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
  0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
  0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
  0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
  0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
  0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
  0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
  0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
  0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
  0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
  0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
  0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
  0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
  0x2d02ef8dL
};

//---------------------------------------------------------------------------
// Windows routines to replicate ones used by Dos code
//---------------------------------------------------------------------------

void __fastcall ms_delay(int ticks)
{
   // Dos version Returns after roughly ticks*0.8381 microseconds
   // we convert figures used for Dos version to no of cycles.

   // Currently set so 1,000,000 = 1 second

   LARGE_INTEGER Count,EndCount;

   QueryPerformanceCounter(&Count);
   EndCount.QuadPart = Count.QuadPart + (ticks * PerformanceFrequency.QuadPart * 0.0000008381);
   while (Count.QuadPart < EndCount.QuadPart) QueryPerformanceCounter(&Count);
}

void ChecksumBuf(void)
{
	unsigned int crc;
    int y, grot;

	if(PIC)
    {
		grot=20000;
	} else
    {
		grot=BufferSize;
	}

    if(frmEprom->cbCheckType->ItemIndex == 0)
    {
      crc = 0;

      for( y=0; y<grot; y++ )
      {
          crc = Buffer[y];
      }

      crc &= 0xffff;
      frmEprom->edChecksum->Text = Format("%4.4x",ARRAYOFCONST(((int)crc)));
    }
    else
    {
      crc = 0xffffffff;
      for(y=0; y<grot; y++)
        crc = crc_table[((int)crc ^ (Buffer[y])) & 0xff] ^ (crc >> 8);

      frmEprom->edChecksum->Text = Format("%8.8x",ARRAYOFCONST(((int)(crc ^ 0xffffffff))));
    }
}

void winstat(void)
{
    // Updates display with current settings for device
    // and generally sorts out all of the display options.

    int Count,ImageNo;
    char buffer[12];

    // Update Dip Switches from Dos style string
    for(Count=1;Count<13;Count++)
    {
      switch(dip[Count*2])
      {
          case 32  : ImageNo = 0;
                     break;

          case 48  : ImageNo = 2;
                     break;

          case 49  : ImageNo = 1;
                     break;
      }
      frmEprom->ilDip->GetBitmap(ImageNo,DipSwitch[Count]->Picture->Bitmap);
      DipSwitch[Count]->Refresh();
    }

    // Device Panel
    ChecksumBuf();
    sprintf(buffer, "%4.4x", BufferSize);
    frmEprom->edBuffer->Text = buffer;

    frmEprom->p24CSettings->Visible = ((eep==1) || (eep==2));
    frmEprom->pF28Settings->Visible = (eep==3);
    frmEprom->pPICSettings->Visible = (PIC);
    frmEprom->p27CSettings->Visible = (eep+C93+PIC+C50 == 0);

    frmEprom->imgFlip2->Visible =  frmEprom->p24CSettings->Visible
                                || frmEprom->pF28Settings->Visible
                                || frmEprom->pPICSettings->Visible
                                || frmEprom->p27CSettings->Visible;

    // Settings and Position Panel
    if(PIC)
    {
        // PIC Settings Panel
        if(C50)
        {
            frmEprom->lPIC2->Caption = "Master Clear";
            frmEprom->cbOscillator->Items->Strings[2] = "iRC";
            frmEprom->cbOscillator->Items->Strings[3] = "eRC";
        }
        else
        {
            frmEprom->lPIC2->Caption = "Powerup Timer";
            frmEprom->cbOscillator->Items->Strings[2] = "HS";
            frmEprom->cbOscillator->Items->Strings[3] = "RS";
        }
        frmEprom->SetFuses();
    }

    // ZIF Position Panel
    frmEprom->img32->Visible = False;
    frmEprom->img28->Visible = False;
    frmEprom->img24->Visible = False;
    frmEprom->img12c50->Visible = False;
    frmEprom->img16x84->Visible = False;
    frmEprom->img24xx->Visible = False;
    frmEprom->img25xx->Visible = False;
    frmEprom->img93xx->Visible = False;

    switch (ZIFImage)
    {
      case 32 : frmEprom->img32->Visible=True;
                break;

      case 28 : frmEprom->img28->Visible=True;
                break;

      case 24 : frmEprom->img24->Visible=True;
                break;

      case 240 : frmEprom->img24xx->Visible=True;
                break;

      case 25 : frmEprom->img25xx->Visible=True;
                break;

      case 93 : frmEprom->img93xx->Visible=True;
                break;

      case 12 : frmEprom->img12c50->Visible=True;
                break;

      case 16 : frmEprom->img16x84->Visible=True;
                break;
    }

    // Hex Grids
    frmEprom->HexGrid->Refresh();
    frmEprom->ProgGrid->Refresh();

    // Get rid of Progress Bar
    frmEprom->gProcess->Visible = FALSE;
    frmEprom->btnCancel->Visible = FALSE;
}

void Dip(int DIPid)
{
    switch(DIPid)
    {
        case  0 : strcpy(dip," | | | | | | | | | | | | |");break;

        case  1 : strcpy(dip," |0|0|1|0|1|0|1|1|0|1|0|1|");break;
        case  2 : strcpy(dip," |0|0|1|0|1|0|1|1|0|1|0|1|");break;
        case  3 : strcpy(dip," |0|0|1|1|0|0|1|0|0|1|0|1|");break;

        case  4 : strcpy(dip," |1|1|0|1|0|1|0|0|0|1|0|1|");break;

        case  5 : strcpy(dip," 111010110101");break;

        case  6 : strcpy(dip," 111100100101");break;
        case  7 : strcpy(dip," 110100111001");break;
        case  8 : strcpy(dip," 110100100101");break;
        case  9 : strcpy(dip," 111111000101");break;

        case 10 : strcpy(dip," 110100101010");break;
        case 11 : strcpy(dip," 110100100110");break;
        case 12 : strcpy(dip," 110100111010");break;
    }

    winstat();
}

//---------------------------------------------------------------------------
// Output to LPT port routines
//---------------------------------------------------------------------------



void __fastcall Outport(int IOadd, byte data)
{
    frmEprom->DLPortIO->Port[IOadd] = data;
}

byte __fastcall Inport(int IOadd)
{
    return frmEprom->DLPortIO->Port[IOadd];
}

//---------------------------------------------------------------------------
// Routines from Dos version
//---------------------------------------------------------------------------

void __fastcall clock_out(unsigned long bit)
{
         bit? adrhi: adrlo;asadres;
         ms_delay(1);
         clkhi, asadres;
         ms_delay(1);
         clklo, asadres;
         ms_delay(1);
         adrlo, asadres;
}

void __fastcall eclock_out(int bit)			// DW: ANSIfied
{
	 bit? outhi:outlo;asadres;
	 ms_delay(2);
	 adrhi;asadres;
	 ms_delay(2);
	 adrlo; asadres;
	 ms_delay(2);
	 outlo;asadres;
}

void __fastcall r_adres(long adnr)
{
    unsigned long a,r,adr;
    unsigned long een=1;
    int b;
    adr=(unsigned long)adnr;

    for ( b=18; b>=0; --b ){                // DW: used to start at 17
        r=(unsigned long)(een<<b);
        a=(adr&r);clock_out(a);
    }
}

int clock_in(void)			// DW: added void
{
	 int b;
	 ms_delay(1);
	 b = inbit? 0: 1;
	 clahi, asadres;
	 ms_delay(1);
	 clalo;asadres;
	 return b;
}

int eclock_in(void)			// DW: added void
{
	 int b;

	 outhi;adrlo; asadres;
	 ms_delay(2);
	 adrhi;asadres;
	 ms_delay(2);
	 b = einbit? 0: 1;
	 ms_delay(2);
	 adrlo;asadres;
	 return b;
}

void __fastcall pclock_out(int bit)	// DW: ANSIfied
{
	bit? outhi: outlo, adrhi, asadres;
	ms_delay(2);
	adrlo, asadres;
	ms_delay(2);
	outlo, asadres;
}

int pclock_in(void)			// DW: added void
{
	int b;

	outhi, adrhi, asadres;
	ms_delay(2);
	adrlo, asadres;
	b = einbit? 0: 1;
	ms_delay(2);
	return b;
}

void __fastcall out_word(unsigned char w)			// DW: added void
{
	 int b;

	 for ( b=7; b>=0;--b )
		 eclock_out(w&(1<<b));
}

int in_word(void)			// DW: added void
{
	 int b, w;
	 adrhi;asadres;
	 ms_delay(1);
	 clahi;asadres;
	 ms_delay(1);
	 adrlo;asadres;clalo;asadres;

	 for ( w=0, b=7; b>=0; --b)
		 w += clock_in()<<b;
	 return w;
}

int ein_word(void)			// DW: added void
{
	int b, w;

	for ( w=0, b=7; b>=0; --b )
		w += eclock_in()<<b;

	return w;
}

void __fastcall pout_word(int w)		// DW: ANSIfied
{
	int b;

	pclock_out(0);
	for ( b=0; b<14; ++b )
	  pclock_out(w&(1<<b));
	pclock_out(0);
}

int pin_word(void)			// DW: added void
{
	int b, w;

	(void) pclock_in();
	for ( w=0, b=0; b<14; ++b )
	  w += pclock_in()<<b;
	(void) pclock_in();
	return w;
}

void __fastcall command(int cmd)		// DW: ANSIfied
{
	int b;

	outlo, asadres;
	ms_delay(5);
	for ( b=0; b<6; ++b )
		pclock_out(cmd&(1<<b));
	outhi, asadres;
	ms_delay(5);
}

void power_down(void)     // ARW: added proc to set all lines to low on
{                           //exit to avoid cmos latchup & allow chip removal
    r_adres(0);
    read,adrlo,pgrlo,ppoff,powoff,clklo,clalo,assert;asadres;
    r_adres(0);
}

void setup(void)        // DW: added void
{
    r_adres(0);             //ARW: added, make sure we know where

    read,adrlo,pgrlo,ppoff,powoff,clklo,clalo,assert;asadres;
    ms_delay(25000);powon;assert;       //ARW: increased delays was 5,50
    ms_delay(30000);
}

void start(void)			// DW: added void
{
	outlo;asadres;ms_delay(5);
	adrlo;asadres;ms_delay(5);
}

void stop(void)			    // DW: added void
{
	adrhi;asadres;ms_delay(5);
	outhi;asadres;ms_delay(5);
}

int strct(void)			// DW: added void
{
	start(),out_word(CTW);
	if(!eclock_in()) return 1;
	else return 0;
}

int strctR(void)			// DW: added void
{
	start(),out_word(CTR);
	if(!eclock_in()) return 1;
	else return 0;
}

void prog_mode(void)			// DW: added void
{
	ppoff, powon, adrlo, outlo, assert;asadres;
	ms_delay(TICKS(1000));
	ppon, assert;
	ms_delay(12000);
}

void idle_mode(void)				// DW: added void
{
	ppoff, adrlo, outlo, assert;asadres;
	ms_delay(TICKS(10000));
	powoff, assert;
}

void p5_mode(void)			// DW: added void
{
	ppoff, clklo, outlo, assert;
	ms_delay(3000);

	power_down();		//ARW: added
	//powoff, assert;
}

int mclock_in(void)			// DW: added void
{
	 int b;

	 ms_delay(2);
	 adrhi;asadres;
	 ms_delay(2);
	 b = einbit? 0: 1;
	 ms_delay(2);
	 adrlo;asadres;
	 return b;
}

int min_word(void)			// DW: added void
{
	int b, w=0;
	adrlo;asadres;
	for (b=7;b>=0;b-- )
	    w = w |( mclock_in()<<b);
    adrlo;asadres;
	return w;
}

void __fastcall mclock_out(int bit)			// DW: ANSIfied
{
	bit? outlo:outhi;asadres;
	ms_delay(2);
	adrhi;asadres;
	ms_delay(2);
	outhi;asadres;
    ms_delay(2);
	adrlo; asadres;
}

void __fastcall mout_word(unsigned char w)			// DW: added void
{
	 int b, di;

	 for ( b=7; b>=0;--b )
	 {
         di=(w>>b) & 0x01;
		 mclock_out(di);
     }
}

int __fastcall W_BYTE(unsigned char out)			// DW: ANSIfied
{
	out_word(out);
	if(!eclock_in()) return 1;
	else return 0;
}

void __fastcall send_instruction(int instr)
{
    int n, di;
    if(B46)
    {
        for(n=8;n>=0; n--)
        {
            di=(instr>>n) & 0x01;
            mclock_out(di);
        }
    }
    else if(A46)
    {
        for(n=9;n>=0; n--)
        {
            di=(instr>>n) & 0x01;
            mclock_out(di);
        }
    }
    else if(B56)
    {
        for(n=11;n>=0; n--)
        {
            di=(instr>>n) & 0x01;
            mclock_out(di);
        }
    }
    else if(C66)
    {
        for(n=12;n>=0; n--)
        {
            di=(instr>>n) & 0x01;
            mclock_out(di);
        }
    }
}

int test_hw(void)           //DW: added void
{
    int result;
    int l,r;

    setup();
    OE_Low;
    powon;assert;ms_delay(15000);
    outhi;asadres;ms_delay(1000);
    r=einbit;
    outlo;asadres;ms_delay(1000);
    l=einbit;

    if(l==128 && r==0)
    {
        result = 1;
        frmEprom->Message("Hardware present ",2);
    }
    else
    {
        result = 0;
        frmEprom->Message("Hardware Error :  Check Power && connection",1);
    }
    power_down();           //ARW: was powoff
    OE_High;            //ARW: duplication

    return result;
}

void __fastcall pr622(int i)
{
	command(LDPROG);
	ms_delay(1200);
	pout_word(picbuf[i]);
	ms_delay(4);
	command(BEGPRG);
	ms_delay(119);
	command(ENDPROG);
}

void error(String Notice)
{
	if(test_hw())
      frmEprom->Message(Notice,1);

	power_down();           //ARW: was powoff
	VPP_Off;           //ARW duplication also in power_down
}

void getoff()
{
  // Confirm that we wish to run with the Offset specified
  if (MessageDlg(Format("Use offset of $%4.4x",ARRAYOFCONST(((int)(st)))), mtConfirmation, TMsgDlgButtons() << mbYes << mbNo, 0) == mrNo)
  {
    frmEprom->edOffset->Text = "0000";
    st = 0;
  }
}

int config(void)			// DW: added void
{
	int i;
    fuses = picbuf[0x2007];
	prog_mode();
	command(LDCONF);
	pout_word(fuses);
	for ( i=0; i<7; ++i )
		command(INCADD);
	command(LDPROG);
	pout_word(fuses);
	command(BEGPRG);
	ms_delay(TICKS(12000));
	command(RDPROG);
	if ( fuses != (pin_word()&0x1F) )
    {
        error("Error blowing fuses");
		idle_mode();
		return 0;
	}
	return 1;
}

void ClearBuf(void)
{
  frmEprom->Clear1Click(NULL);
}

//---------------------------------------------------------------------------
// Check Empty
//---------------------------------------------------------------------------

int Check(void)
{
    int Answer=-1;
	long i;
	unsigned char h;

  	if(st!=0) getoff();

    frmEprom->SetProcessGauge("Checking Device is Empty",BufferSize,clWhite);

	setup();

	if(epr64)
    {
		WE_High;
	}

	for(i=0;i<BufferSize;i++)
    {
        frmEprom->gProcess->Progress = i;
   		r_adres(i+st);

		h=in_word();
		if(h!=0xFF)
        {
            Answer = i+st;
			break;
		}

        Application->ProcessMessages();
        if(cancel)
        {
          Answer = -2;
          break;
        }
	}

	power_down();
    winstat();
    return Answer;
}

//---------------------------------------------------------------------------
// Verify Device Routines
//---------------------------------------------------------------------------

int verify(void)
{
	int i, w, mask, rdcmd;	// DW: also had *buf, ldcmd, n but they are unused

    frmEprom->SetProcessGauge("verifying PIC program area",PSIZE-1,clGreen);

	OE_Low;

	mask = 0x3FFF;

	rdcmd = RDPROG;
	prog_mode();
	for ( i=0; i<PSIZE; ++i )
    {
		command(rdcmd);
		w = pin_word() & mask;
		if ( w != picbuf[i])
        {
            error("Error verifying PIC");
			idle_mode();
			return 0;
		}

		command(INCADD);

        frmEprom->gProcess->Progress = i;
	}

	mask = 0xFF;

	rdcmd = RDDATA;
	prog_mode();
	ms_delay(12000);

    frmEprom->SetProcessGauge("verifying PIC data area",DSIZE-1,clGreen);

	for ( i=0; i<DSIZE; ++i )
    {
		command(rdcmd);
		w = pin_word() & mask;
		if ( w != picbuf[i+0x2100] )
        {
            error("Error verifying PIC");
			idle_mode();
			return 0;
		}

		command(INCADD);

        frmEprom->gProcess->Progress = i;
	}

	idle_mode();
	OE_High;

    frmEprom->Message("Device Verified OK",2);
    winstat();
	return 1;
}

int verify50(void)
{
	int i, w, mask, rdcmd;		//DW: *buf, ldcmd, n unused

    frmEprom->SetProcessGauge("verifying PIC program area",(BufferSize/2)-1,clGreen);

	OE_Low;
	mask = 0x3FFF;

	rdcmd = RDPROG;
	prog_mode();
	command(INCADD);
	for ( i=0; i<(BufferSize/2)-1; ++i )
    {
		command(rdcmd);
		ms_delay(1200);
		w = pin_word() & mask;
		if ( w != picbuf[i])
        {
            error(Format("Error at $%6x",ARRAYOFCONST(((int)(i)))));
        	idle_mode();
			return 0;
		}

		command(INCADD);

        frmEprom->gProcess->Progress = i;
	}

	idle_mode();
	OE_High;
	return 1;
}

void Compare(void)
{
	int  i, y;
    bool OK = True;

  	if(st!=0)getoff();

	setup();

	if(epr64)
    {
		WE_High;
	}

	if(grot>=BufferSize) grot=BufferSize-1;

    frmEprom->SetProcessGauge("Comparing chip",grot-1,clGreen);

	for(i=0;i<=grot;i++)
    {
		y=i;

		r_adres(i+st);
		if(in_word()!=Buffer[y])
        {
            error(Format("Error at $%4.4x",ARRAYOFCONST(((int)(i+st)))));
			power_down();/*assert;*/         //ARW: was powoff
            OK = False;
			break;//for i
		}

        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
        {
          frmEprom->Message("program interrupted!",1);
          OK = False;
          break;
        }
	}

    if((OK) && (!cancel)) frmEprom->Message("Device Compares OK",2);

	power_down();/*assert;*/          //ARW: was powoff
    winstat();
}

void verify_65(void)
{
	unsigned char adrby1, adrby2;
	int i;
    bool OK=True;

	unsigned char g;

    frmEprom->SetProcessGauge("Comparing chip",grot,clGreen);

	outhi,adrhi;asadres;

    if(pin7)WE_High
    else WE_Low;

	powon;assert; OE_Low;
	ms_delay(15000);

	for(i=0;i<grot;i++)
    {
		adrby1=i>>8;
		adrby2=i;

		if(strct())
        {
			if(W_BYTE(adrby1))
            {
				if(W_BYTE(adrby2))
                {
					start();
					stop();
					if(strctR())
                    {
						g=ein_word();eclock_out(1);stop();
					}
					else		//DW: clean up a bit
                    {
						error("Unknown Error");
                        OK=False;
						break;
					}
				}
				else
                {
					error("Unknown Error");
                    OK=False;
					break;
				}
			}
			else
            {
				error("Unknown Error");
                OK=False;
				break;
			}
		}
		else
        {
			error("Unknown Error");
            OK=False;
			break;
		}

		if(g!=Buffer[i])
        {
            error(Format("Error at $%4.4x",ARRAYOFCONST(((int)(i)))));
            OK=False;
			break;
		}

        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
        {
          frmEprom->Message("program interrupted!",1);
          break;
        }
	}

    if ((OK) && (!cancel)) frmEprom->Message("Device Compares OK",2);

	idle_mode();
    winstat();
}

void verify_16(void)		//DW: added void
{
	unsigned char adrby1, adrby2;
	int i;
    bool OK=True;
	unsigned char g;

    frmEprom->SetProcessGauge("Comparing chip",grot,clGreen);

	outhi,adrhi;asadres;

    if(pin7)WE_High
    else WE_Low;

	powon;assert;

	OE_Low;
	ms_delay(15000);
	for(i=0;i<grot;i++)
    {
		adrby1=i>>7;
		adrby1&=126;
		adrby1=adrby1+CTW;
		adrby2=i;

		start();
		if(W_BYTE(adrby1))
        {
			if(W_BYTE(adrby2))
            {
				start();
				stop();
				adrby1=adrby1+1;
				start();
				if(W_BYTE(adrby1))
                {
					g=ein_word();
					eclock_out(1);
					stop();
				}
				else
                {
					error("Unknown Error");
                    OK=False;
					break;
				}
			}
			else
            {
				error("Unknown Error");
                OK=False;
				break;
			}
		}
		else
        {
			error("Unknown Error");
            OK=False;
			break;
		}

		if(g!=Buffer[i])
        {
            error(Format("Error at $%4.4x",ARRAYOFCONST(((int)(i)))));
            OK=False;
			break;
		}

        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
        {
          frmEprom->Message("program interrupted!",1);
          break;
        }
	}

    if ((OK) && (!cancel)) frmEprom->Message("Device Compares OK",2);

	idle_mode();
    winstat();
}

//---------------------------------------------------------------------------
// Erase Device Routines
//---------------------------------------------------------------------------

void P_erase(void)
{
	int i;
	OE_Low;

	prog_mode();

	command(LDCONF);
	pout_word(0x3FFF);
	for ( i=0; i<7; ++i )
		command(INCADD);
	command(1);
	command(7);
	command(BEGPRG);
	ms_delay(TICKS(12000));
	command(1);
	command(7);
	idle_mode();
	OE_High;
}

void erase_93(void)
{
  /* erases all bytes */

  int wt=0;
  WE_High;

  if(B46)       send_instruction(0x120);
  else if(A46)  send_instruction(0x240);
  else if(C56)  send_instruction(0x480);
  else if(C66)  send_instruction(0x900);

  WE_Low;
  ms_delay(5);
  WE_High;
  while(!einbit)
  {
    ms_delay(1000);
    wt++;
    if(wt==50) break;
  }
  WE_Low;
}


// Called by Erase routine
void prog_f28(void);

void erase_f28(void)
{
	int  q=1,i=0;
    bool OK = True;

	if(st!=0) getoff();

	erase=1;
	if(!qe)
    {
      prog_f28();
    }
    else
    {
      frmEprom->SetProcessGauge("erasing chip",BufferSize,clWhite);

      setup();
      WE_High;
      r_adres(0);
      VPP_On;

      F:
      WE_High;

      OE_Low;
      ms_delay(5);
      Outport(poort,0x20);

      ms_delay(5);
      WE_Low;ms_delay(TICKS(6));WE_High;

      ms_delay(1);
      Outport(poort,0x20);
      WE_Low;ms_delay(TICKS(6));WE_High;
      ms_delay(10000);
      do
      {
          Outport(poort,0xA0);
          ms_delay(5);
          WE_Low;ms_delay(TICKS(6));WE_High;
          ms_delay(TICKS(6));
          adrlo;clklo;clalo;asadres;
          ms_delay(5);
          OE_High;
          ms_delay(5);
          if(in_word()!=0xff)
          {
              if(q==1000)
              {
                  error("Error erasing chip");
                  OK = False;
                  ppoff;
                  power_down();/*assert;*/         //ARW: was powoff
                  break;
              }
              else
              {
                  q++;
                  goto F;
              }
          }

          if(qe) i=i+20;
          else i++;

          r_adres(i+st);

          frmEprom->gProcess->Progress = i;

          Application->ProcessMessages();
          if(cancel)
          {
              frmEprom->Message("Erase interrupted!",1);
              OK = False;
              break;
          }

      } while(i<=BufferSize);
    }

    if(OK) frmEprom->Message("Erase Completed",2);

	power_down();                  //ARW was powoff
	VPP_Off;                  //ARW: duplication
	erase=0;
    winstat();
}

void erase_f29(void)
{
	int test;

	if(st!=0)getoff();

    frmEprom->SetProcessGauge("erasing chip",1,clWhite);

	OE_High;
	r_adres(0);
	WE_High;
	powon;VPP_Off;
	ms_delay(555);
	r_adres(0x5555);
	OE_Low;
	ms_delay(5);
	Outport(poort,0xAA);
	ms_delay(5);
	WE_Low;ms_delay(5);WE_High;
	ms_delay(5);
	OE_High;
	ms_delay(5);
	r_adres(0x2AAA);
	OE_Low;
	ms_delay(5);
	Outport(poort,0x55);
	WE_Low;ms_delay(5);WE_High;
	ms_delay(5);
	OE_High;
	ms_delay(5);
    r_adres(0x5555);

	ms_delay(5);
	OE_Low;
	Outport(poort,0x80);
	ms_delay(5);
	WE_Low;ms_delay(5);WE_High;
	ms_delay(5);
	Outport(poort,0xAA);
	ms_delay(5);
	WE_Low;ms_delay(5);WE_High;
	ms_delay(5);
	OE_High;
	ms_delay(5);
	r_adres(0x2AAA);
	OE_Low;
	ms_delay(5);
	Outport(poort,0x55);
	WE_Low;ms_delay(5);WE_High;
	ms_delay(5);
	OE_High;
	ms_delay(5);
	r_adres(0x5555);
	OE_Low;
	ms_delay(5);
	Outport(poort,0x10);
	WE_Low;ms_delay(5);WE_High;
	ms_delay(12000);
	adrlo;clklo;clalo;asadres;assert;
	OE_High;

    ms_delay(105);
    test=0;

    while((test!=255))
    {
        Application->ProcessMessages();

        if(cancel)
        {
            frmEprom->Message("erase interrupted!",1);
            idle_mode(); return;
        }

        test=in_word();
    }

    /*{if(in_word()&16){
    ms_delay(1);if((in_word()&128))goto k;
    tcu_wprintf(&window4,"              error");
    power_down();              //ARW: was powoff
    getch();tcu_close_window(&window4);
    idle_mode(); return;}
    k: */

    frmEprom->Message("erase completed",2);
    power_down();/*assert;*/        //ARW: was powoff
    idle_mode();   //ARW: duplication
    winstat();
}

//---------------------------------------------------------------------------
// Write Device Routines
//---------------------------------------------------------------------------

void PIC508_pr(void)
{
	int i, x, w,te, mask, rdcmd;
	int q=0;

    if(Application->MessageBox("This routine is known to have problems!", "Warning", MB_OKCANCEL + MB_DEFBUTTON1) == IDOK)
    {
      frmEprom->SetProcessGauge("programming PIC program area",(BufferSize/2)-1,clRed);

      mask = 0x3FFF;

      rdcmd = RDPROG;
      OE_Low;
      prog_mode();
      ms_delay(14);
      command(INCADD);

      for ( i=0; i<(BufferSize/2)-1; ++i )
      {
          W:
              pr622(i);
              ms_delay(4);
              command(rdcmd);
              ms_delay(4);
              w = pin_word() & mask;
              if ( w != picbuf[i])
              {
                  q++;
                  if(q>25)
                  {
                      error("Error writing program area");
                      idle_mode();
                      return;
                  }
                  else goto W;
              }
              for(x=0;x<(q*3);x++)
                  pr622(i);
              command(INCADD);

              frmEprom->gProcess->Progress = i;
              q=0;
      }

      if(verify50())
      {
          frmEprom->SetProcessGauge("programming PIC config area",1,clRed);

          fuses=picbuf[0x2007];

          p5_mode();
          ms_delay(5000);
          prog_mode();

          ms_delay(4000);

          OE_Low;
          for ( i=0; i<100; ++i )
              pr622(0x2007);

          p5_mode();
          ms_delay(5000);
          prog_mode();
          command(RDPROG);
          te = pin_word()&0x1F;
          if ( fuses != te)
          {
              p5_mode();
              error("Error writing configuration");
              idle_mode();
              return;
          }

          ms_delay(35000);
      }
    }
	idle_mode();
	OE_High;
    winstat();
}

void PIC_pr(void)
{
	int i, w, mask, ldcmd, rdcmd;

	P_erase();

    frmEprom->SetProcessGauge("programming PIC program area",PSIZE,clRed);

	mask = 0x3FFF;
	ldcmd = LDPROG;
	rdcmd = RDPROG;
	prog_mode();
	OE_Low;

	ms_delay(5000);
	for ( i=0; i<PSIZE; ++i )
    {
		command(ldcmd);
		ms_delay(100);
		pout_word(picbuf[i]);
		ms_delay(100);
		command(BEGPRG);
		ms_delay(TICKS(12000));
		command(rdcmd);
		ms_delay(100);
		w = pin_word() & mask;
		if ( w != picbuf[i])
        {
            error("Error writing PIC program");
			idle_mode();
			return;
		}
		ms_delay(100);
		command(INCADD);
		ms_delay(100);

        frmEprom->gProcess->Progress = i;
	}

    frmEprom->SetProcessGauge("programming PIC data area",DSIZE,clRed);

	mask = 0xFF;
	ldcmd = LDDATA;
	rdcmd = RDDATA;
	prog_mode();

	for ( i=0; i<DSIZE; ++i )
    {
		command(ldcmd);
		pout_word(picbuf[i+0x2100]);
		command(BEGPRG);
		ms_delay(TICKS(12000));
		command(rdcmd);
		w = pin_word() & mask;
		if ( w != picbuf[i+0x2100] )
        {
            error("Error writing PIC data");
			idle_mode();
			return;
		}
		command(INCADD);

        frmEprom->gProcess->Progress = i;
	}

	if(verify())
    {
		OE_Low;
        frmEprom->SetProcessGauge("programming PIC config area",1,clRed);
		config();
	}

	ms_delay(35000);
	power_down();VPP_Off;
	OE_High;
    winstat();
}

void prog_93(void)
{
	int i, g;

	if(grot>BufferSize) grot=BufferSize;

    frmEprom->SetProcessGauge("programming chip",grot,clRed);

	OE_Low;
	outhi,adrlo;pgrlo;asadres;assert;

	powon;assert;
	ms_delay(30000);
	WE_High;
	if(B46) 	 send_instruction(0x130);
	else if(A46) send_instruction(0x260);
	else if(C56) send_instruction(0x4C0);
	else if(C66) send_instruction(0x980);
	WE_Low;
	erase_93();
	ms_delay(3000);
	WE_High;
	if(B46)	     send_instruction(0x130);
	else if(A46) send_instruction(0x260);
	else if(C56) send_instruction(0x4C0);
	else if(C66) send_instruction(0x980);

	WE_Low;
	ms_delay(3000);
	for(i=0;i<grot;i++){
		g=Buffer[i];
		WE_High;

		if(B46)       send_instruction(0x140+i/2);
		else  if(A46) send_instruction(0x280+i);
		else  if(C56) send_instruction(0x500+i);
		else  if(C66) send_instruction(0xA00+i);
		mout_word(g);
		if(B46||B56)
        {
          i++;
		  g=Buffer[i];
		  mout_word(g);
        }

		WE_Low;
		ms_delay(5);
		WE_High;
		ms_delay(12000);
		WE_Low;

        frmEprom->gProcess->Progress = i;
	}

	WE_Low;
	power_down();
	OE_High; 		//ARW: duplication but left for clarity
    winstat();
}

void Program8751(void)
{
	int i, y;
	unsigned char g;
	int q=1,q0=0;   //ARW: added q0
    bool OK=TRUE;

//  if(!ep) fast=1;	//ARW: added change default normal to fast

	if(st!=0) getoff();

 	if(grot>=BufferSize) grot=BufferSize-1;

    frmEprom->SetProcessGauge("programming chip",grot,clRed);

	setup();

	WE_High;
	r_adres(0);

	for(i=0;i<=grot;i++)
    {
		y=i;

		g=Buffer[y];
  	    if((frmEprom->cbSkipFF->ItemIndex==1) || (g!=0xFF))
        {
          r:
		  WE_High;
		  r_adres(i+st);
		  OE_Low;
		  ms_delay(15);               //ARW: OE to Vpp delay increased for 8751
		  Outport(poort,g);
		  ms_delay(15);					//ARW: added allow bus to settle
		  if(!pe)
          {
			VPP_On;
		  }
		  ms_delay(150);

		  WE_Low;
		  if(pe)
          {
		    ms_delay(TICKS(1000));
		    WE_High;
		  }
		  else
          {
			if(ep)
            {
              ms_delay(119);            //ARW: express , delay 100usecs (snap mode)
			  WE_High;
			  ms_delay(15);             //ARW: delay at least 10usecs between pgm pulses
			  WE_Low;             //ARW: do another pulse now it saves a lot of time
			  ms_delay(119);            //ARW: max downside is one pulse too many
			  q++;				        //ARW: upside is nearly halved program time
			}                           //ARW: inc pulse acc we did an extra pulse here
			else if(!fast)ms_delay(burn*1200);                   //ARW normal but must not exceed 50ms total
				 else if(q<burn) ms_delay((int)((q-q0)*1200));   //ARW fast intelligent 1ms pulse
					  else ms_delay((int)((burn-q0)*1200));      //doubles every time till done
															     //q is acc burn time after this pulse
		  }
		  WE_High;

		  if(pe) ms_delay(TICKS(1000));
		  else ms_delay(150);

		  adrlo;clklo;clalo;asadres;
		  ms_delay(5);
		  if((!epr64)||e8751)           //ARW need Vpp at 5v for reading 8751
          {
			VPP_Off;
		  }
		  OE_High;
		  ms_delay(5);
		  if(!epr64&&!e8751)            //ARW: need pgrhi for 8751 read
          {
			WE_Low;               //ARW: not sure about this for 2732/2716
		  }
		  ms_delay(150);

		  if(g!=(unsigned char)in_word())
          {
			if(q>=burn||(!ep&&!fast))    //ARW: changed to GE
            {                            //gives user some control
										 //now burn means max no of  .1 or 1ms pulses
										 //depending on mode express/fast
										 //ARW: also if normal must only do one
										 //pulse or will really burn it!!!!
              frmEprom->Message(Format("Error at $%4.4x",ARRAYOFCONST(((int)(i+st)))),1);
			  VPP_Off;
			  power_down();			     //ARW: was powoff before assert
			  goto PE;
			}
			else
            {
			  if(!fast)                  //ARW:
              {
                q++;
				goto r;
              }
			  else                       //ARW: must be fast intelligent mode
			  {
                q0=q;                    //save last q to calculate new delay
				q=2*q;			         //ARW: q= accumulated burn time in msecs
				//q=int(q*3/2)+1;	     // this algorithm can be changed as requ
				//q++;
				goto r;
              }
			}
		  }
        }

//      if(epr64){if(er<200){er++;goto r;}}

        frmEprom->gProcess->Progress = i;
        Application->ProcessMessages();
		q=1;
        if(cancel)
        {
          frmEprom->Message("program interrupted!",1);
          OK = False;
          break;
        }
	}

	ppoff; assert;		//DW: added. Used to leave program power on!

    if(OK)
    {
      frmEprom->SetProcessGauge("verifying chip",grot,clGreen);

      for(i=0;i<=grot;i++)
      {
          y=i;

          r_adres(i+st);
          if(in_word()!=Buffer[y])
          {
            frmEprom->Message(Format("Error at $%4.4x",ARRAYOFCONST(((int)(i+st)))),1);
            power_down();/*assert;*/			//ARW: was powoff
            goto PE;
          }

          frmEprom->gProcess->Progress = i;
          Application->ProcessMessages();
          if(cancel)
          {
            frmEprom->Message("program interrupted!",1);
            OK = False;
            break;
          }
      }

      if (OK) frmEprom->Message("Device Programmed and Verified OK",2);
    }

	PE:                            //ARW jump to here avoids reset_timer???
	power_down();
    winstat();
}

void Program_89(void)
{
	int i,control=0; //ARW:added control
	unsigned char g;
	int q=1, p12, at, manufacturer,sig30, sig31;
    AnsiString DeviceType;

	if(st!=0) getoff();

	setup();pgrhi;OE_High;r_adres(0);

	r_adres(0x30);  // ARW: need to set up control lines for signature read??
				   	// unless adaptor defaults to signature read
				    // but must default to normal otherwise
					// standard read/compare routines wont work?

	manufacturer = in_word();
	sig30=manufacturer;
	r_adres(0x31);
	sig31=in_word();

    DeviceType = Format("Chip (30/31 = $%2x%2x)",ARRAYOFCONST((sig30,sig31)));

	if(sig30==0x15)      DeviceType = "Phillips";	  //its a Phillips  //ARW: added this lot probably not much help
	else if(sig30==0x89) DeviceType = "Intel";        //its an Intel    //there must be several other  manufacturers
	else if(sig30==0xE0) DeviceType = "Siemens";      //its a Siemens   // and device types. would need a lookup table
	else if(sig30==0x1E) DeviceType = "Amtel";        //its an Amtel    // at best could put it on screen
	else if(sig30==0x97) DeviceType = "TI";           //its a Texas Ins
	else DeviceType = "";         //Unknown manufacture

	if((sig30==0x15)&&(sig31==0x92))      DeviceType = "Phillips 87C51";   	//its a Phillips 87c51
	else if((sig30==0x89)&&(sig31==0x58)) DeviceType = "Intel 87C51";		//its an Intel 87c51
	else if((sig30==0xE0)&&(sig31==0x71)) DeviceType = "Siemens C501-1E"; 	//its a Siemens C501-1E
	else if((sig30==0x1E)&&(sig31==0x51)) DeviceType = "Amtel AT89C51";		//its an Amtel AT89c51
	else;										                            //unknown device

    frmEprom->SetProcessGauge("programming "+DeviceType,grot,clRed);

	r_adres(0x32);
	p12=in_word();
	if(p12==0xff)
    {
	  VPP_On;
	}
	else p12=0;

	//r_adres(0x4000);      //ARW: set A14 altered so bits can be set or cleared
	control+=A14;			//but  must add it to  subsequent r_adres ???
	r_adres(control);		//could be set adaptor to not signature mode P3.6 P3.7

	WE_Low;ms_delay(12000);WE_High;
	VPP_Off;

	r_adres(0);
	for(i=0;i<=grot;i++)
    {
		g=Buffer[i];
		w89:

		control+=A15; 					//ARW: need to add 0x4000??? otherwise
		control+=A16;
		control+=A16;
		r_adres(st+i+control); 			//A14 cleared??
		if(p12){VPP_On;}           // DW: Is this right?  //ARW: curly braces added
		OE_Low;
		Outport(poort,g);

		ms_delay(15);                   //ARW: added longer delay
		WE_Low;

		ms_delay(120);                  //but need 100usecs for 87c51 and would be
		WE_High;                   //compatible with 89c51 also


		ms_delay(15);
		VPP_Off;adrlo;clklo;clalo;asadres;

		ms_delay(TICKS(2000));
		OE_High;

		control-=A15;					//ie A15 probably used for output enable
		r_adres(st+i+control);			//but why not use read/write signal already there
										//A16,17 probably control bits P3.6,P3.7
										//need one signal for chip erase cntrl (P2.6)


		at=in_word();
		if(at!=g){
			if(q>30)
            {
              frmEprom->Message(Format("Error at $%4x",ARRAYOFCONST(((int)(i+st)))),1);
			  power_down();
 			  VPP_Off;        //ARW: duplication
 			  goto P89E;
			}
			else
            {
				q++;
				goto w89;
			}
		}
        frmEprom->gProcess->Progress = i;
	    q=0;
	}
	P89E:
	power_down();/*assert;*/        //ARW: was powoff
    winstat();
}

void prog_f28(void)
{
	int i, y;
	unsigned char g;
	int q=1;
    int hgrot= grot;
    bool OK = True;

	if(st!=0) getoff();

	if(grot>=BufferSize) grot=BufferSize-1;

	if(erase)
    {
      grot=BufferSize-1;
      frmEprom->SetProcessGauge("Erasing chip",grot,clWhite);
    }
    else
      frmEprom->SetProcessGauge("programming chip",grot,clRed);

	setup();

	WE_High;
	r_adres(0);
	VPP_On;

	for(i=0;i<=grot;i++)
    {
		y=i;

		if(erase) g=0xff;
		else g=Buffer[y];

		r:
		WE_High;

		r_adres(i+st);

		OE_Low;
		ms_delay(5);
		Outport(poort,0x40);

		ms_delay(100);
		WE_Low;ms_delay(TICKS(6));WE_High;

		ms_delay(1);
		Outport(poort,g);
		WE_Low;ms_delay(TICKS(6));WE_High;
		ms_delay(10);
		Outport(poort,0xC0);
		ms_delay(5);
		WE_Low;ms_delay(TICKS(6));WE_High;
		ms_delay(TICKS(6));
		adrlo;clklo;clalo;asadres;
		ms_delay(5);
		OE_High;
		ms_delay(5);
		if(g!=(unsigned char)in_word())
        {
			if(q==20)
            {
				error("error");
				power_down();		//ARW: was powoff
				VPP_Off;       //ARW: duplication
				goto P8E;
			}
			else
            {
				q++;
				goto r;
			}
		}

        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
        {
          frmEprom->Message("program interrupted!",1);
          break;
        }
		q=1;
	}

	VPP_Off;

	if(erase)
    {
   	  grot=hgrot;
      return;
    }

    if(!cancel)
    {
      frmEprom->SetProcessGauge("verifying chip",grot,clGreen);

      for(i=0;i<=grot;i++)
      {
          y=i;

          r_adres(i+st);
          if(in_word()!=Buffer[y])
          {
              frmEprom->Message(Format("Error at $%6x",ARRAYOFCONST(((int)(i+st)))),1);
              power_down();
              OK = False;
              break;
          }

          frmEprom->gProcess->Progress = i;
      }

      if(OK) frmEprom->Message("Device Compares OK",2);
    }

	P8E:
	grot=hgrot;
	power_down();/*assert;*/ 			//ARW: was powoff
    winstat();
}

void prog_f29(void)
{
	int i, y;
	int  test;
	unsigned char g;
    int hgrot=grot;
    bool OK = True;

	if(grot>=BufferSize) grot=BufferSize-1;

	if(st!=0) getoff();

    frmEprom->SetProcessGauge("programming chip",grot,clRed);

	WE_High;
	powon;ppoff;read;WE_High;
	r_adres(0);
	OE_Low;

	for(i=0;i<=grot;i++)
    {
		y=i;

		g=Buffer[y];
		WE_High;
		OE_High;

		r_adres(0x5555);
		OE_Low;
		ms_delay(2);
		Outport(poort,0xaa);
		ms_delay(zes);       // zes is never set to anything!
		WE_Low;ms_delay(1);WE_High;
		OE_High;
		ms_delay(2);
		r_adres(0x2AAA);
		OE_Low;
		ms_delay(2);
		Outport(poort,0x55);
		WE_Low;ms_delay(1);WE_High;
		OE_High;
		ms_delay(2);
		r_adres(0x5555);
		OE_Low;
		ms_delay(2);
		Outport(poort,0xa0);
		ms_delay(2);
		WE_Low;ms_delay(1);WE_High;
		OE_High;
		ms_delay(2);
		r_adres(i+st);
		OE_Low;
		Outport(poort,g);
		ms_delay(2);
		WE_Low;ms_delay(1);WE_High;
		ms_delay(2);
		adrlo;clklo;clalo;asadres;
		OE_High;
		ms_delay(2);

		test=in_word();
		while(test!=g)
        {
          Application->ProcessMessages();
          if(cancel)
          {
            frmEprom->Message("program interrupted!",1);
            idle_mode();
            OK = False;
            break;
          }
		  ms_delay(1);test=in_word();
		}

        frmEprom->gProcess->Progress = i;
	}

    if(!cancel)
    {
      frmEprom->SetProcessGauge("verifying chip",grot,clGreen);

      for(i=0;i<=grot;i++)
      {
          y=i;

          r_adres(i+st);
          if(in_word()!=Buffer[y])
          {
              frmEprom->Message(Format("Error at $%6x",ARRAYOFCONST(((int)(i+st)))),1);
              power_down();
              OK = False;
              break;
          }

          frmEprom->gProcess->Progress = i;
      }

      if(OK) frmEprom->Message("Device Compares OK",2);
    }

    grot = hgrot;
	power_down();
    winstat();
}

void eprogram65(void)
{
	unsigned char adrby1, adrby2;
	int i;
	int w=0;
	unsigned char g,k;
    bool OK=True;

    frmEprom->SetProcessGauge("programming chip",BufferSize,clRed);

	outhi,adrhi;asadres;

    if(pin7)WE_High
    else WE_Low;

	OE_Low;
	powon;assert;

	ms_delay(30000);
	for(i=0;i<BufferSize;i++)
    {
    	g=Buffer[i];

		adrby1=i>>8;
		adrby2=i;

		if(strct())
        {
			if(W_BYTE(adrby1))
            {
				if(W_BYTE(adrby2))
                {
					if(W_BYTE(g))
                    {
						outlo;asadres;stop();
					}
					else
                    {
						error("Unknown Error");
                        OK=False;
						break;
					}
				}
			}
			else
            {
				error("Unknown Error");
                OK=False;
				break;
			}
		}
		else
        {
			error("Unknown Error");
            OK=False;
			break;
		}
		W:
		if(strct())
        {
			if(W_BYTE(adrby1))
            {
				if(W_BYTE(adrby2))
                {
					start();
					stop();
					if(strctR())
                    {
						k=ein_word();eclock_out(1);stop();
					}
					else
                    {
						error("Unknown Error");
                        OK=False;
						break;
					}
				}
				else
                {
					error("Unknown Error");
                    OK=False;
					break;
				}
			}
			else
            {
				error("Unknown Error");
                OK=False;
				break;
			}
		}
		else
        {
			ms_delay(TICKS(1000));
			w++;
			if(w==10)
            {
				error("Too many Retries - would not program");
                OK=False;
                break;
			}
			stop();
			goto W;
		}
		if(k!=g)
        {
			error("Verify Error");
            OK=False;
			break;
		}
		w=0;

        frmEprom->gProcess->Progress = i;
	}

    if (OK) frmEprom->Message("Device Programmed OK",2);

	power_down();		//ARW: replaces powoff
	OE_High;        //duplication but left for clarity
    winstat();
}

void prog_16(void)
{
	unsigned char adrby1, adrby2;
	int i;
	int w=0;
	int g,k;
    bool OK=True;

    frmEprom->SetProcessGauge("programming chip",BufferSize,clRed);

	OE_Low;
	outhi,adrhi;asadres;

    if(pin7)WE_High
    else WE_Low;

	powon;assert;
	ms_delay(30000);
	for(i=0;i<BufferSize;i++)
    {
		g=Buffer[i];

		adrby1=i>>7;
		adrby1&=126;
		adrby1=adrby1+CTW;
		adrby2=i;

		start();

		if(W_BYTE(adrby1))
        {
			if(W_BYTE(adrby2))
            {
				if(W_BYTE(g))
                {
					outlo;asadres;stop();ms_delay(TICKS(1000));
				}
				else
                {
					error("Unknown Error");
                    OK=False;
					break;
				}
			}
		}
		else
        {
			error("Unknown Error");
            OK=False;
			break;
		}
		W:
		start();
		if(W_BYTE(adrby1))
        {
			if(W_BYTE(adrby2))
            {
				start();
				stop();
				adrby1=adrby1+1;
				start();
				if(W_BYTE(adrby1))
                {
					k=ein_word();
					eclock_out(1);
					stop();
				}
				else
                {
					error("Unknown Error");
                    OK=False;
					break;
				}
			}
			else
            {
				error("Unknown Error");
                OK=False;
				break;
			}
		}
		else
        {
			ms_delay(TICKS(1000));
			w++;
			if(w==10)
            {
				error("Unknown Error");
                OK=False;
				break;
			}
			stop();
			goto W;
		}
		if(k!=g)
        {
			error("Verify Error");
            OK=False;
			break;
		}
		w=1;

        frmEprom->gProcess->Progress = i;
	}

	power_down();
	OE_High; 		//ARW: duplication but left for clarity

    if (OK) frmEprom->Message("Device Programmed OK",2);
    winstat();
}

//---------------------------------------------------------------------------
// Read Device Routines
//---------------------------------------------------------------------------

void P50read(void)
{
	int i, mask, rdcmd;

    frmEprom->SetProcessGauge("Reading PIC program area",(BufferSize/2),clBlack);

	OE_Low;

	mask = 0x3FFF;

	rdcmd = RDPROG;
	prog_mode();
	command(INCADD);

	for (i=0;i<(BufferSize/2);++i)
    {
		command(rdcmd);
		picbuf[i] = pin_word() & mask;
		command(INCADD);
        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
        {
          frmEprom->Message("program interrupted!",1);
          idle_mode();break;
        }
	}

    if(!cancel)
    {
  	  idle_mode();
	  prog_mode();
	  command(RDPROG);
      picbuf[0x2007] = pin_word()&0x1F;
	  fuses = picbuf[0x2007];
    }

	idle_mode();OE_High;
    if(!cancel)
      frmEprom->Message("PIC dumped OK",0);
    else
      frmEprom->Message("Process was Cancelled",1);
    winstat();
}

void Pread(void)
{
	int i, mask, rdcmd;

    frmEprom->SetProcessGauge("Reading PIC program area", BufferSize,clBlack);

	OE_Low;

	mask = 0x3FFF;

	rdcmd = RDPROG;
	prog_mode();
	for ( i=0; i<BufferSize; ++i ) {
		command(rdcmd);
		picbuf[i] = pin_word() & mask;
		command(INCADD);
        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
          break;
	}

    if(!cancel)
    {
      mask = 0xFF;

      frmEprom->SetProcessGauge("Reading Data program area", DSIZE,clBlack);

      rdcmd = RDDATA;
      prog_mode();

      for ( i=0; i<DSIZE; ++i ) {
          command(rdcmd);
          picbuf[i + 0x2100]= pin_word() & mask;
          command(INCADD);
          frmEprom->gProcess->Progress = i;

          Application->ProcessMessages();
          if(cancel)
            break;
      }
    }

    if(!cancel)
    {
      command(LDCONF);
      pout_word(fuses);
      for ( i=0; i<7; ++i )
           command(INCADD);
      command(RDPROG);
      picbuf[0x2007] = pin_word()&0x1F;
      fuses = picbuf[0x2007];
    }

	idle_mode();OE_High;

    if(!cancel)
      frmEprom->Message("PIC dumped OK",0);
    else
      frmEprom->Message("Process Cancelled",1);
    winstat();
}

void read_93(void)
{
	unsigned int instr;
	unsigned char g;
    int i;

	OE_Low;

    frmEprom->SetProcessGauge("Reading Chip", BufferSize,clBlack);

	outhi,adrlo;pgrlo;asadres;assert;

	powon;assert;
	ms_delay(300);

	for(i=0;i<BufferSize;i++)
    {
		WE_High;

		ms_delay(1);
		if(B46)instr=0x180+i/2;
		else if(A46)instr=0x300+i;
		else if(C56)instr=0x600+i;
		else if(C66)instr=0xc00+i;
		send_instruction(instr);

		ms_delay(10);

		g=min_word();
		Buffer[i]=g;
		if(B46)
        {
            i++;
		    g=min_word();
    		Buffer[i]=g;
        }

        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
          break;

		WE_Low;
	}

	WE_Low;
	power_down();
	outhi;asadres;
	//powoff,
	OE_High; 		    //ARW: duplication but left for clarity

    if(!cancel)
    {
      frmEprom->Message("Chip dumped OK",0);
      frmEprom->Caption = "Willem Eprom Programmer - " + frmEprom->edDevice->Caption + " Image";
    }
    else
      frmEprom->Message("Process Cancelled",1);
    winstat();
}

void CRead(void)
{
  int i, y;

  ClearBuf();

  if(st!=0)getoff();

  frmEprom->SetProcessGauge("Reading Chip", BufferSize,clBlack);

  setup();

  if(epr64)
  {
   	WE_High;
  }

  r_adres(0);
  for(i=0;i<BufferSize;i++)
  {
  	y=i+st;
  	r_adres(i+st);
  	Buffer[y]=in_word();
    frmEprom->gProcess->Progress = i;

    Application->ProcessMessages();
    if(cancel)
      break;
  }

  power_down();/* */assert;          //ARW: was powoff , assert;
  if(!cancel)
  {
    frmEprom->Message("Chip read OK",0);
    frmEprom->Caption = "Willem Eprom Programmer - " + frmEprom->edDevice->Caption + " Image";
    grot = BufferSize;
  }
  else
    frmEprom->Message("Process Cancelled",1);

  winstat();
}

void read_65(void)		//DW: added void
{
	unsigned char adrby1, adrby2;
	unsigned char g;
	int i;

	OE_Low;

    frmEprom->SetProcessGauge("Reading Chip", BufferSize,clBlack);

	outhi,adrhi;asadres;

    if(pin7)WE_High
    else WE_Low;

	powon;assert;
	ms_delay(15000);
	for(i=0;i<=BufferSize;i++)
    {
		adrby1=i>>8;
		adrby2=i;

		if(strct())
        {
			if(W_BYTE(adrby1))
            {
				if(W_BYTE(adrby2))
                {
					start();
					stop();
					if(strctR())
                    {
						g=ein_word();eclock_out(1);stop();
					}
					else
                    {        // DW: removed some unused code
						error("Unknown Error");
						break;
					}
				}
				else
                {
					error("Unknown Error");
					break;
				}
			}
			else
            {
				error("Unknown Error");
				break;
			}
		}
		else
        {
			error("Unknown Error");
			break;
		}
		Buffer[i]=g;
        frmEprom->gProcess->Progress = i;

        Application->ProcessMessages();
        if(cancel)
          break;
	}

	power_down();              //ARW:
	//powoff,
	read,assert;               //ARW:    duplication

    if(!cancel)
    {
      frmEprom->Message("Chip read OK",0);
      frmEprom->Caption = "Willem Eprom Programmer - " + frmEprom->edDevice->Caption + " Image";
    }
    else
      frmEprom->Message("Process Cancelled",1);

    winstat();
}

void read_16(void)
{
  unsigned char adrby1, adrby2;
  unsigned char g;
  int i;

  frmEprom->SetProcessGauge("Reading Chip", BufferSize,clBlack);

  OE_Low;
  outhi,adrhi;asadres;

  if(pin7)WE_High
  else WE_Low;

  powon;assert;
  ms_delay(15000);
  for(i=0;i<BufferSize;i++)
  {
	adrby1=i>>7;
	adrby1&=126;
	adrby1=adrby1+CTW;
	adrby2=i;
	start();
	if(W_BYTE(adrby1))
    {
      if(W_BYTE(adrby2))
      {
	  	start();
	  	stop();
	  	adrby1=adrby1+1;
	  	start();
	  	if(W_BYTE(adrby1))
        {
		  g=ein_word();clock_out(1);stop();
        }
		else
        {
		  error("Unknown Error");
		  break;
    	}
	  }
	  else
      {
	  	error("Unknown Error");
	  	break;
	  }
	}
	else
    {
	  error("Unknown Error");
	  break;
	}
    frmEprom->gProcess->Progress = i;
	Buffer[i]=g;

    Application->ProcessMessages();
    if(cancel)
      break;
  }

  power_down();
  read,assert;      //ARW: duplication but left for clarity
  if(!cancel)
  {
    frmEprom->Message("Chip read OK",0);
    frmEprom->Caption = "Willem Eprom Programmer - " + frmEprom->edDevice->Caption + " Image";
  }
  else
    frmEprom->Message("Process Cancelled",1);

  winstat();
}

//---------------------------------------------------------------------------
// Load File Routines
//---------------------------------------------------------------------------

void Load_S(bool Autoswitch);

// Process line of Intel Hex file

int segment;

unsigned Process_Line (unsigned char *inp_line)
{
	int file_chksum, calc_chksum;
	int i, status;
	int count, type;
	int address, addhi, addlo;
	unsigned char  *inp_ptr;

	inp_line=strstr((char *)inp_line, ":") ; //ARW: reset inp_line to point to colon
	if ( inp_line != NULL)                   //ARW: avoids errors due to LFCR / CRLF
	{									     //ARW: warning now accepts bytes before the colon
		sscanf(inp_line, ":%2X%2lX%2lX%2X", &count, &addhi, &addlo, &type);
		calc_chksum = count + addhi + addlo + type;
		inp_ptr = inp_line + 9;
		for (i=0; i<count; i++) {
			sscanf(inp_ptr, "%2X", &bytes[i]);
			calc_chksum += bytes[i];
			inp_ptr +=2;
		}
		calc_chksum =-(calc_chksum & 0xFF) & 0xFF;

		sscanf(inp_ptr, "%2X", &file_chksum);

		if (calc_chksum == file_chksum)
        {
          switch(type)
          {
              case 0  : // Data
                        address = segment + (addhi << 8) + addlo;
                       	for (i=0; i<count; i++)
            				Buffer[address+i] = bytes[i];
            			image_highwater=MAX(image_highwater,segment+address+i);
                        break;

              case 1  : // Terminator
                        break;

              case 2  : // 20 bit Segment Address
                        segment = ((bytes[0] << 12) + bytes[1]) << 4;
                        break;

              case 4  : // 32 bit Segment Address
                        segment = ((bytes[0] << 24) + bytes[1] << 16);
                        break;
          }
          status = 1;
		}
		else
        {
          error("Checksum Error reading Hexfile");
          status = 0;
		}
	}
	else
    {
	  error("Error reading Hexfile");
      status = 0;
	}
	return status;
}

unsigned Process_Line_S (unsigned char *inp_line)
{
	unsigned file_chksum, calc_chksum;
	int i, status;
	int count=0;
	int address, addhi;
	unsigned char  *inp_ptr;

    // Only process S1,S2 and S3 lines

	if(strstr((char *)inp_line, "S1") != NULL)
    {
      sscanf(inp_line+2, "%2X%4X", &count, &addhi);
	  calc_chksum = count + (addhi & 0xFF) + ((addhi >> 8) & 0xFF);
	  inp_ptr = inp_line + 8;
	  count=count-3;
    }
    else
    {
   	  if(strstr((char *)inp_line, "S2") != NULL)
      {
        sscanf(inp_line+2, "%2X%6X", &count, &addhi);
        calc_chksum = count + (addhi & 0xFF) + ((addhi >> 8) & 0xFF) + ((addhi >> 16) & 0xFF);
        inp_ptr = inp_line + 10;
        count=count-4;
      }
      else
      {
   	    if(strstr((char *)inp_line, "S3") != NULL)
        {
          sscanf(inp_line+2, "%2X%8X", &count, &addhi);
          calc_chksum = count + (addhi & 0xFF) + ((addhi >> 8) & 0xFF) + ((addhi >> 16) & 0xFF)+ ((addhi >> 24) & 0xFF);
          inp_ptr = inp_line + 12;
          count=count-5;
        }
      }
    }

    if(count > 0)
    {
	  for(i=0; i<count; i++)
      {
	    sscanf(inp_ptr, "%2X", &bytes[i]);
	    calc_chksum += bytes[i];
	    inp_ptr +=2;
	  }
	  calc_chksum =~calc_chksum;
	  calc_chksum=calc_chksum & 0xFF;
	  sscanf(inp_ptr, "%2X", &file_chksum);

	  if (calc_chksum == file_chksum)
      {
	  	address = addhi;

      	for (i=0; i<count; i++)
		  Buffer[address+i]= bytes[i];

 		image_highwater=MAX(image_highwater,address+i);
		status = 1;
      }
      else
      {
        error("Checksum Error reading Hexfile");
        status = 0;
      }
	}
	return status;
}

void Load_Hex(bool Autoswitch)
{
	int status=1;

    segment=0;
	fp=fopen((frmEprom->FileOpen->FileName.c_str()),"r");

	if(fp==NULL)
    {
      error("Error opening file");
      return;
    }

	ClearBuf();

	while ((fgets(inp_line,80,fp) != NULL) && (status))
      status = Process_Line (inp_line);

	fclose(fp);

	if(status)
    {
        grot = image_highwater;
        image_highwater = 0;
        frmEprom->Message("Hex File loaded OK",0);
        frmEprom->Caption = "Willem Eprom Programmer - " + ExtractFileName(frmEprom->FileOpen->FileName);
	}
    // If failed, check that file is not really S record
    else if((Autoswitch) && ((strstr((char *)inp_line, "S") != NULL)))
        Load_S(False);
}

void Load_Bin(void)
{
    fp=fopen((frmEprom->FileOpen->FileName.c_str()),"rb");

	if(fp==NULL)
    {
      error("Error opening file");
      return;
    }

	ClearBuf();

    grot = fread(Buffer,1,MAXROMSIZE,fp);

	fclose(fp);

    if(grot)
    {
      frmEprom->Message("Binary File loaded OK",0);
      frmEprom->Caption = "Willem Eprom Programmer - " + ExtractFileName(frmEprom->FileOpen->FileName);
    }
}

void Load_S(bool Autoswitch)
{
    int status=1;

    fp=fopen((frmEprom->FileOpen->FileName.c_str()),"rb");

	if(fp==NULL)
    {
      error("Error opening file");
      return;
    }

	ClearBuf();

	while ((fgets(inp_line,80,fp) != NULL) && (status))
      status=Process_Line_S(inp_line);

	fclose(fp);

	if(status)
    {
        grot = image_highwater;
        image_highwater = 0;
        frmEprom->Message("S Record File loaded OK",0);
        frmEprom->Caption = "Willem Eprom Programmer - " + ExtractFileName(frmEprom->FileOpen->FileName);
	}
    // If failed, check that file is not really Intel Hex
    else if((Autoswitch) && (strstr((char *)inp_line, ":") != NULL))
        Load_Hex(False);
}

//---------------------------------------------------------------------------
// Save File Routines
//---------------------------------------------------------------------------

void Save_Bin(void)
{
    if(PIC)
    {
      ShowMessage("This format will not preserve any Fuse and Data information");
    }

    if(UpperCase(ExtractFileExt(frmEprom->FileSave->FileName)) != ".BIN")
      frmEprom->FileSave->FileName = frmEprom->FileSave->FileName + ".BIN";

    fp=fopen((frmEprom->FileSave->FileName.c_str()),"wb");

    if(fp==NULL)
    {
      error("Error opening file");
      return;
    }

    if(PIC)
    {
        fwrite(Buffer,1,BufferSize * 2,fp);
    }
    else
    {
        fwrite(Buffer,1,BufferSize,fp);
    }
    fclose(fp);

    frmEprom->Caption = "Willem Eprom Programmer - " + ExtractFileName(frmEprom->FileSave->FileName);
}

static void d_hexbyte(FILE *fp, int b)
{
		fprintf(fp,"%02X",b);
		check += b;
}

static void d_hexword(FILE *fp, int w)
{
		d_hexbyte(fp,w>>8);
		d_hexbyte(fp,w&0xFF);
}

static void hexrec(FILE *fp, long address, long nw)
{
	int i, n;

    segment=0;
	while ( nw > 0 )
    {
        if (segment != (address >> 16))
        {
            // write new segment offset
    		check = 0;
            segment = (address >> 16);
	    	fprintf(fp,":");
            d_hexbyte(fp,2);
    		d_hexword(fp,0);
            d_hexbyte(fp,4);
    		d_hexbyte(fp,segment >> 8);         // write hi-lo format
    		d_hexbyte(fp,segment & 0xFF);
    		d_hexbyte(fp,(-check)&0xFF);
    		fprintf(fp,"\n");
        }

		check = 0;
		fprintf(fp,":");
		n = (nw > 16)? 16: nw;

		d_hexbyte(fp,n);
		d_hexword(fp,(address & 0xffff));

		d_hexbyte(fp,0);
		for ( i=0; i < n; ++i ) {
			d_hexbyte(fp,Buffer[address+i]);
		}
		d_hexbyte(fp,(-check)&0xFF);
		fprintf(fp,"\n");
		nw -= 16;
		address += 16;
	}
}

void Save_Hex(void)
{
    if(UpperCase(ExtractFileExt(frmEprom->FileSave->FileName)) != ".HEX")
      frmEprom->FileSave->FileName = frmEprom->FileSave->FileName + ".HEX";

    fp=fopen((frmEprom->FileSave->FileName.c_str()),"w");

	if(fp==NULL)
    {
      error("Error opening file");
      return;
    }

    if(PIC){
        hexrec(fp,0,BufferSize * 2);
        hexrec(fp,0x400E,1);
        hexrec(fp,0x4200,128);
    }
    else
    {
        hexrec(fp,0,BufferSize);
    }
    fprintf(fp,":00000001FF\n");
    fclose(fp);

    frmEprom->Caption = "Willem Eprom Programmer - " + ExtractFileName(frmEprom->FileSave->FileName);
}

static void S_rec(FILE *fp, long address, long nw, int S3)
{
  long i, n;

  while ( nw > 0 )
  {
  	n = (nw > 16)? 16: nw;

  	if(S3)
    {
      // 4 bytes address, 1 byte checksum
      fprintf(fp,"S3%02X%08X",n+5,address);
      check = n + 1 + 5 + (address & 0xFF) + ((address >> 8) & 0xFF) + ((address >> 16) & 0xFF)+ ((address >> 24) & 0xFF);
    }
    else
    {
      // 2 bytes address, 1 byte checksum
      fprintf(fp,"S1%02X%04X",n+3,address);
      check = n + 1 + 3 + (address & 0xFF) + ((address >> 8) & 0xFF);
    }

	for ( i=0; i < n; ++i )
	  d_hexbyte(fp,Buffer[address+i]);

	d_hexbyte(fp,(-check) & 0xFF);
	fprintf(fp,"\n");
	nw -= 16;
	address += 16;
  }
}

void Save_S(void)
{
  // We could use S1,S2 and S3 dependant on address size
  // but for simplicity we force the entire file to be
  // S1 or S3 type records.

  int S3=0;

  if(BufferSize > 0xFFFF)
  {
    S3 = -1;
  }

  if(UpperCase(ExtractFileExt(frmEprom->FileSave->FileName)) != ".S")
    frmEprom->FileSave->FileName = frmEprom->FileSave->FileName + ".S";

  fp=fopen((frmEprom->FileSave->FileName.c_str()),"w");

  if(fp==NULL)
  {
    error("Error opening file");
    return;
  }

  if(PIC){
    S_rec(fp,0,BufferSize * 2,S3);
    S_rec(fp,0x400E,1,S3);
    S_rec(fp,0x4200,128,S3);
  }
  else
  {
    S_rec(fp,0,BufferSize,S3);
  }
  fprintf(fp,"S9030000FC\n");
  fclose(fp);

  frmEprom->Caption = "Willem Eprom Programmer - " + ExtractFileName(frmEprom->FileSave->FileName);
}
//---------------------------------------------------------------------------
// GUI interface routines
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::Message(String Notice, int Level)
{
    switch (Level)
    {
      case 1 : StatusBar->Font->Color = clRed;
               MessageBeep(MB_ICONEXCLAMATION);
               break;

      case 2 : StatusBar->Font->Color = clBlack;
               Beep();
               break;

      default: StatusBar->Font->Color = clBlue;
    }

    StatusBar->Caption = Notice;
    StatusBar->Refresh();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::SetProcessGauge(String Notice, int Max, TColor Colour)
{
  Message(Notice,0);
  cancel = FALSE;
  btnCancel->Visible  = TRUE;

  if(Colour == clWhite)
    gProcess->BackColor = clNavy;
  else
    gProcess->BackColor = clWhite;

  gProcess->Progress  = 0;
  gProcess->MaxValue  = Max;
  gProcess->ForeColor = Colour;
  gProcess->Visible   = TRUE;
  gProcess->Refresh();
  btnCancel->Refresh();
  btnCancel->SetFocus();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::SetBufferSize(int Size)
{
    TGridRect myRect;
    BufferSize = Size;

    // Change size of HexGrid (I always want a scrollbar!)
    if((BufferSize / 16) < 28)
        HexGrid->RowCount = 29;
    else
        HexGrid->RowCount = (BufferSize / 16) + 1;

    // Sort out size of ProgGrid
    if (PIC)
        ProgGrid->RowCount = (BufferSize / 8) + 1;
    else
        ProgGrid->RowCount = 2;

    // Sort out what we need to see
    HexGrid->Visible  = FALSE;
    ProgGrid->Visible = FALSE;
    if(PIC)
    {
        if(!C50)
        {
            // Need Both Grids
            ProgGrid->Height     = frmEprom->Height - 160;
            HexGrid->Cells[0][0] = " Data";
            HexGrid->Height      = 112;

            HexGrid->Visible     = TRUE;
            ProgGrid->Visible    = TRUE;
        }
        else
        {
            // Just Program Grid
            ProgGrid->Height  = frmEprom->Height - 26;
            ProgGrid->Visible = TRUE;
        }
    }
    else
    {
        // Just Data Grid
        HexGrid->Cells[0][0] = "Offset";
        HexGrid->Top         = 0;
        HexGrid->Height      = frmEprom->Height - 26;
        HexGrid->Visible     = TRUE;
    }

    // Lose Selected Boxes
	myRect.Left   = -1;
	myRect.Top    = -1;
	myRect.Right  = -1;
	myRect.Bottom = -1;
	HexGrid->Selection  = myRect;
    ProgGrid->Selection = myRect;

    HexGrid->Refresh();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::SetFuses()
{
    // Set the comboboxes correctly for the fuse data

    FuseUpdate = TRUE;

   	cp=(fuses&CP)? CP: 0;
	wd=(fuses&WDTE)? WDTE: 0;
	if(F84)
        pt=(fuses&PWRTE)? 0: PWRTE;
    else
    	pt=(fuses&PWRTE)? PWRTE: 0;

	osc=(fuses&3);

	if(!cp) cbCodeProtect->ItemIndex = 1;
	else cbCodeProtect->ItemIndex = 0;
	if(pt)cbPic2->ItemIndex = 1;
	else cbPic2->ItemIndex = 0;
	if(wd)cbWatchDog->ItemIndex = 1;
	else cbWatchDog->ItemIndex = 0;
    cbOscillator->ItemIndex = osc;

    FuseUpdate = FALSE;
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::mnuTestClick(TObject *Sender)
{
    test_hw();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::About1Click(TObject *Sender)
{
    ShowMessage("A Windows version of Willem's Eprom Programmer Software\n"
                " constructed by Mike Coates (mike.coates@btinternet.com)\n"
                "\n"
                " Hints, Bug Reports, Improvements, please let either one\n"
                " of us know. For full details of the eprom programmer and\n"
                "             other projects visit www.willem.org\n"
                "\n"
                "DriverLINX Port I/O Driver for Win95 and WinNT\n"
                "(C) Copyright 1996, Scientific Software Tools, Inc.\n"
                "All Rights Reserved.\n"
                "\n"
                "TDLPortIO: DriverLINX Port IO Driver wrapper component\n"
                "Copyright (c) 1999 John Pappas (DiskDude). All rights reserved.\n"
                "\n"
                "                         Version Beta 0.85\n");
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::FormCreate(TObject *Sender)
{
    // Get performance details for timing
    QueryPerformanceFrequency(&PerformanceFrequency);

    // Port Access DLL's for Windows 95 and NT
    DLPortIO->DriverPath=ExtractFileDir(ParamStr(0));
    DLPortIO->OpenDriver();

    if (!DLPortIO->ActiveHW)
       MessageDlg("Could not open the DriverLINX driver.",mtError, TMsgDlgButtons() << mbOK, 0);

    // Default controls to correct values (add registry settings later!)
    cbPort->ItemIndex = 0;
    cbMethod->ItemIndex = 0;
    cb24CPin7->ItemIndex = 0;
    cbFastErase->ItemIndex = 0;
    cbCheckType->ItemIndex = 0;
    cbSkipFF->ItemIndex = 1;

    edOffset->Text = "0000";

    // Assign dip switch elements to array for ease of use
    DipSwitch[1]  = dip1;
    DipSwitch[2]  = dip2;
    DipSwitch[3]  = dip3;
    DipSwitch[4]  = dip4;
    DipSwitch[5]  = dip5;
    DipSwitch[6]  = dip6;
    DipSwitch[7]  = dip7;
    DipSwitch[8]  = dip8;
    DipSwitch[9]  = dip9;
    DipSwitch[10] = dip10;
    DipSwitch[11] = dip11;
    DipSwitch[12] = dip12;

    // Setup headings for Hex Grid - Widths are 52,16 * 16, 114
    HexGrid->Cells[1][0] = "0";
    HexGrid->Cells[2][0] = "1";
    HexGrid->Cells[3][0] = "2";
    HexGrid->Cells[4][0] = "3";
    HexGrid->Cells[5][0] = "4";
    HexGrid->Cells[6][0] = "5";
    HexGrid->Cells[7][0] = "6";
    HexGrid->Cells[8][0] = "7";
    HexGrid->Cells[9][0] = "8";
    HexGrid->Cells[10][0] = "9";
    HexGrid->Cells[11][0] = "A";
    HexGrid->Cells[12][0] = "B";
    HexGrid->Cells[13][0] = "C";
    HexGrid->Cells[14][0] = "D";
    HexGrid->Cells[15][0] = "E";
    HexGrid->Cells[16][0] = "F";
    HexGrid->Cells[17][0] = "Ascii";

    ProgGrid->Cells[0][0] = "Program";
    ProgGrid->Cells[1][0] = "0";
    ProgGrid->Cells[2][0] = "1";
    ProgGrid->Cells[3][0] = "2";
    ProgGrid->Cells[4][0] = "3";
    ProgGrid->Cells[5][0] = "4";
    ProgGrid->Cells[6][0] = "5";
    ProgGrid->Cells[7][0] = "6";
    ProgGrid->Cells[8][0] = "7";
    ProgGrid->Cells[9][0] = "Ascii";

    power_down();
    SetBufferSize(0);
    ClearBuf();

    LoadSettings();

    winstat();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::mnuExitClick(TObject *Sender)
{
    // Leave programmer in safe state for chip removal etc.
    power_down();
    SaveSettings();
    Application->Terminate();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::FormClose(TObject *Sender, TCloseAction &Action)
{
    // Leave programmer in safe state for chip removal etc.
    power_down();
    SaveSettings();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::cbPortChange(TObject *Sender)
{
  switch(cbPort->ItemIndex)
  {
    case 0:     // 378
            {
                poort  = 0x378;
                ipoort = 0x379;
                spoort = 0x37a;
                break;
            }
    case 1:     // 278
            {
                poort  = 0x278;
                ipoort = 0x279;
                spoort = 0x27a;
                break;
            }
    case 2:     // 3bc
            {
                poort  = 0x3bc;
                ipoort = 0x3bd;
                spoort = 0x3be;
                break;
            }
  }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::cbMethodChange(TObject *Sender)
{
  if(cbMethod->ItemIndex == 0)
    ep = 1;
  else
    ep = 0;
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::cb24CPin7Change(TObject *Sender)
{
    pin7 = cb24CPin7->ItemIndex;
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::cbFastEraseChange(TObject *Sender)
{
    qe = cbFastErase->ItemIndex;
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::MaskEdit1MouseDown(TObject *Sender,
      TMouseButton Button, TShiftState Shift, int X, int Y)
{
    TPoint P, Q;

    // Popup Device Menu

    if(Button==mbLeft)
    {
        P.x = X + edDevice->Left + Panel4->Left;
	    P.y = Y + edDevice->Top + Panel4->Top;
	    Q = ClientToScreen(P);
        mnuDevice->Popup(Q.x,Q.y);
    }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M24C_Click(TObject *Sender)
{
    // Setup for selected 24Cxx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;

    edDevice->Caption = Selected->Caption;
    C622=0;PIC=0;C89=0;F84=0;C50=0;C93=0;
    ZIFImage = 240;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(256);
                 eep=1;
                 break;

        case 2 : SetBufferSize(512);
                 eep=1;
                 break;

        case 3 : SetBufferSize(1024);
                 eep=1;
                 break;

        case 4 : SetBufferSize(2048);
                 eep=1;
                 break;

        case 5 : SetBufferSize(4096);
                 eep=2;
                 break;

        case 6 : SetBufferSize(8192);
                 eep=2;
                 break;

        case 7 : SetBufferSize(16384);
                 eep=2;
                 break;

        case 8 : SetBufferSize(32768);
                 eep=2;
                 break;

        case 9 : SetBufferSize(65536);
                 eep=2;
                 break;
    }
    Dip(0);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M27C_Click(TObject *Sender)
{
    // Setup for selected 27Cxx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;

    edDevice->Caption = Selected->Caption;
    C622=0;PIC=0;C89=0;F84=0;C50=0;C93=0;
    eep=0;pe=0;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(4096);
                 epr64=0;
                 ZIFImage=24;
                 Dip(9);
                 break;

        case 2 : SetBufferSize(8192);
                 epr64=1;
                 ZIFImage=28;
                 Dip(1);
                 break;

        case 3 : SetBufferSize(16384);
                 epr64=1;
                 ZIFImage=28;
                 Dip(2);
                 break;

        case 4 : SetBufferSize(32768);
                 epr64=0;
                 ZIFImage=28;
                 Dip(3);
                 break;

        case 5 : SetBufferSize(65536);
                 epr64=0;
                 ZIFImage=28;
                 Dip(4);
                 break;

        case 6 : SetBufferSize(131072);
                 epr64=0;
                 ZIFImage=32;
                 Dip(7);
                 break;

        case 7 : SetBufferSize(262144);
                 epr64=0;
                 ZIFImage=32;
                 Dip(7);
                 break;

        case 8 : SetBufferSize(0x400000L/8);
                 epr64=0;
                 ZIFImage=32;
                 Dip(10);
                 break;
    }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M28C_Click(TObject *Sender)
{
    // Setup for selected 28Cxx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;

    edDevice->Caption = Selected->Caption;
    C622=0;PIC=0;C89=0;F84=0;C50=0;C93=0;
    eep=0;pe=1;

    // Is this Needed ? - Reported as required for 28C65 at least
    epr64=1;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(8192);
                 ZIFImage=28;
                 Dip(5);
                 break;

        case 2 : SetBufferSize(16384);
                 ZIFImage=28;
                 Dip(5);
                 break;

        case 3 : SetBufferSize(32768);
                 ZIFImage=28;
                 Dip(6);
                 break;

        case 4 : SetBufferSize(65536);
                 ZIFImage=28;
                 Dip(8);
                 break;

        case 5 : SetBufferSize(131072);
                 ZIFImage=32;
                 Dip(8);
                 break;

        case 6 : SetBufferSize(262144);
                 ZIFImage=32;
                 Dip(8);
                 break;

        case 7 : SetBufferSize(0x400000/8);
                 ZIFImage=32;
                 Dip(11);
                 break;
    }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M28F_Click(TObject *Sender)
{
    // Setup for selected 28Fxx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;

    edDevice->Caption = Selected->Caption;
    C622=0;PIC=0;C89=0;F84=0;C50=0;C93=0;
    eep=3;pe=0;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(8192);
                 ZIFImage=28;
                 Dip(7);
                 break;

        case 2 : SetBufferSize(16384);
                 ZIFImage=28;
                 Dip(7);
                 break;

        case 3 : SetBufferSize(32768);
                 ZIFImage=28;
                 Dip(7);
                 break;

        case 4 : SetBufferSize(65536);
                 ZIFImage=28;
                 Dip(7);
                 break;

        case 5 : SetBufferSize(131072);
                 ZIFImage=32;
                 Dip(7);
                 break;

        case 6 : SetBufferSize(262144);
                 ZIFImage=32;
                 Dip(7);
                 break;

        case 7 : SetBufferSize(0x400000/8);
                 ZIFImage=32;
                 Dip(12);
                 break;
    }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M29F_Click(TObject *Sender)
{
    // Setup for selected 29Fxx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;

    edDevice->Caption = Selected->Caption;
    C622=0;PIC=0;C89=0;F84=0;C50=0;C93=0;
    eep=4;pe=0;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(8192);
                 ZIFImage=28;
                 Dip(1);
                 break;

        case 2 : SetBufferSize(16384);
                 ZIFImage=28;
                 Dip(2);
                 break;

        case 3 : SetBufferSize(32768);
                 ZIFImage=28;
                 Dip(7);
                 break;

        case 4 : SetBufferSize(65536);
                 ZIFImage=28;
                 Dip(7);
                 break;

        case 5 : SetBufferSize(131072);
                 ZIFImage=32;
                 Dip(7);
                 break;

        case 6 : SetBufferSize(262144);
                 ZIFImage=32;
                 Dip(7);
                 break;

        case 7 : SetBufferSize(0x400000/8);
                 ZIFImage=32;
                 Dip(12);
                 break;
    }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M12C_Click(TObject *Sender)
{
    // Setup for selected 12C5xx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;

    edDevice->Caption = Selected->Caption;
    C622=0;C50=1;C89=0;F84=0;PIC=1;C93=0;eep=0;
    fuses=picbuf[0x2007];
    ZIFImage = 12;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(512);
                 break;

        case 2 : SetBufferSize(1024);
                 break;
    }
    Dip(0);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M16C_Click(TObject *Sender)
{
    // Setup for selected 16C6xx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;
    ZIFImage = 16;

    edDevice->Caption = Selected->Caption;
    C622=1;C50=0;C89=0;F84=0;PIC=1;C93=0;eep=0;
    fuses=picbuf[0x2007];
    ZIFImage = 16;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(512);
                 break;

        case 2 : SetBufferSize(1024);
                 break;

        case 3 : SetBufferSize(2048);
                 break;
    }
    Dip(0);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::M93C_Click(TObject *Sender)
{
    // Setup for selected 93Cxx device

    TMenuItem *Selected = (TMenuItem *)(Sender);
    int DeviceID = Selected->Tag;

    edDevice->Caption = Selected->Caption;
    C622=0;C50=0;C89=0;F84=0;PIC=0;C93=1;
    ZIFImage = 93;

    switch(DeviceID)
    {
        case 1 : SetBufferSize(128);
                 C66=0;C56=0;A46=1;B46=0;
                 break;

        case 2 : SetBufferSize(128);
                 C66=0;C56=0;B46=1;A46=0;
                 break;

        case 3 : SetBufferSize(128);
                 C66=0;C56=0;B46=1;A46=0;
                 break;

        case 4:  SetBufferSize(256);
                 C56=1;C66=0;B46=1;A46=0;
                 break;

        case 5:  SetBufferSize(512);
                 C56=0;C66=1;B46=0;A46=0;
                 break;
    }
    Dip(0);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::N16C841Click(TObject *Sender)
{
    C89=0;F84=0;C93=0;C622=0;PIC=1;C50=0;eep=0;
    edDevice->Caption = "16C84";
    SetBufferSize(1024);
    fuses=picbuf[0x2007];
    ZIFImage = 16;
    Dip(0);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::N16F841Click(TObject *Sender)
{
    C89=0;F84=1;PIC=1;C50=0;C93=0;C622=0;eep=0;
    edDevice->Caption = "16F84";
    SetBufferSize(1024);
    fuses=picbuf[0x2007];
    ZIFImage = 16;
    Dip(0);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::mnuEmptyClick(TObject *Sender)
{
    // Check eprom empty

    int ErrorLocation;

    if((BufferSize > 0) && (PIC == 0))
    {
        if((ErrorLocation=Check()) == -1)
        {
          Message("Device is Empty",2);
        }
        else
        {
          if(ErrorLocation== -2)
            Message("Device Check was Cancelled",1);
          else
            Message(Format("Device is NOT Empty ($%4.4x)",ARRAYOFCONST((ErrorLocation))),1);
        }
    }
    else
        Message("Cannot check this Device",1);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::Clear1Click(TObject *Sender)
{
    frmEprom->Caption = "Willem Eprom Programmer";
    memset(Buffer,255,524288);     // Pic Data or Parallel Eprom
    fuses = 0;
    winstat();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::HexGridDrawCell(TObject *Sender, int Col,
      int Row, TRect &Rect, TGridDrawState State)
{
  // Data Grid (Parallel Eprom & Data from PICS)
  int  DataSize;
  int  Offset = (Row - 1) * 16;
  int  DataStart = 0;
  int  Count;
  char ch[17];

  if(PIC)
  {
    DataSize  = 128;
    DataStart = 0x4200;
  }
  else
    DataSize = BufferSize;

  if(Row > 0)
  {
    if(Offset < DataSize)
    {
      switch(Col)
      {
          case 0  : HexGrid->Canvas->TextRect(Rect, Rect.Left + 1, Rect.Top + 1, Format("$%6.6x",ARRAYOFCONST((Offset))));
                    break;

          case 17 : ch[16]=0;
                    for(Count=0;Count<16;Count++)
                    {
                      if((Buffer[DataStart+Offset+Count] < 33) || (Buffer[DataStart+Offset+Count] > 127))
                        ch[Count] = '.';
                      else
                        ch[Count] = Buffer[DataStart+Offset+Count];
                    }

                    HexGrid->Canvas->TextRect(Rect, Rect.Left + 1, Rect.Top + 1, ch);
                    break;

          default : Offset += (Col - 1);
                    if(Offset <= DataSize)
                    {
                      HexGrid->Canvas->TextRect(Rect, Rect.Left + 1, Rect.Top + 1,Format("%2.2x",ARRAYOFCONST((Buffer[DataStart+Offset]))));
                    }
                    break;
      }
    }
  }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::ProgGridDrawCell(TObject *Sender, int Col,
      int Row, TRect &Rect, TGridDrawState State)
{
  // Program Grid (Program code from PICS)
  int  Offset = (Row - 1) * 16;
  int  Count;
  char ch[17];

  if(Row > 0)
  {
    if(Offset < (BufferSize * 2))
    {
      switch(Col)
      {
          case 0  : ProgGrid->Canvas->TextRect(Rect, Rect.Left + 1, Rect.Top + 1, Format("$%6.6x",ARRAYOFCONST((Offset/2))));
                    break;

          case 9  : ch[16]=0;
                    for(Count=0;Count<16;Count++)
                    {
                      if((Buffer[Offset+Count] < 33) || (Buffer[Offset+Count] > 127))
                        ch[Count] = '.';
                      else
                        ch[Count] = Buffer[Offset+Count];
                    }

                    ProgGrid->Canvas->TextRect(Rect, Rect.Left + 1, Rect.Top + 1, ch);
                    break;

          default : Offset += (Col - 1);
                    if(Offset <= (BufferSize * 2))
                    {
                      ProgGrid->Canvas->TextRect(Rect, Rect.Left + 1, Rect.Top + 1,Format("%4.4x",ARRAYOFCONST((picbuf[(Row - 1) * 8 + (Col - 1)]))));
                    }
                    break;
      }
    }
  }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::mnuReadClick(TObject *Sender)
{
    if(BufferSize > 0)
    {
      if(C50)P50read();
      else if(PIC)Pread();
      else if(C93)read_93();
      else if(!eep) { if(test_hw()) CRead(); }
      else if(eep==3||eep==4) { if(test_hw()) CRead(); }
      else if(eep==2) read_65();
      else read_16();
    }
    else
        Message("Please select a Device Type",1);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::mnuProgramClick(TObject *Sender)
{
  // Write Buffer to Device
  if(BufferSize > 0)
  {
    if(grot > 0)
    {
      if(C50) PIC508_pr();
      else if(PIC) PIC_pr();
      else if(C93) prog_93();
      else if(!eep){ if(test_hw()) Program8751(); }
      else if(C89) { if(test_hw()) Program_89(); }
      else if(eep==3) prog_f28();
      else if(eep==4) prog_f29();
      else if(eep==2) eprogram65();
      else prog_16();
    }
    else
      Message("Please load some information to program",1);
  }
  else
    Message("Please select a Device Type",1);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::mnuCompareClick(TObject *Sender)
{
  if(BufferSize > 0)
  {
    if(grot > 0)
    {
      if(C50) verify50();
      else if(PIC) { if(test_hw()) verify(); }              //ARW: added test_hw
      else if(!eep) { if(test_hw()) Compare(); }            //ARW: added test_hw
      else if(eep==3||eep==4) { if(test_hw()) Compare(); }  //ARW: added test_hw
      else if(eep==2) verify_65();
      else verify_16();
    }
    else
      Message("Please load some information to compare against",1);
  }
  else
    Message("Please select a Device Type",1);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::cbPICChange(TObject *Sender)
{
    // Calculate new Fuses value

    if(!FuseUpdate)
    {
      cp=(cbCodeProtect->ItemIndex==1) ? 0: CP;

      wd=(cbPic2->ItemIndex==1)? WDTE: 0;

      if(F84)
          pt=(cbWatchDog->ItemIndex==1)? 0: PWRTE;
      else
          pt=(cbWatchDog->ItemIndex==1)? PWRTE: 0;

      osc=cbOscillator->ItemIndex;

      fuses = cp | wd | pt | osc;
      picbuf[0x2007] = fuses;
    }
}
//---------------------------------------------------------------------------


void __fastcall TfrmEprom::mnuLoadClick(TObject *Sender)
{
  // Get filename & load selected file
  AnsiString Extension;

  if (FileOpen->Execute())
  {
    Extension = UpperCase(ExtractFileExt(FileOpen->FileName));

    if(Extension==".S")
    {
      Load_S(True);
    }
    else
    {
      if(Extension==".HEX")
        Load_Hex(True);
      else
        Load_Bin();
    }

    if(PIC)
    {
      BufferSize=1024;
      fuses=picbuf[0x2007];
    }
    winstat();

    // Default start Directory for future loads & Saves
    FileOpen->InitialDir = ExtractFilePath(FileOpen->FileName);
    FileSave->InitialDir = FileOpen->InitialDir;
    FileSave->FileName   = FileOpen->FileName;
  }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::mnuSaveClick(TObject *Sender)
{
  if(PIC) FileSave->FilterIndex = 2;
  else FileSave->FilterIndex = 1;

  if (FileSave->Execute())
  {
    switch (FileSave->FilterIndex)
    {
      case 1 : Save_Bin();
               break;

      case 2 : Save_Hex();
               break;

      case 3 : Save_S();
               break;
    }
  }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::btnCancelClick(TObject *Sender)
{
  // Cancel Button - Set Flag to Exit Processing
  cancel = TRUE;
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::btnCancelKeyPress(TObject *Sender, char &Key)
{
  // Cancel Button - Set Flag to Exit Processing
  if(Key == 27)
  {
    cancel = True;
    Key = 0;
  }
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::EraseClick(TObject *Sender)
{
  if(BufferSize > 0)
  {
      if(PIC)         P_erase();
      else if(eep==4) erase_f29();
      else if(eep==3) erase_f28();
      else
        Message("Cannot Erase this Device",1);
  }
  else
    Message("Please select a Device Type",1);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::edOffsetExit(TObject *Sender)
{
  edOffset->Text = Format("%4.4x",ARRAYOFCONST(((int)st)));
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::edOffsetChange(TObject *Sender)
{
  sscanf(edOffset->Text.c_str(),"%X",&st);
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::LoadSettings(void)
{
    int    ct1,ct2;
    String DeviceType;

    TRegistry *MyKey = new TRegistry;

    if(MyKey->OpenKey("\\Software\\Willem\\Eprom\\",True))
    {
      if(MyKey->ValueExists("PORT"))
      {
        cbPort->ItemIndex = MyKey->ReadInteger("PORT");
        cbPortChange(0);
      }

      if(MyKey->ValueExists("METHOD"))
      {
        cbMethod->ItemIndex = MyKey->ReadInteger("METHOD");
        cbMethodChange(0);
      }

      if(MyKey->ValueExists("PIN7"))
      {
        cb24CPin7->ItemIndex = MyKey->ReadInteger("PIN7");
        cb24CPin7Change(0);
      }

      if(MyKey->ValueExists("ERASE"))
      {
        cbFastErase->ItemIndex = MyKey->ReadInteger("ERASE");
        cbFastEraseChange(0);
      }

      if(MyKey->ValueExists("CHECKSUM"))
      {
        cbCheckType->ItemIndex = MyKey->ReadInteger("CHECKSUM");
      }

      if(MyKey->ValueExists("SKIP-FF"))
      {
        cbSkipFF->ItemIndex = MyKey->ReadInteger("SKIP-FF");
      }

      if(MyKey->ValueExists("DEVICE"))
      {
        // Loop through device menu, look for this entry
        // Will work with 1 level of submenu only!

        DeviceType = MyKey->ReadString("DEVICE");

        for(ct1=0;ct1<mnuDevice->Items->Count;ct1++)
        {
            if(mnuDevice->Items->Items[ct1]->Count > 0)
            {
                for(ct2=0;ct2<mnuDevice->Items->Items[ct1]->Count;ct2++)
                {
                    if(mnuDevice->Items->Items[ct1]->Items[ct2]->Caption == DeviceType)
                        mnuDevice->Items->Items[ct1]->Items[ct2]->Click();
                }
            }
            else
            {
                if(mnuDevice->Items->Items[ct1]->Caption == DeviceType)
                    mnuDevice->Items->Items[ct1]->Click();
            };
        }
      }

      if(MyKey->ValueExists("FILES"))
      {
        FileOpen->InitialDir = MyKey->ReadString("FILES");
        FileSave->InitialDir = FileOpen->InitialDir;
      }

      if(MyKey->ValueExists("BURN"))
        burn = MyKey->ReadInteger("BURN");

      if(MyKey->ValueExists("E8751"))
        e8751 = MyKey->ReadInteger("E8751");
    }
    MyKey->CloseKey();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::SaveSettings(void)
{
    TRegistry *MyKey = new TRegistry;

    if(MyKey->OpenKey("\\Software\\Willem\\Eprom\\",True))
    {
      // Form Controls
      MyKey->WriteInteger("PORT",cbPort->ItemIndex);
      MyKey->WriteInteger("METHOD",cbMethod->ItemIndex);
      MyKey->WriteInteger("PIN7",cb24CPin7->ItemIndex);
      MyKey->WriteInteger("ERASE",cbFastErase->ItemIndex);
      MyKey->WriteInteger("CHECKSUM",cbCheckType->ItemIndex);
      MyKey->WriteInteger("SKIP-FF",cbSkipFF->ItemIndex);
      MyKey->WriteString("DEVICE",edDevice->Caption);
      MyKey->WriteString("FILES",FileOpen->InitialDir);

      // Hidden Options (at the Moment)
      MyKey->WriteInteger("BURN",burn);
      MyKey->WriteInteger("E8751",e8751);
    }
    MyKey->CloseKey();
}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::imgFlip1Click(TObject *Sender)
{
    // Flip between PIC settings and ZIF Position
    if (PIC)
      pPICSettings->Visible = !pPICSettings->Visible;

    if ((eep==1) || (eep==2))
      p24CSettings->Visible = !p24CSettings->Visible;

    if (eep==3)
      pF28Settings->Visible = !pF28Settings->Visible;

    if (eep+C93+PIC+C50 == 0)
      p27CSettings->Visible = !p27CSettings->Visible;

}
//---------------------------------------------------------------------------

void __fastcall TfrmEprom::cbCheckTypeChange(TObject *Sender)
{
  winstat();    
}
//---------------------------------------------------------------------------

