﻿using System;
using System.Collections.Generic;
using System.Text;
using ZXMAK2;
using ZXMAK2.Engine.Z80;
using ZXMAK2.Entities;
using ZXMAK2.Hardware.General;
using ZXMAK2.Interfaces;
using ZXMAK2.Serializers;
using ZXMAK2.Serializers.TapeSerializers;

namespace TapeRecorderPlugin
{
	public class TapeRecorder : BusDeviceBase
	{
		enum Tape_Status { ReadyForPilotTone = 1, ReadyForSync1, ReadyForSync2, ReadyForData };
		Tape_Status status = Tape_Status.ReadyForPilotTone;

		private Z80CPU m_cpu;
		private TapeDevice m_tapeDevice;

		const int PILOT_TSTATE = 2168;
		const int SYNC1_TSTATE = 667;
		const int SYNC2_TSTATE = 735;
		const int ONE_TSTATE = 1710;
		const int ZERO_TSTATE = 855;

		List<byte> data = new List<byte>();
		byte received_byte;
		int received_bits_count;
		long last_tstate_save = -1;

		byte last_tape_bit = 0;
		long last_tstate = -1;
		int pilot_count = 0;

		public override string Name
		{
			get { return "Tape recorder"; }
		}

		public override string Description
		{
			get { return "Simple tape (.TAP) recorder (c) Eltaron"; }
		}

		public override BusDeviceCategory Category
		{
			get { return BusDeviceCategory.Tape; }
		}

		public override void BusInit(IBusManager bmgr)
		{
			m_cpu = bmgr.CPU;
			m_tapeDevice = bmgr.FindDevice<TapeDevice>();

			if (m_tapeDevice != null)
			{
				bmgr.SubscribeWrIo(0x0001, 0x0000, writePortFE);
				bmgr.SubscribePreCycle(busPreCycle);

				// Delete TAP serializer. Wanna keep recorder as plugin, so need reflection-based hack here.
				var field = typeof(ZXMAK2.Engine.BusManager).GetField("m_loadManager", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic | System.Reflection.BindingFlags.GetField);
				var loadManager = (LoadManager)field.GetValue(bmgr);
				field = typeof(SerializeManager).GetField("_formats", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic | System.Reflection.BindingFlags.GetField);
				var formats = (Dictionary<string, FormatSerializer>)field.GetValue(loadManager);
				if (formats.ContainsKey("TAP"))
					formats.Remove("TAP");

				bmgr.AddSerializer(new TapWriteSerializer(m_tapeDevice));
			}
			else
				LogAgent.Warn("Tape recorder disabled because no Tape Player found");
		}

		private void busPreCycle(int frameTact)
		{
			if (last_tstate_save > 0 && m_cpu.Tact - last_tstate_save > 70000) // FIXME: stupid check to determine end of saving
			{
				var tap = TapSerializer.getBlockPeriods(data.ToArray(), 0, data.Count, 2168, 667, 735, 855, 1710, (data[0] < 4) ? 8064 : 3220, 1000, 8);

				m_tapeDevice.Blocks.Add(new TapeBlock()
				{
					Periods = tap,
					Description = TapSerializer.getBlockDescription(data.ToArray(), 0, data.Count),
					TapData = data.ToArray(),
					TzxId = -1
				});

				data.Clear();
				last_tstate_save = -1;
			}
		}

		public override void BusConnect()
		{
			
		}

		public override void BusDisconnect()
		{
			
		}

		private void writePortFE(ushort addr, byte value, ref bool iorqge)
		{
			byte tape_bit = (value & 0x04) == 0x04 ? (byte)1 : (byte)0;
			if (tape_bit != last_tape_bit)
			{
				var length = m_cpu.Tact - last_tstate;

				if (length > 0.95 * PILOT_TSTATE && length < 1.05 * PILOT_TSTATE)
					pilot_count++;
				else if (length > 0.95 * SYNC1_TSTATE && length < 1.05 * SYNC1_TSTATE)
				{
					if (pilot_count > 1000) // FIXME: veeeeeery bad check!
						status = Tape_Status.ReadyForSync2;
				}
				else if (length > 0.95 * SYNC2_TSTATE && length < 1.05 * SYNC2_TSTATE)
				{
					status = Tape_Status.ReadyForData;
					received_bits_count = 0;
					received_byte = 0;
					data.Clear();
				}
				else if (status == Tape_Status.ReadyForData)
				{
					byte v = 0;
					if (length > 0.95 * ONE_TSTATE && length < 1.05 * ONE_TSTATE)
						v = 1;
					else if (length > 0.95 * ZERO_TSTATE && length < 1.05 * ZERO_TSTATE)
						v = 0;
					else
					{
						// R TAPE LOADING ERROR
						status = Tape_Status.ReadyForPilotTone;
						pilot_count = 0;
					}

					if (tape_bit == 1) // store only front edge
					{
						received_bits_count++;
						received_byte = (byte)((received_byte << 1) | v);

						if (received_bits_count == 8)
						{
							data.Add(received_byte);
							last_tstate_save = m_cpu.Tact;
							received_bits_count = 0;
							received_byte = 0;
						}
					}
				}
				else
				{
					status = Tape_Status.ReadyForPilotTone;
					pilot_count = 0;
				}

				last_tstate = m_cpu.Tact;
				last_tape_bit = tape_bit;
			}
		}
	}

	public class TapWriteSerializer : TapSerializer
	{
		public override bool CanSerialize { get { return true; } }

		private ITapeDevice _tape;

		public TapWriteSerializer(ITapeDevice tape) : base(tape)
		{
			_tape = tape;
		}

		public override void Serialize(System.IO.Stream stream)
		{
			foreach (var block in _tape.Blocks)
			{
				var length = block.TapData.Length;
				stream.WriteByte((byte)(length % 256));
				stream.WriteByte((byte)(length / 256));
				stream.Write(block.TapData, 0, length);
			}
		}
	}
}
