#include <avr/pgmspace.h>
#include <util/delay.h>
#include "lcd1100.h"

#define LCD_WIDTH 95
#define LCD_HEIGHT 64
#define LCD_FONTHEIGHT 8
#define LCD_FONTWIDTH 5
#define LCD_ROWS 8
#define LCD_COLUMNS 16

//char bitaddr;
// lookup table, here you can modify the font

static const unsigned char FontLookup[96][5] PROGMEM ={ 
{0x00, 0x00, 0x00, 0x00, 0x00},// (space) 
{0x00, 0x00, 0x5F, 0x00, 0x00},// ! 
{0x00, 0x07, 0x00, 0x07, 0x00},// " 
{0x14, 0x7F, 0x14, 0x7F, 0x14},// # 
{0x24, 0x2A, 0x7F, 0x2A, 0x12},// $ 
{0x23, 0x13, 0x08, 0x64, 0x62},// % 
{0x36, 0x49, 0x55, 0x22, 0x50},// & 
{0x00, 0x05, 0x03, 0x00, 0x00},// ' 
{0x00, 0x1C, 0x22, 0x41, 0x00},// ( 
{0x00, 0x41, 0x22, 0x1C, 0x00},// ) 
{0x08, 0x2A, 0x1C, 0x2A, 0x08},// * 
{0x08, 0x08, 0x3E, 0x08, 0x08},// + 
{0x00, 0x50, 0x30, 0x00, 0x00},// , 
{0x08, 0x08, 0x08, 0x08, 0x08},// - 
{0x00, 0x30, 0x30, 0x00, 0x00},// . 
{0x20, 0x10, 0x08, 0x04, 0x02},// / 
{0x3E, 0x51, 0x49, 0x45, 0x3E},// 0 
{0x00, 0x42, 0x7F, 0x40, 0x00},// 1 
{0x42, 0x61, 0x51, 0x49, 0x46},// 2 
{0x21, 0x41, 0x45, 0x4B, 0x31},// 3 
{0x18, 0x14, 0x12, 0x7F, 0x10},// 4 
{0x27, 0x45, 0x45, 0x45, 0x39},// 5 
{0x3C, 0x4A, 0x49, 0x49, 0x30},// 6 
{0x01, 0x71, 0x09, 0x05, 0x03},// 7 
{0x36, 0x49, 0x49, 0x49, 0x36},// 8 
{0x06, 0x49, 0x49, 0x29, 0x1E},// 9 
{0x00, 0x36, 0x36, 0x00, 0x00},// : 
{0x00, 0x56, 0x36, 0x00, 0x00},// ; 
{0x00, 0x08, 0x14, 0x22, 0x41},// < 
{0x14, 0x14, 0x14, 0x14, 0x14},// = 
{0x41, 0x22, 0x14, 0x08, 0x00},// > 
{0x02, 0x01, 0x51, 0x09, 0x06},// ? 
{0x32, 0x49, 0x79, 0x41, 0x3E},// @ 
{0x7E, 0x11, 0x11, 0x11, 0x7E},// A 
{0x7F, 0x49, 0x49, 0x49, 0x36},// B 
{0x3E, 0x41, 0x41, 0x41, 0x22},// C 
{0x7F, 0x41, 0x41, 0x22, 0x1C},// D 
{0x7F, 0x49, 0x49, 0x49, 0x41},// E 
{0x7F, 0x09, 0x09, 0x01, 0x01},// F 
{0x3E, 0x41, 0x41, 0x51, 0x32},// G 
{0x7F, 0x08, 0x08, 0x08, 0x7F},// H 
{0x00, 0x41, 0x7F, 0x41, 0x00},// I 
{0x20, 0x40, 0x41, 0x3F, 0x01},// J 
{0x7F, 0x08, 0x14, 0x22, 0x41},// K 
{0x7F, 0x40, 0x40, 0x40, 0x40},// L 
{0x7F, 0x02, 0x04, 0x02, 0x7F},// M 
{0x7F, 0x04, 0x08, 0x10, 0x7F},// N 
{0x3E, 0x41, 0x41, 0x41, 0x3E},// O 
{0x7F, 0x09, 0x09, 0x09, 0x06},// P 
{0x3E, 0x41, 0x51, 0x21, 0x5E},// Q 
{0x7F, 0x09, 0x19, 0x29, 0x46},// R 
{0x46, 0x49, 0x49, 0x49, 0x31},// S 
{0x01, 0x01, 0x7F, 0x01, 0x01},// T 
{0x3F, 0x40, 0x40, 0x40, 0x3F},// U 
{0x1F, 0x20, 0x40, 0x20, 0x1F},// V 
{0x7F, 0x20, 0x18, 0x20, 0x7F},// W 
{0x63, 0x14, 0x08, 0x14, 0x63},// X 
{0x03, 0x04, 0x78, 0x04, 0x03},// Y 
{0x61, 0x51, 0x49, 0x45, 0x43},// Z 
{0x00, 0x00, 0x7F, 0x41, 0x41},// [ 
{0x02, 0x04, 0x08, 0x10, 0x20},// "\" 
{0x41, 0x41, 0x7F, 0x00, 0x00},// ] 
{0x04, 0x02, 0x01, 0x02, 0x04},// ^ 
{0x40, 0x40, 0x40, 0x40, 0x40},// _ 
{0x00, 0x01, 0x02, 0x04, 0x00},// ` 
{0x20, 0x54, 0x54, 0x54, 0x78},// a 
{0x7F, 0x48, 0x44, 0x44, 0x38},// b 
{0x38, 0x44, 0x44, 0x44, 0x20},// c 
{0x38, 0x44, 0x44, 0x48, 0x7F},// d 
{0x38, 0x54, 0x54, 0x54, 0x18},// e 
{0x08, 0x7E, 0x09, 0x01, 0x02},// f 
{0x08, 0x14, 0x54, 0x54, 0x3C},// g 
{0x7F, 0x08, 0x04, 0x04, 0x78},// h 
{0x00, 0x44, 0x7D, 0x40, 0x00},// i 
{0x20, 0x40, 0x44, 0x3D, 0x00},// j 
{0x00, 0x7F, 0x10, 0x28, 0x44},// k 
{0x00, 0x41, 0x7F, 0x40, 0x00},// l 
{0x7C, 0x04, 0x18, 0x04, 0x78},// m 
{0x7C, 0x08, 0x04, 0x04, 0x78},// n 
{0x38, 0x44, 0x44, 0x44, 0x38},// o 
{0x7C, 0x14, 0x14, 0x14, 0x08},// p 
{0x08, 0x14, 0x14, 0x18, 0x7C},// q 
{0x7C, 0x08, 0x04, 0x04, 0x08},// r 
{0x48, 0x54, 0x54, 0x54, 0x20},// s 
{0x04, 0x3F, 0x44, 0x40, 0x20},// t 
{0x3C, 0x40, 0x40, 0x20, 0x7C},// u 
{0x1C, 0x20, 0x40, 0x20, 0x1C},// v 
{0x3C, 0x40, 0x30, 0x40, 0x3C},// w 
{0x44, 0x28, 0x10, 0x28, 0x44},// x 
{0x0C, 0x50, 0x50, 0x50, 0x3C},// y 
{0x44, 0x64, 0x54, 0x4C, 0x44},// z 
{0x00, 0x08, 0x36, 0x41, 0x00},// { 
{0x00, 0x00, 0x7F, 0x00, 0x00},// | 
{0x00, 0x41, 0x36, 0x08, 0x00},// } 
{0x08, 0x08, 0x2A, 0x1C, 0x08},// -> 
{0x08, 0x1C, 0x2A, 0x08, 0x08} // <- 
}; 



// clear LCD
void lcd_clear(void)
{
	unsigned int i;
	lcd_gotoxy(0,0);
	lcd_write(CMD,0xae); // disable display;
	for(i=0;i<864;i++) lcd_write(DATA,0x00);
	lcd_write(CMD,0xaf); // enable display;
}

// init LCD
void lcd_init(void)
{
	SPIDDR |= (1<<SDA)|(1<<CLK)|(1<<CS)|(1<<RST); //Port-Direction Setup
	SCLK0                           // Standby SCLK
	CS1                         // Disable CS
	SDA1
	CS0
	RST0
	_delay_ms(100);			// 5mS so says the stop watch(less than 5ms will not work)
	RST1
	lcd_write(CMD,0x21); // write VOP register
	lcd_write(CMD,0x9A); 
	lcd_write(CMD,0xA4); // all on/normal display
	lcd_write(CMD,0x2F); // Power control set(charge pump on/off)
	lcd_write(CMD,0x40); // set start row address = 0
	lcd_write(CMD,0xb0); // set Y-address = 0
	lcd_write(CMD,0x10); // set X-address, upper 3 bits
	lcd_write(CMD,0x0);  // set X-address, lower 4 bits
	lcd_write(CMD,0xC8); // mirror Y axis (about X axis)
	lcd_write(CMD,0xa1); // Invert screen in horizontal axis
	lcd_write(CMD,0xac); // set initial row (R0) of the display
	lcd_write(CMD,0x00);
	//lcd_write(CMD,0xF9); // 
	lcd_write(CMD,0xaf); // display ON/OFF
	lcd_clear(); // clear LCD
	lcd_write(CMD,0xa7); // invert display
	_delay_ms(100);				// 1/2 Sec delay
	lcd_write(CMD,0xa6); // normal display (non inverted)
	_delay_ms(100);				// 1 Sec delay
}


void lcd_write(char cd,unsigned char c)
{
	unsigned char i;
	CS0
	SCLK0
	if (cd==0)
	{
		SDA0
	}
	else
	{
		SDA1
	}
	SCLK1
	for(i=0;i<8;i++)
	{
		SCLK0
		if(c & 0x80)
		{
			SDA1
		}
		else
		{
			SDA0
		}
		SCLK1
		c <<= 1;
		//_delay_ms(20);
	}
	SCLK0
	CS1
}


void lcd_setx(unsigned char x)
{
	lcd_write(CMD,0x10 | ((x>>4)&0x7));
	lcd_write(CMD,x & 0x0f);
}

void lcd_sety(unsigned char y)
{
	char aux = 0;
	while(y>7)
	{
		y -=8;
		aux ++;
	}
	//bitaddr = y-1;
	lcd_write(CMD,0xB0 | (aux & 0x0f));
}

void lcd_gotoxy(unsigned char x,unsigned char y)
{
	lcd_write(CMD,(0xB0|(y&0x0F)));		 // Y axis initialisation: 0100 yyyy			
	lcd_write(CMD,(0x00|(x&0x0F)));		 // X axis initialisation: 0000 xxxx ( x3 x2 x1 x0)
	lcd_write(CMD,(0x10|((x>>4)&0x07))); // X axis initialisation: 0010 0xxx  ( x6 x5 x4)
}


void lcd_vline(unsigned char x)
{
	unsigned char dy = 0;
    while (dy<9) 
    {
		lcd_gotoxy(x,dy);
		lcd_write(DATA,0xFF);
		dy++;
    }
}


void lcd_hline(unsigned char y, unsigned char mask)
{
	unsigned char dx = 0;
	lcd_gotoxy(dx,y);
    while (dx<LCD_WIDTH) 
    {
		lcd_write(DATA,mask);
		dx++;
    }
}

void lcd_clear_line(unsigned char y)
{
	lcd_hline(y,0);
}

	
void lcd_print_char(unsigned char c)
{
	int i;
	for ( i = 0; i < 5; i++ )
	{
		lcd_write(DATA,pgm_read_byte(&FontLookup[c - 32][i]) << 1);
    }
	lcd_write(DATA,0x00);
}

void lcd_print_string_f(unsigned char c,unsigned char l, const unsigned char *string)
{	
	lcd_gotoxy(6*c,l);
	while (pgm_read_byte(&(*string)))
	lcd_print_char(pgm_read_byte(&(*string++)));
}


void lcd_print_string(unsigned char c,unsigned char l, unsigned char* message)
{
	unsigned char x;
	x=6*c;
	if (x>LCD_WIDTH) return;
	lcd_gotoxy(x,l);
	while (*message)
	{
		x+=6;
		if (x>LCD_WIDTH) 
		{
			lcd_print_char('>');
			return;
		}
		else
		{
			lcd_print_char(*message++);
		}
	}
}

void lcd_print_char_inv(unsigned char c)
{
	int i;
	for ( i = 0; i < 5; i++ )
	{
		lcd_write(DATA,~(pgm_read_byte(&FontLookup[c - 32][i]) << 1));
	}
	lcd_write(DATA,0xFF);
}


void lcd_print_string_inv(unsigned char c,unsigned char l, unsigned char* message)
{
	unsigned char x;
	x=6*c;
	if (x>LCD_WIDTH) return;
	lcd_gotoxy(x,l);
	while (*message)
	{
		x+=6;
		if (x>LCD_WIDTH) 
		{
			lcd_print_char('>');
			return;
		}
		else
		{
			lcd_print_char_inv(*message++);
		}
	}
}

/* Clears an area on a line */
void lcd_clear_area(unsigned char line, unsigned char startX, unsigned char endX)
{  
	lcd_gotoxy(startX,line);
	unsigned int i;
    for(i=startX;i<endX;i++) 
	{
		lcd_write(DATA,0x00);
    }
}
