#if 0
The MIT License (MIT)

Copyright (c) 2014 inmensabolademanteca@gmail.com

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
#endif

; ----------------------------------------------------------------------------
; CIDLESA's Altair arcade (1981) port to the ZX Spectrum.
;
; Key handling routines. Joysticks.
; ----------------------------------------------------------------------------

KEY_CAP	.equ %00001000
KEY_Z	.equ %00010000
KEY_X	.equ %00100000
KEY_C	.equ %01000000
KEY_V	.equ %10000000
KEY_A	.equ %00001001
KEY_S	.equ %00010001
KEY_D	.equ %00100001
KEY_F	.equ %01000001
KEY_G	.equ %10000001
KEY_Q	.equ %00001010
KEY_W	.equ %00010010
KEY_E	.equ %00100010
KEY_R	.equ %01000010
KEY_T	.equ %10000010
KEY_1	.equ %00001011
KEY_2	.equ %00010011
KEY_3	.equ %00100011
KEY_4	.equ %01000011
KEY_5	.equ %10000011
KEY_0	.equ %00001100
KEY_9	.equ %00010100
KEY_8	.equ %00100100
KEY_7	.equ %01000100
KEY_6	.equ %10000100
KEY_P	.equ %00001101
KEY_O	.equ %00010101
KEY_I	.equ %00100101
KEY_U	.equ %01000101
KEY_Y	.equ %10000101
KEY_RET	.equ %00001110
KEY_L	.equ %00010110
KEY_K	.equ %00100110
KEY_J	.equ %01000110
KEY_H	.equ %10000110
KEY_SPC	.equ %00001111
KEY_SYM	.equ %00010111
KEY_M	.equ %00100111
KEY_N	.equ %01000111
KEY_B	.equ %10000111

; Keyboard port.
KBPORT	.equ $fe

; Real state of keys for each of the 8 rows (bit set pressed).
rkeys	.fill 8, 0

; LAst frame keys state.
skeys	.fill 8, 0

; First pressed keys. Keys that have been pressed and weren't last frame.
fkeys	.fill 8, 0

; ---------------------
; 'pollk' Poll keyboard
; ---------------------
;	Get the state of all rows in two buffers, rkeys and fkeys.
;

pollk	

	push ix
	ld hl,skeys
	ld de,rkeys
	ld ix,fkeys

; In C we are going to rotate a 0 bit to select one of the 8 keyboard rows.

	ld c,$fe

; 8 keyboard rows.

	ld b,8

pollk1
	
; Reset first pressed keys.

	ld (ix+0),0

; Save real keys in saved keys.

	ld a,(de)
	ld (hl),a

; Get row state.

	ld a,c
	in a,(KBPORT)

; Always 1 in 3 upper rows, just in case...

	or $e0

; Complement, only 1 for pressed keys, and save in real keys buffer.

	cpl
	ld (de),a

; Calculate which keys are pressed and weren't the last time, and save in
; first pressed keys.

	xor (hl)
	ex de,hl
	and (hl)
	ex de,hl
	ld (ix+0),a

; Next row.

	inc hl
	inc de
	inc ix
	rlc c
	djnz pollk1
	pop ix
	ret

; ------------------------------
; 'iskeyfp' Is key first pressed
; ------------------------------
;	Checks if a key has been pressed this frame and wasn't last frame.
;
; In	A bits aaaaa bbb, where b is keyboard row to check and aaaaa has one
;	bit 1 and the rest 0 for the key we want to check.
; Out	ZF=0 if the key is pressed.
; Saves	BC, DE, HL.

iskeyfp	push hl
	ld hl,fkeys
	call chkkey
	pop hl
	ret

; ---------
; 'keydown'
; ---------
;	Checks if a key is pressed.
;
; In	A bits aaaaa bbb, where b is keyboard row to check and aaaaa has one
;	bit 1 and the rest 0 for the key we want to check.
; Out	ZF=0 if the key is pressed.
; Saves	BC, DE, HL.

keydown	push hl
	ld hl,rkeys
	call chkkey
	pop hl
	ret

; ------------------
; 'chkkey' Check key
; ------------------
;	Checks for a key in real keys or first pressed keys.
;
; In	A bits aaaaa bbb, where b is keyboard row to check and aaaaa has one
;	bit 1 and the rest 0 for the key we want to check.
;	HL either rkeys or fkeys.
; Out	ZF=0 if the key is pressed.
; Saves	BC, DE.

chkkey	push bc
	push af
	and 7
	ld c,a
	ld b,0
	add hl,bc
	pop af
	rrca
	rrca
	rrca
	and $1f
	and (hl)
	pop bc
	ret
	
; --------
; 'getkey'
; --------
;	Gets the keycode of a key that has been pressed and wasn't before.
;
; Out	A 0 if no key first pressed, or key code if key first pressed.
; Saves	BC, DE, HL.

getkey	

	push bc
	push hl

	ld hl,fkeys
	ld b,8

getkey_loop

	xor a
	or (hl)
	jr nz,getkey_any
	inc hl
	djnz getkey_loop
	jr getkey_end

getkey_any
	
; A key was pressed, take first.

; Ultra trick to select the leftmost bit that is active and only leave it and
; reset the others.

	ld c,a
	dec a
	and c
	xor c

; Build key code (A << 3) | (8 - B)

	rlca
	rlca
	rlca
	ld c,a
	ld a,8
	sub b
	or c

getkey_end
	pop hl
	pop bc
	ret

; ----------------------------------------------------------------------------
; Gameplay key flags.
;
; The game can call 'polli' each frame and then check the logical inputs with:
;
; ld a,(rinput)
; and K_UP
; jr z,up_is_down
;
; Or if it wants to check if a key is first pressed this frame:
;
; ld a,(finput)
; and K_FIRE
; jr z,fire_is_pressed_now
; ----------------------------------------------------------------------------

; We order the bits as the Kempston joystick expects.

K_RIGHT	.equ 1
K_LEFT	.equ 2
K_DOWN	.equ 4
K_UP	.equ 8
K_FIRE	.equ 16
K_SERVA	.equ 32
K_SERVB	.equ 64
K_SERVC	.equ 128

; ------------------
; 'polli' Poll input
; ------------------
;
;	This intends to be a faster way to check input than pollk, for
; gameplay. Uses rinput and finput to set only 8 actions that can be on or off.
; Handles joysticks and keyboard, so to be transparent for the code.
; 'poll_handler' must contain the function to call: poll_keyboard,
; poll_sinclair1, poll_kempston.

polli	ld hl,(poll_handler)
	jp (hl)

; Handler to set for keyboard or joysticks.

poll_handler	.dw poll_keyboard

; Real state of inputs, first pressed inputs, saved inputs.

rinput	.db 0
finput	.db 0
sinput	.db 0

; Key table to set by game (set_keys).
; Ordered as Kempston joystick.

keyt	
	.db %10111111, %00000100	; K
	.db %10111111, %00010000	; H
	.db %10111111, %00001000	; J
	.db %11011111, %00001000	; U
	.db %11111101, %00000001	; A
	.db %11111101, %00000010	; S
	.db 0, 0
	.db 0, 0
	.db 0, 0

; Valid entries in keyt.

KEYTSZ	.equ 6

; For the Sinclair Joystick, that maps on keys, we use this table.
; 'set_keys' modify this table as well.
; Ordered as Kempston joystick.

sincl1_keyt
	.db %11101111, %00001000	; 7
	.db %11101111, %00010000	; 6
	.db %11101111, %00000100	; 8
	.db %11101111, %00000010	; 9
	.db %11101111, %00000001	; 0
	.db %11111101, %00000010	; S
	.db 0, 0
	.db 0, 0
	.db 0, 0

; ---------------
; 'poll_keyboard'
; ---------------
;	Keyboard poll handler.

poll_keyboard
	ld hl,keyt
	jp poll_kb

; ----------------
; 'poll_sinclair1'
; ----------------
;	Sinclair 1 joystick handler.

poll_sinclair1
	ld hl,sincl1_keyt
	jp poll_kb

; ---------
; 'poll_kb'
; ---------
;	Checks keyboard.

; In	HL table to check.

poll_kb call input_pre
	ld e,0
	ld bc,+(KEYTSZ<<8)+1
	call poll_key_table
	jp input_post

; ---------------
; 'poll_kempston'
; ---------------
;	Kempston joystick handler.

poll_kempston

	call input_pre

; Read kempston.

	ld bc,31
	in a,(c)
	and $1f
	ld e,a

#if KEYTSZ > 5

; Poll the rest of keys.

	ld hl,keyt+10
	ld bc,+((KEYTSZ-5)<<8)+(1<<5)
	call poll_key_table

#endif

	jp input_post

; ----------------
; 'poll_key_table'
; ----------------
;
; In	HL key table. C first bit to set in inputs. B n keys to check.
;	E should be 0 on entry or already contain some bits on.
; Out	E bits sets on keys pressed.

poll_key_table

; Read row.

	ld a,(hl)
	in a,(KBPORT)
	inc hl
	cpl

; Check specific key.

	and (hl)
	inc hl
	jr z,poll_key_table_2

; The key is on, set bit in e.

	ld a,e
	or c
	ld e,a

poll_key_table_2

; Next key.

	rlc c
	djnz poll_key_table
	ret

; -----------
; 'input_pre'	
; -----------

input_pre

; Reset first pressed input.

	xor a
	ld (finput),a

; Save input.

	ld a,(rinput)
	ld (sinput),a
	ret

; ------------
; 'input_post'
; ------------
;
; In	E bits set for active inputs.

input_post

; Set the real key states.

	ld a,e
	ld (rinput),a

; Set keys first pressed.

	ld hl,sinput
	xor (hl)
	ld hl,rinput
	and (hl)
	ld (finput),a
	ret

; ----------
; 'set_keys'
; ----------
;	Sets the keys to use for gameplay (in keyt and sincl_keyt).
;
; In	HL list of keycodes orderer as in kempston joystick.

set_keys

	ld de,keyt
	ld c,KEYTSZ

set_keys_loop

	ld a,(hl)

; Select bits for row number (0-7).

	and 7

; Set row numer from 1-8.

	inc a

; We will rotate row times.

	ld b,a

; Put leftmost bit 0.

	ld a,$7f

set_keys_rotate

; Calculate row, all bits 1 except the numbered B.

	rlca
	djnz set_keys_rotate

; Set in table the row.

	ld (de),a
	inc de

; Now set the key code.

	ld a,(hl)
	rrca
	rrca
	rrca
	and $1f
	
; Load in table.

	ld (de),a
	inc de

; Next key.

	inc hl
	dec c
	jr nz,set_keys_loop

; Now, copy in sinclair joystick table.

	ld hl,keyt+10
	ld de,sincl1_keyt+10
	ld bc,6
	ldir
	ret

; ---------
; 'kfeedbk'
; ---------
;
; Sound to make when keys pressed in menus etc.

kfeedbk push bc
	push de
	push hl
	ld hl,965
	ld de,8
	call beep
	pop hl
	pop de
	pop bc
	ret
