// Author: Dmitry Tselikov   http://bashkiria-2m.narod.ru/
//
// Modified for vector06cc with 144 MHz and cache: Ivan Gorodetsky


module SDRAM_Controller(
	input			clk24,				//  Clock 24 MHz
	input			reset,					//  System reset
	inout	[15:0]	DRAM_DQ,				//	SDRAM Data bus 16 Bits
	output	reg[11:0]	DRAM_ADDR,			//	SDRAM Address bus 12 Bits
	output	reg		DRAM_LDQM,				//	SDRAM Low-byte Data Mask 
	output	reg		DRAM_UDQM,				//	SDRAM High-byte Data Mask
	output	reg		DRAM_WE_N,				//	SDRAM Write Enable
	output	reg		DRAM_CAS_N,				//	SDRAM Column Address Strobe
	output	reg		DRAM_RAS_N,				//	SDRAM Row Address Strobe
	output			DRAM_CS_N,				//	SDRAM Chip Select
	output			DRAM_BA_0,				//	SDRAM Bank Address 0
	output			DRAM_BA_1,				//	SDRAM Bank Address 0
	input	[21:0]	iaddr,
	input	[15:0]	idata,
	input			rd,
	input			we_n,
	output	reg [15:0]	odata,
	output	reg [15:0]	odata2,
	output reg memcpubusy,
	output reg rdcpu_finished,
	output reg memvidbusy,
	input rdv
);

parameter ST_RESET0 = 5'd0;
parameter ST_RESET1 = 5'd1;
parameter ST_IDLE   = 5'd2;
parameter ST_RAS0   = 5'd3;
parameter ST_RAS1   = 5'd4;
parameter ST_READ0  = 5'd5;
parameter ST_READ1  = 5'd6;
parameter ST_READ2  = 5'd7;
parameter ST_WRITE0 = 5'd8;
parameter ST_WRITE1 = 5'd9;
parameter ST_WRITE2 = 5'd10;
parameter ST_REFRESH0 = 5'd11;
parameter ST_REFRESH1 = 5'd12;
parameter ST_READ3  = 5'd13;
parameter ST_WRITE3  = 5'd14;
parameter ST_READV  = 5'd15;
parameter ST_REFRESH2 = 5'd16;
parameter ST_REFRESH3 = 5'd17;
parameter ST_REFRESH4 = 5'd18;
parameter ST_READ4  = 5'd19;
parameter ST_WRITE4  = 5'd20;
parameter ST_READV0  = 5'd21;
parameter ST_RAS2   = 5'd22;
parameter ST_REFRESH5 = 5'd30;
parameter ST_REFRESH6 = 5'd31;
parameter ST_REFRESH7 = 5'd29;
parameter ST_READ5  = 5'd28;
parameter ST_CREAD0  = 5'd27;
parameter ST_CREAD1  = 5'd26;



reg[1:0] rst;
reg[4:0] state;
reg[21:0] addr;
reg[7:0] data;
reg exrd,exwen,lsb,rdvid;
reg cwr;
//reg[10:0] cindex;//for 2k
reg[11:0] cindex;//for 4k
reg[7:0] cdatai,ctdatai;
wire[7:0] cdatao,ctdatao;

ram4k ctag(
	.address(cindex),
	.clock(clk24),
	.data(ctdatai),
	.wren(cwr),
	.q(ctdatao)
	);

ram4k cdata(
	.address(cindex),
	.clock(clk24),
	.data(cdatai),
	.wren(cwr),
	.q(cdatao)
	);

assign DRAM_DQ[7:0] = (state==ST_WRITE0)&&(lsb==0) ? data : 8'bZZZZZZZZ;
assign DRAM_DQ[15:8] = (state==ST_WRITE0)&&(lsb==1) ? data : 8'bZZZZZZZZ;

assign DRAM_CS_N = 1'b0;
assign DRAM_BA_0 = addr[20];
assign DRAM_BA_1 = addr[21];

always @(*) begin
	case (state)
	ST_RESET0: DRAM_ADDR = 12'b000000110000;//CAS Latency 3
	ST_RAS0:   DRAM_ADDR = addr[19:8];

	ST_READ0: if(rdvid==1) DRAM_ADDR = {4'b0000,addr[7:0]};
							else DRAM_ADDR = {4'b0100,addr[7:0]};
	ST_READ1: if(rdvid==1) DRAM_ADDR = {4'b0100,addr[7:1],1'b1};
	ST_WRITE0:  DRAM_ADDR = {4'b0100,addr[7:0]};
	ST_CREAD0: DRAM_ADDR[10] = 1'b1;//PRECHARGE ALL BANKS
	endcase
	case (state)
	ST_RESET0:   {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N} = 3'b000;//LOAD MODE REGISTER
	ST_RAS0:     {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N} = 3'b011;//ACTIVE
	ST_READ0: 	 {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N,DRAM_UDQM,DRAM_LDQM} = 5'b10100;
	ST_READ1: 	 if(rdvid==1)  {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N,DRAM_UDQM,DRAM_LDQM} = 5'b10100;
	ST_WRITE0:   {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N,DRAM_UDQM,DRAM_LDQM} = {3'b100,~lsb,lsb};
	ST_WRITE4:	 {DRAM_UDQM,DRAM_LDQM} = 2'b00;
	ST_REFRESH0: {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N} = 3'b001;//AUTO REFRESH
	ST_CREAD0:	 {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N} = 3'b010;//PRECHARGE
	default:     {DRAM_RAS_N,DRAM_CAS_N,DRAM_WE_N} = 3'b111;//NOP
	endcase
end

always @(posedge clk24) begin
	if (reset) begin
		state <= ST_RESET0; exrd <= 0; exwen <= 1'b1;rdvid<=1'b0;
		memcpubusy<=1'b0; memvidbusy<=1'b0;
	end else begin
		case (state)
		ST_RESET0: state <= ST_RESET1;
		ST_RESET1: state <= ST_IDLE;
		ST_IDLE:
		begin
			memcpubusy<=1'b0;
			memvidbusy<=1'b0;
			rdcpu_finished<=1'b0;
			cwr<=1'b0;
			if(rdv==0) begin exrd <= rd; exwen <= we_n;
			end
			addr[17:0] <= iaddr[18:1]; lsb<=iaddr[0]; data <= idata[7:0];rdvid<=rdv;
			casex ({rd,exrd,we_n,exwen,rdv})
//for 4k
			5'b10110: {state,memcpubusy,cindex} <= {ST_RAS0,1'b1,iaddr[13:2]};//CPU READ
			5'b00010: {state,memcpubusy,cindex} <= {ST_RAS0,1'b1,iaddr[13:2]};//CPU WRITE
			5'bxxxx1: {state,memvidbusy} <= {ST_RAS0,1'b1};//VIDEO READ
			default: state <= ST_IDLE;
			endcase
		end
		ST_RAS0: state <= ST_RAS1;
		ST_RAS1:	state <= ST_RAS2;		
		ST_RAS2:
			casex ({exrd,exwen,rdvid})
			3'b110://CPU READ
//for 4k
			if (ctdatao=={1'b0,addr[17:13],addr[0],lsb}) {state,odata[7:0],rdcpu_finished}<={ST_CREAD0,cdatao,1'b1};
			else state<=ST_READ0;
			3'b000: //CPU WRITE
//for 4k
			{state,ctdatai,cdatai,cwr} <= {ST_WRITE0,{1'b0,addr[17:13],addr[0],lsb},data,1'b1};
			3'bxx1: state <= ST_READ0;//VIDEO READ
			default: state <= ST_IDLE;
			endcase
		ST_CREAD0: state <= ST_CREAD1;
		ST_CREAD1: state <= ST_IDLE;
		ST_READ0: state <= ST_READ1;
		ST_READ1: state <= ST_READ2;
		ST_READ2: state <= ST_READ3;
		ST_READ3: state <= ST_READ4;//144
		ST_READ4: begin
		case(rdvid)
		1'b0:begin
		 rdcpu_finished<=1'b1;
		 state<=ST_IDLE;
		 if (lsb==0) begin
		 {ctdatai,cdatai,cwr} <= {{1'b0,addr[17:13],addr[0],lsb},DRAM_DQ[7:0],1'b1};
		 odata[7:0]<=DRAM_DQ[7:0];
		 end
		 else begin
 		 {ctdatai,cdatai,cwr} <= {{1'b0,addr[17:13],addr[0],lsb},DRAM_DQ[15:8],1'b1};
		 odata[7:0]<=DRAM_DQ[15:8];
		 end
		end
		1'b1:{state,odata} <= {ST_READV,DRAM_DQ[15:0]};
		endcase
		end
		ST_READV: {state,odata2} <= {ST_REFRESH0,DRAM_DQ[15:0]};

		ST_WRITE0: {state,cwr} <= {ST_WRITE1,1'b0};
		ST_WRITE1: state <= ST_WRITE2;
		ST_WRITE2: state <= ST_WRITE3;
		ST_WRITE3: state <= ST_WRITE4;//144
		ST_WRITE4: state <= ST_IDLE;
		ST_REFRESH0: state <= ST_REFRESH1;
		ST_REFRESH1: state <= ST_REFRESH2;//72
		ST_REFRESH2: state <= ST_REFRESH3;
		ST_REFRESH3: state <= ST_REFRESH4;
		ST_REFRESH4: state <= ST_REFRESH5;
		ST_REFRESH5: state <= ST_REFRESH7;//144
//		ST_REFRESH6: state <= ST_REFRESH7;
		ST_REFRESH7: state <= ST_IDLE;
		default: state <= ST_IDLE;
		endcase
	end
end
	
endmodule
