/*---------------------------------------------------------------*/
/* FAT file system module test program R0.0. (c)2007 Serge       */
/*---------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <cpm.h>

#include "ff.h"
#include "diskio.h"
#include "wildcomp.h"

#define MAX_BUFF 2048

#define die(rres) { perror("I/O", rres); return 1; }

#define	MEMSET(ptr,val,cnt)  for (jj=0; jj<cnt; jj++) (ptr)[jj]=(val)

typedef FRESULT (*dir_callback)(FILINFO* fileinfo);

typedef struct {
  BYTE	dr;		/* drive code */
  char	name[8];	/* file name */
  char	ft[3];		/* file type */
  BYTE	ex;		/* file extent */
  BYTE	fil[2];		/* not used */
  BYTE	rc;		/* number of records in present extent */
  BYTE	dm[16];		/* CP/M disk map */
} FCB;


typedef struct {
  BOOL  OSType;     /* TRUE=FAT, FALSE=CPM  */
  FILE* fileCPM;    /* CP/M file descriptor */
  FIL   fileFAT;    /* FAT file structure   */
} OSFILE;

extern int rename(char * name1, char * name2);

OSFILE fsrc, fdst;

FILINFO finfo;
FRESULT res;         /* FatFs function common result code */
FATFS   fatfs[2];    /* File system object for each logical drive */

DWORD DriveSize[2];
int  CurrentDrive=0;
int  TotalDrives=0;

static int jj;
static char buffer[MAX_BUFF+1];

static BYTE toAlt[]={
  0xC4, 0xB3, 0xDA, 0xBF, 0xC0, 0xD9, 0xC3, 0xB4, 0xC2, 0xC1, 0xC5, 0xDF, 0xDC, 0xDB, 0xDD, 0xDE,
  0xB0, 0xB1, 0xB2, 0xF4, 0xFE, 0xF9, 0xFB, 0xF7, 0xF3, 0xF2, 0xFF, 0xF5, 0xF8, 0xFD, 0xFA, 0xF6,
  0xCD, 0xBA, 0xD5, 0xF1, 0xD6, 0xC9, 0xB8, 0xB7, 0xBB, 0xD4, 0xD3, 0xC8, 0xBE, 0xBD, 0xBC, 0xC6,
  0xC7, 0xCC, 0xB5, 0xF0, 0xB6, 0xB9, 0xD1, 0xD2, 0xCB, 0xCF, 0xD0, 0xCA, 0xD8, 0xD7, 0xCE, 0xFC,
  0xEE, 0xA0, 0xA1, 0xE6, 0xA4, 0xA5, 0xE4, 0xA3, 0xE5, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE,
  0xAF, 0xEF, 0xE0, 0xE1, 0xE2, 0xE3, 0xA6, 0xA2, 0xEC, 0xEB, 0xA7, 0xE8, 0xED, 0xE9, 0xE7, 0xEA,
  0x9E, 0x80, 0x81, 0x96, 0x84, 0x85, 0x94, 0x83, 0x95, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E,
  0x8F, 0x9F, 0x90, 0x91, 0x92, 0x93, 0x86, 0x82, 0x9C, 0x9B, 0x87, 0x98, 0x9D, 0x99, 0x97, 0x9A};

static BYTE toKoi[]={
  0xE1, 0xE2, 0xF7, 0xE7, 0xE4, 0xE5, 0xF6, 0xFA, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 
  0xF2, 0xF3, 0xF4, 0xF5, 0xE6, 0xE8, 0xE3, 0xFE, 0xFB, 0xFD, 0xFF, 0xF9, 0xF8, 0xFC, 0xE0, 0xF1, 
  0xC1, 0xC2, 0xD7, 0xC7, 0xC4, 0xC5, 0xD6, 0xDA, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 
  0x90, 0x91, 0x92, 0x81, 0x87, 0xB2, 0xB4, 0xA7, 0xA6, 0xB5, 0xA1, 0xA8, 0xAE, 0xAD, 0xAC, 0x83, 
  0x84, 0x89, 0x88, 0x86, 0x80, 0x8A, 0xAF, 0xB0, 0xAB, 0xA5, 0xBB, 0xB8, 0xB1, 0xA0, 0xBE, 0xB9, 
  0xBA, 0xB6, 0xB7, 0xAA, 0xA9, 0xA2, 0xA4, 0xBD, 0xBC, 0x85, 0x82, 0x8D, 0x8C, 0x8E, 0x8F, 0x8B, 
  0xD2, 0xD3, 0xD4, 0xD5, 0xC6, 0xC8, 0xC3, 0xDE, 0xDB, 0xDD, 0xDF, 0xD9, 0xD8, 0xDC, 0xC0, 0xD1, 
  0xB3, 0xA3, 0x99, 0x98, 0x93, 0x9B, 0x9F, 0x97, 0x9C, 0x95, 0x9E, 0x96, 0xBF, 0x9D, 0x94, 0x9A};


char* alt2koi(char* strn, BYTE* codetable)
{
  BYTE* st;
  for (st=(BYTE*)strn; *st; st++)
    if (*st>127) *st=codetable[*st - 128];
  return strn;
}

void UpperCase(char* st)
{
  while (*st) {
    if ( (*st>='a') && (*st<='z') ) *st=*st-32;
    st++;
  }
}

void perror(char* txt, FRESULT res)
{
  printf("\nError: %s (%d)\n\n", txt, res);
}

BOOL IsCPMpath(char* path)   /*  0:A:filename.ext  |  A:filename.ext  */
{
  return (path) && (strchr(path, '/')==0) &&
         ( ( (strlen(path)>3) && isdigit(*path) && (path[3]==':') ) ||
           ( (strlen(path)>1) && isalpha(*path) && (strrchr(path,':')==path+1) )
         );  
}

BOOL IsFATpath(char* path)  /*  0:/path/fname.ext  |  /path/fname.ext  |  fname.ext */
{
  char* pos;
  return (! IsCPMpath(path)) && 
         ((pos=strchr(path, ':'))==strrchr(path, ':')) &&
         ( ( pos==0 ) || ( isdigit(*path) && (pos==path+1) ) );
}

FRESULT OS_open(OSFILE* FileObject, char* path, BYTE Flags)
{
  if (IsCPMpath(path))
    FileObject->OSType=FALSE; 
  else if (IsFATpath(path))
    FileObject->OSType=TRUE; 
  else
    return FR_INVALID_NAME;
  if (FileObject->OSType)                             /* FAT if OSType=TRUE  */
    return f_open(&FileObject->fileFAT, path, Flags);
                                                      /* CPM if OSType=FALSE */
  FileObject->fileCPM=NULL;
  if (Flags & FA_CREATE_ALWAYS)
    FileObject->fileCPM=fopen(path, "wb+");
  else
    FileObject->fileCPM=fopen(path, "rb+");
  if (FileObject->fileCPM) return FR_OK;
  return FR_NO_FILE;
}
 
FRESULT OS_close(OSFILE* FileObject)
{
  if (FileObject->OSType)                                   /* FAT if OSType=TRUE  */
    return f_close(&FileObject->fileFAT);
                                                            /* CPM if OSType=FALSE */
  if ((FileObject->fileCPM) && (fclose(FileObject->fileCPM)==0)) {
    return FR_OK;
  }
  return FR_INVALID_OBJECT;
}

FRESULT OS_read(OSFILE* FileObject, void* buf, WORD cnt, int* readed)
{  
  if (FileObject->OSType)                                   /* FAT if OSType=TRUE  */
    return f_read(&FileObject->fileFAT, buf, cnt, (WORD*)readed);
  if (FileObject->fileCPM) {
    *readed=fread(buf, 1, cnt, FileObject->fileCPM); 
    if (*readed>=0)
      return FR_OK;
  }
  return FR_INVALID_OBJECT;
}

FRESULT OS_write(OSFILE* FileObject, void* buf, WORD cnt, int* written)
{
  *written=0;
  if (FileObject->OSType)                                   /* FAT if OSType=TRUE  */
    return f_write(&FileObject->fileFAT, buf, cnt, (WORD*)written);
  if ((FileObject->fileCPM) && 
     ((*written=fwrite(buf, 1, cnt, FileObject->fileCPM))>=0)) {
    return FR_OK;
  }
  return FR_INVALID_OBJECT;
}

FRESULT OS_seek(OSFILE* FileObject, DWORD offset)
{
  if (FileObject->OSType)                                   /* FAT if OSType=TRUE  */
    return f_lseek(&FileObject->fileFAT, offset);
  if ((FileObject->fileCPM) &&
      (fseek(FileObject->fileCPM, offset, 0)==offset) ) {
    return FR_OK;
  }
  return FR_INVALID_OBJECT;
}

void ShowTitle()
{
  printf("\n FAT v 1.3.  (c) 2007 Serge.  ?=HELP \n\n");
}

void ShowPrompt()
{
  printf("\nFAT> "); 
}

void do_help()
{
  printf("\n FAT v 1.3 usefull for serving files, stored on FAT12/FAT16/FAT32 volumes\n");
  printf("\nUsage: FAT <command> [<filemask> [<filemask>]]");
  printf("\n       where <command>: dir   <filemask> - show files list");
  printf("\n                        type  <filemask> - type file(s) content");
  printf("\n                        atype <filemask> - type alternate(cp1251) coded file(s)");
  printf("\n                        del   <filemask> - delete file(s) or catalog(s)");
  printf("\n                        copy  <inpmask> <outmask> - copy from inp to out");
  printf("\n                        ren   <inpmask> <outmask> - rename inp to out");
  printf("\n                        mkdir <dirname>  - create catalog by full path");
  printf("\n                        info  <drive>    - show drive (filesystem) information");
  printf("\n                        stat  <filemask> - show file(s) attributes");
  printf("\n                        chmod <filemask> - change file(s) attributes");
  printf("\n              <filemask> - file specification with full dirrectory path and(or)");
  printf("\n                           wildcards ?*");
  printf("\n              CP/M filemask: 0:A:*.ext - all files with any names and extension");
  printf("\n                                       `ext` in user 0 of drive A. `0:`-optional"); 
  printf("\n              FAT filemask:  0:/dir/dir2/file1.* - all files with `file1` name");
  printf("\n                                        in `dir/dir2` catalog of master drive,");
  printf("\n                                        catalog and drive names are optional");
  printf("\n Example: FAT copy 1:/test/aaa.bbb 3:B:ccc.ddd\n");  
  return;           
}

FILINFO* FcbToFInfo(FCB* pfcb, FILINFO* finfo)
{
  char* pos1;

  finfo->fsize=0;        
  finfo->fdate=0;
  finfo->ftime=0;
  finfo->fattrib=0;
  if (pfcb->ft[0] & 128) {
    finfo->fattrib|=AM_RDO;
    pfcb->ft[0]&=127;
  }
  if (pfcb->ft[1] & 128) {
    finfo->fattrib|=AM_HID;
    pfcb->ft[1]&=127;
  }
  MEMSET(finfo->fname, 32, 12);
  strncpy(finfo->fname, pfcb->name, 8);
  if (! (pos1=strchr(finfo->fname, ' ')) ) pos1=&finfo->fname[8];
  *pos1='.';
  strncpy(pos1+1, pfcb->ft, 3);
  return finfo;
}

FCB* PathToFcb(char* path, FCB  *xfcb)
{ 
  char *pos1, ch;
  int   i;

  xfcb->dr=xfcb->ex=xfcb->fil[0]=xfcb->fil[1]=xfcb->fil[2]=xfcb->rc=0;

  if (pos1=strchr(path, ':')) {
    xfcb->dr=(BYTE)(*path-'A'+1);
    path=pos1+1;
  }
  MEMSET(xfcb->name, '?', 11);
  if (*path) {
    ch=0;
    for(i=0; i<8; i++) {
      if ((! *path) || (*path=='.')) {
        if (! ch) ch=' ';
      }
      else {
        if (*path=='*') ch='?';
        path++;
      }
      xfcb->name[i]=(ch ? ch : *(path-1)); 
    }
    if (*path) {
      if (*path=='.') path++;
      ch=0;
      for(i=0; i<3; i++) {
        if (! *path) {
          if (! ch) ch=' ';
        }
        else {
          if (*path=='*') ch='?';
          path++;
        }
        xfcb->ft[i]=(ch ? ch : *(path-1)); 
      }
    }
  }
  return xfcb;
}

FRESULT scanCPM(char* path, dir_callback OnFile)
{
  BYTE uid;
  BYTE idx;
  char *pos1, *pos2;
  FCB  *dma, myfcb;

  if ((! path) || (! *path)) return FR_INVALID_OBJECT;

  if ( (pos1=strchr(path, ':')) != (pos2=strrchr(path, ':')) ) {     
    uid=getuid();  
    setuid((int)(*path-'0'));
    PathToFcb(pos1+1, &myfcb);
  }
  else 
    PathToFcb(path, &myfcb);

  dma=(void*)&buffer[0];
  bdos(CPMSDMA, &dma[0]);                                         
  idx=bdos(CPMFFST, &myfcb);                                     
  while (idx<4) {
    if (OnFile(FcbToFInfo(&dma[idx], &finfo))!=FR_OK) break;
    idx=bdos(CPMFNXT, &myfcb);                                   
  }
  if (pos1!=pos2) setuid(uid);
  return FR_OK; 
}

FRESULT scanFAT(char* path, dir_callback OnFile)
{
  DIR dir;
  char *pos1, *pos2, *mask;
  if ((! path) || (! *path)) return FR_INVALID_OBJECT;
  strcpy(buffer, path);
  mask=NULL;
  pos2=strchr(buffer, '?');
  pos1=strchr(buffer, '*');
  if ((pos1) || (pos2)) {
    mask=pos2;
    if ((! mask) || ((mask>pos1)&&(pos1))) mask=pos1;
  }
  for ( ; (mask)&&(&mask[0]>&buffer[0])&&(*mask!='/'); mask--);
  if (mask==buffer) mask=NULL;
  if (mask) {
    *mask=0;
    mask++;
  }
  if ((res = f_opendir(&dir, buffer)) == FR_OK)
    while (((res = f_readdir(&dir, &finfo)) == FR_OK) && finfo.fname[0])
      if (mask) {
        if ( WildStringCompare(finfo.fname, mask) && (OnFile(&finfo)!=FR_OK) ) break;
      }
      else if (OnFile(&finfo)!=FR_OK) break;
  return res;
}

BOOL do_info(char* path)
{
  DWORD p2;
  FATFS *fs;

  if (IsFATpath(path)) {
    printf("\n GetInfo... please wait... \n");
    res=f_getfree(path, &p2, &fs);
    if (res) {
	perror("f_getfree(path, ...)", res);
	return 0;
    }
    if (fs->fs_type==FS_FAT12) strcpy(buffer, "FAT12");
    else if (fs->fs_type==FS_FAT16) strcpy(buffer, "FAT16");
    else if (fs->fs_type==FS_FAT32) strcpy(buffer, "FAT32");
    else strcpy(buffer, "UNKNOWN");

    printf("\nFAT type = %s\nBytes/Cluster = %lu\nNumber of FATs = %u\n",
           buffer, (DWORD)fs->csize * 512, (WORD)fs->n_fats);
    printf("Root DIR entries = %u\nSectors/FAT = %lu\nNumber of clusters = %lu\n",
           fs->n_rootdir, fs->sects_fat, (DWORD)fs->max_clust - 2);
    printf("FAT start (lba) = %lu\nDIR start (lba,cluster) = %lu\nData start (lba) = %lu",
           fs->fatbase, fs->dirbase, fs->database);
    printf("\n%lu KB total.\n%lu KB available.\n",
           ((DWORD)fs->csize / 2)*(fs->max_clust -2), p2 * (fs->csize / 2) );
    return TRUE;
  }
  else if (IsCPMpath(path)) {
    printf("\n CPM filesystem\n");
    return TRUE;
  }
  return FALSE;
}


FRESULT dirCPMfile(FILINFO* fileinfo)
{
  printf("  %s\n", fileinfo->fname);
  return FR_OK;
}

FRESULT dirFATfile(FILINFO* fileinfo)
{
  if (! fileinfo) return FALSE;
  printf("%02d\.%02d\.%04d  %02d:%02d   ",
         fileinfo->fdate & 31, (fileinfo->fdate >>5) & 15,
         (WORD)1980 + (WORD)((fileinfo->fdate >>9) & 127) ,
         (fileinfo->ftime >>11) & 31, (fileinfo->ftime >>5) & 63);
  if (fileinfo->fattrib & AM_DIR) 
    printf("<DIR>       ");
  else
    printf("%12lu", fileinfo->fsize);
  printf("  %s\n", alt2koi(fileinfo->fname, toKoi));
  return FR_OK;
}

BOOL do_dir(char* path)
{
  if (IsCPMpath(path))
    return scanCPM(path, dirCPMfile);
  else if (IsFATpath(path))
    return scanFAT(alt2koi(path, toAlt), dirFATfile); 
  else
    do_help();
  return FALSE;
}

BOOL do_mkdir(char* path)
{
  if (IsFATpath(path))
    return (f_mkdir(alt2koi(path, toAlt))==FR_OK); 
  else
    do_help();
  return FALSE;
}

BOOL do_del(char* path)
{
  if (IsCPMpath(path))
    return (remove(path)!=-1);
  else if (IsFATpath(path))
    return (f_unlink(alt2koi(path, toAlt))==FR_OK); 
  else
    do_help();
  return FALSE;
}

BOOL do_ren(char* src, char* dst)
{
  if (IsCPMpath(src) && IsCPMpath(dst))
    return (rename(src, dst)!=-1);
  else if (IsFATpath(src) && IsFATpath(dst)) {
    if (dst[1]==':') dst=dst+2;
    if (*dst=='/') dst++;
    res=f_rename(alt2koi(src, toAlt), alt2koi(dst, toAlt));
    return (res==FR_OK); 
  }
  else
    do_help();
  return FALSE;
}

BOOL do_type(char* path, BOOL alt)
{
  char ch;
  int br, bw;        
  if (IsFATpath(path)) alt2koi(path, toAlt);
  res = OS_open(&fsrc, path, FA_OPEN_EXISTING | FA_READ);
  if (res) perror("f_open()", res);
  for (;;) {
    res = OS_read(&fsrc, buffer, MAX_BUFF, &br);
    buffer[MAX_BUFF]=0; 
    if (alt)
      alt2koi(buffer, toKoi); 
    for (bw=0; bw<br; bw++) {
      ch=buffer[bw];
      if (ch==0x1A) break; 
      putchar(ch);
    }
    if (res || br == 0) break;     
  }
  OS_close(&fsrc);
  return TRUE;
}

BOOL do_copy(char* src, char* dst)
{
    int br, bw;         
 
    if (IsFATpath(src)) alt2koi(src, toAlt);
    if (IsFATpath(dst)) alt2koi(src, toAlt);
    
    res = OS_open(&fsrc, src, FA_OPEN_EXISTING | FA_READ); 
    if (res) die(res);
    res = OS_open(&fdst, dst, FA_CREATE_ALWAYS | FA_WRITE); 
    if (res) die(res);
    for (;;) {                                             
        res = OS_read(&fsrc, buffer, MAX_BUFF, &br);
        if (res || br == 0) break;                        
        res = OS_write(&fdst, buffer, br, &bw);
        if (res || bw < br) break;                         
    }
    OS_close(&fsrc);
    OS_close(&fdst);
    return TRUE;
}

BOOL eq(char* st1, char* st2, BOOL param_ok)
{
  if (strcmp(st1, st2)==0) {
    if (param_ok) 
      return TRUE;
    else
      do_help();
  }
  return FALSE;
}

void CheckIDE()
{
  DriveSize[0]=DriveSize[1]=0;
  if ((disk_ioctl(0, GET_SECTOR_COUNT, &DriveSize[0]) == RES_OK) && 
      (DriveSize[0]>1)) 
    TotalDrives++;
  else
    CurrentDrive=1; 
  if ((disk_ioctl(1, GET_SECTOR_COUNT, &DriveSize[1]) == RES_OK) &&
      (DriveSize[1]>1)) 
    TotalDrives++;
  if (TotalDrives<1)
    { printf("No IDE drives or IDEBDOS driver (V1.3 or higher) not installed\n");
      exit (-1); }
}

int main (int argc, char* argv[])
{
  BYTE DriveOne=255;
  BYTE DriveTwo=255;
  char  path[128];

  ShowTitle();

  if (argc<2) {
    do_help();
    return 1;
  }  
  CheckIDE();

  if ((argc>=3) && IsFATpath(argv[2])) {
    if (! isdigit(argv[2][0])) DriveOne=0;
    else DriveOne=argv[2][0] - '0';
    if (f_mount(DriveOne, &fatfs[0])) { 
/*	perror("f_mount(DriveOne)", 255);
*/	return 1;
    }
  } 
  if ((argc>=4) && IsFATpath(argv[3])) {
    if (! isdigit(argv[3][0])) DriveTwo=0;
    else DriveTwo=argv[3][0] - '0';
    if (f_mount(DriveTwo, &fatfs[1])) { 
/*	perror("f_mount(DriveTwo)", 255);
*/	return 1;
    }
  } 
  strcpy(path, argv[1]);
  UpperCase(path);

  if (eq(path, "INFO", argc==3))
    do_info(argv[2]);
  else if (eq(path, "HELP", 0) || eq(path, "?", 0))
    do_help();
  else if (eq(path, "DIR", argc==3))
    do_dir(argv[2]);
  else if (eq(path, "TYPE", argc==3))
    do_type(argv[2], FALSE);
  else if (eq(path, "ATYPE", argc==3))
    do_type(argv[2], TRUE);
  else if (eq(path, "COPY", argc==4))
    do_copy(argv[2], argv[3]);
  else if (eq(path, "MKDIR", argc==3))
    do_mkdir(argv[2]);
  else if (eq(path, "DEL", argc==3))
    do_del(argv[2]);
  else if (eq(path, "REN", argc==4))
    do_ren(argv[2], argv[3]);

  /* Unregister a work areas before discard it */
  if (DriveOne != 255) f_mount(DriveOne, NULL);
  if (DriveTwo != 255) f_mount(DriveTwo, NULL);

  return 0;
}

