#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <string.h>
//#include <dos.h>


#define CPM_TYPE        0x02        //   CP/M
#define MAX_DIR         0x10        //     

//    
#define DEFAULT_ALV         256         // 256  = 1024 
#define DEFAULT_DIRBLOCKS   2
#define DEFAULT_RESTRACKS   0


typedef unsigned char       UINT8;
typedef unsigned short int  UINT16;
typedef unsigned long int   UINT32;

typedef signed char         INT8;
typedef signed short int    INT16;
typedef signed long int     INT32;


#pragma pack (1)


typedef struct {
    UINT8   Active;     // 0x80 - active partion
    UINT8   Side;       // head
    UINT16  Addr;       // cylinder/sector
    UINT8   Type;       // type partion
    UINT8   SideEnd;
    UINT16  AddrEnd;
    UINT32  RelAddr;    //   
    UINT32  Size;       //    
} PARTION;

typedef struct {
    char    Active;     //   
    char    Name[18];
    char    NumChan;
    char    NumDev;
    UINT16  Head;
    UINT16  Cyl;
    UINT16  Sec;
    UINT32  Size;       // in sectors
} DEVICE;

typedef struct {
    UINT16  SPT;        // sectors per track
    UINT8   BSH;        // block shift
    UINT8   BLM;        // block mask
    UINT8   EXM;        // extent mask
    UINT16  DSM;        // disk maximum
    UINT16  DRM;        // directory maximum
    UINT8   AL0;        // allocation vector
    UINT8   AL1;
    UINT16  CKS;        // checksum vector size
    UINT16  OFF;        // track offset
    UINT8   res;
} DPB;

typedef struct {
    char    Sign[8];    //  "CP/M"
    DPB     dpb;
    UINT8   res[486];   // 
    UINT16  parSign;    // 0xAA55;
} SYSSEC;

typedef struct {
    char    user;
    char    name[8];
    char    ext[3];
    char    ex;
    UINT16  res;
    char    rc;
    UINT16  map[8];
} DIRREC;


typedef struct {
    UINT32  ALV;        //     
    UINT32  DirBlocks;  //    
    UINT32  ResTracks;  //   (    )
} FILESYS;

#pragma pack ()


#define DEVICE          FILE

#define DIRINSEC        (512 / sizeof(DIRREC))


/*
==============================================================================

                                 DISK IO

==============================================================================
*/

char ide_ReadSector(DEVICE *p, UINT32 Sector, void* buff)
{
    Sector *= 512;
    if (fseek(p, Sector, SEEK_SET) == -1)
        return 0;
    if (!fread(buff, 512, 1, p))
        return 0;
    return -1;
}

char ide_WriteSector(DEVICE *p, UINT32 Sector, void* buff)
{
    Sector *= 512;
    if (fseek(p, Sector, SEEK_SET) == -1)
        return 0;
    if (!fwrite(buff, 512, 1, p))
        return 0;
    return -1;
}


/*
==============================================================================

                                  FORMAT

==============================================================================
*/

char hdd_CheckSign(char buff[])
{
    if ((buff[0x1FE] != 0x55) || (buff[0x1FF] != 0xAA))
        return 0;
    else
        return -1;
}

//
//     BLS   
//  size -   
//
UINT16 calck_BlockSize(UINT32 DiskSize, UINT32 alv)
{
    UINT32 BLS, tmp;

    tmp = DiskSize / ((alv - 1) * 8);       //   
    BLS = 2048;                             //   
    while (BLS < tmp)
        BLS <<= 1;
    return BLS;
}


//
// DiskSize     -    
// BlockSize    -  
// ResTracks    - . 
// SecPerTrack  -     ( 128 )
// DirBlock     -    
//
void make_DPB(DPB *dpb, UINT32 DiskSize, UINT32 BlockSize, UINT32 ResTracks, UINT32 SecPerTrack, int DirBlock)
{
    UINT32  Tracks;
    UINT32  SecInBlock;
    int     w;

    Tracks = DiskSize / (SecPerTrack * 128);
    SecInBlock = BlockSize / 128;
    dpb->SPT = SecPerTrack;
    dpb->OFF = ResTracks;
    dpb->CKS = 0;

    dpb->BLM = SecInBlock - 1;
    dpb->BSH = 3;
    dpb->EXM = 0;
    w = BlockSize/1024;
    while (w > 1)
    {
        dpb->BSH++;
        dpb->EXM = (dpb->EXM << 1) | 0x01;
        w >>= 1;
    }
    dpb->DSM = ((Tracks-ResTracks) * SecPerTrack) / SecInBlock;
    if (dpb->DSM > 256)
        dpb->EXM >>= 1;
    dpb->DSM--;
    w = 0x0000;
    dpb->DRM = DirBlock * (BlockSize / 32) - 1;
    do
    {
        w = (w >> 1) | 0x8000;
        DirBlock--;
    }
    while (DirBlock > 0);
    dpb->AL0 = (w >> 8) & 0xFF;
    dpb->AL1 = w & 0xFF;
}


char hdd_doFormatCPM(DEVICE *p, UINT32 AbsAddr, UINT32 TotalSec, FILESYS *fsys)
{
    UINT32  DiskSize;
    SYSSEC  sec;
    int     i,n;
    UINT32  DSM, BLS;

    memset(&sec, 0, sizeof(SYSSEC));

    //  
    DiskSize = TotalSec * 512;
    BLS = calck_BlockSize(DiskSize, fsys->ALV);
    make_DPB(&sec.dpb, DiskSize, BLS, fsys->ResTracks, 128, fsys->DirBlocks);
    DSM =(sec.dpb.DSM+1);
    DiskSize = (DSM * BLS) / 1024;
    printf("        First sector   : 0x%08lX\n", AbsAddr);
    printf("        Size           : %luKb\n", DiskSize);
    printf("        Cluster size   : %lu\n", BLS);
    printf("        Block shift    : 0x%02hX\n", sec.dpb.BSH);
    printf("        Block mask     : 0x%02hX\n", sec.dpb.BLM);
    printf("        Extent mask    : 0x%02hX\n", sec.dpb.EXM);
    printf("        Num clusters   : %hu\n", sec.dpb.DSM+1);
    printf("        Num dir entries: %hu\n", sec.dpb.DRM+1);
    printf("        AL0            : 0x%02hX\n", sec.dpb.AL0);
    printf("        AL1            : 0x%02hX\n", sec.dpb.AL1);

    //  
    memcpy(sec.Sign, "CP/M    ", 8);
    sec.parSign = 0xAA55;

    if (!ide_WriteSector(p, AbsAddr, &sec))
    {
        printf("      *error* - can't write DPB at 0x%08lX!\n", AbsAddr);
        return 0;
    }
    // "" 
    n = (sec.dpb.DRM+1) / DIRINSEC;
    memset((char*) &sec, 0xE5, sizeof(SYSSEC));
    AbsAddr++;
    for (i = 0; i < n; i++)
    {
        ide_WriteSector(p, AbsAddr, &sec);
        AbsAddr++;
    }

    return -1;
}



//
//  SMBR     CP/M
//  relAdd -      SMBR
//
void hdd_ParseSMBR(DEVICE *p, UINT32 relAddr, char *lastDev, FILESYS *fsys)
{
    PARTION    *par;
    PARTION    *nxt;
    UINT8       buff[512];
    UINT32      base = relAddr;     //    SMBR
    char        c;

    do
    {
        //  SMBR
        if (!ide_ReadSector(p, relAddr, &buff))
        {
            printf("    *error* - can't read SMBR at 0x%08lX!\n", relAddr);
            return;
        }
        if (!hdd_CheckSign(buff))
        {
            printf("    *error* - SMBR at 0x%08lX is corrupt!\n", relAddr);
            return;
        }
        par = (PARTION *) &buff[0x1BE];
        nxt = (PARTION *) &buff[0x1CE];
        switch (par->Type)
        {
            case 0x01:
            case 0x04:
            case 0x06:
            case 0x0B:
            case 0x0E: {        // DOS logic disk
                (*lastDev)++;
                printf("    -found DOS disk [%c]    size: %9luKb\tFormat drive (Yes/No)? ", (*lastDev)+'A', par->Size / 2);
                fflush(stdout);
                c = getch();
                if ((c == 'y') || (c == 'Y'))
                {
                    printf("\r                                                                               \r");
                    printf("    -created CP/M disk [%c]\n", (*lastDev)+'A');
                    if (hdd_doFormatCPM(p, par->RelAddr+relAddr, par->Size, fsys))
                    {
                        par->Type = CPM_TYPE;
                        ide_WriteSector(p, relAddr, &buff);
                    }
                } else {
                    printf("\r                                                                               \r");
                    printf("    -skip DOS disk [%c]   size: %9luKb\n", (*lastDev)+'A', par->Size / 2);
                }
                break;
            }
            case CPM_TYPE: {
                (*lastDev)++;
                printf("    -found CP/M disk [%c]   size: %9luKb\tUnformat drive (Yes/No)? ", (*lastDev)+'A', par->Size / 2);
                fflush(stdout);
                c = getch();
                if ((c == 'y') || (c == 'Y'))
                {
                    par->Type = 1;
                    if ((par->Size / 2) > 15*1024)
                        par->Type += 3;
                    if ((par->Size / 2) > 32*1024)
                        par->Type += 2;
                    ide_WriteSector(p, relAddr, &buff);
                    printf("\r                                                                               \r");
                    printf("    -unformat disk [%c] to DOS disk\n", (*lastDev)+'A');
                } else {
                    printf("\r                                                                               \r");
                    printf("    -skip CP/M disk [%c]   size: %9luKb\n", (*lastDev)+'A', par->Size / 2);
                }
                break;
            }
        } // switch
        relAddr = nxt->RelAddr+base;
    } while (nxt->Type != 0);
}



//
//   CP/M     dos
//
void hdd_ParseMBR(DEVICE *p, FILESYS *fsys)
{
    PARTION    *par;
    UINT8       buff[512];
    int         i;
    char        lastDev;

    //  MBR
    if (!ide_ReadSector(p, 0, &buff))
    {
        printf("  *error* - can't read MBR!\n");
        return;
    }
    if (!hdd_CheckSign(buff))
    {
        printf("  *error* - MBR is corrupt!\n");
        return;
    }
    //    MBR
    par = (PARTION*) &buff[0x1BE];
    lastDev = -1;
    for(i = 0; i < 4; i++)
    {
        if (par->Active)
        {
            printf("    -skip primary DOS partion.\n");
        } else {
            if ((par->Type == 0x05) || (par->Type == 0x0C) || (par->Type == 0xF))
                hdd_ParseSMBR(p, par->RelAddr, &lastDev, fsys);
        }
        par++;
    }
}



/*
==============================================================================

                                   MENU

==============================================================================
*/


//      
FILE *hdd_Find(char *imgfile)
{
    return fopen(imgfile, "r+b");
}


void do_Usage()
{
    printf("Usage: F8000d.EXE file.img [-options | /options]\n");
    printf("  file.img - filename of image HDD.\n");
    printf("  options:\n");
    printf("    h|?    - this help\n");
    printf("    a<xxx> - size ALV with 'xxx', [128..512] (default 256)\n");
    printf("    d<x>   - size directory in clusters [2..16] (default 2)\n");
    printf("    t<x>   - reserved tracks [0..9] (default 0)\n");
}

char do_argv(int argc, char *argv[], char *DiskName, FILESYS *fsys)
{
    int     i;
    long    n;

    fsys->ALV = DEFAULT_ALV;
    fsys->DirBlocks = DEFAULT_DIRBLOCKS;
    fsys->ResTracks = DEFAULT_RESTRACKS;
    if (argc < 2)
    {
        do_Usage();
        return 0;
    }
    strcpy(DiskName, &argv[1][0]);

    i = 2;
    while (i < argc)
    {
        if ((argv[i][0] == '-') || (argv[i][0] == '/'))
        {
            switch (argv[i][1])
            {
                case 'A':
                case 'a': {     // ALV size
                    n = atol(&argv[i][2]);
                    if ((n >= 128) && (n <= 512))
                        fsys->ALV = n;
                    else
                        printf("  *error* - parameter '-a' is bad! Use default.\n");
                    break;
                }
                case 'D':
                case 'd': {
                    n = atol(&argv[i][2]);
                    if ((n >= 2) && (n <= 16))
                        fsys->DirBlocks = n;
                    else
                        printf("  *error* - parametr '-d' is bad! Use default.\n");
                    break;
                }
                case 'T':
                case 't': {
                    n = atol(&argv[i][2]);
                    if ((n >= 0) && (n <= 9))
                        fsys->ResTracks = n;
                    else
                        printf("  *error* - parametr '-t' is bad! Use default.\n");
                    break;
                }
            } // switch

        }   // if
        i++;
    }
    return -1;
}


void do_Format(int argc, char *argv[])
{
    FILESYS fsys;
    DEVICE         *p;
    char           *DiskName;

    while (kbhit()) getch();

    DiskName = (char*) malloc(_MAX_PATH);
    if (DiskName == NULL)
    {
        printf("*error* - not enought memory!\n");
        return;
    }
    if (!do_argv(argc, argv, DiskName, &fsys))
        return;
    if ((p = hdd_Find(DiskName)) == 0)
    {
        printf("*error* - hard disk's not found!\n");
        return;
    }

    //    
    hdd_ParseMBR(p, &fsys);

    free(DiskName);
    while (kbhit()) getch();
}


int main(int argc, char *argv[])
{
    printf("Format for PK8000.\n");
    do_Format(argc, argv);
    printf("Bye!\n");
    return 0;
}






