#!/usr/bin/perl
#
# Encoding data from Nemiga MD DSK file to HxC hfe
#
# Supported DSK types:
# 80 tracks, 1 side , Sector 128 bytes, 23 sectors per track, 235 520 bytes
#
# (c) 2104 PortaOne, Inc.
# Oleksandr Kapitanenko kapitan@portaone.com
#
# Usage:
# nemiga-dsk-to-hfe.pl <dsk file> <hfe file>
# or
# nemiga-dsk-to-hfe.pl <dsk file> <dsk file> <hfe file>
#
# DSK file may be:
# - standard RT-11 DSK
# - Nemiga MD = standard RT-11 DSK with 128 byte track-sector table
# - RAW file of 250 000 bytes = 80* FM track 3125 bytes

#
use POSIX qw(ceil);
$| = 1;

# Replace MD Header with the standard one
#
my $replace_md_header = 0;

# Debug flag
#
#my $debug = 'FM';

my $n_files = $#ARGV;
if( $n_files != 1 && $n_files != 2) {
   die <<EOF;

Usage:
nemiga-dsk-to-hfe.pl <dsk file> <hfe file>
or
nemiga-dsk-to-hfe.pl <dsk file> <dsk file> <hfe file>

EOF
}
 
die "Output and input files are the same!" if $n_files == 1 && $ARGV[0] eq $ARGV[1];
die "Output and input files are the same!" if $n_files == 2 && $ARGV[0] eq $ARGV[3] || $ARGV[1] eq $ARGV[3];
unlink($ARGV[$#ARGV]);

my @in_files;
my $dsk_size = -s $ARGV[0];
if( $dsk_size > 235520 && $dsk_size != 250000 ) {
   die "DSK file must be <= 235 520 bytes. Or RAW file must be 250 000 bytes exactly.";
}
open(IN1, "<$ARGV[0]")
   || die "File $ARGV[0] not found.";
push( @in_files, *IN1 );
if( $n_files == 2 ) {
$dsk_size = -s $ARGV[1];
   if( $dsk_size > 235520 && $dsk_size != 250000 ) {
      die "DSK file must be <= 235 520 bytes. Or RAW file must be 250 000 bytes exactly.";
   }
   open(IN2, "<$ARGV[1]")
   || die "File $ARGV[1] not found.";
   push @in_files, *IN2;
}
open(OUT, ">$ARGV[$#ARGV]")
   || die "Can not create output file $ARGV[$#ARGV]";

# Standard MD Header
#
my $md_header = [];
push $md_header, ( 23 ) x 80;
push $md_header, ( 0x30, 0x07 );
push $md_header, ( 0xFF ) x 46;

# Read DSK file
#
my $DSK_Tracks, $DSK_Sides;
$DSK_Tracks = 80;
$DSK_Sides = $n_files;

print "Reading DSK  ";
my @dsk, $buff, $dsk_track, $track, $side, $file, @md_headers, $header_buff;
for( $track = 0; $track < $DSK_Tracks*2; $track++ ) {
   $dsk_track = [];
   $side = $track & 0x01;
   if( $DSK_Sides == 1 && $side == 1 ) {
      push $dsk_track, ( 0x00 ) x 128*23;
      print "_";
   } else {
    if( -s $ARGV[$side] != 250000 ) {
      if( int($track/2) == 0 ) {
         $header_buff = []; 

         sysread( $in_files[$side], $buff, 128 );
         push $header_buff, unpack( 'C128', $buff);

         if( $$header_buff[0] == 0xA0 ) {
            #
            # DSK File
            #
            $md_header[$side] = $md_header;
            push $dsk_track, @$md_header;

            push $dsk_track, @$header_buff;

            sysread( $in_files[$side], $buff, 128*21 );
            push $dsk_track, unpack( 'C2688', $buff);
         } else {
            #
            # MD File
            #
            if( $replace_md_header == 1 ) {
               $md_header[$side] = $md_header;
               push $dsk_track, @$md_header;
            } else {
               $md_header[$side] = $header_buff;
               push $dsk_track, @$header_buff;
            }

            sysread( $in_files[$side], $buff, 128*22 );
            push $dsk_track, unpack( 'C2816', $buff);
         }
         print "D";
      } else {
         if( sysread( $in_files[$side], $buff, 128*23 ) ) {
            push $dsk_track, unpack( 'C2944', $buff); 
            print "D";
         } else {
            push $dsk_track, ( 0x00 ) x 128*23;
            print "0";
         }
      }
    } else {
       push $dsk_track, ( 0x00 ) x 128*23;
       print "R";
    }

   }
   $dsk[$track] = $dsk_track;
}

# DSK to TRK
#
print "\nDSK to TRK ";
my $data_byte, $MSB, $LSB, $crc, $dsk_byte, $ph_track, $sec_per_track;

# Glogal Nemiga Logical Sectro Number
#
my @global_sector = ( 0, 0 );


for( $track = 0; $track < $DSK_Tracks*2; $track++ ) {
   $dsk_track = [];
   $side = $track & 0x01;

 if( -s $ARGV[$side] != 250000 ) {
   $dsk_byte = 0;
   $crc = 0;

   # GAP1
   #
   push $dsk_track, ( 0x00 ) x 8;

   # Marker
   #
   push $dsk_track, ( 0xF3 );
   $data_byte = 0xFFF3;
   $crc = ( $crc + $data_byte ) & 0xFFFF;

   # Track Number
   #
   $ph_track = int($track/2);
   push $dsk_track, ( $ph_track );
   $data_byte = $ph_track;
   $data_byte = 0xFF00 | $data_byte if $data_byte & 0x80;
   $crc = ( $crc + $data_byte ) & 0xFFFF;

   # Number of Sectors on Track
   #
   $sec_per_track = $md_header[$side][$ph_track];

   push $dsk_track, ( $sec_per_track , 0 );
   $data_byte = $sec_per_track;
   $crc = ( $crc + $data_byte ) & 0xFFFF;
   $data_byte = 0x0000;
   $crc = ( $crc + $data_byte ) & 0xFFFF;

   # Number of First Sector on Track Word
   #
   $MSB = $global_sector[$side] >> 8;
   $LSB = $global_sector[$side] & 0xFF;
   push $dsk_track, ( $LSB, $MSB );
   $global_sector[$side] += $sec_per_track;

   $data_byte = $LSB;
   $data_byte = 0xFF00 | $data_byte if $data_byte & 0x80;
   $crc = ( $crc + $data_byte ) & 0xFFFF;
   $data_byte = $MSB;
   $data_byte = 0xFF00 | $data_byte if $data_byte & 0x80;
   $crc = ( $crc + $data_byte ) & 0xFFFF;

   # Status Word
   #
   push $dsk_track, ( 0xFF, 0xFF );
   $data_byte = 0xFFFF;
   $crc = ( $crc + $data_byte ) & 0xFFFF;
   $data_byte = 0xFFFF;
   $crc = ( $crc + $data_byte ) & 0xFFFF;

   # Checksum Word
   #
   $MSB = $crc >> 8;
   $LSB = $crc & 0xFF;
   push $dsk_track, ( $LSB, $MSB );

   for($sector = 0; $sector < 23; $sector++) {

      # DATA
      #
      $crc = 0;

      for ($count = 128; $count >= 1; $count--) {
         $data_byte = $dsk[$track][$dsk_byte++];
         $data_byte = 0xFF00 | $data_byte if $data_byte & 0x80;

         $crc = ( $crc + $data_byte ) & 0xFFFF;

         push $dsk_track, ( $data_byte );
      }

      # CRC
      #
      $MSB = $crc >> 8;
      $LSB = $crc & 0xFF;
      push $dsk_track, ( $LSB, $MSB );
   }

   # Total of 3008. Total FM track is 3125 bytes.
   # Remaider is 117 bytes 
   #
   push $dsk_track, ( 0x00 ) x 117;
 } else {
   sysread( $in_files[$side], $buff, 3125 );
   push $dsk_track, unpack( 'C3125', $buff);
 }

   $dsk[$track] = $dsk_track;
   print ".";
}

# Generate FM code
#
print "\nEncoding FM ";

$fm_code[0] = 0x20;
$fm_code[1] = 0xA0;
 
my $mfm_track, $byte, $bit, $bit_count, $fm_i;
for( $track = 0;  $track < $DSK_Tracks*2; $track++ ) {
   $fm_i =0;

   for( $i = 0;  $i < $#{$dsk[$track]}+1; $i++ ) {
      $byte = $dsk[$track][$i];

      $bit_count = 0;
      $mfm_track[$track][$fm_i] = 0;
      while( $bit_count++ < 8 ) {
         $bit = ($byte & 0x80) >> 7;
         $byte = $byte << 1;
         if( $bit_count == 3 || $bit_count == 5 || $bit_count == 7 ) {
            
            # Advance to next FM byte
            #
            $fm_i++;
            $mfm_track[$track][$fm_i] = 0;
         } else {
            $mfm_track[$track][$fm_i] = $mfm_track[$track][$fm_i] >> 4;
         }
         $mfm_track[$track][$fm_i] = $mfm_track[$track][$fm_i] | $fm_code[$bit];
      }
      $fm_i++;
   }
   print ".";
}

print "\nCreating HFE ";

# Create HxC header
#
print OUT "HXCPICFE";
print OUT chr(0);           # Revision 0
print OUT chr($DSK_Tracks); # Number of tracks
print OUT chr($DSK_Sides);  # Number of sides (Not used by the emulator)
print OUT chr(2);           # Track Encoding mode FM
print OUT chr(250), chr(0); # Bitrate in Kbit/s. Ex : 250=250000bits/s for FM true_bitrate*2
print OUT chr(0), chr(0);   # Rotation per minute (Not used by the emulator)
print OUT chr(7);           # Floppy interface mode
print OUT chr(0);           # Write protected
print OUT chr(1), chr(0);   # Offset of the track list LUT in block of 512 bytes (Ex: 1=0x200)

# Advance to 0x200
#
print OUT chr(0xFF)x(0x200-20);

# Create track list
#
my $dsk_track_len = $#{$dsk[0]}+1;
my $offset = 2; # Start offset 2x512 bytes blocks
for( $track = 0; $track < $DSK_Tracks; $track++ ) {
   $track_len = $dsk_track_len * 4 *2;
   print OUT chr($offset&0x00ff), chr($offset>>8), chr($track_len&0x00ff), chr($track_len>>8);
   $offset += ceil($track_len/512);
}

# Advance to 0x400
#
print OUT chr(0xFF)x(0x200-$DSK_Tracks*4); 

# Dump FM code to hfe file
#
my $i1, $i2, $s0_last_byte, $s1_last_byte;
$hfe_track_len = ceil( $dsk_track_len * 4 * 2 / 512 ) * 512;

for( $track = 0; $track < $DSK_Tracks*2; $track += 2 ) {
   print "..";
   $i = 0;
   $i1 = 0;
   $i2 = 0;
   while( $i < $hfe_track_len ) {

      # Alternate 512 byte blocks. Pad track with last byte if needed.
      #
      if( $i & 0x0100 ) {
         #
         # Side 1
         #
         if( $i2 < $dsk_track_len*4 ) {
            print OUT chr( $mfm_track[$track+1][$i2] );
            $s1_last_byte = $mfm_track[$track+1][$i2];
            $i2++;
         } else {
            print OUT chr( $s1_last_byte );
         }
      } else {
         #
         # Side 0
         #
         if( $i1 < $dsk_track_len*4 ) {
            print OUT chr( $mfm_track[$track][$i1] );
            $s0_last_byte = $mfm_track[$track][$i1];
            $i1++;
         } else {
            print OUT chr( $s0_last_byte );
         }
      }
   $i++;
   }
}
print "\nDone!\n";

close(OUT);

foreach $file (@in_files)
{
   close($file);
}

