module SRAM_Controller(
	input			clk,				//  Clock 120MHz
	input			reset,					//  System reset
	inout	[15:0]	SRAM_DQ,				//	SRAM Data bus 16 Bits
	output reg[17:0]	SRAM_ADDR,			//	SRAM Address bus 18 Bits
	output reg			SRAM_LB_N,				//	SRAM Low-byte Data Mask 
	output reg			SRAM_UB_N,				//	SRAM High-byte Data Mask
	output	reg		SRAM_WE_N,				//	SRAM Write Enable
	output		SRAM_OE_N,				//	SRAM Output Enable
	output			SRAM_CE_N,				//	SRAM Chip Enable
	input	[18:0]	iaddr,
	input	[7:0]	idata,
	input			rd,
	input			we_n,
	output	reg [15:0]	odata,
	output	reg [15:0]	odata2,
	output reg memcpubusy,
	output reg rdcpu_finished,
	output reg memvidbusy,
	input rdv
);

parameter ST_RESET = 4'd0;
parameter ST_IDLE   = 4'd1;
parameter ST_PRERW   = 4'd2;
parameter ST_READ0  = 4'd3;
parameter ST_READ1  = 4'd4;
parameter ST_READ2  = 4'd5;
parameter ST_WRITE0 = 4'd6;
parameter ST_WRITE1 = 4'd7;
parameter ST_WRITE2 = 4'd8;
parameter ST_READV0  = 4'd9;
parameter ST_READV1  = 4'd10;
parameter ST_READV2  = 4'd11;

reg[3:0] state;
reg[17:0] addr;
reg[7:0] data;
reg exrd,exwen,lsb,rdvid;

assign SRAM_DQ[7:0] = (state==ST_WRITE1) ? data : 8'bZZZZZZZZ;
assign SRAM_DQ[15:8] = (state == ST_WRITE1) ? data : 8'bZZZZZZZZ;
assign SRAM_OE_N = 0;
assign SRAM_CE_N = reset;


always @(posedge clk) begin
	if (reset) begin
//	if (0) begin
		state <= ST_RESET; exrd <= 0; exwen <= 1'b1;
	end else begin
		case (state)
		ST_RESET: {state,SRAM_WE_N} <= {ST_IDLE,1'b1};
		ST_IDLE: begin
			{memcpubusy,memvidbusy,rdcpu_finished}<=3'b000;
			if(rdv==0){exrd,exwen} <= {rd,we_n};
			{addr[17:0],lsb,data,rdvid}<={iaddr[18:1],iaddr[0],idata,rdv};
			SRAM_WE_N <= 1'b1;
			casex ({rd,exrd,we_n,exwen,rdv})
			5'b10110: {state,memcpubusy} <= {ST_PRERW,1'b1};
			5'b00010: {state,memcpubusy} <= {ST_PRERW,1'b1};
			5'bxxxx1: {state,memvidbusy} <= {ST_PRERW,1'b1};
			default: state <= ST_IDLE;
			endcase
		end
		ST_PRERW:
			casex ({exrd,exwen,rdvid})
			3'b110: {state,SRAM_ADDR,SRAM_UB_N,SRAM_LB_N} <= {ST_READ0,addr,~lsb,lsb};
			3'b000: {state,SRAM_ADDR,SRAM_WE_N,SRAM_UB_N,SRAM_LB_N} <= {ST_WRITE0,addr,1'b0,~lsb,lsb};
			3'bxx1: {state,SRAM_ADDR,SRAM_UB_N,SRAM_LB_N} <= {ST_READ0,{addr[17:1],1'b0},2'b00};
			default: state <= ST_IDLE;
			endcase
		ST_READ0: state <= ST_READ1;
		ST_READ1: state <= ST_READ2;
		ST_READ2:
		case(rdvid)
		1'b0:{state,rdcpu_finished,odata[7:0]}<={ST_IDLE,1'b1,lsb?SRAM_DQ[15:8]:SRAM_DQ[7:0]};
		1'b1:{state,odata,SRAM_ADDR} <= {ST_READV0,SRAM_DQ[15:0],{addr[17:1],1'b1}};
		endcase
		ST_READV0: state <= ST_READV1;
		ST_READV1: state <= ST_READV2;
		ST_READV2: {state,odata2} <= {ST_IDLE,SRAM_DQ[15:0]};
		ST_WRITE0: state <= ST_WRITE1;
		ST_WRITE1: state <= ST_WRITE2;
		ST_WRITE2: state <= ST_IDLE;
		default: state <= ST_IDLE;
		endcase
	end
end

endmodule
