#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* Spectrum screen compressor
   Based on Burial Laser Compact 1.03
   Nikita Burnashev, 2005 */

void reorder(unsigned char *pIn, unsigned char *pOut)
{
    int i;
    unsigned char *po = pOut;

    for (i = 0; i < 6144; i++)
    {
        *po++ = pIn[i & 0xf800 | (i & 7) << 8 | (i & 0x38) << 2 | (i & 0x7c0) >> 6];
    }
    memcpy(po, pIn + 6144, 768);
}

unsigned char *po, *pb;
int bbuf;

void bit(int value)
{
    if (bbuf == 1) pb = po++;
    bbuf = (bbuf << 1) | value;
    if (bbuf & 0x100)
    {
        *pb = bbuf & 0xff;
        bbuf = 1;
    }
}

void vlc(int num)
{
    if (num == 0) bit(1);
    else if (num <= 2)
    {
        bit(0);
        bit(num & 1);
        bit(1);
    }
    else if (num <= 6)
    {
        num = 6 - num;
        bit(0);
        bit(num >> 1);
        bit(0);
        bit(num & 1);
        bit(1);
    }
    else if (num <= 22)
    {
        num = 22 - num;
        bit(0);
        bit(num >> 3);
        bit(0);
        bit(num >> 2 & 1);
        bit(0);
        bit(num >> 1 & 1);
        bit(num & 1);
    }
}

int compress(unsigned char *pIn, unsigned char *pInEnd, unsigned char *pOut, int wnd)
{
    unsigned char *pi = pIn;

    po = pOut;
    bbuf = 1;

    *po++ = *pi++;
    do
    {
        unsigned char *ps, *pw;
        int mtype, msize, mofs;

        /* find match */
        mtype = msize = 0;

        ps = pi - wnd;
        if (ps < pIn) ps = pIn;

        pw = pi - 1;
        while (1)
        {
            unsigned char *pwc, *pic;
            int nsize, minsize;

            while (pw >= ps && *pw != *pi) pw--;
            if (pw < ps) break;

            /* forward match */
            pwc = pw + 1;
            pic = pi + 1;
            nsize = 0;
            while (pic < pInEnd && nsize < 255 && *pwc == *pic)
            {
                pwc++;
                pic++;
                nsize++;
            }

            minsize = (pi - pw >= 0x300) ? 2 : 1;
            if (nsize > msize && nsize >= minsize)
            {
                mtype = 2;
                msize = nsize;
                mofs = pi - pw;
            }

            /* backward match */
            pwc = pw - 1;
            pic = pi + 1;
            nsize = 0;
            while (pic < pInEnd && pwc >= pIn && nsize < 255 && *pwc == *pic)
            {
                pwc--;
                pic++;
                nsize++;
            }

            minsize = (pi - pw >= 0x300) ? 2 : 1;
            if (nsize > msize && nsize >= minsize)
            {
                mtype = 3;
                msize = nsize;
                mofs = pi - pw;
            }

            pw--;
        }

#ifdef DEBUG
        if (mtype == 0) printf("0x%04x 0x%04x 0 0x%02x\n", pi - pIn, po - pOut, *pi);
        else printf("0x%04x 0x%04x %d %d 0x%04x\n", pi - pIn, po - pOut, mtype, msize, 0x10000 - mofs);
#endif

        /* store char or match */
        if (mtype == 0)
        {
            bit(1);
            *po++ = *pi++;
        }
        else
        {
            pi += msize + 1;
            bit(0);
            if (mofs >= 0x300) msize--;
            if (msize >= 23)
            {
                vlc(6);
                *po++ = (-msize) & 0xff;
            }
            else
            {
                if (msize < 7) msize--;
                vlc(msize);
            }
            vlc((mofs - 1) >> 8);
            bit(mtype & 1);
            *po++ = (-mofs) & 0xff;
        }
    }
    while (pi < pInEnd);

    bit(0);
    vlc(6);
    *po++ = 0;

    if (bbuf != 1)
    {
        while (!(bbuf & 0x100)) bbuf <<= 1;
        *pb = bbuf & 0xff;
    }

    return (po - pOut);
}

unsigned char bufScr[6912];
unsigned char bufIn[6912];
unsigned char bufOut[7000];

int main(int argc, char *argv[])
{
    int i, wnd;
    char *nameIn, *nameOut, *p;
    FILE *pfi, *pfo;
    int size;

    nameIn = NULL;
    nameOut = NULL;
    wnd = 0x1700;
    for (i = 1; i < argc; i++)
    {
        p = argv[i];
        if (*p == '-')
        {
            if (strcmp(p, "-wnd") == 0) wnd = strtol(argv[++i], &p, 0);
        }
        else if (!nameIn) nameIn = p;
        else if (!nameOut) nameOut = p;
    }
    if (!nameIn || !nameOut || wnd < 1 || wnd > 0x1700)
    {
        printf("Usage: laser [-wnd <[0x]num>] <input.scr> <output.bin>\n"
            "Default window is 0x1700\n");
        return 1;
    }

    pfi = fopen(nameIn, "rb");
    if (!pfi)
    {
        printf("Can't open %s for reading", nameIn);
        return 2;
    }
    fread(bufScr, 1, 6912, pfi);
    fclose(pfi);

    reorder(bufScr, bufIn);
    size = compress(bufIn, bufIn + 6912, bufOut, wnd);

    pfo = fopen(nameOut, "wb");
    fwrite(bufOut, 1, size, pfo);
    fclose(pfo);
    return 0;
}
