#include <stdio.h>
#include <stdarg.h>

#include <windows.h>
#include <setupapi.h>

#ifdef _DEBUG
//for debugging purposes
void PrintDebug(char *fmt, ...)
{
	char buf[4096];
	va_list args;
	va_start( args, fmt );
	vsprintf( buf, fmt, args );
	va_end( args );
	OutputDebugString(buf);
}
#endif

//Handles for accessing USB device
HANDLE hWrite = INVALID_HANDLE_VALUE;
HANDLE hRead  = INVALID_HANDLE_VALUE; 

//Modify this value to match the VID and PID in your USB device descriptor.
//Use the formatting: "Vid_xxxx&Pid_xxxx" where xxxx is a 16-bit hexadecimal number.
#define MY_DEVICE_ID  "VID_4FFD&PID_1001" //in up case

//Globally Unique Identifier (GUID) for HID class devices.  Windows uses GUIDs to identify things.
GUID InterfaceClassGuid = {0x4d1e55b2, 0xf16f, 0x11cf, 0x88, 0xcb, 0x00, 0x11, 0x11, 0x00, 0x00, 0x30}; 

BOOL FindDevice(HANDLE* phRead, HANDLE* phWrite)
{
	HDEVINFO DeviceInfoTable = INVALID_HANDLE_VALUE;
	SP_DEVICE_INTERFACE_DATA InterfaceDataStructure;
	SP_DEVICE_INTERFACE_DETAIL_DATA* DetailedInterfaceDataStructure = 0;
	SP_DEVINFO_DATA DevInfoData;

	DWORD InterfaceIndex = 0;
	DWORD StatusLastError = 0;
	DWORD dwRegType;
	DWORD dwRegSize;
	DWORD StructureSize = 0;
	PBYTE PropertyValueBuffer = 0;
	BOOL MatchFound = FALSE;
	BOOL StopFound = FALSE;
	DWORD ErrorStatus = ERROR_SUCCESS;

	//First populate a list of plugged in devices (by specifying "DIGCF_PRESENT"), which are of the specified class GUID. 
	DeviceInfoTable = SetupDiGetClassDevs(&InterfaceClassGuid, NULL, NULL, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);

	//Now look through the list we just populated.  We are trying to see if any of them match our device. 
	do
	{
		InterfaceDataStructure.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);
		if(SetupDiEnumDeviceInterfaces(DeviceInfoTable, NULL, &InterfaceClassGuid, InterfaceIndex, &InterfaceDataStructure))
		{
			ErrorStatus = GetLastError();
#ifdef _DEBUG
			PrintDebug("%i:SetupDiEnumDeviceInterfaces OK errorstatus %i\r\n",__LINE__,ErrorStatus);
#endif
			if(ERROR_NO_MORE_ITEMS == ErrorStatus)	//Did we reach the end of the list of matching devices in the DeviceInfoTable?
			{	//Cound not find the device.  Must not have been attached.
				break;		
			}
		}
		else	//Else some other kind of unknown error ocurred...
		{
			ErrorStatus = GetLastError();
#ifdef _DEBUG
			PrintDebug("%i:SetupDiEnumDeviceInterfaces FAIL errorstatus %i\r\n",__LINE__,ErrorStatus);
#endif
			break;	
		}

		//Now retrieve the hardware ID from the registry.  The hardware ID contains the VID and PID, which we will then 
		//check to see if it is the correct device or not.

		//Initialize an appropriate SP_DEVINFO_DATA structure.  We need this structure for SetupDiGetDeviceRegistryProperty().
		DevInfoData.cbSize = sizeof(SP_DEVINFO_DATA);
		if ( !SetupDiEnumDeviceInfo(DeviceInfoTable, InterfaceIndex, &DevInfoData) )
		{
			ErrorStatus = GetLastError();
#ifdef _DEBUG
			PrintDebug("%i:SetupDiEnumDeviceInfo FAIL errorstatus %i\r\n",__LINE__,ErrorStatus);
#endif	
		}
		else
		{
			//First query for the size of the hardware ID, so we can know how big a buffer to allocate for the data.
			SetupDiGetDeviceRegistryProperty(DeviceInfoTable, &DevInfoData, SPDRP_HARDWAREID, &dwRegType, NULL, 0, &dwRegSize);
#ifdef _DEBUG
			PrintDebug("%i:dwRegSize=%lu\r\n",__LINE__, dwRegSize);
#endif	

			//Allocate a buffer for the hardware ID.
			PropertyValueBuffer = (BYTE *) malloc (dwRegSize);
			if(PropertyValueBuffer == NULL)	//if null, error, couldn't allocate enough memory
			{	
				ErrorStatus = GetLastError();
#ifdef _DEBUG
				PrintDebug("%i:malloc fail %i\r\n",__LINE__,ErrorStatus);
#endif	
			}
			else
			{
				//Retrieve the hardware IDs for the current device we are looking at.  PropertyValueBuffer gets filled with a 
				//REG_MULTI_SZ (array of null terminated strings).  To find a device, we only care about the very first string in the
				//buffer, which will be the "device ID".  The device ID is a string which contains the VID and PID, in the example 
				//format "Vid_04d8&Pid_003f".
				if ( !SetupDiGetDeviceRegistryProperty(DeviceInfoTable, &DevInfoData, SPDRP_HARDWAREID, &dwRegType, PropertyValueBuffer, dwRegSize, NULL) )
				{
#ifdef _DEBUG
					ErrorStatus = GetLastError();
					PrintDebug("%i:SetupDiGetDeviceRegistryProperty FAIL errorstatus %i\r\n",__LINE__, ErrorStatus);
#endif	
				}
				else
				{
					//Now check if the first string in the hardware ID matches the device ID of my USB device.
#ifdef _DEBUG
					PrintDebug("%i:PropertyValueBuffer:'%s'\r\n",__LINE__,PropertyValueBuffer);
#endif
					strupr(PropertyValueBuffer);
					if ( strstr(PropertyValueBuffer,MY_DEVICE_ID) )
					{
						//Device must have been found.  Open read and write handles.  In order to do this, we will need the actual device path first.
						//We can get the path by calling SetupDiGetDeviceInterfaceDetail(), however, we have to call this function twice:  The first
						//time to get the size of the required structure/buffer to hold the detailed interface data, then a second time to actually 
						//get the structure (after we have allocated enough memory for the structure.)
						StopFound = TRUE;

						//First call populates "StructureSize" with the correct value
						SetupDiGetDeviceInterfaceDetail(DeviceInfoTable, &InterfaceDataStructure, NULL, NULL, &StructureSize, NULL);	
#ifdef _DEBUG
						PrintDebug("%i:StructureSize=%lu\r\n",__LINE__,StructureSize);
#endif	

						//Allocate enough memory
						DetailedInterfaceDataStructure = (SP_DEVICE_INTERFACE_DETAIL_DATA*)(malloc(StructureSize));		
						if(DetailedInterfaceDataStructure == NULL)	//if null, error, couldn't allocate enough memory
						{	//Can't really recover from this situation, just exit instead.
#ifdef _DEBUG
							PrintDebug("%i:malloc fail\r\n",__LINE__);
#endif	
						}
						else
						{
							DetailedInterfaceDataStructure->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

							//Now call SetupDiGetDeviceInterfaceDetail() a second time to receive the goods.  
							if ( !SetupDiGetDeviceInterfaceDetail(DeviceInfoTable, &InterfaceDataStructure, DetailedInterfaceDataStructure, StructureSize, NULL, NULL) )
							{
								ErrorStatus = GetLastError();
#ifdef _DEBUG
								PrintDebug("%i:SetupDiGetDeviceInterfaceDetail FAIL errorstatus %i\r\n",__LINE__, ErrorStatus);
#endif	
							}
							else
							{
#ifdef _DEBUG
								PrintDebug("%i:DevicePath:'%s'\r\n",__LINE__, DetailedInterfaceDataStructure->DevicePath);
#endif	
								//We now have the proper device path, and we can finally open read and write handles to the device.
								//We store the handles in the global variables "WriteHandle" and "ReadHandle", which we will use later to actually communicate.
								if ( phWrite!=NULL )
								{
									*phWrite = CreateFile((DetailedInterfaceDataStructure->DevicePath), GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, 0);
#ifdef _DEBUG
									ErrorStatus = GetLastError();
									PrintDebug("%i:CreateFile(GENERIC_WRITE) errorstatus %i\r\n",__LINE__,ErrorStatus);
#endif	
								}
								if ( phRead!=NULL )
								{
									*phRead = CreateFile((DetailedInterfaceDataStructure->DevicePath), GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, 0);
#ifdef _DEBUG
									ErrorStatus = GetLastError();
									PrintDebug("%i:CreateFile(GENERIC_READ) errorstatus %i\r\n",__LINE__,ErrorStatus);
#endif	
								}
								MatchFound = TRUE;
							}

							free(DetailedInterfaceDataStructure);
						}
					}
				}

				free(PropertyValueBuffer);		//No longer need the PropertyValueBuffer, free the memory to prevent potential memory leaks
			}
		}

		InterfaceIndex++;	
	}
	while( !StopFound );

	SetupDiDestroyDeviceInfoList(DeviceInfoTable);	//Clean up the old structure we no longer need.
	return MatchFound;
}


BOOL ExchangeDevice(HANDLE hWrite, HANDLE hRead, BYTE funcNumber, BYTE* sendBuffer, BYTE sendSize, BYTE* readBuffer, BYTE readSize)
{
	DWORD BytesWritten = 0;
	DWORD BytesRead = 0;
	BYTE OutputPacketBuffer[65];	//Allocate a memory buffer equal to our endpoint size + 1
	BYTE InputPacketBuffer[65];	//Allocate a memory buffer equal to our endpoint size + 1

	InputPacketBuffer[0] = 0;				//The first byte is the "Report ID" and does not get transmitted over the USB bus.  Always set = 0.
	OutputPacketBuffer[0] = 0;				//The first byte is the "Report ID" and does not get transmitted over the USB bus.  Always set = 0.

	OutputPacketBuffer[1] = funcNumber;
	memcpy(&OutputPacketBuffer[2],sendBuffer,(sendSize<sizeof(OutputPacketBuffer)-2)?sendSize:sizeof(OutputPacketBuffer)-2);
	//For simplicity, we will leave the rest of the buffer uninitialized, but you could put real
	//data in it if you like.

	//The basic Windows I/O functions WriteFile() and ReadFile() can be used to read and write to HID class USB devices 
	//(once we have the read and write handles to the device, which are obtained with CreateFile()).

	//The following call to WriteFile() sends 64 bytes of data to the USB device.
	WriteFile(hWrite, &OutputPacketBuffer, sizeof(OutputPacketBuffer), &BytesWritten, 0);	//Blocking function, unless an "overlapped" structure is used

	if( readBuffer )
	{
		//Now get the response packet from the firmware.
		//The following call to ReadFile() retrieves 64 bytes of data from the USB device.
		ReadFile(hRead, &InputPacketBuffer, sizeof(InputPacketBuffer), &BytesRead, 0);		//Blocking function, unless an "overlapped" structure is used	

		//InputPacketBuffer[0] is the report ID, which we don't care about.
		//InputPacketBuffer[1] is an echo back of the command.
		memcpy(readBuffer, &InputPacketBuffer[2], (readSize<sizeof(InputPacketBuffer)-2)?readSize:sizeof(InputPacketBuffer)-2 );
	}

	return TRUE;
}

//for playing function
BYTE* data;
DWORD size;
DWORD counter;
HANDLE hEvent;

VOID CALLBACK play_ym3_proc(UINT uTimerID, UINT uMsg, DWORD_PTR dwUser, DWORD_PTR dw1, DWORD_PTR dw2)
{
	BYTE i = 0;
	BYTE playData[1+(14*2)];
	DWORD takts = (size-4)/14; 
	if ( counter > takts )
	{
		//end of data
		SetEvent(hEvent);
	}

	//data for playing is must set
	// 0     - registers count (n)
	// 1     - 1 register address 
	// 2     - 1 register data
	// (n-1)*2+1 - n register address
	// (n-1)*2+2 - n register data

	//prepare registers
	playData[0] = 14;
	for ( i=0; i<14; i++ )
	{
		//register number
		playData[(i*2) + 1] = i;
		playData[(i*2) + 2] = data[4 + i*takts + counter];
	}
	if ( playData[28] == 0xFF )
	{
		playData[0] = 13;
	}

	//send data to USB
	ExchangeDevice(hWrite, hRead, 0x20, playData, sizeof(playData), 0, 0);

	//if ( counter%50 == 0 )
	//{
	//	printf("\r\n.");
	//}
	counter ++;
}

void play_ym3(char* fileName)
{
	//open file
	HANDLE hFile = CreateFile(fileName, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, 0);
	if ( hFile == INVALID_HANDLE_VALUE )
	{
		printf("\r\nERROR: CreateFile(READ) failed '%s'",fileName);
	}
	else
	{
		//get size
		size = GetFileSize(hFile, 0);
		if ( size == INVALID_FILE_SIZE )
		{
			printf("\r\nERROR: GetFileSize failed %i", GetLastError());
		}
		else
		{
			//get mem
			data = (BYTE*)LocalAlloc(LPTR,size);
			if ( data==0 )
			{
				printf("\r\nERROR: LocalAlloc(%lu) failed %i",size,GetLastError());
			}
			else
			{
				//read file
				DWORD tmp;
				if ( ReadFile(hFile, data, size, &tmp, 0) == 0 )
				{
					printf("\r\nERROR: ReadFile failed %i",GetLastError());
				}
				else
				{
					//check format
					if ( memcmp(data,"YM3!",4)!=0 )
					{
						printf("\r\nERROR: wrong data file (only YM3 supported)");
					}
					else 
					if ( (size-4)%14 != 0 )
					{
						printf("\r\nERROR: wrong file size (%lu)",size);
					}
					else
					{
						//prepare playing
						printf("\r\nModul is %i takts (~%i seconds)",(size-4)/14, ((size-4)/14)/50 );
						counter = 0;
						hEvent = CreateEvent(0,TRUE,FALSE,0);
						if ( hEvent == 0 )
						{
							printf("\r\nERROR: CreateEvent failed %i",GetLastError());
						}
						else
						{
							//create timer
							MMRESULT timer = timeSetEvent(20,1,play_ym3_proc,0,TIME_PERIODIC);
							if ( timer == 0 )
							{
								printf("\r\nERROR: timeSetEvent failed %i",GetLastError());
							}
							else
							{
								WaitForSingleObject(hEvent,INFINITE);
								timeKillEvent(timer);
								printf("\r\nStop play");
							}

							//free event
							CloseHandle(hEvent);
						}
					}
				}
				//free mem
				LocalFree( data );
			}
		}
		//free file
		CloseHandle(hFile);
	}
}


int main (int argc, char* argv[])
{
	//get handles for exchanging data with USB device
	if ( !FindDevice(&hRead,&hWrite) )
	{
		printf("\r\nERROR: Device not found.");
	}
	else
	{
		int i;
		BYTE about[65];
		printf("\r\nDevice found.");

		//get ABOUT string
		// command code - 0x10 
		// return char string  
		ExchangeDevice(hWrite, hRead, 0x10, 0, 0, about, 64);
		about[65]=0;
		if ( strlen(about)>0 ) printf("\r\n%s",about);

		i = 1;
		while ( i<argc )
		{
			printf("\r\nStart play file '%s'",argv[i]);
			play_ym3(argv[i]);
			i++;
		}
	}

	if ( hWrite!=INVALID_HANDLE_VALUE ) CloseHandle(hWrite);
	if ( hRead !=INVALID_HANDLE_VALUE ) CloseHandle(hRead);

	return 0;
}
