//
//  i8080.c
//
//  Intel i8080 CPU emulation
//  Short and clean
//
//  Created by Alexander Medvedev on 03/04/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>
#include <memory.h>

#include "types.h"

#include "i8080.h"
#include "i8080debug.h"

#define SP              i8080->sp
#define PC              i8080->pc
#define A               i8080->af.b.h
#define F               i8080->af.b.l
#define B               i8080->bc.b.h
#define C               i8080->bc.b.l
#define D               i8080->de.b.h
#define E               i8080->de.b.l
#define H               i8080->hl.b.h
#define L               i8080->hl.b.l
#define AF              i8080->af.w
#define BC              i8080->bc.w
#define DE              i8080->de.w
#define HL              i8080->hl.w
#define IFF             i8080->iff

//
// Flag register [ S | Z | 0 | H | 0 | P | 1 | C ]
//

#define FLAGS   I8080FLAGS     // negative
#define FLAGZ   I8080FLAGZ     // zero
#define FLAG5   I8080FLAG5     // 0
#define FLAGH   I8080FLAGH     // halfcarry
#define FLAG3   I8080FLAG3     // 0
#define FLAGP   I8080FLAGP     // even
#define FLAG1   I8080FLAG1     // 1
#define FLAGC   I8080FLAGC     // carry / borrow

//
// Fast flag calculation tables:
//
// FSZPT    - SZ000P10              256 bytes
// FINCT    - SZ0H0P10 for INC      256 bytes
// FDECT    - SZ0H0P10 for DEC      256 bytes
//
// Next two are big, turn off I8080BIGTABLES if
// you code for microcontroller or make precalculated
// ones for ROM
//
// FADDT    - SZ0H0P1C for ADD/ADC  128 Kbytes
// FSUBT    - SZ0H0P1C for SUB/SBC  128 Kbytes
//

//            R
byte FSZPT  [256];
byte FINCT  [256];
byte FDECT  [256];

#ifdef I8080BIGTABLES

//        (FLAGC) S    R
byte FADDT  [2] [256][256];
byte FSUBT  [2] [256][256];

#endif

int i8080tablesReady = 0;

void i8080tables(void)
{    
    byte f;
    int val;
    
    for (int new = 0; new < 256; new++) {
        
        // SZ
        
        f = new ? (new & FLAGS) : FLAGZ;
        f |= FLAG1;
        
        val = 0;
        
        if (new &   1) val++;
        if (new &   2) val++;
        if (new &   4) val++;
        if (new &   8) val++;
        if (new &  16) val++;
        if (new &  32) val++;
        if (new &  64) val++;
        if (new & 128) val++;
        
        if (!(val&1)) f |= FLAGP;
        
        FSZPT[new] = f;
        
        // INC
        
        f = FSZPT[new];
        
        if ((new & 0x0F) == 0) f |=FLAGH;
        
        FINCT[new] = f;
        
        // DEC
        
        f = FSZPT[new];
        
        if ((new & 0x0F) != 0xF) f |=FLAGH;
        
        FDECT[new] = f;
    }
    
#ifdef I8080BIGTABLES
    
    for (int old = 0; old < 256; old++) {
        for (int new = 0; new < 256; new++) {
            
            // ADD
            
            val = new - old; // new = old + val
            
            if (new == 0) f = FLAGZ; else f = (new & FLAGS);
            if (new < old) f |= FLAGC;
            if ((new & 0x0F) < (old & 0x0F)) f |= FLAGH;
            f |= FSZPT[new] & FLAGP;
            f |= FLAG1;
            
            FADDT[0][old][new] = f;
            
            // ADD with C set
            
            val = new - old - 1; // new = old + val + 1
            
            if (new == 0) f = FLAGZ; else f = (new & FLAGS);
            if (new <= old) f |= FLAGC;
            if ((new & 0x0F) <= (old & 0x0F)) f |= FLAGH;
            f |= FSZPT[new] & FLAGP;
            f |= FLAG1;
            
            FADDT[1][old][new] = f;
            
            // SUB
            
            val = old - new; // new = old - val
            
            if (new == 0) f = FLAGZ; else f = (new & FLAGS);
            if (new > old) f |= FLAGC;
            if ((new & 0x0F) <= (old & 0x0F)) f |= FLAGH;
            f |= FSZPT[new] & FLAGP;
            f |= FLAG1;
            
            FSUBT[0][old][new] = f;
            
            // SUB with C set
            
            val = old - new - 1;  // new = old - val - 1
            
            if (new == 0) f = FLAGZ; else f = (new & FLAGS);
            if (new >= old) f |= FLAGC;
            if ((new & 0x0F) < (old & 0x0F)) f |= FLAGH;
            f |= FSZPT[new] & FLAGP;
            f |= FLAG1;
            
            FSUBT[1][old][new] = f;
        }
    }
    
#endif
    
    i8080tablesReady = 1;
}

void memory_write_word(word addr, word data)
{
    memory_write(addr, data & 255);
    memory_write(addr+1, data >> 8);
}

word memory_read_word(word addr)
{
    return memory_read(addr) + (memory_read(addr+1) << 8);
}

#define PUSH(x)     { SP-=2; memory_write_word(SP, x); }
#define POP(x)      { x = memory_read_word(SP); SP+=2; }

#define RB(x)       memory_read(x)
#define WB(x,b)     memory_write(x, b)

#define RW(x)       memory_read_word(x)
#define WW(x,w)     memory_write_word(x, w)

#define RP(x)       port_read(x)
#define WP(x,b)     port_write(x, b)

#define INR(x)      { ++(x); F=(F&FLAGC)|FINCT[x]; }    // SZ0H0P1- OK
#define DCR(x)      { --(x); F=(F&FLAGC)|FDECT[x]; }    // SZ0H0P1- OK

#ifdef I8080BIGTABLES

 #define ADD(x)     { b=A; A+=x; F=FADDT[0][b][A]; }    // SZ0H0P1C OK
 #define SUB(x)     { b=A; A-=x; F=FSUBT[0][b][A]; }    // SZ0H0P1C OK

 #define CMP(x)     { b=A-x; F=FSUBT[0][A][b]; }        // SZ0H0P1C OK

 #define ADC(x)     { b=(F&FLAGC); w=A; A+=x+b; F=FADDT[b][w][A]; }     // SZ0H0P1C OK
 #define SBC(x)     { b=(F&FLAGC); w=A; A-=x+b; F=FSUBT[b][w][A]; }     // SZ0H0P1C OK

#else

 #define ADD(x)     { w=A+x; F=((w>>8)&FLAGC)|((A^w^x)&FLAGH); A=w&0xff; F|=FSZPT[A]; }     // SZ0H0P1C
 #define SUB(x)     { w=A-x; F=((w>>8)&FLAGC)|(~(A^w^x)&FLAGH); A=w&0xff; F|=FSZPT[A]; }  // SZ0H0P1C

 #define CMP(x)     { w=A-x; F=((w>>8)&FLAGC)|((~(A^w^x))&FLAGH); F|=FSZPT[w&0xff]; }       // SZ0H0P1C

 #define ADC(x)     { w=A+x+(F&FLAGC); F=((w>>8)&FLAGC)|((A^w^x)&FLAGH); A=w&0xff; F|=FSZPT[A]; }     // SZ0H0P1C
 #define SBC(x)     { w=A-x-(F&FLAGC); F=((w>>8)&FLAGC)|((~(A^w^x))&FLAGH); A=w&0xff; F|=FSZPT[A]; }  // SZ0H0P1C

#endif

#define DAD(x)      { d=(int)HL+x; F&=~FLAGC; F|=(d>>16)&FLAGC; HL=d&0xFFFF; }   // --0-0-1C OK

#define ANA(x)      { b=((A|x)<<1)&FLAGH; A&=x; F=FSZPT[A]|b; } // SZ0H0P00 OK

#define XRA(x)      { A^=x; F=FSZPT[A]; }       // SZ000P00 OK
#define ORA(x)      { A|=x; F=FSZPT[A]; }       // SZ000P00 OK

// --000-0C OK
#define RLCA        { A=(A<<1)|(A>>7); F&=~FLAGC; F|=A&FLAGC; }
#define RRCA        { F&=~FLAGC; F|=A&FLAGC; A=(A>>1)|(A<<7); }
#define RAL         { b=(A>>7); A=(A<<1)|(F&FLAGC); F&=~FLAGC; F|=b; }
#define RAR         { b=(A&1); A=(A>>1)|(F<<7); F&=~FLAGC; F|=b; }

#define CALL        { PUSH(PC+2); PC=RW(PC); }
#define JP          { PC=RW(PC); }
#define RET         { POP(PC); }

#define T(x)        { i8080->t-=x; }

byte TACTT[256] = {
 4, 10,7, 5, 5, 5, 7, 4, 4, 10,7, 5, 5, 5, 7, 4,
 4, 10,7, 5, 5, 5, 7, 4, 4, 10,7, 5, 5, 5, 7, 4,
 4, 10,16,5, 5, 5, 7, 4, 4, 10,16,5, 5, 5, 7, 4,
 4, 10,13,5, 10,10,10,4, 4, 10,13,5, 5, 5, 7, 4,
 5, 5, 5, 5, 5, 5, 7, 5, 5, 5, 5, 5, 5, 5, 7, 5,
 5, 5, 5, 5, 5, 5, 7, 5, 5, 5, 5, 5, 5, 5, 7, 5,
 5, 5, 5, 5, 5, 5, 7, 5, 5, 5, 5, 5, 5, 5, 7, 5,
 7, 7, 7, 7, 7, 7, 7, 7, 5, 5, 5, 5, 5, 5, 7, 5,
 4, 4, 4, 4, 4, 4, 7, 4, 4, 4, 4, 4, 4, 4, 7, 4,
 4, 4, 4, 4, 4, 4, 7, 4, 4, 4, 4, 4, 4, 4, 7, 4,
 4, 4, 4, 4, 4, 4, 7, 4, 4, 4, 4, 4, 4, 4, 7, 4,
 4, 4, 4, 4, 4, 4, 7, 4, 4, 4, 4, 4, 4, 4, 7, 4,
 5, 10,10,10,11,11,7, 11,5, 10,10,10,11,11,7, 11,
 5, 10,10,10,11,11,7, 11,5, 10,10,10,11,11,7, 11,
 5, 10,10,18,11,11,7, 11,5, 5, 10,5, 11,11,7, 11,
 5, 10,10,4, 11,11,7, 11,5, 5, 10,4, 11,11,7, 11
};

void i8080init(i8080state * i8080)
{
    memset(i8080, 0, sizeof(i8080state));

    if (! i8080tablesReady) i8080tables();
    
    i8080->iff = 0;
    i8080->Trap = 0xFFFF;
    
    i8080->DebugName = "i8080";
}

int i8080execute(i8080state * i8080)
{
    byte    b, m, opcode;
    word    w;
    int     d;
    
    while (1) {
        
#ifdef I8080TRAPS
        if (i8080->Debug && (PC == i8080->Trap)) return 2;  // 2 - Trap reached
#endif
        if (i8080->Ticks && (i8080->t <= 0)) return 0;      // 0 - No more tacts left
        
#ifdef I8080DUMP
        if (i8080->DebugInfo>2) i8080dump(i8080);
#endif
        
#ifdef I8080ENGINE
        if (i8080->EngineState != 0) i8080->Engine(i8080->EngineState, i8080->t);
#endif

        opcode = RB(PC); PC++;
        
        T(TACTT[opcode]);
        
        switch (opcode) {
            
            case 0x00:
            case 0x10:
            case 0x20:
            case 0x30:
            case 0x08:
            case 0x18:
            case 0x28:
            case 0x38: break; // NOP
                
            case 0x01: BC = RW(PC); PC+=2; break; // LXI BC,####
            case 0x11: DE = RW(PC); PC+=2; break; // LXI DE,####
            case 0x21: HL = RW(PC); PC+=2; break; // LXI HL,####
            case 0x31: SP = RW(PC); PC+=2; break; // LXI SP,####
                
            case 0x02: WB(BC,A); break; // STAX BC
            case 0x12: WB(DE,A); break; // STAX DE
                
            case 0x22: WW(RW(PC), HL); PC+=2; break; // SHLD ####
            case 0x32: WB(RW(PC), A);  PC+=2; break; // STA ####
                
            case 0x03: BC++; break; // INX BC
            case 0x13: DE++; break; // INX DE
            case 0x23: HL++; break; // INX HL
            case 0x33: SP++; break; // INX SP
                
            case 0x0B: BC--; break; // DCX BC
            case 0x1B: DE--; break; // DCX DE
            case 0x2B: HL--; break; // DCX HL
            case 0x3B: SP--; break; // DCX SP
                
            case 0x0A: A = RB(BC); break; // LDAX BC
            case 0x1A: A = RB(DE); break; // LDAX DE
                
            case 0x2A: HL = RW(RW(PC)); PC+=2; break; // LHLD ####
            case 0x3A: A  = RB(RW(PC)); PC+=2; break; // LDA ####
                
            case 0x06: B = RB(PC); PC++; break; // MVI B,##
            case 0x16: D = RB(PC); PC++; break; // MVI D,##
            case 0x26: H = RB(PC); PC++; break; // MVI H,##
                
            case 0x36: WB(HL, RB(PC)); PC++; break; // MVI M,##
                
            case 0x0E: C = RB(PC); PC++; break; // MVI C,##
            case 0x1E: E = RB(PC); PC++; break; // MVI E,##
            case 0x2E: L = RB(PC); PC++; break; // MVI L,##
            case 0x3E: A = RB(PC); PC++; break; // MVI A,##
                
            case 0x04: INR(B); break; // INR B
            case 0x14: INR(D); break; // INR D
            case 0x24: INR(H); break; // INR H
                
            case 0x34: m = RB(HL); INR(m); WB(HL, m); break; // INR (HL)
                
            case 0x0C: INR(C); break; // INR C
            case 0x1C: INR(E); break; // INR E
            case 0x2C: INR(L); break; // INR L
            case 0x3C: INR(A); break; // INR A
                
            case 0x05: DCR(B); break; // DCR B
            case 0x15: DCR(D); break; // DCR D
            case 0x25: DCR(H); break; // DCR H
                
            case 0x35: m = RB(HL); DCR(m); WB(HL, m); break; // DCR (HL)
                
            case 0x0D: DCR(C); break; // DCR C
            case 0x1D: DCR(E); break; // DCR E
            case 0x2D: DCR(L); break; // DCR L
            case 0x3D: DCR(A); break; // DCR A
                
            case 0x09: DAD(BC); break; // DAD BC
            case 0x19: DAD(DE); break; // DAD DE
            case 0x29: DAD(HL); break; // DAD HL
            case 0x39: DAD(SP); break; // DAD SP
                
            case 0x07: RLCA; break; // RLC
            case 0x17: RAL;  break; // RAL
                
            case 0x27: // FCKN DAA ;-)
                
                b = A;
                
                if ((F & FLAGH) | ((A & 0xf) > 9))  b+=0x06;
                if ((F & FLAGC) | (A > 0x99))       b+=0x60;
                
                F = (F&FLAGC) | ((A>0x99)&FLAGC) | ((A^b)&FLAGH) | FSZPT[b];
                
                A = b;

                break;
                
            case 0x37: F |= FLAGC; break; // STC
                
            case 0x0F: RRCA; break; // RRC
            case 0x1F: RAR;  break; // RAR
            case 0x2F: A = ~ A;         break; // CMA
            case 0x3F: F = F ^ FLAGC;   break; // CMC
                
            case 0x40: B = B; break; // MOV B,B
            case 0x41: B = C; break; // MOV B,C
            case 0x42: B = D; break; // MOV B,D
            case 0x43: B = E; break; // MOV B,E
            case 0x44: B = H; break; // MOV B,H
            case 0x45: B = L; break; // MOV B,L
            case 0x46: B = RB(HL); break; // MOV B,(HL)
            case 0x47: B = A; break; // MOV B,A
                
            case 0x50: D = B; break; // MOV D,B
            case 0x51: D = C; break; // MOV D,C
            case 0x52: D = D; break; // MOV D,D
            case 0x53: D = E; break; // MOV D,E
            case 0x54: D = H; break; // MOV D,H
            case 0x55: D = L; break; // MOV D,L
            case 0x56: D = RB(HL); break; // MOV D,(HL)
            case 0x57: D = A; break; // MOV H,A
                
            case 0x60: H = B; break; // MOV H,B
            case 0x61: H = C; break; // MOV H,C
            case 0x62: H = D; break; // MOV H,D
            case 0x63: H = E; break; // MOV H,E
            case 0x64: H = H; break; // MOV H,H
            case 0x65: H = L; break; // MOV H,L
            case 0x66: H = RB(HL); break; // MOV H,(HL)
            case 0x67: H = A; break; // MOV C,A
                
            case 0x70: WB(HL,B); break; // MOV (HL),B
            case 0x71: WB(HL,C); break; // MOV (HL),C
            case 0x72: WB(HL,D); break; // MOV (HL),D
            case 0x73: WB(HL,E); break; // MOV (HL),E
            case 0x74: WB(HL,H); break; // MOV (HL),H
            case 0x75: WB(HL,L); break; // MOV (HL),L
                
            case 0x76:                  // HLT == MOV (HL),(HL)
                --PC;
                if (i8080->Halt) return 1;  // 1 - if HLT reached
                break;
                
            case 0x77: WB(HL,A); break; // MOV (HL),A
                
            case 0x48: C = B; break; // MOV C,B
            case 0x49: C = C; break; // MOV C,C
            case 0x4A: C = D; break; // MOV C,D
            case 0x4B: C = E; break; // MOV C,E
            case 0x4C: C = H; break; // MOV C,H
            case 0x4D: C = L; break; // MOV C,L
            case 0x4E: C = RB(HL); break; // MOV C,(HL)
            case 0x4F: C = A; break; // MOV C,A
                
            case 0x58: E = B; break; // MOV E,B
            case 0x59: E = C; break; // MOV E,C
            case 0x5A: E = D; break; // MOV E,D
            case 0x5B: E = E; break; // MOV E,E
            case 0x5C: E = H; break; // MOV E,H
            case 0x5D: E = L; break; // MOV E,L
            case 0x5E: E = RB(HL); break; // MOV E,(HL)
            case 0x5F: E = A; break; // MOV E,A
                
            case 0x68: L = B; break; // MOV L,B
            case 0x69: L = C; break; // MOV L,C
            case 0x6A: L = D; break; // MOV L,D
            case 0x6B: L = E; break; // MOV L,E
            case 0x6C: L = H; break; // MOV L,H
            case 0x6D: L = L; break; // MOV L,L
            case 0x6E: L = RB(HL); break; // MOV L,(HL)
            case 0x6F: L = A; break; // MOV L,A
                
            case 0x78: A = B; break; // MOV A,B
            case 0x79: A = C; break; // MOV A,C
            case 0x7A: A = D; break; // MOV A,D
            case 0x7B: A = E; break; // MOV A,E
            case 0x7C: A = H; break; // MOV A,H
            case 0x7D: A = L; break; // MOV A,L
            case 0x7E: A = RB(HL); break; // MOV A,(HL)
            case 0x7F: A = A; break; // MOV A,A
                
            case 0x80: ADD(B); break; // ADD B
            case 0x81: ADD(C); break; // ADD C
            case 0x82: ADD(D); break; // ADD D
            case 0x83: ADD(E); break; // ADD E
            case 0x84: ADD(H); break; // ADD H
            case 0x85: ADD(L); break; // ADD L
            case 0x86: m=RB(HL); ADD(m); break; // ADD (HL)
            case 0x87: ADD(A); break; // ADD A
                
            case 0x88: ADC(B); break; // ADC B
            case 0x89: ADC(C); break; // ADC C
            case 0x8A: ADC(D); break; // ADC D
            case 0x8B: ADC(E); break; // ADC E
            case 0x8C: ADC(H); break; // ADC H
            case 0x8D: ADC(L); break; // ADC L
            case 0x8E: m=RB(HL); ADC(m); break; // ADC (HL)
            case 0x8F: ADC(A); break; // ADC A
                
            case 0x90: SUB(B); break; // SUB B
            case 0x91: SUB(C); break; // SUB C
            case 0x92: SUB(D); break; // SUB D
            case 0x93: SUB(E); break; // SUB E
            case 0x94: SUB(H); break; // SUB H
            case 0x95: SUB(L); break; // SUB L
            case 0x96: m=RB(HL); SUB(m); break; // SUB (HL)
            case 0x97: SUB(A); break; // SUB A
                
            case 0x98: SBC(B); break; // SBB B
            case 0x99: SBC(C); break; // SBB C
            case 0x9A: SBC(D); break; // SBB D
            case 0x9B: SBC(E); break; // SBB E
            case 0x9C: SBC(H); break; // SBB H
            case 0x9D: SBC(L); break; // SBB L
            case 0x9E: m=RB(HL); SBC(m); break; // SBB (HL)
            case 0x9F: SBC(A); break; // SBB A
                
            case 0xA0: ANA(B); break; // ANA B
            case 0xA1: ANA(C); break; // ANA C
            case 0xA2: ANA(D); break; // ANA D
            case 0xA3: ANA(E); break; // ANA E
            case 0xA4: ANA(H); break; // ANA H
            case 0xA5: ANA(L); break; // ANA L
            case 0xA6: m=RB(HL); ANA(m); break; // ANA (HL)
            case 0xA7: ANA(A); break; // ANA A
                
            case 0xA8: XRA(B); break; // XRA B
            case 0xA9: XRA(C); break; // XRA C
            case 0xAA: XRA(D); break; // XRA D
            case 0xAB: XRA(E); break; // XRA E
            case 0xAC: XRA(H); break; // XRA H
            case 0xAD: XRA(L); break; // XRA L
            case 0xAE: m=RB(HL); XRA(m); break; // XRA (HL)
            case 0xAF: XRA(A); break; // XRA A
                
            case 0xB0: ORA(B); break; // ORA B
            case 0xB1: ORA(C); break; // ORA C
            case 0xB2: ORA(D); break; // ORA D
            case 0xB3: ORA(E); break; // ORA E
            case 0xB4: ORA(H); break; // ORA H
            case 0xB5: ORA(L); break; // ORA L
            case 0xB6: m=RB(HL); ORA(m); break; // ORA (HL)
            case 0xB7: ORA(A); break; // ORA A
                
            case 0xB8: CMP(B); break; // CMP B
            case 0xB9: CMP(C); break; // CMP C
            case 0xBA: CMP(D); break; // CMP D
            case 0xBB: CMP(E); break; // CMP E
            case 0xBC: CMP(H); break; // CMP H
            case 0xBD: CMP(L); break; // CMP L
            case 0xBE: m=RB(HL); CMP(m); break; // CMP (HL)
            case 0xBF: CMP(A); break; // CMP A
                
            case 0xC6: ADD(RB(PC)); PC++; break; // ADI ##
            case 0xCE: ADC(RB(PC)); PC++; break; // ACI ##
            case 0xD6: SUB(RB(PC)); PC++; break; // SUI ##
            case 0xDE: SBC(RB(PC)); PC++; break; // SBI ##
            case 0xE6: ANA(RB(PC)); PC++; break; // ANI ##
            case 0xEE: XRA(RB(PC)); PC++; break; // XRI ##
            case 0xF6: ORA(RB(PC)); PC++; break; // ORI ##
            case 0xFE: CMP(RB(PC)); PC++; break; // CPI ##
                
            case 0xC0: if (!(F&FLAGZ)) { T(6); RET; } break; // RNZ
            case 0xD0: if (!(F&FLAGC)) { T(6); RET; } break; // RNC
            case 0xE0: if (!(F&FLAGP)) { T(6); RET; } break; // RPO
            case 0xF0: if (!(F&FLAGS)) { T(6); RET; } break; // RP
                
            case 0xC8: if ((F&FLAGZ)) { T(6); RET; } break; // RZ
            case 0xD8: if ((F&FLAGC)) { T(6); RET; } break; // RC
            case 0xE8: if ((F&FLAGP)) { T(6); RET; } break; // RPE
            case 0xF8: if ((F&FLAGS)) { T(6); RET; } break; // RM
                
            case 0xC1: POP(BC); break; // POP BC
            case 0xD1: POP(DE); break; // POP DE
            case 0xE1: POP(HL); break; // POP HL
                
            case 0xF1: POP(AF); F&=~(FLAG5|FLAG3); F|=FLAG1; break; // POP AF
                
            case 0xC2: if (!(F&FLAGZ)) { JP; } else PC+=2; break; // JNZ ####
            case 0xD2: if (!(F&FLAGC)) { JP; } else PC+=2; break; // JNC ####
            case 0xE2: if (!(F&FLAGP)) { JP; } else PC+=2; break; // JPO ####
            case 0xF2: if (!(F&FLAGS)) { JP; } else PC+=2; break; // JP  ####
                
            case 0xCA: if (F&FLAGZ) { JP; } else PC+=2; break; // JZ  ####
            case 0xDA: if (F&FLAGC) { JP; } else PC+=2; break; // JC  ####
            case 0xEA: if (F&FLAGP) { JP; } else PC+=2; break; // JPE ####
            case 0xFA: if (F&FLAGS) { JP; } else PC+=2; break; // JM  ####
                
            case 0xC5: PUSH(BC); break; // PUSH BC
            case 0xD5: PUSH(DE); break; // PUSH DE
            case 0xE5: PUSH(HL); break; // PUSH HL
            case 0xF5: PUSH(AF); break; // PUSH AF
                
            case 0xC4: if (!(F&FLAGZ)) { T(6); CALL; } else PC+=2; break; // CNZ ####
            case 0xD4: if (!(F&FLAGC)) { T(6); CALL; } else PC+=2; break; // CNC ####
            case 0xE4: if (!(F&FLAGP)) { T(6); CALL; } else PC+=2; break; // CPO ####
            case 0xF4: if (!(F&FLAGS)) { T(6); CALL; } else PC+=2; break; // CP  ####
                
            case 0xCC: if ((F&FLAGZ))  { T(6); CALL; } else PC+=2; break; // CZ  ####
            case 0xDC: if ((F&FLAGC))  { T(6); CALL; } else PC+=2; break; // CC  ####
            case 0xEC: if ((F&FLAGP))  { T(6); CALL; } else PC+=2; break; // CPE ####
            case 0xFC: if ((F&FLAGS))  { T(6); CALL; } else PC+=2; break; // CM  ####
                
            case 0xCD:
            case 0xDD:
            case 0xED:
            case 0xFD: CALL; break; // CALL ####
                
            case 0xC3:
            case 0xCB: JP; break; // JP ####
                
            case 0xD3: WP(RB(PC), A);   PC++; break; // OUT ##
            case 0xDB: A = RP(RB(PC));  PC++; break; // IN  ##
                
            case 0xF3: IFF = 0; break; // DI
            case 0xFB: IFF = 1; break; // EI
                
            case 0xE3: w = HL; HL = RW(SP); WW(SP, w); break; // XTHL
            case 0xEB: w = HL; HL = DE; DE = w; break; // XCHG
                
            case 0xC9: 
            case 0xD9: RET; break; // RET
                
            case 0xE9: PC = HL; break; // PCHL
            case 0xF9: SP = HL; break; // SPHL
                
            case 0xC7:
            case 0xCF:
            case 0xD7:
            case 0xDF:
            case 0xE7:
            case 0xEF:
            case 0xF7:
            case 0xFF: PUSH(PC); PC = opcode & 0x38; break; // RST XX
        }
    }
}

int i8080interrupt(i8080state * i8080, byte data)
{
    
#ifdef I8080DEBUG
    if (i8080->DebugInfo) printf("INT REQUEST IFF=%d\n", IFF);
#endif
    
    if (!IFF) return -1;
    
    IFF = 0;
    
    PUSH(PC);
    
    PC = data & 0x38;   // TODO: Yet we assume RST XX on bus
    
    T(13);
    
#ifdef I8080DEBUG
    if (i8080->DebugInfo) printf("INT DATABUS=%02X ADDR=%04X\n", data, PC);
#endif
    
    return 0;
}