//
//  i8080debug.c
//
//  Disassembler / Debuger for i8080
//
//  Created by Alexander Medvedev on 19/05/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "types.h"
#include "i8080.h"
#include "i8080debug.h"

extern word memory_read_word(word addr);

#define RB(x)       memory_read(x)
#define RW(x)       memory_read_word(x)

// Execution history

i8080state i8080history[I8080HISTORYSIZE];

int i8080historyP;
int i8080historyN;

void i8080remember(i8080state * i8080)
{
    i8080history[i8080historyP++]=*i8080;
    i8080historyP %= I8080HISTORYSIZE;
    i8080historyN++;
    if (i8080historyN > I8080HISTORYSIZE) i8080historyN = I8080HISTORYSIZE;
}

//
// Disassembler
//

// TODO: Z80 Mnemonics yet

// %b immediate byte
// %w immediate word

char * CMD[256] =
{
    "NOP","LD BC,%w","LD (BC),A","INC BC","INC B","DEC B","LD B,%b","RLCA",
    "NOP*","ADD HL,BC","LD A,(BC)","DEC BC","INC C","DEC C","LD C,%b","RRCA",
    "NOP*","LD DE,%w","LD (DE),A","INC DE","INC D","DEC D","LD D,%b","RLA",
    "NOP*","ADD HL,DE","LD A,(DE)","DEC DE","INC E","DEC E","LD E,%b","RRA",
    "NOP*","LD HL,%w","LD (%w),HL","INC HL","INC H","DEC H","LD H,%b","DAA",
    "NOP*","ADD HL,HL","LD HL,(%w)","DEC HL","INC L","DEC L","LD L,%b","CPL",
    "NOP*","LD SP,%w","LD (%w),A","INC SP","INC (HL)","DEC (HL)","LD (HL),%b","SCF",
    "NOP*","ADD HL,SP","LD A,(%w)","DEC SP","INC A","DEC A","LD A,%b","CCF",
    "LD B,B","LD B,C","LD B,D","LD B,E","LD B,H","LD B,L","LD B,(HL)","LD B,A",
    "LD C,B","LD C,C","LD C,D","LD C,E","LD C,H","LD C,L","LD C,(HL)","LD C,A",
    "LD D,B","LD D,C","LD D,D","LD D,E","LD D,H","LD D,L","LD D,(HL)","LD D,A",
    "LD E,B","LD E,C","LD E,D","LD E,E","LD E,H","LD E,L","LD E,(HL)","LD E,A",
    "LD H,B","LD H,C","LD H,D","LD H,E","LD H,H","LD H,L","LD H,(HL)","LD H,A",
    "LD L,B","LD L,C","LD L,D","LD L,E","LD L,H","LD L,L","LD L,(HL)","LD L,A",
    "LD (HL),B","LD (HL),C","LD (HL),D","LD (HL),E","LD (HL),H","LD (HL),L","HALT","LD (HL),A",
    "LD A,B","LD A,C","LD A,D","LD A,E","LD A,H","LD A,L","LD A,(HL)","LD A,A",
    "ADD B","ADD C","ADD D","ADD E","ADD H","ADD L","ADD (HL)","ADD A",
    "ADC B","ADC C","ADC D","ADC E","ADC H","ADC L","ADC (HL)","ADC A",
    "SUB B","SUB C","SUB D","SUB E","SUB H","SUB L","SUB (HL)","SUB A",
    "SBC B","SBC C","SBC D","SBC E","SBC H","SBC L","SBC (HL)","SBC A",
    "AND B","AND C","AND D","AND E","AND H","AND L","AND (HL)","AND A",
    "XOR B","XOR C","XOR D","XOR E","XOR H","XOR L","XOR (HL)","XOR A",
    "OR B","OR C","OR D","OR E","OR H","OR L","OR (HL)","OR A",
    "CP B","CP C","CP D","CP E","CP H","CP L","CP (HL)","CP A",
    "RET NZ","POP BC","JP NZ,%w","JP %w","CALL NZ,%w","PUSH BC","ADD %b","RST 00H",
    "RET Z","RET","JP Z,%w","JP*","CALL Z,%w","CALL %w","ADC %b","RST 08H",
    "RET NC","POP DE","JP NC,%w","OUTA (%b)","CALL NC,%w","PUSH DE","SUB %b","RST 10H",
    "RET C","RET*","JP C,%w","INA (%b)","CALL C,%w","CALL*","SBC %b","RST 18H",
    "RET PO","POP HL","JP PO,%w","EX HL,(SP)","CALL PO,%w","PUSH HL","AND %b","RST 20H",
    "RET PE","LD PC,HL","JP PE,%w","EX DE,HL","CALL PE,%w","CALL*","XOR %b","RST 28H",
    "RET P","POP AF","JP P,%w","DI","CALL P,%w","PUSH AF","OR %b","RST 30H",
    "RET M","LD SP,HL","JP M,%w","EI","CALL M,%w","CALL*","CP %b","RST 38H"
};

int i8080disasm(char * string, word addr)
{
    char *p;
    
    int bytes = 0;
    
    byte opcode=RB(addr); addr++; bytes++;
        
    p = CMD[opcode];
                
    while (*p) {
        
        if (*p=='%')
            
            switch (*++p) {
                    
                case 'b':
                    sprintf(string, "%02XH", RB(addr));
                    string+=3;
                    bytes++; addr++;
                    break;
                    
                case 'w':
                    sprintf(string, "%04XH", RW(addr));
                    string+=5;
                    bytes+=2; addr+=2;
                    break;
                    
                default:
                    break;
            }
        
        else
            *string++=*p;
        
        ++p;
    }
    
    *string=0;
    
    return bytes;
}

void i8080dump(i8080state * i8080)
{
    char mnemonic[32];
    
    int bytes = i8080disasm(mnemonic, i8080->pc);
    
    printf("%s A=%02X F=%02X BC=%04X DE=%04X HL=%04X SP=%04X %s | %c%c0%c0%c1%c | %04X: ",
           i8080->DebugName,
           i8080->af.b.h,
           i8080->af.b.l,
           i8080->bc.w,
           i8080->de.w,
           i8080->hl.w,
           i8080->sp,
           i8080->iff ? "EI" : "DI",
           (i8080->af.b.l&I8080FLAGS)  ? 'S' : '-',
           (i8080->af.b.l&I8080FLAGZ)  ? 'Z' : '-',
           (i8080->af.b.l&I8080FLAGH)  ? 'H' : '-',
           (i8080->af.b.l&I8080FLAGP)  ? 'P' : '-',
           (i8080->af.b.l&I8080FLAGC)  ? 'C' : '-',
           i8080->pc
           );
    
    for (int i=0; i<bytes; i++) printf("%02X", RB(i8080->pc+i));
    for (int i=0; i<(5-bytes); i++) printf("  ");
    
    printf("%s\n", mnemonic);
}

void memory_dump(word addr, word addr2)
{
    int i, j;
    byte c;
    
    addr >>= 4;
    addr2 >>= 4;
    
    for(i = addr; i <= addr2; i++) {
        printf("\n%04X ", i << 4 );
        for(j=0;j<16;j++) printf("%02X ", memory_read((i<<4)+j));
        for(j=0;j<16;j++) {
            c = memory_read((i<<4)+j);
            if (c<32)  c = '.';
            if (c>126) c = '.';
            printf("%c", c);
        }
    }
    printf("\n");
}

void i8080debug(i8080state * i8080)
{
    printf("Debugger\nEnter 'H' for help\n");
    
    while(1) {

        char command = 0;
        int  p1 = i8080->pc, p2 = 0, p3 = 0;
        long long counter, tacts;
        char str[256];
        int n;
        
        i8080dump(i8080);
        i8080remember(i8080);
        
        if (i8080->pc == 5) {
    
            printf("COUT:\n");
            
            switch (i8080->bc.b.l) {
                case 2: putchar(i8080->de.b.l); break;
                case 9: while('$' != (n = memory_read(i8080->de.w++))) putchar(n);
            }

            printf("\nCOUT END\n");
        }
        
        printf("%s> ", i8080->DebugName);
        
        gets(str);
        
        n = sscanf(str, "%c %4x %4x %4x\n", &command, &p1, &p2, &p3);
        
        if (n <= 0) {
            
            i8080->Halt = 0;
            i8080->Debug = 0;
            i8080->Trap = 0;
            i8080->Ticks = 1;
            i8080->t = 1;
            i8080execute(i8080);
            
        } else
            switch (toupper(command)) {
                    
                case 'T':
                    
                    if (n == 2) {
            
                        i8080->Halt = 0;
                        i8080->Debug = 1;
                        i8080->Trap = p1;
                        i8080->Ticks = 0;
                        i8080->t = 0;
                        i8080execute(i8080);
                    
                    } else
                        if (n == 1) {
                    
                            i8080->Halt = 0;
                            i8080->Debug = 0;
                            i8080->Trap = 0;
                            i8080->Ticks = 1;
                            i8080->t = 1;
                            i8080execute(i8080);
                        }
                    
                    break;
                    
                case 'S':
                    
                    if (n==2) i8080->pc = p1;
                    
                    i8080->Halt = 1;
                    i8080->Debug = 0;
                    i8080->Trap = 0;
                    i8080->Ticks = 0;
                    i8080->t = 0;
                    i8080execute(i8080);
                    
                    break;
                    
                    
                case 'X':
                    
                    printf("Auto trace...\n");
                    p3 = -1;
                    
                case 'G':
                    
                    if (n==2) i8080->pc = p1;
                    
                    counter = 0;
                    tacts = 0;
                    
                    do {
                        
                        i8080->Halt = 1;
                        i8080->Debug = 0;
                        i8080->Trap = 0;
                        i8080->Ticks = 1;
                        i8080->t = 1;
                        
                        if (p3 != -1) {
                            printf("T "); i8080dump(i8080);
                        }
                        
                        if (1==i8080execute(i8080)) break;
                        if ((n==3)&&(i8080->pc==p2)) break;
                        
                        i8080remember(i8080);
                        
                        ++ counter;
                        tacts += 1 - i8080->t;
                        
                        // CP/M BDOS Char (2) and String (9) output
                        
                        if (i8080->pc == 5) {
                            
                            if (p3 != -1) printf("COUT:\n");
                            
                            switch (i8080->bc.b.l) {
                                case 2: putchar(i8080->de.b.l); break;
                                case 9: while('$' != (n = memory_read(i8080->de.w++))) putchar(n);
                            }
                            
                            if (p3 != -1) printf("\nCOUT END\n");
                        }
                    } while (1);
                    
                    printf("\nElapsed %lld commands, %lld tacts\n", counter, tacts);
                    
                    break;
                    
                case 'H':
                case '?':
                    
                    printf("S - Start running non stop\n"
                           "[ENTER] or T - Trace one instruction\n"
                           "T <ADDR> - Trace up to PC == <ADDR>\n"
                           "G [<ADDR> [<END>]] - Trace automatically until HLT or PC == <END>, from <ADDR>\n"
                           "X [<ADDR> [<END>]] - Execute automatically until HLT or PC == <END>, from <ADDR>\n"
                           "D <ADDR1> [<ADDR2>] - Memory dump\n"
                           "L [<STEPS>] - List trace history\n"
                           "P [<ADDR> [<STACK> [<IFF>]]] - set PC at <ADDR> and SP at <STACK> and IFF <IFF>\n"
                           "A [<VALUE> [<FLAGS>]] - set A to <VALUE> and F to <FLAGS>\n"
                           "R [<BC> [<DE> [<HL>]]] - set register values\n"
                           "Q - Quit\n"
                           "H or ? - This help.\n");
                    
                    break;
                    
                case 'R':
                    
                    if (n >= 2) i8080->bc.w = p1;
                    if (n >= 3) i8080->de.w = p2;
                    if (n >= 4) i8080->hl.w = p3;
                    
                    break;
                    
                case 'P':
                    
                    if (n >= 2) i8080->pc = p1;
                    if (n >= 3) i8080->sp = p2;
                    if (n >= 4) i8080->iff = p3 & 1;
                    
                    break;
                    
                case 'A':
                    
                    if (n >= 2) i8080->af.b.h = p1 & 255;
                    if (n >= 3) i8080->af.b.l = p2 & 255;
                    
                    break;
                    
                case 'L':
                    
                    p3 = I8080HISTORYSIZE;
                    
                    if (n == 2) p3 = p1;
                    
                    if (i8080historyN < p3) p3 = i8080historyN;
                    
                    p1 = (i8080historyP + (I8080HISTORYSIZE - p3));
                    
                    for (int i=0;i<p3;i++,p1++) i8080dump(i8080history+(p1%I8080HISTORYSIZE));
                    
                    break;
                    
                case 'D':
                    
                    if (n==2) p2=p1;
                    
                    memory_dump(p1, p2);
                    
                    break;
                    
                case 'Q':
                    
                    return;
                    
                default:
                    
                    printf("Enter 'H' or '?' for help.\n");
                    
                    break;
            }
        
    } while (1);
    
}

