//
//  i8255.c
//
//  Intel i8255 PPI Emulation
//
//  Created by Alexander Medvedev on 02/06/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>
#include <memory.h>

#include "types.h"
#include "i8255.h"

//
// 8255 PPI Registers
//
//  PPIA  0
//  PPIB  1
//  PPIC  2
//  PPICW 3
//
//  A: 8 bit,
//  B: 8 bit,
//  C1+C2: 4+4 bit
//
//  CW: [ ON | BAC2 | MAC2 | AIO | C2IO | MBC1 | BIO | C1IO ]
//        ON: 1 for setup, 0 for bit set/reset
//        bit set/reset in C: 1XXXNNNB - NNN - bit number, B - 1 for set, 0 for reset
//        BAC2 - biderectional mode for A/C2
//        MAC2: mode for A/C2 - 0 for basic, 1 for strobed
//        MBC1: same for B/C1
//        AIO, C2IO, BIO, C1IO: 0 for output, 1 for input
//


void i8255init(i8255state * i8255)
{
    memset(i8255, 0, sizeof(i8255state));

    i8255->DebugName = "i8255";
}

void i8255write(i8255state * i8255, int A, byte data)
{
    switch (A) {
        
        case 0:
            
            if ((! i8255->Ain) || i8255->AC2bidir) i8255->A = data;

#ifdef I8255DEBUG
            if (i8255->DebugInfo > 1) printf("%s WRITE A=%02X\n", i8255->DebugName, i8255->A);
#endif
            break;

        case 1:
            
            if (! i8255->Bin) i8255->B = data;

#ifdef I8255DEBUG
            if (i8255->DebugInfo > 1) printf("i8255%s WRITE B=%02X\n", i8255->DebugName, i8255->B);
#endif
            break;
            
        case 2:

            if (! i8255->C1in) i8255->C1 = data & 15;
            if ((! i8255->C2in) || i8255->AC2bidir) i8255->C2 = (data >> 4) & 15;

#ifdef I8255DEBUG
            if (i8255->DebugInfo > 1) printf("i8255%s WRITE C1=%1X C2=%1X\n", i8255->DebugName, i8255->C1, i8255->C2);
#endif
            break;
            
        case 3:

            if (data & 128) {
                
                i8255->C1in         = data & 1;
                i8255->Bin          = data & 2;
                i8255->BC1strobe    = data & 4;
                i8255->C2in         = data & 8;
                i8255->Ain          = data & 16;
                i8255->AC2strobe    = data & 32;
                i8255->AC2bidir     = data & 64;
                
#ifdef I8255DEBUG
                if (i8255->DebugInfo) printf("%s WRITE CW "
                                             "AIN=%s C2IN=%s AC2STROBE=%s AC2BIDIR=%s "
                                             "BIN=%s C1IN=%s BC1STROBE=%s\n",
                                             i8255->DebugName,
                                             i8255->Ain  ? "ON" : "OFF",
                                             i8255->C2in ? "ON" : "OFF",
                                             i8255->AC2strobe ? "ON" : "OFF",
                                             i8255->AC2bidir  ? "ON" : "OFF",
                                             i8255->Bin  ? "ON" : "OFF",
                                             i8255->C1in ? "ON" : "OFF",
                                             i8255->BC1strobe ? "ON" : "OFF" );
#endif
            } else {
                
                int bit = (data >> 1) & 7;
                
                if (bit<4) {
                    
                    i8255->C1 &= ~ (1<<bit);
                    i8255->C1 |= data & 1;
                    
#ifdef I8255DEBUG
                    if (i8255->DebugInfo > 1) printf("%s BITSET C1[%d]=%s\n", i8255->DebugName, bit, (data & 1) ? "ON" : "OFF");
#endif
                } else {
                
                    bit -= 4;
                    
                    i8255->C2 &= ~ (1<<bit);
                    i8255->C2 |= data & 1;

#ifdef I8255DEBUG
                    if (i8255->DebugInfo > 1) printf("%s BITSET C2[%d]=%s\n", i8255->DebugName, bit, (data & 1) ? "ON" : "OFF");
#endif
                }
            }
            break;
            
        default:
            break;
    }
}

byte i8255read(i8255state * i8255, int A)
{
    switch (A) {
        case 0:
            
#ifdef I8255DEBUG
            if (i8255->DebugInfo > 1) printf("%s READ A=%02X\n", i8255->DebugName, i8255->A);
#endif
            return i8255->A;

        case 1:
            
#ifdef I8255DEBUG
            if (i8255->DebugInfo > 1) printf("%s READ B=%02X\n", i8255->DebugName, i8255->B);
#endif
            return i8255->B;
            
        case 2:
            
#ifdef I8255DEBUG
            if (i8255->DebugInfo > 1) printf("%s READ C1=%1X C2=%1X\n", i8255->DebugName, i8255->C1, i8255->C2);
#endif
            return i8255->C1 + (i8255->C2 << 4);
            
        default:
            
            return 255;
    }
}

void i8255dump(i8255state * i8255)
{
    printf("%s: "
           "A %02X AIN=%s C2 %1X C2IN=%s AC2STROBE=%s AC2BIDIR=%s\n"
           "B %02X BIN=%s C1 %1X C1IN=%s BC1STROBE=%s\n",
           i8255->DebugName,
           i8255->A,  i8255->Ain  ? "ON" : "OFF",
           i8255->C2, i8255->C2in ? "ON" : "OFF",
           i8255->AC2strobe ? "ON" : "OFF",
           i8255->AC2bidir  ? "ON" : "OFF",
           i8255->B,  i8255->Bin  ? "ON" : "OFF",
           i8255->C1, i8255->C1in ? "ON" : "OFF",
           i8255->BC1strobe ? "ON" : "OFF" );
}
