//
//  i8257.c
//  rk86
//
//  Created by Alexander Medvedev on 20/05/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>
#include <memory.h>

#include "types.h"
#include "i8257.h"

extern byte memory_read(word addr);
extern void memory_write(word addr, byte data);

//
// i8257 DMA controller
//
// ADDR FLIP=0  FLIP=1
// 0    CH0AL   CH0AH   AH=[A15..A8]     AL=[A7..A0]
// 1    CH0TL   CH0TH   TH=[R|W|T13..T8] TL=[T7..T0]
// 2    CH1AL   CH1AH
// 3    CH1TL   CH1TH   R W
// 4    CH2AL   CH2AH   0 0 Verify
// 5    CH2TL   CH2TH   1 0 Read
// 6    CH3AL   CH3AH   0 1 Write
// 7    CH3TL   CH3TH   1 1 Illegal
//
// 8W   MODE
// 8R   STATUS
//
// MODE is [ AL | TCS | EW | RP | E3 | E2 | E1 | E0 ]
//
// AL: 1 autoreload CH2 -> CH1 / 0 none  TCS: 1 stop on terminal count 0 none
// EW: 1 extended write time 0 none RP: 1 rotating priority 0 fixed
// E3..E0: 1 channel on 0 off
//
// STATUS is [ 0 0 0 | UP | TC3 | TC2 | TC1 | TC0 ]
//
// UP: 1 CH1 was reloaded from CH2 0 not
// TC3..TC0: 1 Terminal count reached 0 not


void i8257init(i8257state * i8257)
{
    int d = i8257->DebugInfo;
    
    memset(i8257, 0, sizeof(i8257state));

    i8257->DebugInfo = d;
    
    if (i8257->DebugInfo) printf("i8257 HARDWARE RESET\n");
}

void i8257write(i8257state * i8257, int A, byte data)
{
    switch (A) {
        case 0:
            if (i8257->Flip)
                i8257->Address[0] = (data << 8) + (i8257->Address[0] & 255);
            else
                i8257->Address[0] = (i8257->Address[0] & 0xFF00) + data;
            
            i8257->Flip = ! i8257->Flip;
            
            if (i8257->DebugInfo > 1) printf("i8257 CH0 ADDR=%04X\n", i8257->Address[0]);
            break;
            
        case 1:
            if (i8257->Flip)
                i8257->Terminate[0] = ((data&63) << 8) + (i8257->Terminate[0] & 255),
                i8257->Write[0] = data & 64, i8257->Read[0] = data & 128;
            else
                i8257->Terminate[0] = (i8257->Terminate[0] & 0xFF00) + data;
            
            i8257->Flip = ! i8257->Flip;
            
            if (i8257->DebugInfo > 1) printf("i8257 CH0 TERMINAL=%d MODE=%c%c\n",
                                             i8257->Terminate[0],
                                             i8257->Read[0]?'R':'-',
                                             i8257->Write[0]?'W':'-');
            break;
            
        case 2:
            if (i8257->Flip)
                i8257->Address[1] = (data << 8) + (i8257->Address[1] & 255);
            else
                i8257->Address[1] = (i8257->Address[1] & 0xFF00) + data;
            
            i8257->Flip = ! i8257->Flip;

            if (i8257->DebugInfo > 1) printf("i8257 CH1 ADDR=%04X\n", i8257->Address[1]);
            break;
            
        case 3:
            if (i8257->Flip)
                i8257->Terminate[1] = ((data&63) << 8) + (i8257->Terminate[1] & 255),
                i8257->Write[1] = data & 64, i8257->Read[1] = data & 128;
            else
                i8257->Terminate[1] = (i8257->Terminate[1] & 0xFF00) + data;
            
            i8257->Flip = ! i8257->Flip;

            if (i8257->DebugInfo > 1) printf("i8257 CH1 TERMINAL=%d MODE=%c%c\n",
                                             i8257->Terminate[1],
                                             i8257->Read[1]?'R':'-',
                                             i8257->Write[1]?'W':'-');
            break;
            
        case 4:
            if (i8257->Flip)
                i8257->Address[2] = (data << 8) + (i8257->Address[2] & 255);
            else
                i8257->Address[2] = (i8257->Address[2] & 0xFF00) + data;
            
            i8257->Flip = ! i8257->Flip;

            if (i8257->DebugInfo > 1) printf("i8257 CH2 ADDR=%04X\n", i8257->Address[2]);
            break;
            
        case 5:
            if (i8257->Flip)
                i8257->Terminate[2] = ((data&63) << 8) + (i8257->Terminate[2] & 255),
                i8257->Write[2] = data & 64, i8257->Read[2] = data & 128;
            else
                i8257->Terminate[2] = (i8257->Terminate[2] & 0xFF00) + data;
            i8257->Flip = ! i8257->Flip;
            if (i8257->DebugInfo > 1) printf("i8257 CH2 TERMINAL=%d MODE=%c%c\n",
                                             i8257->Terminate[2],
                                             i8257->Read[2]?'R':'-',
                                             i8257->Write[2]?'W':'-');
            break;
            
        case 6:
            if (i8257->Flip)
                i8257->Address[3] = (data << 8) + (i8257->Address[3] & 255);
            else
                i8257->Address[3] = (i8257->Address[3] & 0xFF00) + data;

            i8257->Flip = ! i8257->Flip;
            
            if (i8257->DebugInfo > 1) printf("i8257 CH3 ADDR=%04X\n", i8257->Address[3]);
            break;
            
        case 7:
            if (i8257->Flip)
                i8257->Terminate[3] = ((data&63) << 8) + (i8257->Terminate[3] & 255),
                i8257->Write[3] = data & 64, i8257->Read[3] = data & 128;
            else
                i8257->Terminate[3] = (i8257->Terminate[3] & 0xFF00) + data;

            i8257->Flip = ! i8257->Flip;
            
            if (i8257->DebugInfo > 1) printf("i8257 CH3 TERMINAL=%d MODE=%c%c\n",
                                             i8257->Terminate[3],
                                             i8257->Read[3]?'R':'-',
                                             i8257->Write[3]?'W':'-');
            break;
            
        case 8:
            
            i8257->Flip = 0;
            
            i8257->Enable[0] = data & 1;
            i8257->Enable[1] = data & 2;
            i8257->Enable[2] = data & 4;
            i8257->Enable[3] = data & 8;
            
            i8257->TerminalStop = data & 64;
            i8257->Autoload = data & 128;
            
            if (i8257->DebugInfo) printf("i8257 MODE ENABLE=%c%c%c%c AUTOLOAD=%s TERMINALSTOP=%s\n",
                                         i8257->Enable[0]?'0':'-',
                                         i8257->Enable[1]?'1':'-',
                                         i8257->Enable[2]?'2':'-',
                                         i8257->Enable[3]?'3':'-',
                                         i8257->Autoload?"ON":"OFF",
                                         i8257->TerminalStop?"ON":"OFF");
            break;
    }
}

byte i8257read(i8257state * i8257, int A)
{
    switch (A) {
            
            // TODO
            
        default:
            
            return
            ((i8257->Terminate[0]<=0)?1:0) +
            ((i8257->Terminate[1]<=0)?2:0) +
            ((i8257->Terminate[2]<=0)?4:0) +
            ((i8257->Terminate[3]<=0)?8:0) +
            (i8257->Up?16:0);
            
            break;
    }
}

//  0 no data
//  1 data
// -1 data & TC

int i8257dmaread(i8257state * i8257, byte * data, int channel)
{
    if ((channel == 1) && i8257->Autoload && (! i8257->Enable[1])) {
        i8257->Terminate[1] = i8257->Terminate[2];
        i8257->Address[1]   = i8257->Address[2];
        i8257->Read[1] = i8257->Read[2];
        i8257->Write[1] = i8257->Write[2];
        i8257->Up = 1;
        i8257->Enable[1] = i8257->Enable[2];
        if (i8257->DebugInfo > 2) printf("i8257 RELOADED CH1 %04X %d %c%c ENABLED=%s \n",
                                         i8257->Address[1], i8257->Terminate[1],
                                         i8257->Read[1]?'R':'-', i8257->Write[1]?'W':'-',
                                         i8257->Enable[1]?"ON":"OFF");

    }
        
    if (! i8257->Enable[channel]) return 0;
    if (! i8257->Write[channel]) return 0;
    
    *data = memory_read(i8257->Address[channel]++);
 
    if (i8257->Terminate[channel] < 0) {
        if (i8257->TerminalStop || ((channel == 1) && i8257->Autoload)) i8257->Enable[channel] = 0;
        if (i8257->DebugInfo > 2) printf("i8257 TERMINATED CH%d %04X %c%c ENABLED=%s\n",
                                         channel,
                                         i8257->Address[1],
                                         i8257->Read[1]?'R':'-',
                                         i8257->Write[1]?'W':'-',
                                         i8257->Enable[1]?"ON":"OFF");
        return -1;
    }
    
    -- i8257->Terminate[channel];
    return 1;
}

int i8257dmawrite(i8257state * i8257, int channel, byte data)
{
    if ((channel == 1) && i8257->Autoload && (! i8257->Enable[1])) {
        i8257->Terminate[1] = i8257->Terminate[2];
        i8257->Address[1]   = i8257->Address[2];
        i8257->Read[1] = i8257->Read[2];
        i8257->Write[1] = i8257->Write[2];
        i8257->Up = 1;
        i8257->Enable[1] = i8257->Enable[2];
        if (i8257->DebugInfo > 2) printf("i8257 RELOADED CH1 %04X %d %c%c ENABLED=%s \n",
                                         i8257->Address[1],
                                         i8257->Terminate[1],
                                         i8257->Read[1]?'R':'-',
                                         i8257->Write[1]?'W':'-',
                                         i8257->Enable[1]?"ON":"OFF");
    }
    
    if (! i8257->Enable[channel]) return 0;
    if (! i8257->Read[channel]) return 0;
    
    memory_write(i8257->Address[channel]++, data);

    if (i8257->Terminate[channel] < 0) {
        if (i8257->TerminalStop || ((channel == 1) && i8257->Autoload)) i8257->Enable[channel] = 0;
        if (i8257->DebugInfo > 2) printf("i8257 TERMINATED CH%d %04X %c%c ENABLED=%s\n",
                                         channel,
                                         i8257->Address[1],
                                         i8257->Read[1]?'R':'-',
                                         i8257->Write[1]?'W':'-',
                                         i8257->Enable[1]?"ON":"OFF");
        return -1;
    }
        
    -- i8257->Terminate[channel];
    return 1;
}

void i8257dump(i8257state * i8257)
{
    printf("i8257:\n"
           "CH0 %04X %d %c%c\n"
           "CH1 %04X %d %c%c\n"
           "CH2 %04X %d %c%c\n"
           "CH3 %04X %d %c%c\n"
           "ENABLED %c%c%c%c\n"
           "AUTOLOAD=%d TERMINALSTOP=%d\n"
           "UP=%d FLIP=%d\n",
           i8257->Address[0], i8257->Terminate[0], i8257->Read[0]?'R':'-', i8257->Write[0]?'W':'-',
           i8257->Address[1], i8257->Terminate[1], i8257->Read[1]?'R':'-', i8257->Write[1]?'W':'-',
           i8257->Address[2], i8257->Terminate[2], i8257->Read[2]?'R':'-', i8257->Write[2]?'W':'-',
           i8257->Address[3], i8257->Terminate[3], i8257->Read[3]?'R':'-', i8257->Write[3]?'W':'-',
           i8257->Enable[0]?'0':'-',
           i8257->Enable[1]?'1':'-',
           i8257->Enable[2]?'2':'-',
           i8257->Enable[3]?'3':'-',
           i8257->Autoload,
           i8257->TerminalStop,
           i8257->Up,
           i8257->Flip);
}