//
//  i8275.c
//
//  Created by Alexander Medvedev on 19/05/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>
#include <memory.h>

#include "types.h"
#include "i8275.h"

i8275state i8275;

extern byte i8275fetchDMA(int Reset);
extern byte i8275fetchROM(int Char, int Line);

void i8275init(i8275state * i8275, int CharW)
{
    int d = i8275->DebugInfo;
    
    memset(i8275, 0, sizeof(i8275state));
    
    i8275->DebugInfo = d;
    i8275->CharW = CharW;
    
    if (i8275->DebugInfo) printf("i8275 HARDWARE RESET, CHAR WIDTH=%d\n", CharW);
}

byte i8275read(i8275state * i8275, int A0)
{
    byte result = 0;
    
    if (A0 == 1) {
        
        result =
        
        (i8275->InterruptsEnabled  ? 64 : 0) +
        (i8275->InterruptRequested ? 32 : 0) +
        (i8275->LightPenCaptured   ? 16 : 0) +
        (i8275->VideoEnabled       ?  4 : 0);

        if (i8275->DebugInfo > 1) printf("i8275 STATUS=0%c%c%c0%c00\n",
                                    i8275->InterruptsEnabled  ? 'I' : '0',
                                    i8275->InterruptRequested ? 'R' : '0',
                                    i8275->LightPenCaptured   ? 'P' : '0',
                                    i8275->VideoEnabled       ? 'V' : '0');

        i8275->InterruptRequested = 0;

    } else
        if (3 == i8275->Command) {
            
            if(0 == i8275->Index) result = i8275->LightPenCharacter;
            if(1 == i8275->Index) result = i8275->LightPenRow;
            
            if (i8275->DebugInfo) printf("i8275 LIGHTPEN READ (data) %c=%d\n", i8275->Index ? 'R' : 'C', result);
                
            ++ i8275->Index;
        }
    
    return result;
}

void i8275write(i8275state * i8275, int A0, byte data)
{
    if (A0 == 1) {
    
        i8275->Command = (data >> 5);
        i8275->Index = 0;
        
        switch (i8275->Command)
        {
            case 0: // RESET

                i8275->InterruptsEnabled = 0;
                i8275->DMAEnabled = 0;
                i8275->VideoEnabled = 0;
                
                if (i8275->DebugInfo) printf("i8275 RESET\n");
                
                break;
            
            case 1: // START
                
                i8275->DMABurst = 1 << ((data & 3) - 1);
                
                i8275->InterruptsEnabled = 1;
                i8275->DMAEnabled = 1;
                i8275->VideoEnabled = 1;
                
                i8275->Width  = i8275->Characters * i8275->CharW;
                i8275->Height = i8275->Rows * i8275->CharH * (i8275->SpacedRows ? 2 : 1);
                
                if (i8275->DebugInfo) printf("i8275 START W=%d H=%d Burst=%d\n", i8275->Width, i8275->Height, i8275->DMABurst);
                
                break;
                
            case 2: // STOP
                
                i8275->InterruptsEnabled = 1;
                i8275->DMAEnabled = 0;
                i8275->VideoEnabled = 0;
                
                if (i8275->DebugInfo) printf("i8275 STOP\n");
                
                break;
                
            case 3: // READ LIGHT PEN
                
                i8275->LightPenCaptured = 0;
                
                if (i8275->DebugInfo) printf("i8275 LIGHTPEN READ\n");

                break;
                
            case 5: // ENABLE INTERRUPT
                
                i8275->InterruptsEnabled = 1;
                
                if (i8275->DebugInfo) printf("i8275 INTERRUPT ENABLE\n");
                
                break;

            case 6: // DISABLE INTERRUPT
                
                i8275->InterruptsEnabled = 0;
                
                if (i8275->DebugInfo) printf("i8275 INTERRUPT DISABLE\n");

                break;
                
        }
        
    } else {

        switch (i8275->Command) {
                
            case 0: // RESET
                
                if (i8275->Index == 0) {
                    i8275->SpacedRows = (data >> 7);
                    i8275->Characters = (data & 127) + 1;
                    
                    if (i8275->DebugInfo) printf("i8275 RESET SPACED=%d CHARACTERS=%d\n", i8275->SpacedRows, i8275->Characters);
                }

                if (i8275->Index == 1) {
                    i8275->Rows = (data & 63) + 1;
                    i8275->VRTC = (data >> 6);
                    
                    if (i8275->DebugInfo) printf("i8275 RESET ROWS=%d VRTC=%d\n", i8275->Rows, i8275->VRTC);
                }

                if (i8275->Index == 2) {
                    i8275->CharH = (data & 15) + 1;
                    i8275->Underline = (data >> 4);
                    i8275->BlankFirstLast = (data >> 7);
                    
                    if (i8275->DebugInfo) printf("i8275 RESET CHARH=%d UNDERLINE=%d BLANKFIRSTLAST=%d\n", i8275->CharH, i8275->Underline, i8275->BlankFirstLast);
                }
                
                if (i8275->Index == 3) {
                    i8275->LineOffset = (data >> 7);
                    i8275->TransparentAttribute = ! ((data >> 6) & 1);
                    i8275->CursorBlinking = ! ((data >> 5) & 1);
                    i8275->CursorReverse = ! ((data >> 4) & 1);
                    i8275->HRTC = ((data & 15) + 1) * 2;
                    
                    if (i8275->DebugInfo)
                        printf("i8275 RESET LINEOFFSET=%d TRANSPATTR=%d CURSOR BLINKING=%d REVERSE=%d HRTC=%d\n",
                                i8275->LineOffset, i8275->TransparentAttribute,
                                i8275->CursorBlinking, i8275->CursorReverse, i8275->HRTC);
                }

                break;
                
            case 4: // SET CURSOR
                
                if (i8275->Index == 0) i8275->CursorCharacter = data;
                if (i8275->Index == 1) i8275->CursorRow       = data;
                if (i8275->DebugInfo > 1) printf("i8275 SET CURSOR CHARACTER=%d ROW=%d\n", i8275->CursorCharacter, i8275->CursorRow);
                
                break;
                
        }
        
        ++ i8275->Index;
    }
}

// Redefine here you hardware nuances:

void i8275line(i8275state * i8275, byte* where, byte line, int inverse, int highlight, int general1, int general2)
{
    int F = highlight ? I8275GRADEHIGH : I8275GRADE1;
    
    if (inverse) line = ~ line;
    
    for (int c = 1; c <= i8275->CharW; c++, where++)
        *where = (line & (1 << (i8275->CharW - c))) ? F : I8275GRADE0;
}

int i8275draw(i8275state * i8275, byte* buffer)
{
    ++ i8275->Frame;
    
    if (i8275->InterruptsEnabled) i8275->InterruptRequested = 1;
    
    memset(buffer, I8275GRADE0, i8275->Width * i8275->Height);

    if (! i8275->VideoEnabled) return 0;
    
    byte data = 0;
    byte * bp;
    
    if (i8275->DMAEnabled) data = i8275fetchDMA(1);

    if (i8275->DebugInfo > 1)
        printf("i8275 DRAW SCREEN %dx%d (%dx%d)\n", i8275->Characters, i8275->Rows, i8275->Width, i8275->Height);

    int inverse = 0;
    int highlight = 0;
    int underline = 0;
    int blink = 0;
    int general1 = 0, general2 = 0;
    
    int supressframe = 0;
    int supressframedma = 0;
    
    for (int R=0; R < i8275->Rows; R++) {

        int supressline = 0, supresslinedma = 0;

        for (int C=0; C < i8275->Characters; C++) {
            
            if (data & 128) {
                
                switch (data) {
                    case 0xF1: supresslinedma = 1;
                    case 0xF0: supressline = 1;
                        if (i8275->DebugInfo > 2) printf("i8275 STOP LINE, DMA=%s\n", supresslinedma ? "OFF" : "ON" );
                        break;
                    case 0xF3: supressframedma = 1;
                    case 0xF2: supressframe = 1;
                        if (i8275->DebugInfo > 2) printf("i8275 STOP FRAME, DMA=%s\n", supressframedma ? "OFF" : "ON" );
                        break;
                }
                
                if (2 == (data >> 6)) {
                    
                    underline = data & 32;
                    inverse = data & 16;
                    general1 = data & 8;
                    general2 = data & 4;
                    blink = data & 2;
                    highlight = data & 1;
                    
                    if (i8275->TransparentAttribute) C--;

                    if (i8275->DebugInfo > 2)
                        printf("i8275 ATTR 10%c%c%c%c%c%c @ (%d,%d)\n",
                               (underline ? 'U' : '0'),
                               (inverse   ? 'I' : '0'),
                               (general1  ? 'G' : '0'),
                               (general2  ? 'G' : '0'),
                               (blink     ? 'B' : '0'),
                               (highlight ? 'H' : '0'),
                               C, R);
                    
                } else {
 
                    printf("i8275 ATTR %02X\n", data);
                    
                }
                                
            } else {

                bp = buffer + (R * i8275->Width * i8275->CharH) * (i8275->SpacedRows ? 2 : 1) + (C * i8275->CharW);
    
                int cursor = (i8275->CursorRow == R) && (i8275->CursorCharacter == C);
                
                int cblink = i8275->CursorBlinking && ((i8275->Frame >> 4) & 1);
                
                int chblink = blink && ((i8275->Frame >> 5) & 1);
                              
                int cinverse = cursor && i8275->CursorReverse && (! cblink);
                              
                for (int L=0; L < i8275->CharH; L++) {
                    
                    int l = L;
                    unsigned int line = 0;
                    
                    if (i8275->LineOffset) l = (L == 0) ? (i8275->CharH - 1) : (L - 1);
        
                    line = i8275fetchROM(data & 127, l);
                    
                    if ( chblink || supressline || supressframe || (i8275->BlankFirstLast && ((L == 0) || (L == (i8275->CharH - 1))))) line = 0;
                    
                    if ( ( underline || (cursor && (! i8275->CursorReverse) && (! cblink))) && (L == i8275->Underline)) line = ~ 0;

                    i8275line(i8275, bp + L * i8275->Width, line, inverse || cinverse, highlight, general1, general2);
                }
            }
            
            if (i8275->DMAEnabled && (! supresslinedma) && (! supressframedma)) data = i8275fetchDMA(0);
        }
    }
    return 1;
}

void i8275dump(i8275state * i8275)
{
    printf("i8275:\n"
           "Width=%d Height=%d\n"
           "Characters=%d Rows=%d\n"
           "Char Width=%d Height=%d\n"
           "Underline=%d BlankFirstLast=%d\n"
           "SpacedRows=%d LineOffset=%d TransparentAttribute=%d"
           "Cursor Blinking=%d Reverse=%d"
           "VRTC=%d HRTC=%d\n"
           "VideoEnabled=%d InterruptsEnabled=%d DMAEnabled=%d InterruptRequested=%d\n"
           "LightPen Character=%d Row=%d Captured=%d\n"
           "Cursor Character=%d Row=%d\n"
           "DMABurst=%d\n"
           "Command=%d Index=%d\n",
           i8275->Width, i8275->Height,
           i8275->Characters, i8275->Rows,
           i8275->CharW, i8275->CharH,
           i8275->Underline, i8275->BlankFirstLast,
           i8275->SpacedRows, i8275->LineOffset, i8275->TransparentAttribute,
           i8275->CursorBlinking, i8275->CursorReverse,
           i8275->VRTC, i8275->HRTC,
           i8275->VideoEnabled, i8275->InterruptsEnabled, i8275->DMAEnabled, i8275->InterruptRequested,
           i8275->LightPenCharacter, i8275->LightPenRow, i8275->LightPenCaptured,
           i8275->CursorCharacter, i8275->CursorRow, i8275->DMABurst,
           i8275->Command, i8275->Index);
}