//
//  i8275.h
//
//  Created by Alexander Medvedev on 19/05/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#ifndef i8275_h
#define i8275_h

//
// 8275 Command Set:
//
// [R/W][A0] [BITS]
//
// 0 = RESET CMD
//
// W1 00000000
// W0 SHHHHHHH    S: 0 normal / 1 spaced
//                H: chars per row (-1) 1..80
// W0 VVRRRRRR    V: vertical retrace rows (-1)
//                R: vertical rows per frame (-1) 1..64
// W0 UUUULLLL    U: underline line number (-1) 1..16
//    X              + if X=1 then top and bottom lines blanked
//                L: lines per character (-1) 1..16
// W0 MFCCZZZZ    M: 0 normal line counter / 1 +1 line offset
//                F: 0 transparent field attribute / 1 non transparent
//                C: 00 blinking + reverse
//                   01 blinking + underline
//                   10 nonblinking + reverse
//                   11 nonblinking + underline
//                Z: horisontal retrace count (/2-1) 2..32
//
// After:         DMA stops, interrupts disabled, video blanked
//
// 1 = START CMD
//
// W1 001SSSBB    S: Character clock between dma burst
//                   as 0 7 15 23 31 39 47 55
//                B: Cycles per burst
//                   as 1 2 4 8
//
// After:         DMA starts, interrupts enabled, video enabled
//
// 2 = STOP CMD
//
// W1 01000000
//
// After:         DMA stops, interrupts enabled, video disabled
//
// 3 = READ LIGHT PEN
//
// W1 01100000
// R0 CCCCCCCC    C: estimate character number
// R0 RRRRRRRR    R: estimate row number
//
// 4 = LOAD CURSOR POSITION
//
// W1 10000000
// W0 CCCCCCCC    C: char number
// W0 RRRRRRRR    R: row number
//
// 5 = ENABLE INTERRUPT
//
// W1 10100000
//
// After:         interrupts enabled
//
// 6 = DISABLE INTERRUPT
//
// W1 11000000
//
// After:         interrupts enabled
//
// 7 = PRESET COUNTERS
//
// W1 11100000
//
// After:         all counters reset to frame begin
//
// STATUS READING
//
// R1 0IRPEVDF    I: 1 interrupts enabled / 0 disabled
//                R: 1 interrupt request set at last line,
//                   0 reset then by status read
//                P: 1 light pen data captured / 0 none
//                E: 1 error command
//                V: 1 video enabled / 0 disabled
//                D: 1 dma underrun / 0 ok
//                F: 1 FIFO overrun / 0 ok
//

// Values for frame buffer

#define I8275GRADE0         10
#define I8275GRADE1         212
#define I8275GRADEHIGH      255

// I8275 itself

typedef struct {

    int Width, Height; // in pixels
    int Characters, Rows;
    int CharW, CharH;
    int Underline, BlankFirstLast;
    int SpacedRows, LineOffset, TransparentAttribute;
    int CursorBlinking, CursorReverse;
    int VRTC, HRTC;
    int VideoEnabled, InterruptsEnabled, DMAEnabled, InterruptRequested;
    int LightPenCharacter, LightPenRow, LightPenCaptured;
    int CursorCharacter, CursorRow;
    int DMABurst;
    
    // Internals
    
    int Command; int Index;
    
    long Frame;
    
    int DebugInfo;
    
} i8275state;

void i8275init(i8275state * i8275, int CharW);

byte i8275read(i8275state * i8275, int A0);
void i8275write(i8275state * i8275, int A0, byte data);

int i8275draw(i8275state * i8275, byte* buffer);

void i8275dump(i8275state * i8275);

#endif
