//
//  main.c
//  rk86
//
//  RK86 emulation for SDL2
//
//  Created by Alexander Medvedev on 17-05-2014.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>
#include <string.h>

#include <SDL2/SDL.h>

#include "types.h"

#include "i8080.h"
#include "i8275.h"
#include "i8257.h"
#include "i8255.h"

#include "i8080debug.h"

#include "rkkeys.h"

//
// RK86 32K Memory:
//
// RAM 32K
// ROM 2K (mirrored 4 times)
// + ZG ROM 1K (unvisible)
//
// Adressed as:
//           R       W
// 0x0000 [ RAM ] [ RAM ]
// 0x8000 [ PPI ] [ PPI ] <- Keyboard PPI
// 0xA000 [ PPI ] [ PPI ] <- ROM disk PPI
// 0xC000 [ VG  ] [ VG  ]
// 0xE000 [ DMA ] [ ROM ] <- 1st ROM Mirror
// 0xE800         [ ROM ] <- 2nd
// 0xF000         [ ROM ] <- 3rd
// 0xF800         [ ROM ] <- Official ROM address

#define RAM_SIZE        0x8000
#define ROM_SIZE        0x0800
#define ROM_BEGIN       0xE000
#define CG_SIZE         0x0800

byte ram[RAM_SIZE];     // RAM
byte rom[ROM_SIZE];     // ROM (Monitor)

byte cgrom[CG_SIZE];    // Character generator ROM (unvisible)

//
// RK86 CPU:
//
// i8080 at 1750000 Hz
// Video refresh @ 50Hz (70000ts)
//

#define CPUHZ    1750000
#define FPS      50
#define TICKS    (CPUHZ/FPS)

long long IntCounter = 0;
unsigned int LastIntTime;

//
// RK86 Screen
//
// Default:
//      78x30 characters by default (80x64 maximum)
//      6x10 character size (6x16 maximum),
//      only 6x8 active (in CG ROM)
//      64x24 chars default used by monitor

#define SCREENX  (78*6)
#define SCREENY  (30*10)

// Hardware

i8080state i8080;       // CPU
i8275state i8275;       // VG
i8257state i8257;       // DMA
i8255state i8255_1;     // PPI1 Keyboard / Tape
i8255state i8255_2;     // PPI2 ROM-disk / extension
RKkeyboard keyboard;    // RK Keyboard

void memory_write(word addr, byte data)
{
    if (addr<RAM_SIZE) {
        ram[addr] = data;
        return;
    }
    
    switch (addr >> 13) {
        case 4: // 0x8000-0x9FFF: PPI1
            
            i8255write(&i8255_1, addr & 3, data);
            
            // Keyboard index port - A
            if (0 == (addr & 3)) KeyboardUpdate(&keyboard, &i8255_1);
            
            break;
            
        case 5: // 0xA000-0xBFFF: PPI2
            
            i8255write(&i8255_1, addr & 3, data);
            
            break;
            
        case 6: // 0xC000-0xDFFF: VG
            
            i8275write(&i8275, addr & 1, data);
            
            break;
            
        case 7: // 0xE000-0xFFFF: DMA
            
            i8257write(&i8257, addr & 15, data);
            
            break;
    }
}

byte memory_read(word addr)
{
    if (addr<RAM_SIZE) return ram[addr];
    if (addr>=0xE000)  return rom[addr & (ROM_SIZE-1)];
    
    switch (addr >> 13) {
        case 4: // 0x8000-0x9FFF: PPI1
            
            KeyboardUpdate(&keyboard, &i8255_1);
            
            return i8255read(&i8255_1, addr & 3);
            
        case 5: // 0xA000-0xBFFF: PPI2
            
            return i8255read(&i8255_2, addr & 3);
            
        case 6: // 0xC000-0xDFFF: VG
            
            return i8275read(&i8275, addr & 1);

            break;
    }
    
    return 0xFF;
}

void port_write(byte addr, byte data)
{
    // No ports
}

byte port_read(byte addr)
{
    // No ports
    
    return 0;
}

int load_mem(void * where, char * name, int limit)
{
    int size = 0;
    
    FILE* f = fopen(name, "r");
    if (NULL == f) return -1;
    size = (int) fread(where, 1, limit, f);
    fclose(f);
    return size;
}

int load_rki(word* from, word* to, char* name)
{
    int size = 0;
    word begin = 0;
    word end = 0;
    byte header[5];
    
    FILE* f = fopen(name, "r");
    if (NULL == f) return -1;
    
    size = (int) fread(&header, 1, 5, f);
    if (5 != size) return -2;
    
    if( header[0] != 0xE6 ) return -3;
    
    begin = (header[1]<<8)+header[2];
    end = (header[3]<<8)+header[4];
    
    if (begin>end) return -4;
    if (begin>=RAM_SIZE) return -5;
    if (end>=RAM_SIZE) return -6;
    
    size = (int) fread(ram + begin, 1, end - begin + 1, f);
    if ((end - begin +1) != size) return -7;
    
    fclose(f);
    
    *from = begin;
    *to = end;
    
    return size;
}

int load_rkr(word* from, word* to, char* name)
{
    int size = 0;
    word begin = 0;
    word end = 0;
    byte header[4];
    
    FILE* f = fopen(name, "r");
    if (NULL == f) return -1;
    
    size = (int) fread(&header, 1, 4, f);
    if (4 != size) return -2;
    
    begin = (header[0]<<8)+header[1];
    end = (header[2]<<8)+header[3];
    
    if (begin>end) return -4;
    if (begin>=RAM_SIZE) return -5;
    if (end>=RAM_SIZE) return -6;
    
    size = (int) fread(ram + begin, 1, end - begin + 1, f);
    if ((end - begin +1) != size) return -7;
    
    fclose(f);
    
    *from = begin;
    *to = end;
    
    return size;
}

int RKInitHAL(void)
{
    i8080init(&i8080);
    
    keyboard.DebugInfo = 0;
    KeyboardInit(&keyboard);
    
    i8255_1.DebugInfo = 0;
    i8255init(&i8255_1);

    i8255_2.DebugInfo = 0;
    i8255init(&i8255_2);
    
    i8275.DebugInfo = 0;
    i8275init(&i8275, 6);
    
    i8257.DebugInfo = 0;
    i8257init(&i8257);
    
    if ( 0 > load_mem(rom, "/Users/medvdv/Documents/Programming/rk86/mon32.rom", ROM_SIZE)) return -1;
    if ( 0 > load_mem(cgrom, "/Users/medvdv/Documents/Programming/rk86/font.rom", CG_SIZE)) return -1;
    
    IntCounter = 0;
    LastIntTime = SDL_GetTicks();

    return 0;
}

byte i8275fetchDMA(int Reset)
{
    byte data = 0;
    
    i8257dmaread(&i8257, &data, 1);
    
    return data;
}

byte i8275fetchROM(int Char, int Line)
{
    return ~ cgrom [Char * 8 + Line];
}

SDL_Window * Window;
SDL_Surface * Screen;

void RefreshScreenWindow()
{
    byte * buffer = malloc(i8275.Width * i8275.Height);
    
    i8275draw(&i8275, buffer);
    
    byte * p = buffer;
    
    SDL_LockSurface(Screen);
    
    for (int y = 0; y < i8275.Height; y++) {
        unsigned int * pixels = Screen->pixels + y * Screen->w * 4;
        for (int x = 0; x < i8275.Width; x++, p++) *pixels++ = SDL_MapRGB(Screen->format, *p, *p, *p);
    }
    SDL_UnlockSurface(Screen);
    
    free(buffer);
    
    SDL_Surface * ws = SDL_GetWindowSurface(Window);
    SDL_BlitScaled(Screen, NULL, ws, NULL);
    SDL_UpdateWindowSurface(Window);
}

int FullScreen = 0;

int RKStuff()
{
    RefreshScreenWindow();
    
    SDL_Event event;
    
    while (SDL_PollEvent(&event)) {
        
        switch( event.type ) {
                
            case SDL_KEYDOWN:
                
                if (! event.key.repeat) if (KeyboardPC(&keyboard, &event)) break;
                
                // Here emulator control keys
                
                switch (event.key.keysym.sym) {
                        
                    case SDLK_F5:
                        
                        if ( ! FullScreen) {
                            SDL_SetWindowFullscreen(Window, SDL_WINDOW_FULLSCREEN_DESKTOP);
                            FullScreen = 1;
                        } else {
                            SDL_SetWindowFullscreen(Window, 0);
                            FullScreen = 0;
                        }
                        
                        break;
                        
                    case SDLK_F6:
                        
                    {
                        word from, to; load_rki(&from, &to, "/Users/medvdv/Documents/RK/TENNIS.GAM");
                        i8080.pc = from;
                    }
                        break;
                        
                    case SDLK_F7:
                        
                    {
                        word from, to; load_rkr(&from, &to, "/Users/medvdv/Documents/RK/CHESS2.RKR");
                        i8080.pc = from;
                    }
                        break;
                        
                    case SDLK_F8:
                        
                    {
                        word from, to; load_rkr(&from, &to, "/Users/medvdv/Documents/RK/klad.rkr");
                        i8080.pc = from;
                    }
                        break;
                    
                    case SDLK_F9:
                        
                    {
                        word from, to; load_rki(&from, &to, "/Users/medvdv/Documents/RK/diverse.gam");
                        i8080.pc = from;
                    }
                        break;
                        
                    case SDLK_F10:
                        
                    {
                        i8275dump(&i8275);
                    }
                        break;
                        
                    default:
                        break;
                }
                
                break;
                
            case SDL_KEYUP:
                
                if (! event.key.repeat) if (KeyboardPC(&keyboard, &event)) break;
                break;
                
            case SDL_QUIT:
                
                // Ask to quit emulation loop
                
                return 1;
                
            default:
                break;
        }
    }
    
    LastIntTime += (1000 / FPS);
    
    unsigned int current = SDL_GetTicks();
    if (current < LastIntTime) SDL_Delay(LastIntTime - current);
    
    ++ IntCounter;
    
    return 0;
}

int main(int argc, const char * argv[])
{
    printf("\nRADIO-86RK 32K Emulator for Mac OS X"
           "\nVersion 1 (c) 2014 by Alexander Medvedev @medvdv\n");
    
    if (0 > SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO)) {
		printf("Couldn't initialize SDL: %s\n", SDL_GetError());
        return 1;
    }
    
    SDL_Window * window = SDL_CreateWindow("RADIO-86RK",
                                           SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED,
                                           SCREENX*2, SCREENY*2,
                                           0);
    
    if ( ! window) {
        printf("Couldn't create window: %s\n", SDL_GetError());
        return 2;
    }
    
    SDL_Surface * screen  = SDL_CreateRGBSurface(0, SCREENX, SCREENY, 32, 0, 0, 0, 0);
    
    if ( ! screen) {
        printf("Couldn't create screen surface: %s\n", SDL_GetError());
        return 2;
    }
    
    Window = window;
    Screen = screen;
    
    switch (RKInitHAL()) {
        case -1:
            printf("Couldn't load ROM image files\n");
            return 3;
        case 0:
            printf("HW initialised, ROMs loaded\n");
            break;
    }
    
    i8080.pc = 0xF800;
    
    while (1) {
        
        i8080.t = TICKS;
        i8080.Ticks = 1;
        
        i8080execute(&i8080);
        
        if (RKStuff()) {
            
            SDL_FreeSurface(screen);
            SDL_DestroyWindow(window);
            SDL_Quit();
            
            return 0;
        }
    } 
    
}

