//
//  rkkeys.c
//
//  RK-86 / ORION-128 Keyboard
//
//  Created by Alexander Medvedev on 02/06/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>

#include <SDL2/SDL.h>

#include "types.h"
#include "i8255.h"

#include "rkkeys.h"

int RShift = 0;  // Host modifier

int KeyboardMap[][4] = {
    // ch row mask
    { -1,               0, 0x01, 0 }, //   //
    { -2,               0, 0x02, 0 }, //   CTP
    { SDLK_ESCAPE,      0, 0x04, 0 }, //   AP2
    { SDLK_F1,          0, 0x08, 0 }, //   Ф1
    { SDLK_F2,          0, 0x10, 0 }, //   Ф2
    { SDLK_F3,          0, 0x20, 0 }, //   Ф3
    { SDLK_F4,          0, 0x40, 0 }, //   Ф4
    { SDLK_TAB,         1, 0x01, 0 }, //   TAB
    { -3,               1, 0x02, 0 }, //   ПС
    { SDLK_RETURN,      1, 0x04, 0 }, //   BK
    { SDLK_BACKSPACE,   1, 0x08, 0 }, //   ЗБ
    { SDLK_LEFT,        1, 0x10, 0 }, //   LEFT
    { SDLK_UP,          1, 0x20, 0 }, //   UP
    { SDLK_RIGHT,       1, 0x40, 0 }, //   RIGHT
    { SDLK_DOWN,        1, 0x80, 0 }, //   DOWN
    { '0',              2, 0x01, 0 }, //   0
    { '1',              2, 0x02, 0 }, //   1 !
    { '2',              2, 0x04, 0 }, //   2 "
    { '3',              2, 0x08, 0 }, //   3 #
    { '4',              2, 0x10, 0 }, //   4 $
    { '5',              2, 0x20, 0 }, //   5 %
    { '6',              2, 0x40, 0 }, //   6 &
    { '7',              2, 0x80, 0 }, //   7 '
    { '8',              3, 0x01, 0 }, //   8 (
    { '9',              3, 0x02, 0 }, //   9 )
    { '\'',             3, 0x04, 0 }, //   * :
    { ';',              3, 0x08, 0 }, //   ; +
    { ',',              3, 0x10, 0 }, //   , <
    { '-',              3, 0x20, 0 }, //   - =
    { '.',              3, 0x40, 0 }, //   . >
    { '/',              3, 0x80, 0 }, //   / ?
    { '@',              4, 0x01, 0 }, //   @ Ю
    { 'A',              4, 0x02, 0 }, //   A А
    { 'B',              4, 0x04, 0 }, //   B Б
    { 'C',              4, 0x08, 0 }, //   C Ц
    { 'D',              4, 0x10, 0 }, //   D Д
    { 'E',              4, 0x20, 0 }, //   E Е
    { 'F',              4, 0x40, 0 }, //   F Ф
    { 'G',              4, 0x80, 0 }, //   G Г
    { 'H',              5, 0x01, 0 }, //   H Х
    { 'I',              5, 0x02, 0 }, //   I И
    { 'J',              5, 0x04, 0 }, //   J Й
    { 'K',              5, 0x08, 0 }, //   K К
    { 'L',              5, 0x10, 0 }, //   L Л
    { 'M',              5, 0x20, 0 }, //   M М
    { 'N',              5, 0x40, 0 }, //   N Н
    { 'O',              5, 0x80, 0 }, //   O О
    { 'P',              6, 0x01, 0 }, //   P П
    { 'Q',              6, 0x02, 0 }, //   Q Я
    { 'R',              6, 0x04, 0 }, //   R Р
    { 'S',              6, 0x08, 0 }, //   S С
    { 'T',              6, 0x10, 0 }, //   T Т
    { 'U',              6, 0x20, 0 }, //   U У
    { 'V',              6, 0x40, 0 }, //   V Ж
    { 'W',              6, 0x80, 0 }, //   W В
    { 'X',              7, 0x01, 0 }, //   X Ь
    { 'Y',              7, 0x02, 0 }, //   Y Ы
    { 'Z',              7, 0x04, 0 }, //   Z З
    { '[',              7, 0x08, 0 }, //   [ Ш
    { SDLK_BACKSLASH,   7, 0x10, 0 }, //   \ Э
    { ']',              7, 0x20, 0 }, //   ] Щ
    { '^',              7, 0x40, 0 }, //   ^ Ч
    { SDLK_SPACE,       7, 0x80, 0 }  //   SPC
};

void KeyboardInit(RKkeyboard * keyboard)
{
    int d = keyboard->DebugInfo;
    
    memset(keyboard, 0, sizeof(RKkeyboard));
    
    keyboard->DebugInfo = d;
    
    if (keyboard->DebugInfo) printf("KEYBOARD HARDWARE RESET\n");
}

int KeyboardRK(RKkeyboard * keyboard, int Key, int Press)
{
    int i;
    
    for (i = 0; i < ( sizeof(KeyboardMap) / sizeof(int[4]) ); i++)
        
        if (Key == KeyboardMap[i][0]) {
            
            if (keyboard->DebugInfo) printf("KEY %d %s -> %d %02X\n", Key, Press ? "DOWN" : "UP", KeyboardMap[i][1], KeyboardMap[i][2]);
            
            if (Press)
                keyboard->Matrix[KeyboardMap[i][1]] |= KeyboardMap[i][2];
            else
                keyboard->Matrix[KeyboardMap[i][1]] &= ~KeyboardMap[i][2];
            return i;
        }
    
    if (keyboard->DebugInfo) printf("KEY %d %s UNKNOWN\n", Key, Press ? "DOWN" : "UP");
    
    return -1;
}

void KeyboardUpdate(RKkeyboard * keyboard, i8255state * i8255)
{
    byte answer = 0xFF;
    byte index  = i8255->A;
    
    if (0 == (index &   1)) answer &= ~ keyboard->Matrix[0];
    if (0 == (index &   2)) answer &= ~ keyboard->Matrix[1];
    if (0 == (index &   4)) answer &= ~ keyboard->Matrix[2];
    if (0 == (index &   8)) answer &= ~ keyboard->Matrix[3];
    if (0 == (index &  16)) answer &= ~ keyboard->Matrix[4];
    if (0 == (index &  32)) answer &= ~ keyboard->Matrix[5];
    if (0 == (index &  64)) answer &= ~ keyboard->Matrix[6];
    if (0 == (index & 128)) answer &= ~ keyboard->Matrix[7];
    
    i8255->B = answer;
    
    i8255->C2 = 15;
    
    if (keyboard->CC) i8255->C2 &= ~ 2;
    if (keyboard->UC) i8255->C2 &= ~ 4;
    if (keyboard->RL) i8255->C2 &= ~ 8;
}

int KeyboardPC(RKkeyboard * keyboard, SDL_Event * Event)
{
    if ((Event -> type != SDL_KEYDOWN) && (Event -> type != SDL_KEYUP)) return 0;
 
    int Press = 0;
    int Key = Event -> key.keysym.sym;
    
    if (Event -> type == SDL_KEYDOWN) {
        Press = 1;
        RShift = (Event -> key.keysym.mod) & KMOD_RSHIFT;
    }
    
    // Some translation of host keys to RK pairs
    
    switch (Key) {      // TODO!
            
            /*
             case SDLK_1: if (ZXRShift) { ZXSS(Press); ZXKey('1', Press); return 1; } else break;
             case SDLK_COMMA:
             if (ZXRShift) { ZXSS(Press); ZXKey('R', Press); return 1; }
             else { ZXSS(Press); ZXKey('N', Press); return 1; }
             */
    }
    
    // Base keys
    
    if ((Event -> key.keysym.mod) & KMOD_LSHIFT ) keyboard->CC = Press ? 1 : 0;
    if ((Event -> key.keysym.mod) & (KMOD_LALT | KMOD_RALT) ) keyboard->UC = Press ? 1 : 0;
    
    if ( isalpha(Key) ) Key=toupper(Key);
    if ( -1 != KeyboardRK(keyboard, Key, Press) ) return 1;
    
    return 0;
}