#
# Library for creating files for HxC Floppy Emulator
#
# (c) 2015 PortaOne, Inc.
# Oleksandr Kapitanenko kapitan@portaone.com
#
package HxC;

use strict;
use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
use POSIX qw(ceil);

$VERSION     = 1.00;
@ISA         = qw(Exporter);
@EXPORT      = ();
@EXPORT_OK   = qw(hxc_mfm_file);
%EXPORT_TAGS = ( DEFAULT => [qw(&hxc_mfm_file)]);

# Interface mode constants
#
use constant IBMPC_DD_FLOPPYMODE => 0x00;
use constant IBMPC_HD_FLOPPYMODE => 0x01;
use constant ATARIST_DD_FLOPPYMODE => 0x02;
use constant ATARIST_HD_FLOPPYMODE => 0x03;
use constant AMIGA_DD_FLOPPYMODE => 0x04;
use constant AMIGA_HD_FLOPPYMODE => 0x05;
use constant CPC_DD_FLOPPYMODE => 0x06;
use constant GENERIC_SHUGGART_DD_FLOPPYMODE => 0x07;
use constant IBMPC_ED_FLOPPYMODE => 0x08;
use constant MSX2_DD_FLOPPYMODE => 0x09;
use constant C64_DD_FLOPPYMODE => 0x0A;
use constant EMU_SHUGART_FLOPPYMODE => 0x0B;
use constant S950_DD_FLOPPYMODE => 0x0C;
use constant S950_HD_FLOPPYMODE => 0x0D;
use constant DISABLE_FLOPPYMODE => 0xFE;

# Encoding constants
#
use constant ISOIBM_MFM_ENCODING => 0x00;
use constant AMIGA_MFM_ENCODING => 0x01;
use constant ISOIBM_FM_ENCODING => 0x02;
use constant EMU_FM_ENCODING => 0x03;
use constant UNKNOWN_ENCODING => 0xFF;

#
# HFE V1 File
#

sub hxc_hfe_file($$$) {
   my ($OUT, $mfm_track, $sides) = @_;

   my $tracks = @{$mfm_track};
   die "ERROR: Sides $sides is not supported." if $sides != 1 && $sides != 2;

   # Create HxC header
   #
   print $OUT "HXCPICFE";
   print $OUT pack( 'C', 0 );                # Revision 0
   print $OUT pack( 'C', $tracks / $sides ); # Number of tracks
   print $OUT pack( 'C', $sides );           # Number of sides (Not used by the emulator)
   print $OUT pack( 'C', ISOIBM_MFM_ENCODING );                # Track Encoding mode MFM
   print $OUT pack( 'v', 250 );              # Bitrate in Kbit/s. Ex : 250=250000bits/s
   print $OUT pack( 'v', 0 );                # Rotation per minute (Not used by the emulator)
   print $OUT pack( 'C', GENERIC_SHUGGART_DD_FLOPPYMODE );                # Floppy interface mode
   print $OUT pack( 'C', 0 );                # Write protected
   print $OUT pack( 'v', 1 );   # Offset of the track list LUT in block of 512 bytes (Ex: 1=0x200)

   # Advance to 0x200
   #
   print $OUT chr(0xFF)x(0x200-20);

   # Create track list
   #
   my $offset = 2; # Start offset 2x512 bytes blocks
   for( my $track = 0; $track < $tracks/$sides; $track++ ) {
      my $track_len = @{$$mfm_track[$track]} *2;
      print $OUT pack('v', $offset );
      print $OUT pack('v', $track_len );
      $offset += ceil($track_len/512);
   }

   # Advance to 0x400
   #
   print $OUT chr(0xFF)x(0x200-80*4);

   # Dump MFM code to hfe file
   #
   for( my $track = 0; $track < $tracks; $track += 2 ) {
      print "..";
      my $i1 = 0;
      my $i2 = 0;
      my $s0_last_byte = 0;
      my $s1_last_byte = 0;
      my $hfe_track_len = ceil( @{$$mfm_track[$track]} * 2 / 512 ) * 512;

      for( my $i = 0; $i < $hfe_track_len; $i++ ) {

         # Alternate 512 byte blocks. Pad track with last byte if needed.
         #
         if( $i & 0x0100 ) {
            #
            # Side 1
            #
            if( $i2 < @{$$mfm_track[$track]} ) {
               print $OUT chr( $$mfm_track[$track+1][$i2] );
               $s1_last_byte = $$mfm_track[$track+1][$i2];
               $i2++;
            } else {
               print $OUT chr( $s1_last_byte );
            }
         } else {
            #
            # Side 0
            #
            if( $i1 < @{$$mfm_track[$track]} ) {
               print $OUT chr( $$mfm_track[$track][$i1] );
               $s0_last_byte = $$mfm_track[$track][$i1];
               $i1++;
            } else {
               print $OUT chr( $s0_last_byte );
            }
         }
      }
   }

}

#
# HFE V2 File
#

sub hxc_hfe_v2_file($$$) {
   my ($OUT, $mfm_track, $sides) = @_;

   my @v2_padding = ( 0x000, 0x06, 0xAA, 0xAA);
   my $tracks = @{$mfm_track};
   die "ERROR: Sides $sides is not supported." if $sides != 1 && $sides != 2;

   # Create HxC header
   #
   print $OUT "HXCPICFE";
   print $OUT pack( 'C', 1 );                # Revision 1
   print $OUT pack( 'C', $tracks / $sides ); # Number of tracks
   print $OUT pack( 'C', $sides );           # Number of sides (Not used by the emulator)
   print $OUT pack( 'C', ISOIBM_MFM_ENCODING );                # Track Encoding mode MFM
   print $OUT pack( 'v', 250 );              # Bitrate in Kbit/s. Ex : 250=250000bits/s
   print $OUT pack( 'v', 0 );                # Rotation per minute (Not used by the emulator)
   print $OUT pack( 'C', GENERIC_SHUGGART_DD_FLOPPYMODE );                # Floppy interface mode
   print $OUT pack( 'C', 0 );                # Write protected
   print $OUT pack( 'v', 1 );   # Offset of the track list LUT in block of 512 bytes (Ex: 1=0x200)

   # Advance to 0x200
   #
   print $OUT chr(0xFF)x(0x200-20);

   # Create track list
   #
   my $offset = 2; # Start offset 2x512 bytes blocks
   for( my $track = 0; $track < $tracks/$sides; $track++ ) {
      my $track_len = @{$$mfm_track[$track]} *2;
      print $OUT pack('v', $offset );
      print $OUT pack('v', $track_len );
      $offset += ceil($track_len/512);
   }

   # Advance to 0x400
   #
   print $OUT chr(0xFF)x(0x200-80*4);

   # Dump MFM code to hfe file
   #
   for( my $track = 0; $track < $tracks; $track += 2 ) {
      print "..";
      my $i1 = 0;
      my $i2 = 0;
      my $s0_last_byte = 0;
      my $s1_last_byte = 0;
      my $hfe_track_len = ceil( @{$$mfm_track[$track]} * 2 / 512 ) * 512;

      for( my $i = 0; $i < $hfe_track_len; $i++ ) {

         # Alternate 512 byte blocks. Pad track with last byte if needed.
         #
         if( $i & 0x0100 ) {
            #
            # Side 1
            #
            if( $i2 < @{$$mfm_track[$track]} ) {
               print $OUT chr( $$mfm_track[$track+1][$i2] );
               $s1_last_byte = $$mfm_track[$track+1][$i2];
               $i2++;
            } else {
               print $OUT pack( 'C*', @v2_padding );
               $i += $#v2_padding;
            }
         } else {
            #
            # Side 0
            #
            if( $i1 < @{$$mfm_track[$track]} ) {
               print $OUT chr( $$mfm_track[$track][$i1] );
               $s0_last_byte = $$mfm_track[$track][$i1];
               $i1++;
            } else {
               print $OUT pack( 'C*', @v2_padding );
               $i += $#v2_padding;
            }
         }
      }
   }

}

#
# MFM File
#

sub hxc_mfm_file($$$) {
   my ($OUT, $mfm_track, $sides) = @_;

   my $tracks = @{$mfm_track};

   die "ERROR: Sides $sides is not supported." if $sides != 1 && $sides != 2;

   # Create HxC header
   #
   print $OUT "HXCMFM\x00";
   print $OUT pack('v', $tracks / $sides );  # Number of tracks
   print $OUT pack('C', $sides);      # Number of sides (Not used by the emulator)
   print $OUT pack('v', 0);           # Rotation per minute (Not used by the emulator)
   print $OUT pack('v', 250);         # Bitrate in Kbit/s. Ex : 250=250000bits/s
   print $OUT pack('C', GENERIC_SHUGGART_DD_FLOPPYMODE );           # Floppy interface mode
   print $OUT pack('V', 0x13);        # Offset of the MFMTRACKIMG array from the beginning of the file

   # Create MFMTRACKIMG array
   #
   my $offset = 0x800; # Start offset in bytes
   for( my $track = 0; $track < $tracks; $track++ ) {
      my $track_len = @{$$mfm_track[$track]};

      if( $sides == 2) { 
         print $OUT pack('v', int($track/2) );      # Track
      } else {
         print $OUT pack('v', $track );      # Track
      }
      if( ($sides == 2) && ($track & 1) ) {
         print $OUT pack('C', 1);           # Side 1
      } else {
         print $OUT pack('C', 0);           # Side 0
      }
      print $OUT pack('V', $track_len);  # MFM/FM Track size in bytes
      print $OUT pack('V', $offset);     # Offset of the track data from the beg. of the file in bytes
      $offset += 512*ceil($track_len/512);
   }

   # Advance to 0x800
   #
   print $OUT chr(0)x(0x800-0x13-80*2*11);

   # Dump MFM code to mfm file
   #
   for( my $track = 0; $track < $tracks; $track++ ) {
      my $track_len = @{$$mfm_track[$track]};

      my $padding = ceil( $track_len / 512 ) * 512 - $track_len; 
      print $OUT pack( 'C*', @{$$mfm_track[$track]} );
      print $OUT pack( "C$padding", 0);
   }
}

1;
