#!/usr/bin/perl
#
# Encoding data from Agat AIM file to HxC HFE and MFM files
# or AIM file for simulator
#
# Supported AIM commands
# 0x00 Payload
# 0x01 DESYNC 
# 0x80 DESYNC
# 0x03 Index Pulse Start
# 0x02 End of Track
# 0x13 Index Pulse End
#
# (c) 2014 PortaOne, Inc.
# Oleksandr Kapitanenko kapitan@portaone.com
#
# Usage:
# agath-aim-to-hfe.pl <aim file> <hfe, mfm of aim file> [<debug mode>] [<debug track>]
# Output file type selected by extension hfe, mfm or aim.
#
# Debug modes:
#
# 'AIM' Initial analysis after AIM read
# 'GAP' After DESYNC expansion and GAP count
# 'DEF' After GAP deflation
# 'MFM' MFM Encoding
#
# Debug track is assuming Agath / Apple 2 track numbering
#

use POSIX qw(ceil);
use File::Basename qw(dirname);
use Cwd qw(abs_path);
use lib dirname(dirname abs_path $0);
use HxC;
$| = 1;

# Debug mode
#
my $debug = $ARGV[2];

# Debug track
#
my $debug_track = ($ARGV[3] == undef)? 0 : $ARGV[3];


# GAP's equal or lesser than $minimum_gap_len
# are not deflated DEFAULT=5
#
my $minimum_gap_len = 5; 

# Inflate GAP's smaller than $minimum_gap_len to $minimum_gap_len
# DEFAULT=1 (Yes)
#
my $inflate_small_gaps = 1;

# GAP4 length if track must be shifted to avoid PAYLOAD crossing index hole
# DEFAULT=14
#
my $GAP4_len = 14;

# HFE Version
#
my $hfe_version = 2;

# Index hole position allignment
# DEFAULT=1
#
# 0 - Preserve original track alignment (you may need HFE vesion 2 for this)
# 1 - DEFAULT: Largest gap in the begining of the track (Large gap > GAP1+GAP4)
# 2 - Split Sector merged and moved into the END of the track (Index+Data)
# 3 - Split Sector merged and moved into the BEGINNING of the track (Index+Data)
#
my $fix_index_divided_sector_position = 0;

die "Output and input files are the same!" if $ARGV[0] eq $ARGV[1];
die "Target file extension must be HFE, MFM or AIM!" if $ARGV[1] !~ /mfm|hfe|aim$/i;
if( $ARGV[1] =~ /hfe$/i) {
   $bit_order = 'HFE';
} else {
   $bit_order = 'MFM';
}

unlink($ARGV[1]);
open(IN, "<$ARGV[0]")
   || die "File $ARGV[0] not found.";
open(OUT, ">$ARGV[1]")
   || die "Can not create output file $ARGV[1]";

# Read AIM file
#
print "Reading AIM  ";
my @aim, $buff, $track;
for( $track = 0; $track < 160; $track++ ) {
   $aim_track = [];
   sysread(IN, $buff, 6464*2);
   push $aim_track, unpack( 'C12928', $buff);
   $aim[$track] = $aim_track;
   print ".";
}
close(IN);

# Expanding DESYNC byte to 0xA4 0xFF,
# calculating individual track length
# and length of each GAP
#
print "\nExpanding DESYNC ";
my $aim_byte, $aim_command, $track_len, $decoder_state, $gap_len, $gap_index, $byte_decoded,  $expected_sector_len, $index_count, $first_gap_index, $last_gap_index, $index_hole_ptr, $ptr, $trim, $exception_byte;
for( $track = 0; $track < 160; $track++ ) {
   $track_len = 0;
   $aim_track = [];
   $first_gap_index = -1;
   $index_hole_ptr = -1;
   $decoder_state = 'PRE_GAP';
   $aim_track_gap_total_len[$track] = 0;
   $maximum_gap_len[$track] = 0;
   $gap_len = 0;
   $gap_index = 1;
   $exception_byte = 0xA4;
   $index_count = 0;
   print "\n================= TRACK $track as in AIM ======================" if $debug eq 'AIM' && $debug_track == $track;

   for( $i = 0; $i < $#{$aim[$track]}+1; $i += 2 ) {
     ( $aim_byte, $aim_command ) = ( $aim[$track][$i], $aim[$track][$i+1] );
      
     print "\n$decoder_state " if $debug eq 'AIM'  && $debug_track == $track;
     printf "%04X: %02X %02X", $i/2, $aim_byte, $aim_command if $debug eq 'AIM'  && $debug_track == $track;


     $byte_decoded = $aim_byte;
     $aim_command = 0x01 if $aim_command == 0x80;
     if( $aim_command == 0x03 ) {
        print " INDEX Pulse Start AIM Command" if $debug eq 'AIM'  && $debug_track == $track;
        $aim_command = 0x00;
        $index_hole_ptr = $i;
     }
     if( $aim_command == 0x13 ) {
        print " INDEX Pulse End AIM Command" if $debug eq 'AIM'  && $debug_track == $track;
        $aim_command = 0x00;
     }
     if( $aim_command == 0x02 ) {
        print " TRACK End AIM Command.\nSkipping remaining AIM track.\n" if $debug eq 'AIM'  && $debug_track == $track;
        print "\nTRACK End Command is present on track $track";

        # Remove 0xA4 from AIM
        # if it is encoded as payload befoere DESYNC command
        #
        $ptr = $track_len - 2;
        if( $$aim_track[$ptr] == 0xA4 ){
           print " Removing 0xA4 " if $debug eq 'AIM' && $debug_track == $track;
           splice $aim_track, $ptr, 2;
           $gap_len -= 1;
           $track_len -= 2;
        }

        # Remove 0xA4 0xFF from AIM
        # if it is encoded as payload befoere DESYNC command
        #
        $ptr = $track_len - 4;
        if( $$aim_track[$ptr] == 0xA4 && $$aim_track[$ptr+2] == 0xFF ){
           print " Removing 0xA4 0xFF" if $debug eq 'AIM' && $debug_track == $track;
           splice $aim_track, $ptr, 4;
           $gap_len -= 2;
           $track_len -= 4;
        }

        $i = $#{$aim[$track]}+1;
     } else {

      if ( $aim_command > 1 ) {
         print "\nUnsupported AIM command $aim_command on track $track. Ignoring";
         $aim_command = 0x00;
      }

      # Copy protection writes sector data ove disk index hole
      # or some payload at GAP beginning
      #
      if( $decoder_state eq 'PRE_GAP' && $byte_decoded == 0xAA ) {
        $gap_index = $track_len + 1;
        $decoder_state = 'DESYNC';
      } 
      if( $aim_command == 1 ) {
        $decoder_state = 'DESYNC';
      }

      if( $decoder_state eq 'DESYNC' ) {
         if ( $aim_command == 0 ) {
            if( $byte_decoded != 0xAA && $byte_decoded != $exception_byte ) {
               $decoder_state = 'PRE_GAP';
               $first_gap_index = 0;
               $gap_len = 0;
               $exception_byte = 0xA4;
            } else {
               # GAP
               #
               $first_gap_index = $i if $first_gap_index < 0;
               $last_gap_index = $i;
               $gap_len++;
               if( $byte_decoded == 0xA4 ) {
                  $exception_byte = 0xFF;
               } elsif ( $byte_decoded == 0xFF ) {
                  $exception_byte = 0x00;
               }
            }
            push $aim_track, ( $aim_byte, $aim_command );
            $track_len += 2;
         } else {
            $exception_byte = 0xA4;

            # DESYNC
            #
            print " DESYNC AIM Command" if $debug eq 'AIM' && $debug_track == $track;

            # Remove 0xA4 from AIM
            # if it is encoded as payload befoere DESYNC command
            #
            $ptr = $track_len - 2;
            if( $$aim_track[$ptr] == 0xA4 ){
               print " Removing 0xA4 " if $debug eq 'AIM' && $debug_track == $track;
               splice $aim_track, $ptr, 2;
               $gap_len -= 1;
               $track_len -= 2;
            }

            # Remove 0xA4 0xFF from AIM 
            # if it is encoded as payload befoere DESYNC command
            #
            $ptr = $track_len - 4;
            if( $$aim_track[$ptr] == 0xA4 && $$aim_track[$ptr+2] == 0xFF ){
               print " Removing 0xA4 0xFF" if $debug eq 'AIM' && $debug_track == $track;
               splice $aim_track, $ptr, 4;
               $gap_len -= 2;
               $track_len -= 4;
            }

            if( $gap_len > 0 ) {

               # End of GAP
               #
               print " GAP=$gap_len" if $debug eq 'AIM' && $debug_track == $track;

               # Expand smapp GAP to $minimum_gap_len
               #
               if( $inflate_small_gaps == 1 && $gap_len < $minimum_gap_len ) {
                  push $aim_track, ( 0xAA, 0 ) x ($minimum_gap_len - $gap_len);
                  $track_len +=  ($minimum_gap_len - $gap_len)*2;
               }

               $$aim_track[$gap_index] = $gap_len if $gap_len > $minimum_gap_len;
               $aim_track_gap_total_len[$track] += $gap_len if $gap_len > $minimum_gap_len;
               $maximum_gap_len[$track] = $gap_len if $gap_len > $maximum_gap_len[$track];
               $gap_len = -1;
            } else {
               print "\nWARNING: Track=$track GAP expected, No GAP found";
            }
            if( $ARGV[1] =~ /aim$/i) {

               # Simulator AIM file desync encoding
               #
               push $aim_track, ( 0xA4, 1 );
               $track_len += 2;
               $decoder_state = 'MARKER';
               ( $aim_byte, $aim_command ) = ( 0xA4, 0 );
            } else {

               # Disk desync encoding
               #
               push $aim_track, ( 0xA4, 1, 0xFF, 0 );
               $track_len += 4;
               $decoder_state = 'MARKER';         
               ( $aim_byte, $aim_command ) = ( 0xFF, 0 );
            }
         }
      } else {
         push $aim_track, ( $aim_byte, $aim_command );
         $track_len += 2;

         # INDEX or DATA Marker
         #
         if( $decoder_state eq 'MARKER' ) {
            if( $byte_decoded == 0x95 ) {

               # Possible INDEX Marker
               #
               $decoder_state = 'INDEX_MARKER';
               $idam_cursor = 0;
               print " INDEX Mark?" if $debug eq 'AIM' && $debug_track == $track;
            } 
            elsif( $byte_decoded == 0x6A ) {
              
               # Possible DATA Marker
               #
               $decoder_state = 'DATA_MARKER';
               $data_cursor = 0;
               print " DATA Mark?" if $debug eq 'AIM' && $debug_track == $track;
            } else {
               print " WARNING: Expecting marker 0x95 or 0x6A. Copy protection?";
               $gap_len = 0;
               $gap_index = $track_len + 1;
               $decoder_state = 'PRE_GAP';
            }
         }
 
         # INDEX Marker 0x6A
         #
         elsif( $decoder_state eq 'INDEX_MARKER' ) {
            if( $byte_decoded == 0x6A ) {
               $decoder_state = 'SECTOR_HEADER';
               $idam_cursor = 0;
               $index_count++;
               print " INDEX Mark #$index_count" if $debug eq 'AIM' && $debug_track == $track;
            } else {
               print " WARNING: Expecting INDEX marker 0x6A. Copy protection?";
               $gap_len = 0;
               $gap_index = $track_len + 1;
               $decoder_state = 'PRE_GAP';
            }
         }

         # DATA Marker 0x95
         #
         elsif( $decoder_state eq 'DATA_MARKER' ) {
            if( $byte_decoded == 0x95 ) {
               $decoder_state = 'SECTOR_DATA';
               $data_cursor = 0;
               $expected_sector_len = 256;
               print " DATA Mark. Expected $expected_sector_len data bytes." if $debug eq 'AIM' && $debug_track == $track;
            } else {
               print "\nWARNING: Expecting DATA marker 0x95. Copy protection?";
               $gap_len = 0;
               $gap_index = $track_len + 1;
               $decoder_state = 'PRE_GAP';
            }
         }
      
         # INDEX Processing
         #
         # INDEX is 4 bytes: Volume, Track, Sector, 0x5A
         #
         elsif( $decoder_state eq 'SECTOR_HEADER' ) {
            $idam_cursor++;
            if( $idam_cursor == 4 ) {

               # End of INDEX mark
               #
               if( $byte_decoded != 0x5A ) {
                  print "\nERROR: Last INDEX byte must be 0x5A\n";
               } else {
                  print " End Of INDEX Mark" if $debug eq 'AIM' && $debug_track == $track;
               }

               # GAP begin
               #
               $gap_len = 0;
               $gap_index = $track_len + 1;
               $decoder_state = 'PRE_GAP';
            }
         }

         # DATA Processing
         #
         elsif ( $decoder_state eq 'SECTOR_DATA' ) {
            if( $data_cursor < $expected_sector_len ) {
               printf " | %02X: %02X ", $data_cursor, $byte_decoded if $debug eq 'AIM' && $debug_track == $track;

               # Agath codepage translation for printable symbols
               #
               if( ($byte_decoded & 0x7F) >= 0x20 ) {
                  print " ".chr($byte_decoded & 0x7F) if $debug eq 'AIM' && $debug_track == $track;
               } else {
                  print " ." if $debug eq 'AIM'  && $debug_track == $track;
               }
            }
            if( $data_cursor == $expected_sector_len ) {
               print " CRC" if $debug eq 'AIM'  && $debug_track == $track;
            }
            if( $data_cursor == $expected_sector_len+1 ) {

               # End of DATA
               #
               if( $byte_decoded != 0x5A ) {
                  print "\nERROR: Last DATA byte must be 0x5A\n";
               } else {
                  print " End of DATA Mark" if $debug eq 'AIM' && $debug_track == $track;
               }
            }
            if( $data_cursor == $expected_sector_len+2 ) {

               # GAP begin
               #
               $gap_len = 0;
               $gap_index = $track_len + 1;
               $decoder_state = 'PRE_GAP';
            }
            $data_cursor++;
         }
      }
   }
  }

   # End of GAP4
   #
   if( $gap_len > 0 ) {

      # End of GAP
      #
      $$aim_track[$gap_index] = $gap_len; # +1 to avoid clash with DESYNC command
      $aim_track_gap_total_len[$track] += $gap_len if $gap_len > $minimum_gap_len;
      $gap_len = 0;
   }

   # Shifting track data if INDEX Start AIM command found
   #
   if( $index_hole_ptr > 0 ) {

      print "\nINDEX Start AIM command is present on track $track";
      print "\nShifting track $track content ", $index_hole_ptr/2, " MFM bytes";

      # Shift DATA
      #
      $gap_len = 0;
      $shifts_after_last_gap = 0;
      for( $i = 0; $i <= $index_hole_ptr; $i += 2 ) {
         $shifts_after_last_gap++ if $shifts_after_last_gap >= 1;

         $aim_byte = shift $aim_track;
         push $aim_track, $aim_byte;

         $aim_byte = shift $aim_track;
         if( $aim_byte > $minimum_gap_len ) {
            $gap_len = $aim_byte;
            $shifts_after_last_gap = 1;
         }
         push $aim_track, $aim_byte;
      }

      # Correct GAP1 and GAP4 length
      #
      if( $shifts_after_last_gap < $gap_len ) {

         # TODO: GAP $shifts_after_last_gap =1 will be confused with DESYNC
         # this not suppose to happen. May be caused only by badly placed 0x03 command in AIM
         #
         print "\nERROR: GAP wrap of 1 byte will be confused with DESYNC" if $shifts_after_last_gap == 1;
         $$aim_track[1] = $gap_len - $shifts_after_last_gap;
         $gap_len = $shifts_after_last_gap;
         $$aim_track[ $#{$aim_track} - $shifts_after_last_gap*2 + 2 ] = $gap_len; 
         $gap_len = 0;
      } else {
         $gap_len = $shifts_after_last_gap;
      }

      # Correct $first_gap_index value
      #
      $first_gap_index = -1;
      for( $i = 0; $first_gap_index < 0 && $i < $#{$aim_track}+1; $i += 2 ) {
         $first_gap_index = $i if $$aim_track[$i+1] > $minimum_gap_len;
      }
   } 

   # Shift track data to prevent PAYLOAD crossing Index hole on disk
   # only if $fix_index_divided_sector_position = 1 or 2
   #
   if( $decoder_state ne 'DESYNC' ) { 

      print "\nWARNING: PAYLOAD crossing Index hole at track $track. Copy protection?";
      die "\nERROR: Please select option fix_index_divided_sector_position>0 or hfe_version=2, for correct track looping."
         if $ARGV[1] =~ /hfe$/i && $hfe_version == 1 && $fix_index_divided_sector_position == 0;

      print "\nShifting track $track content" if $fix_index_divided_sector_position > 0;

      # Move largest gap into beginning of the track
      #
      if( $fix_index_divided_sector_position == 1 ) {

         # Shift DATA forward to beginning of the largest gap
         #
         $aim_command = 0;
         while( $aim_command != $maximum_gap_len[$track] ) {
            $aim_byte = shift $aim_track;
            $aim_command = shift $aim_track;
            if( $aim_command != $maximum_gap_len[$track] ) {
               push $aim_track, $aim_byte;
               push $aim_track, $aim_command;
            }
         }
 
         # Redistribute GAP1 to GAP4
         #
         $gap_len = $aim_command;
         $aim_command = $GAP4_len;

         push $aim_track, $aim_byte;
         push $aim_track, $aim_command;
         for( $i = 0; $i < ($GAP4_len-1)*2; $i++) {
            push $aim_track,  shift $aim_track;
         }
         $$aim_track[1] = $gap_len - $GAP4_len;
         $aim_track_gap_total_len[$track] -= $GAP4_len if $GAP4_len <= $minimum_gap_len;
      }
    
      # Move split sector to the END of the track (Index + Data)
      #
      if( $fix_index_divided_sector_position == 2 ) {

         # Shift DATA forward to the beginning of GAP3
         # ASSUMING Index-Data gap < $minimum_gap_len+5
         #
         $aim_command = 0;
         while( $aim_command < $minimum_gap_len+5 ) {
            $aim_byte = shift $aim_track;
            $aim_command = shift $aim_track;
            if( $aim_command < $minimum_gap_len+5 ) {
               push $aim_track, $aim_byte;
               push $aim_track, $aim_command;
            }
         }

         # Redistribute GAP3 to GAP4
         #
         $gap_len = $aim_command;
         $aim_command = $minimum_gap_len;

         push $aim_track, $aim_byte;
         push $aim_track, $aim_command;
         for( $i = 0; $i < ($minimum_gap_len-1)*2; $i++) {
            push $aim_track,  shift $aim_track;
         }
         $$aim_track[1] = $gap_len - $minimum_gap_len;
         $aim_track_gap_total_len[$track] -= $minimum_gap_len;
      } 

      # Move split sector to the BEGINNING of the track
      # 
      if( $fix_index_divided_sector_position == 3 ) {

         # Shift DATA backward to the beginning of GAP3
         # ASSUMING Index-Data gap < $minimum_gap_len+5
         #
         $aim_command = 0;
         while( $aim_command < $minimum_gap_len+5 ) {
            $aim_command = pop $aim_track;
            $aim_byte = pop $aim_track;
            if( $aim_command < $minimum_gap_len+5 ) {
               unshift $aim_track, $aim_command;
               unshift $aim_track, $aim_byte;
            }
         }

         # Redistribute GAP3 to GAP4
         #
         $gap_len = $aim_command;
         $aim_command = $minimum_gap_len;

         push $aim_track, $aim_byte;
         push $aim_track, $aim_command;
         for( $i = 0; $i < ($minimum_gap_len-1)*2; $i++) {
            push $aim_track,  shift $aim_track;
         }
         $$aim_track[1] = $gap_len - $minimum_gap_len;
         $aim_track_gap_total_len[$track] -= $minimum_gap_len;

      }
   }

   # Replacing AIM track
   #
   $aim[$track] = $aim_track;
   print ".";
}

# Deflating AIM
# packing each track to exactly 6250 bytes
# decreasing GAP proportionaly
#
print "\nDeflating AIM ";
my $deflation_rate, $rounding_error;
for( $track = 0; $track < 160; $track++ ) {
   $track_len = 0;
   $aim_track = [];
   $first_gap_index = -1;

   # GAP deflation rate
   #
   $deflation_rate = ($aim_track_gap_total_len[$track] - ( ($#{$aim[$track]}+1) - 6250 * 2 )/2)
                   / $aim_track_gap_total_len[$track];
   $rounding_error = 0;

   # Find last big GAP for rounding error correction with EXTRA_GAP
   #
   $last_gap_index = -1;
   for( $i = $#{$aim[$track]}; $last_gap_index < 0 && $i > 0; $i -= 2 ) {
      $last_gap_index = $i-1 if $aim[$track][$i] > $minimum_gap_len;
   }


   $decoder_state = 'PAYLOAD';
   print "\n================= TRACK $track deflation  ======================" if $debug eq 'GAP' && $debug_track == $track;

   for( $i = 0; $i < $#{$aim[$track]}+1 ; $i += 2 ) {
      ( $aim_byte, $aim_command ) = ( $aim[$track][$i], $aim[$track][$i+1] );

      print "\n$decoder_state " if $debug eq 'GAP'  && $debug_track == $track;
      printf "%04X: %02X %02X", $i/2, $aim_byte, $aim_command if $debug eq 'GAP' && $debug_track == $track;

      # Only GAP's longer that $minimum_gap_len bytes deflated
      #
      if( $aim_command > $minimum_gap_len ) {
         $first_gap_index = $i if $first_gap_index < 0;
         $gap_len = $aim_command;
         $deflated_gap_len = int( $gap_len * $deflation_rate );
         $rounding_error += $gap_len * $deflation_rate - $deflated_gap_len;
         if( $rounding_error > 1 || ( $rounding_error > 0 && $last_gap_index == $i ) ) {
            $deflated_gap_len++;
            $rounding_error--;
         }

         $decoder_state = 'GAP';
         print " GAP=$gap_len DEF=$deflated_gap_len" if $debug eq 'GAP' && $debug_track == $track;
      } else {
         print " GAP=$aim_command Ignored" if $aim_command > 1 && $debug eq 'GAP' && $debug_track == $track;
      } 

      print " DESYNC" if $aim_command == 1 && $debug eq 'GAP' && $debug_track == $track;
      $aim_command = 0 if $aim_command > 1;
      
      if( $decoder_state eq 'GAP' ) { 
         push $aim_track, ( $aim_byte, $aim_command );
         $gap_len--;
         $deflated_gap_len--;
         $track_len += 2;
         $decoder_state = 'SKIP' if $deflated_gap_len == 0;
         $decoder_state = 'PAYLOAD' if $gap_len == 0;
      }
      elsif ( $decoder_state eq 'SKIP' ) {
         printf "\nWARNING: Track=$track GAP byte %02X != 0xAA. AIM CMD=$aim_command. Payload in GAP?", $aim_byte if $aim_byte != 0xAA;
         $gap_len--;
         $decoder_state = 'PAYLOAD' if $gap_len == 0 && $i+1 != $#{$aim[$track]};
      } else {
         push $aim_track, ( $aim_byte, $aim_command );
         $track_len += 2;
      }
   }

   # MFM track 250 kbps at 300 RPM is exactly 6250 bytes
   # Not supposed to happen !!!
   #
   die "ERROR: Track $track length ", $track_len/2, " must be exactly 6250 bytes" if $track_len != 6250*2;

   # Replacing AIM track
   #
   $aim[$track] = $aim_track;
   print ".";

if( $debug eq 'DEF' && $debug_track == $track ) {
   print "\n============== TRACK $track after gap deflation len=$track_len ", $#{$aim[$track]}+1, "\n";
   for( $i = 0; $i < $track_len; $i +=2 ) {
      ( $aim_byte, $aim_command ) = ( $aim[$track][$i], $aim[$track][$i+1] );
      printf "%04X: %02X %02X\n", $i/2, $aim_byte, $aim_command;
   }
}

}

# Save new AIM file
#
if( $ARGV[1] =~ /aim$/i) {
   print "\nSaving AIM ";
   for( $track = 0;  $track < 160; $track++ ) {
      print ".";
      $aim_track_len = ( $#{$aim[$track]}+1 ) / 2;
      $padding = (6464 - $aim_track_len) * 2;
      $aim[$track][ $#{$aim[$track]} ] = 2;
      print OUT pack( 'C*', @{$aim[$track]} );
      print OUT pack( "C$padding", 0);
   }
   close OUT;
   die "\nDone.";
}

# Generate MFM code
#
print "\nEncoding MFM ";
my @mfm_code;
$mfm_code[0][1] = 2 << 6;
$mfm_code[0][0] = 1 << 6;
$mfm_code[1][0] = 0 << 6;
$mfm_code[1][1] = 2 << 6;

my $desync_bit = 6; # DESYNC bit position

my $prev_bit, $bit_count;
my $mfm_track;
for( $track = 0;  $track < 160; $track++ ) {
   print ".";
   $prev_bit = 0; 

   print "===================== Track $track =========================\n" if $debug eq 'MFM';
   for( $i = 0;  $i < $#{$aim[$track]}+1; $i++ ) {
      ( $aim_byte, $aim_command ) = ( $aim[$track][$i], $aim[$track][$i+1] );
      printf "%04X: AIM %02X %02X", $i/2, $aim_byte, $aim_command if $debug eq 'MFM';

      if( $aim_command == 1 ) {
         # dirty hack need to be fixed
         $mfm_track[$track][$i++] = 0x22;
         $mfm_track[$track][$i] = 0x09;
         $mfm_track[$track][$i-1] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i-1]
            if $bit_order ne 'HFE';
         $mfm_track[$track][$i] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i]
            if $bit_order ne 'HFE';
      } else {
      $bit_count = 0;
      $mfm_track[$track][$i] = 0;
      while( $bit_count++ < 8 ) {
         $bit = ($aim_byte & 0x80) >> 7;
         $aim_byte = $aim_byte << 1;
         if( $bit_count == 5 ) {
            #
            # Advance to next MFM byte
            #
            $mfm_track[$track][$i] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i]
               if $bit_order ne 'HFE';
            $i++;
            $mfm_track[$track][$i] = 0;
         } else {
            $mfm_track[$track][$i] = $mfm_track[$track][$i] >> 2;
         }
         if ( !($aim_command == 1 && $bit_count == $desync_bit ) ) {
            $mfm_track[$track][$i] = $mfm_track[$track][$i] | $mfm_code[$prev_bit][$bit];
         }
         $prev_bit = $bit;
      }
      $mfm_track[$track][$i] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i]
         if $bit_order ne 'HFE';
      }
      printf " MFM %02X %02X\n", $mfm_track[$track][$i-1], $mfm_track[$track][$i] if $debug eq 'MFM';
   }
}

# Generating Output File
#
if( $ARGV[1] =~ /hfe$/i) {
   print "\nCreating HFE ";
   if( $hfe_version == 1 ) {
      HxC::hxc_hfe_file( \*OUT, \@mfm_track, 2 );
   } else {
      HxC::hxc_hfe_v2_file( \*OUT, \@mfm_track, 2 );
   }
} elsif( $ARGV[1] =~ /mfm$/i) {
   print "\nCreating MFM ";
   HxC::hxc_mfm_file( \*OUT, \@mfm_track, 2 );

} else {
   die "ERROR: Unknown target file type.";
}

print "\nDone!\n";

close(OUT);
