#!/usr/bin/perl
#
# Encoding data from Agat DSK file to HxC HFE or MFM files
# or AIM file for simulator
#
# 840k DSK files supported:
# - 256 bytes header
# - 128 byte epilogue
# -   4 byte epilogue
# - clean 860160 files with NO header or epilogue
#
# 140k DSK files supported
# - 128 byte epilogue
# - 4 byte epilogue
# - clean 143360 files with NO header or epilogue
#
# (c) 2014 PortaOne, Inc.
# Oleksandr Kapitanenko kapitan@portaone.com
#
# Usage:
# agath-dsk-to-hfe.pl <dsk file> <mfm, hfe or aim file>
# Output file type selected by extension hfe or mfm.
#

use POSIX qw(ceil);
use File::Basename qw(dirname);
use Cwd qw(abs_path);
use lib dirname(dirname abs_path $0);
use HxC;
$| = 1;

# Debug flag
#
#my $debug = 'MFM';

die "Output and input files are the same!" if $ARGV[0] eq $ARGV[1];
die "Target file extension must be HFE, MFM or AIM!" if $ARGV[1] !~ /mfm|hfe|aim$/i;
if( $ARGV[1] =~ /hfe$/i) {
   $bit_order = 'HFE';
} else {
   $bit_order = 'MFM';
}

unlink($ARGV[1]);
open(IN, "<$ARGV[0]")
   || die "File $ARGV[0] not found.";
open(OUT, ">$ARGV[1]")
   || die "Can not create output file $ARGV[1]";

# Read DSK file
#
my $DSK140 = 0;
my $dsk_size = -s $ARGV[0];
if( $dsk_size == 860160+256 ) {
    print "DSK 840kB with 256 byte header.\n";
    sysread(IN, $buff, $dsk_size-860160);
}
elsif( $dsk_size == 860160+128 ) {
    print "DSK 840kB with 128 byte epilogue.\n";
}
elsif( $dsk_size == 860160+4 ) {
    print "DSK 840kB with 4 byte epilogue.\n";
}
elsif( $dsk_size == 860160 ) {
    print "DSK 840kB with NO header or epilogue.\n";
}
elsif( $dsk_size == 143360 ) {
    print "DSK 140kB with NO header or epilogue.\n";
    $DSK140 = 1;
}
elsif( $dsk_size == 143360+4 ) {
    print "DSK 140kB with 4 byte epilogue.\n";
    $DSK140 = 1;
}
elsif( $dsk_size == 143360+128 ) {
    print "DSK 140kB with 128 byte epilogue.\n";
    $DSK140 = 1;
} else {
    die "Not DSK file?";
}

print "Reading DSK  ";
my @dsk, $buff, $dsk_track, $track;
if( $DSK140 ) {
   for( $track = 0; $track < 35; $track++ ) {
      $dsk_track = [];
      sysread(IN, $buff, 256*16 ); # Track 256 bytes * 16 sectors
      push $dsk_track, unpack( 'C4096', $buff);
      $dsk[$track] = $dsk_track;
      print ".";
   }
} else {
   for( $track = 0; $track < 160; $track++ ) {
      $dsk_track = [];
      sysread(IN, $buff, 256*21 ); # Track 256 bytes * 21 sectors
      push $dsk_track, unpack( 'C5376', $buff);
      $dsk[$track] = $dsk_track;
      print ".";
   }
}
close(IN);

# Expanding DSK to AIM
#
print "\nExpanding to AIM  ";
my @aim, $aim_track, $sector, $dsk_byte, $data_byte;
for( $track = 0; $track < 160; $track++ ) {
   $aim_track = [];
   $dsk_byte = 0;

   # GAP1 13 bytes
   #
   push $aim_track, ( 0xAA, 0) x 13;

   # unformated MFM track at 300RPM = 6250 bytes 
   #
   for($sector = 0; $sector < 21; $sector++) {

      # DESYNC
      #
      if( $ARGV[1] =~ /aim$/i) {
         push $aim_track, ( 0xA4, 1 );
      } else {
         push $aim_track, ( 0xA4, 1, 0xFF, 0 );
      }

      # INDEX mark
      #
      push $aim_track, ( 0x95, 0, 0x6A, 0 );
   
      # Volume number, Track Number, Sector Number
      #
      push $aim_track, ( 0xFE, 0, $track, 0, $sector, 0 ); 
   
      # End of INDEX mark
      #
      push $aim_track, ( 0x5A, 0 );

      # GAP2 5 bytes 0XAA
      #
      push $aim_track, ( 0xAA, 0 ) x 5;

      # DESYNC
      #
      if( $ARGV[1] =~ /aim$/i) {
         push $aim_track, ( 0xA4, 1 );
      } else {
         push $aim_track, ( 0xA4, 1, 0xFF, 0 );
      }

      # DATA mark
      #
      push $aim_track, ( 0x6A, 0, 0x95, 0 );

      # DATA
      #
      $crc = 0;

      for ($count = 256; $count >= 1; $count--) {
         if( $DSK140 && $sector > 15 ) {
            $data_byte = 0;
         } else {
            $data_byte = $dsk[$track][$dsk_byte++];
         }

         if( $crc > 0xFF ) {
           $crc++;
           $crc &= 0xFF;
         }
         $crc += $data_byte;

         push $aim_track, ( $data_byte, 0 );
      }
      $crc &= 0xFF;

      # CRC
      #
      push $aim_track, ( $crc, 0 );

      # End of DATA
      #
      push $aim_track, ( 0x5A, 0 );

      # GAP3 22 bytes 0XAA
      #
      push $aim_track, ( 0xAA, 0) x 22;
   }

   # Total of 6250 payload bytes

   $aim[$track] = $aim_track;
   $aim_track_len = ( $#{$aim[$track]}+1 ) / 2;
  
   # MFM track 250 kbps at 300 RPM is exactly 6250 bytes
   # 
   die "ERROR: Track length $aim_track_len must be exactly 6250 bytes"
       if $aim_track_len != 6250 && $ARGV[1] !~ /aim$/i;
   print ".";
}

# Save new AIM file
#
if( $ARGV[1] =~ /aim$/i) {
   print "\nSaving AIM ";
   for( $track = 0;  $track < 160; $track++ ) {
      print ".";
      $aim_track_len = ( $#{$aim[$track]}+1 ) / 2;
      $padding = (6464 - $aim_track_len) * 2;
      $aim[$track][ $#{$aim[$track]} ] = 2;
      print OUT pack( 'C*', @{$aim[$track]} );
      print OUT pack( "C$padding", 0);
   }
   close OUT;
   die "\nDone.";
}

# Generate MFM code
#
print "\nEncoding MFM ";
my @mfm_code;
$mfm_code[0][1] = 2 << 6;
$mfm_code[0][0] = 1 << 6;
$mfm_code[1][0] = 0 << 6;
$mfm_code[1][1] = 2 << 6;

my $desync_bit = 6; # DESYNC bit position

my $prev_bit, $bit_count;
my $mfm_track;
for( $track = 0;  $track < 160; $track++ ) {
   print ".";
   $prev_bit = 0; 
   print "===================== Track $track =========================\n" if $debug;

   for( $i = 0;  $i < $aim_track_len*2; $i++ ) {
      ( $aim_byte, $aim_command ) = ( $aim[$track][$i], $aim[$track][$i+1] );
      printf "%04X: AIM %02X %02X", $i/2, $aim_byte, $aim_command if $debug;

      if( $aim_command == 1 ) {
         # dirty hack need to be fixed
         $mfm_track[$track][$i++] = 0x22;
         $mfm_track[$track][$i] = 0x09;
         $mfm_track[$track][$i-1] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i-1]
            if $bit_order ne 'HFE';
         $mfm_track[$track][$i] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i]
            if $bit_order ne 'HFE';
      } else {
      $bit_count = 0;
      $mfm_track[$track][$i] = 0;
      while( $bit_count++ < 8 ) {
         $bit = ($aim_byte & 0x80) >> 7;
         $aim_byte = $aim_byte << 1;
         if( $bit_count == 5 ) {
            #
            # Advance to next MFM byte (1 Data Byte = 2 MFM Bytes)
            #
            $mfm_track[$track][$i] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i]
               if $bit_order ne 'HFE';
            $i++;
            $mfm_track[$track][$i] = 0;
         } else {
            $mfm_track[$track][$i] = $mfm_track[$track][$i] >> 2;
         }
         if ( !($aim_command == 1 && $bit_count == $desync_bit ) ) {
            $mfm_track[$track][$i] = $mfm_track[$track][$i] | $mfm_code[$prev_bit][$bit];
         }
         $prev_bit = $bit;
      }
      $mfm_track[$track][$i] = unpack 'C', pack 'b8', unpack 'B8', pack 'C', $mfm_track[$track][$i]
         if $bit_order ne 'HFE';
      }
      printf " MFM %02X %02X\n", $mfm_track[$track][$i-1], $mfm_track[$track][$i] if $debug;

   }
}

# Generating Output File
#
if( $ARGV[1] =~ /hfe$/i) {
   print "\nCreating HFE ";
   HxC::hxc_hfe_file( \*OUT, \@mfm_track, 2 );

} elsif( $ARGV[1] =~ /mfm$/i) {
   print "\nCreating MFM ";
   HxC::hxc_mfm_file( \*OUT, \@mfm_track, 2 );

} else {
   die "ERROR: Unknown target file type.";
}
print "\nDone!\n";

close(OUT);

