#!/usr/bin/perl
#
# Decoding MFM stream from HxC hfe files one track at a time.
# Only Agath MFM encoding supported
# On Agath track 17 ( track 8 side 1 ) VTOC and CATALOG is decoded
#
# (c) 2014 PortaOne, Inc.
# kapitan@portaone.com
#
# Usage:
#
# Assuming Track 0 Side 0
# hfe-agath-decode.pl <hfe file>
#
# Assuming Agath / Apple 2 Track number
# hfe-agath-decode.pl <hfe file> <Agath track number>
#
# Physical disk track number and side number
# hfe-agath-decode.pl <hfe file> <track number> <side number>
#

my $track_number, $side_number;
$track_number = ($ARGV[1] == undef)? 0 : $ARGV[1];
if( $ARGV[2] == undef ) {
   $side_number = $track_number & 0x01;
   $track_number = $track_number >> 1;
} else {
   $ARGV[2] eq '0' || $ARGV[2] eq '1' || die "Invalide side number '$ARGV[2]'";
   $side_number = $ARGV[2];
}

open(IN, "<$ARGV[0]") 
   || die "File $ARGV[0] not found.";


my $byte, $next_byte;

# Read file header
#
my $header_buff;
sysread(IN, $header_buff, 0x200);
( $file_signature,
  $file_rev,
  $file_track_number,
  $file_side_number,
  $file_encoding,
  $file_bitrate,
  $file_rpm,
  $file_interface_mode,
  $file_write_protected,
  $file_track_list_offset,
  $file_write_allowed,
  $file_single_step,
  $file_track0s0_altencoding,
  $file_track0s0_encoding,
  $file_track0s1_altencoding,
  $file_track0s1_encoding )  = unpack ( 'A8 C C C C S S C C S C C C C C C', $header_buff);

$file_signature eq 'HXCPICFE' 
   || die "Not HXCPICFE file.";

print "HXCPICFE file $ARGV[0] Revision=$file_rev\n";
print "Tracks=$file_track_number Sides=$file_side_number Bitrate=$file_bitrate Kbit/s MFM Encoding\n\n";


$track_number >=0 && $track_number <= $file_track_number-1 
   || die "Invalide track '$track_number' requested";

# Read track allocation table
#
my $table;
if( $file_track_list_offset*512-512 != 0 ) {
   sysread(IN, $table, $file_track_list_offset*512-512);
}
sysread(IN, $table, 0x200);
my @tracks = unpack( 'S160', $table);

# How many zero bytes to look in SYNC
#
my $sync_len = 8;

# Track length MFM encoded two sides in hfe file 256 bytes interlived
#
my $track_len = $tracks[$track_number*2+1];

# Track offset in hfe file
#
my $track_offset = $tracks[$track_number*2] * 512 - 1024;
if( $track_offset ) {
   sysread(IN, $buff, $track_offset);
}

# Reading track data from hfe file
# Extracting required side
#
my $i = 0;
while( $i < $track_len ) {
   if( $side_number == 1 ) {
      sysread(IN, $buff, 256);
   }
   sysread(IN, $buff, 256);
   push @track, unpack('C256', $buff);
   if( $side_number == 0 ) {
      sysread(IN, $buff, 256);
   }
   $i += 512;
}

# Track length MFM encoded
#
$track_len = $track_len/2;

# Forming bitflow from track data
# In reversed order
#
# HxC  bits transmitting order to the FDC is :
# Bit 0-> Bit 1-> Bit 2-> Bit 3-> Bit 4-> Bit 5-> Bit 6-> Bit 7->(next byte)
#
my @bitflow = ();
$i = 0;
while( $i < $track_len ) {
   my $bit = 8;
   my $byte = $track[$i++];
   while( $bit-- ) {
      push @bitflow, $byte & 0x01;
      $byte = $byte>>1;
   }
}

# Track length dencoded
#
$track_len = $track_len/2;

# Start decoding
#
print "Decoding data from track $track_number side $side_number\nGAP1 Looking for SYNC\n";
my @decoded_track =();
my $decoder_state = 'DESYNC';
my $prev_bit = 0;
my $expected_sector_len, $sector_count = 0, $gap_len = -1;
my @sector_len = ( 0, 256, 512, 1024 );

my @mfm_code;
$mfm_code[0][0][0] = 3; # DESYNC bit
$mfm_code[0][0][1] = 1;
$mfm_code[0][1][0] = 0;
$mfm_code[1][0][0] = 0;
$mfm_code[1][0][1] = 1;
$mfm_code[1][1][0] = 4; # ??? DESYNC bit

my @desync_marker = ( 1,0,0,0,1,0,0,1,0,0,0,1,0,0,1 );
my @desync_buff = ( 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 );
my $desync_cursor, $idam_cursor, $data_cursor, $crc_buff, $crc, $hfe_crc, $bit_counter, $byte_decoded, $track_number, $sector_number, $sector_data, $cat_track = -1, $cat_sector =-1, $cat_block = 0;

$count = $track_len;
$i = 0;
my $sync_cursor = $sync_len;
while( $count-- > 0 ) {
START:
   $bit_counter = 8;
   $byte_decoded = 0;
   while( $bit_counter-- ) {
      $byte_decoded = $byte_decoded << 1;
      $prev_bit = $mfm_code[$prev_bit][$bitflow[$i]][$bitflow[$i+1]];


      # Unexpected DESYNC bit handling
      #
      if( $prev_bit == 3 || $prev_bit == 4 ) {
         if( $decoder_state ne 'DESYNC' ) {
            print "\nUnexpected DESYNC bit $prev_bit found, write splice?\n";
         } else {
            print "\nDESYNC at bit $bit_counter\n";
            $decoder_state = 'DESYNC_2';
         }
         $prev_bit = 0;
      }

      # Looking for first non zero bit after SYNC 
      #
      if( $decoder_state eq 'SYNC_2' && $bit_counter != 8 && $prev_bit != 0 ) {
        print "\nByte aligned $bit_counter bits";
        if( $gap_len > 1 ) {
           $gap_len += $sync_len;
           print "\nSYNC length $gap_len bytes";
        }
        $gap_len = 0;
        $decoder_state = 'DESYNC';
        $prev_bit = 0;
        @desync_buff = ( 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 ); 
        goto START; 
      }

      # Search for desync marker 100010010301001 (15 bit)
      #
      push @desync_buff, $bitflow[$i], $bitflow[$i+1];
      splice(@desync_buff, 0, 2);
      $desync_cursor = 0;
      while( $desync_cursor < 15 ) {
         if( $desync_marker[$desync_cursor] != $desync_buff[$desync_cursor] ) {
            $desync_cursor = 25; 
         } else {
            $desync_cursor++;
         }
      }
      
      # Desync marker found. Byte alignment
      # 
      if( $desync_cursor != 25 ) {
         if( $decoder_state eq 'DESYNC' ) {
            print "\nDESYNC sequence found\n";
         } else {
            print "Unexpected DESYNC sequence found, write splice?\n";
         }
      }
      $byte_decoded = $byte_decoded | $prev_bit;
      $i += 2;
   }

   if( $decoder_state eq 'DESYNC' 
    || $decoder_state eq 'DESYNC_2'
    || $decoder_state eq 'DESYNC_3'
    || $decoder_state eq 'SYNC' 
    || $decoder_state eq 'SYNC_2' ) {
      $gap_len++;
   }

   # 0x00 SYNC Seqense search and phase adjustment
   #
   if( $decoder_state eq 'SYNC'
    && ($byte_decoded == 0x00 || $byte_decoded == 0xFF) ) {
      $sync_cursor--;
   } else {
      $sync_cursor = $sync_len;
   }
   
   if( $decoder_state eq 'SYNC' && $sync_cursor == 0 ) {
      $gap_len -= $sync_len;
      print "\nGAP length $gap_len bytes";
      $gap_len = 0;
      print "\nSYNC ";
      if( $byte_decoded == 0xFF ) {
         $i--;
         $prev_bit = 0;
         print "Phase adjusted. ";
      }
      print "Looking for first non zero bit\n";
      $sync_cursor = $sync_len;
      $crc_buff = '';
      $decoder_state = 'SYNC_2';
   }

      # Second DESYNC byte eaten by Agath controller - may be any value
      #
      if( $decoder_state eq 'DESYNC_2' ) {
         $decoder_state = 'DESYNC_3';
      }
      elsif( $decoder_state eq 'DESYNC_3' ) {
         if( $byte_decoded != 0xFF ) {
            print "\n Unexpected byte $byte_decoded after DESYNC";
         }
         $decoder_state = 'MARKER';
      }

      # INDEX Marker 0x95 0x6A or DATA Marker 0x6A 0x95
      #
      elsif( $decoder_state eq 'MARKER' ) {
         if( $byte_decoded == 0x95 ) {
            $gap_len --;
            print "\nGAP length $gap_len bytes\n";
            $decoder_state = 'MARKER_2';
            $idam_cursor = 0;
            $sector_count++;
         }
         elsif( $byte_decoded == 0x6A ) {
            $gap_len --;
            print "\nGAP length $gap_len bytes\n";
            $decoder_state = 'MARKER_2';
            $data_cursor = 0;
         } else {
            print "\nUnexpected byte after DESYNC";
            print "\nGAP3 Continued\n";
            $decoder_state = 'DESYNC';
         }
      }
      elsif( $decoder_state eq 'MARKER_2' ) {
         if( $byte_decoded == 0x6A ) {
            print "\nINDEX Marker found\n";
            $decoder_state = 'SECTOR_HEADER';
            $crc_buff = "";
            $idam_cursor = 0;
            $sector_count++;
            $gap_len = 0;
         }
         elsif( $byte_decoded == 0x95 ) {
            $decoder_state = 'SECTOR_DATA';
            $data_cursor = 0;
            $crc = 0;
            $crc_sprite = 0;
            $crc_buff = "";
            $gap_len = 0;
            $sector_data = [];
            print "\nDATA Marker found\nDATA $expected_sector_len bytes:\n";
         } else {
            print "\nUnexpected byte in INDEX or DATA Mark";
            print "\nGAP3 Continued\n";
            $decoder_state = 'DESYNC';
         } 
      }

   # INDEX Processing
   #
   # INDEX is 4 bytes: Volume, Track, Sector, 0x5A
   #
   elsif( $decoder_state eq 'SECTOR_HEADER' ) {
      $idam_cursor++;
      if( $idam_cursor == 1 ) {
	 printf "\nVolume =%02X\n", $byte_decoded;
      }
      if( $idam_cursor == 2 ) {
         $track_number = $byte_decoded;
         print "\nTrack number=$byte_decoded\n";
      }
      if( $idam_cursor == 3 ) {
         $sector_number = $byte_decoded;
         print "\nSector number=$byte_decoded\n";
      }
      $expected_sector_len = 256;
      if( $idam_cursor == 4 ) {
         if( $byte_decoded != 0x5A ) {
            print "\nERROR: Last INDEX byte must be 0x5A\n";
         }
      }
      if( $idam_cursor == 5 ) {
         $gap_len = 0;
         print "\nGAP2\n";
         $decoder_state = 'DESYNC';
      }

   }

   # DATA Processing
   #
   elsif ( $decoder_state eq 'SECTOR_DATA' ) {
      if( $data_cursor < $expected_sector_len ) {
         push $sector_data, $byte_decoded;
         if( $crc > 0xFF ) {
           $crc++;
           $crc &= 0xFF;
         }
         $crc += $byte_decoded;
         $crc_sprite ^= $byte_decoded;
      } else {
        $crc &= 0xFF;
      }

      if( $data_cursor < $expected_sector_len && ($data_cursor & 0x000F) == 0 ) {
         printf "\n%04X: ", $data_cursor;
      }
      if( $data_cursor == $expected_sector_len ) {
         print "\nCRC\n";
         $hfe_crc = $byte_decoded;
      }
      if( $data_cursor == $expected_sector_len+1 ) {
         printf "\nDecoded    CRC=0x%02X", $hfe_crc;
         printf "\nCalculated CRC=0x%02X", $crc;
         if( $hfe_crc != $crc ) {
            print " CRC ERROR !!! ";
            printf "\nSprite ??? CRC=0x%02X", $crc_sprite;
            if( $hfe_crc != $crc_sprite ) {
               print " CRC ERROR !!! \n"
            }
         }
         print "\n";
         if( $byte_decoded != 0x5A ) {
            print "ERROR: Last DATA byte must be 0x5A\n";
         }
      
         # Sector content analysis
         #
      
         # VTOC is always in track 0x11 (physical track 8 side 1) sector 0x00
         #
         if( $track_number == 0x11 && $sector_number == 0x00 ) {
            print "\n========================= VTOC Found =========================";
            print "\nFirst catalog block track number $$sector_data[1] ( Physical TR=", int($$sector_data[1]/2), " S=", $$sector_data[1] - int($$sector_data[1]/2)*2, " )"; 
            $cat_track = $$sector_data[1];
            print "\nFirst catalog block sector number $$sector_data[2]";
            $cat_sector = $$sector_data[2];
            printf "\nDisk Volume 0x%02X", $$sector_data[0x6];
            print "\nMax. Num. of track/sector pairs $$sector_data[0x27]";
            print "\nAllocation Track Number $$sector_data[0x30]";
            print "\nDirection of track allocation (+1 or -1) $$sector_data[0x31]";
            print "\nNumber of tracks on disk $$sector_data[0x34] ( Physical ", $$sector_data[0x34]/2, " tracks, 2 sides )";
            print "\nNumber of sectors per track $$sector_data[0x35]";
            print "\nNymber of bytes per sector ", $$sector_data[0x37]*256+$$sector_data[0x36];
            print "\n==============================================================\n";  
         }

         # CATALOG block
         #
         if( $track_number == $cat_track && $sector_number == $cat_sector ) {
            $cat_block++;
            print "\n======================= CATALOG Block $cat_block Found =======================";
            if( $$sector_data[1] == 0 && $$sector_data[2] == 0 ) {
               print "\nThis CATALOG block is the last one";
               undef $cat_track, $cat_sector;
            } else {
               print "\nNext catalog block track number $$sector_data[1] ( Physical TR=", int($$sector_data[1]/2), " S=", $$sector_data[1] - int($$sector_data[1]/2)*2, " )";
               $cat_track = $$sector_data[1];
               print "\nNext catalog block sector number $$sector_data[2]";
               $cat_sector = $$sector_data[2];
            }

            $file_type[0x40] = " D";
            $file_type[0x20] = " K";
            $file_type[0x10] = " P";
            $file_type[0x04] = " B";
            $file_type[0x02] = " A";
            $file_type[0x00] = " T";
            $file_type[0xC0] = "*D";
            $file_type[0xA0] = "*K";
            $file_type[0x90] = "*P";
            $file_type[0x84] = "*B";
            $file_type[0x82] = "*A";
            $file_type[0x80] = "*T";

            $cat_ptr = 0x0B;
            print "\n\nAlloc Track Sector\tType    Blocks Name\n";
            for( $file_count = 0; $file_count < 7; $file_count++ ) {
               printf "        %03D     %02D\t", $$sector_data[$cat_ptr], $$sector_data[$cat_ptr+1];
               printf "0x%02X ", $$sector_data[$cat_ptr+2];
               print $file_type[ $$sector_data[$cat_ptr+2] ];
               printf "    %03D ", $$sector_data[$cat_ptr+0x22]*256+$$sector_data[$cat_ptr+0x21];
               for( $file_name_ptr = 0; $file_name_ptr <30; $file_name_ptr++ ) {
                  $agat_char_code = $$sector_data[$cat_ptr+3+$file_name_ptr];

                  # Agath codepage translation for printable symbols
                  #
                  if( ($agat_char_code & 0x7F) >= 0x20 ) {
                     print chr($agat_char_code & 0x7F);
                  } else {
                     print ".";
                  }
               }
               print "\n";
               $cat_ptr += 0x23;
            }
            print "\n=====================================================================\n";
         }
      }
      if( $data_cursor == $expected_sector_len+2 ) {
         $gap_len = 0;
         if( $sector_count == 10 ) {
            print "\nGAP4\n";
         } else {
            print "\nGAP3\n";
         }
         $decoder_state = 'DESYNC';
      }

      $data_cursor++;
   }
   push @decoded_track, $byte_decoded;
   printf "%02X ", $byte_decoded;
}

if( $decoder_state eq 'DESYNC' ) {
   print "\nGAP4 length ", $gap_len+1, " bytes \n";
}
elsif( $decoder_state eq 'SECTOR_DATA' ) {
   print "\nUnexpected track end in DATA block.";
   print "\nOnly  $data_cursor of $expected_sector_len bytes received.\n";
} else {
   print "\nUnexpected track end in state $decoder_state\n";
}

close(IN);
