//
//  psg.c
//
//  AY-3-8910/12 Emulation
//
//  Created by Alexander Medvedev on 05/06/14.
//  Copyright (c) 2014 Alexander Medvedev. All rights reserved.
//

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "types.h"

#include "ay8910.h"

void ay8910init(ay8910state * ay8910, int samples)
{
    if (samples == 0) return;
    
    memset(ay8910, 0, sizeof(ay8910state));
    
    ay8910->EnvelopeStep = -1;
    
    ay8910->OutputA = malloc(samples * sizeof(ay8910->OutputA[0]));
    ay8910->OutputB = malloc(samples * sizeof(ay8910->OutputA[0]));
    ay8910->OutputC = malloc(samples * sizeof(ay8910->OutputA[0]));
    
    ay8910->Samples = samples;
    
    ay8910->Mix[0][0] = 30;
    ay8910->Mix[0][1] = 30;
    ay8910->Mix[0][2] = 30;
    ay8910->Mix[1][0] = 30;
    ay8910->Mix[1][1] = 30;
    ay8910->Mix[1][2] = 30;
    
    ay8910->R[14] = 0xFF;       // TODO
    ay8910->R[15] = 0xFF;

    ay8910->DebugName = "PSG";
    
#ifdef PSGDUMP
    printf("%s INIT\n", ay8910->DebugName);
#endif

}

void ay8910done(ay8910state * ay8910)
{

#ifdef PSGDUMP
    if (ay8910->DebugInfo) printf("%s DONE\n", ay8910->DebugName);
#endif

    free(ay8910->OutputA);
    free(ay8910->OutputB);
    free(ay8910->OutputC);
    
    memset(ay8910, 0, sizeof(ay8910state));
}

void ay8910write(ay8910state * ay8910, int addr, byte data)
{

#ifdef PSGDUMP
    if (ay8910->DebugInfo>1) printf("%s WRITE R[%d]=%02X\n", ay8910->DebugName, addr, data);
#endif

    ay8910->R[addr & 15] = data;
}

byte ay8910read(ay8910state * ay8910, int addr)
{
    
#ifdef PSGDUMP
    if (ay8910->DebugInfo>1) printf("%s READ  R[%d]=%02X\n", ay8910->DebugName, addr, ay8910->R[addr]);
#endif
    
    return ay8910->R[addr & 15];
}

byte PSGVolumes[16] = { 0, 1, 2, 3, 4, 6, 8, 12, 16, 24, 32, 48, 64, 96, 128, 224 };

// PSG Envelope shape parts
#define PSGSHAPEUP    1
#define PSGSHAPEDOWN  2
#define PSGSHAPEHIGH  3
#define PSGSHAPELOW   4

int VolumeShapes[16][3] = {
    { PSGSHAPEDOWN, PSGSHAPELOW,  0 },   // 0000 \_______
    { PSGSHAPEDOWN, PSGSHAPELOW,  0 },   // 0001
    { PSGSHAPEDOWN, PSGSHAPELOW,  0 },   // 0010
    { PSGSHAPEDOWN, PSGSHAPELOW,  0 },   // 0011
    { PSGSHAPEUP,   PSGSHAPELOW,  0 },   // 0100 /_______
    { PSGSHAPEUP,   PSGSHAPELOW,  0 },   // 0101
    { PSGSHAPEUP,   PSGSHAPELOW,  0 },   // 0110
    { PSGSHAPEUP,   PSGSHAPELOW,  0 },   // 0111
    { PSGSHAPEDOWN, PSGSHAPEDOWN, 1 },   // 1000 \\\\\\\\ Cyclic
    { PSGSHAPEUP,   PSGSHAPEUP,   0 },   // 1001 \_______
    { PSGSHAPEDOWN, PSGSHAPEUP,   1 },   // 1010 \/\/\/\/ Cyclic
    { PSGSHAPEUP,   PSGSHAPEDOWN, 1 },   // 1011 \-------
    { PSGSHAPEUP,   PSGSHAPEUP,   1 },   // 1100 //////// Cyclic
    { PSGSHAPEUP,   PSGSHAPEDOWN, 0 },   // 1101 /-------
    { PSGSHAPEUP,   PSGSHAPEDOWN, 1 },   // 1110 /\/\/\/\ Cyclic
    { PSGSHAPEUP,   PSGSHAPEDOWN, 0 }    // 1111 /_______
};

// AY-3-8910 Regs:
//--------------------------------------------
//  0: FAL [ F7 F6 F5 F4 F3 F2 F1 F0 ]
//  1: FAH [ X X X X | F11 F10 F9 F8 ]
//  2: FBL
//  3: FBH
//  4: FCL
//  5: FCH
//  6: FN  [ X X X | F4 F3 F2 F1 F0 ]
//  7: MIX [ InB InA | NC NB NA | FC FB FA ]
//  8: VA  [ X X X | E | V3 V2 V1 V0 ]
//  9: VB  [ X X X | E | V3 V2 V1 V0 ]
// 10: VC  [ X X X | E | V3 V2 V1 V0 ]
// 11: FEL [ F7 F6 F5 F4 F3 F2 F1 F0 ]
// 12: FEH [ F15 F14 F13 F12 F11 F10 F9 F8 ]
// 13: EM  [ X X X X | E3 E2 E1 E0 ]
// 14: A   [ A7 A6 A5 A4 A3 A2 A1 A0 ]
// 15: B   [ B7 B6 B5 B4 B3 B2 B1 B0 ]

void ay8910render(ay8910state * ay8910, int N)
{
    
#ifdef PSGDUMP
    if (ay8910->DebugInfo>2) printf("%s RENDER ABC SAMPLES=%d\n", ay8910->DebugName, ay8910->Samples);
    if (ay8910->Ready) printf("%s ERROR: RENDERING OVER!\n", ay8910->DebugName);
#endif

    int divider[5];
    
    divider[0] = ay8910->R[ 0] + ((ay8910->R[ 1] & 15) << 8);
    divider[1] = ay8910->R[ 2] + ((ay8910->R[ 3] & 15) << 8);
    divider[2] = ay8910->R[ 4] + ((ay8910->R[ 5] & 15) << 8);
    divider[3] =                  (ay8910->R[ 6] & 31);
    divider[4] = ay8910->R[11] +  (ay8910->R[12] << 8);
    
    if (ay8910->Divider[0] != divider[0]) { ay8910->Counter[0] = 0; ay8910->Divider[0] = divider[0]; ay8910->Bit[0] = 0; }
    if (ay8910->Divider[1] != divider[1]) { ay8910->Counter[1] = 0; ay8910->Divider[1] = divider[1]; ay8910->Bit[1] = 0; }
    if (ay8910->Divider[2] != divider[2]) { ay8910->Counter[2] = 0; ay8910->Divider[2] = divider[2]; ay8910->Bit[2] = 0; }
    if (ay8910->Divider[3] != divider[3]) { ay8910->Counter[3] = 0; ay8910->Divider[3] = divider[3]; }
    
//    if (divider[0] == 0) divider[0] = 1;
//    if (divider[1] == 0) divider[1] = 1;
//    if (divider[2] == 0) divider[2] = 1;
//    if (divider[3] == 0) divider[3] = 1;
    
    if ((ay8910->Divider[4] != divider[4]) || (ay8910->EnvelopeMode != ay8910->R[13])) {
        ay8910->Counter[4] = 0;
        ay8910->Divider[4] = divider[4];
        ay8910->EnvelopeStep = 0;
        ay8910->EnvelopeMode = ay8910->R[13];
    }
    
    for (int i = 0; i < N; i++) {
        
        if (0 == ay8910->Counter[0] --) {
            ay8910->Bit[0] ^= 1;
            ay8910->Counter[0] = divider[0];
        }
        
        if (0 == ay8910->Counter[1] --) {
            ay8910->Bit[1] ^= 1;
            ay8910->Counter[1] = divider[1];
        }
        
        if (0 == ay8910->Counter[2] --) {
            ay8910->Bit[2] ^= 1;
            ay8910->Counter[2] = divider[2];
        }
        
        if (0 == ay8910->Counter[3] --) {
            ay8910->Bit[3] = rand() & 1;
            ay8910->Counter[3] = divider[3];
        }
        
        if (0 == ay8910->Counter[4] --) {
            
            int shape = VolumeShapes[ay8910->EnvelopeMode][ay8910->EnvelopeStep/16];
            
            switch (shape) {
                case PSGSHAPELOW:  ay8910->Envelope =  0; break;
                case PSGSHAPEHIGH: ay8910->Envelope = 15; break;
                case PSGSHAPEUP:   ay8910->Envelope = ay8910->EnvelopeStep & 15; break;
                case PSGSHAPEDOWN: ay8910->Envelope = 15 - (ay8910->EnvelopeStep & 15); break;
            }
            
            if (32 == ++ay8910->EnvelopeStep) ay8910->EnvelopeStep = VolumeShapes[ay8910->EnvelopeMode][2] ? 0 : 16;
            
            ay8910->Counter[4] = divider[4];
        }
        
        int volume[3];
        
        volume[0] = (ay8910->R[ 8] & 32) ? ay8910->Envelope : (ay8910->R[ 8] & 15);
        volume[1] = (ay8910->R[ 9] & 32) ? ay8910->Envelope : (ay8910->R[ 9] & 15);
        volume[2] = (ay8910->R[10] & 32) ? ay8910->Envelope : (ay8910->R[10] & 15);
        
        ay8910->OutputA [ay8910->Fill] = (  (ay8910->R[7]       & 1) | ay8910->Bit[0] ) & ( ((ay8910->R[7] >> 3) & 1) | ay8910->Bit[3] );
        ay8910->OutputB [ay8910->Fill] = ( ((ay8910->R[7] >> 1) & 1) | ay8910->Bit[1] ) & ( ((ay8910->R[7] >> 4) & 1) | ay8910->Bit[3] );
        ay8910->OutputC [ay8910->Fill] = ( ((ay8910->R[7] >> 2) & 1) | ay8910->Bit[2] ) & ( ((ay8910->R[7] >> 5) & 1) | ay8910->Bit[3] );
        
        ay8910->OutputA [ay8910->Fill] = PSGVolumes [ (ay8910->OutputA[ay8910->Fill]) ? 0 : volume[0] ];
        ay8910->OutputB [ay8910->Fill] = PSGVolumes [ (ay8910->OutputB[ay8910->Fill]) ? 0 : volume[1] ];
        ay8910->OutputC [ay8910->Fill] = PSGVolumes [ (ay8910->OutputC[ay8910->Fill]) ? 0 : volume[2] ];
        
        ++ ay8910->Fill;
        
        if (ay8910->Fill == ay8910->Samples) { ay8910->Fill = 0; ay8910->Ready = 1; }
    }
}

void ay8910mix(ay8910state * ay8910, short audio[][2], int divider)
{
    int N = ay8910->Samples / divider;
    
#ifdef PSGDUMP
    if (ay8910->DebugInfo>2) printf("%s MIXING AUDIO SAMPLES=%d\n", ay8910->DebugName, N);
    if (! ay8910->Ready) printf("%s ERROR: MIXING UNREADY BUFFER!\n", ay8910->DebugName);
#endif

    for (int i = 0, p = 0; i < N; i++) {
        
        int r = 0, l = 0;
        
        for (int j = 0; j < divider; j++, p++) {
            l += (ay8910->OutputA[p] - 128) * ay8910->Mix[0][0];
            l += (ay8910->OutputB[p] - 128) * ay8910->Mix[0][1];
            l += (ay8910->OutputC[p] - 128) * ay8910->Mix[0][2];
            r += (ay8910->OutputA[p] - 128) * ay8910->Mix[1][0];
            r += (ay8910->OutputB[p] - 128) * ay8910->Mix[1][1];
            r += (ay8910->OutputC[p] - 128) * ay8910->Mix[1][2];
        }
        
        l /= divider;
        r /= divider;

        audio[i][0] += l;
        audio[i][1] += r;
    }
    
    ay8910->Ready = 0;
}
