#!/usr/bin/perl
#
# Print Chameleon DOS directory and extract all COM files to LVT files
# and all EXE files to LVx files.
# MFM disks of 16 sectors of 256 bytes per track and
# FM disks of 9 sectors of 256 bytes per track are supported.
# Limitation: natural linear disk space allocation assumed.
#
# (c) 2014 PortaOne, Inc.
# kapitan@portaone.com
#
# Usage:
# lviv-dsk-to-lvt.pl <dsk file> [FM|MFM]
#
use File::Tee qw(tee);

$| = 1;

# Chameleon maximum number of files for FM and MFM disks
#
$max_files = 134;

open(IN, "<$ARGV[0]")
   || die "File $ARGV[0] not found.";

$total_space = -s $ARGV[0];

# Global sector number cursor
#
$cur_sector = 0;

# Read disk attributes
#
sysread(IN, $buff, 17);
( $disk_name,
  $first_free_sector,
  $first_free_track,
  $unknown_1,
  $hidden_dir,
  $write_protection,
  $unknown_2 )  = unpack ( 'A11 C C C C C C', $buff);

$dir_name = $disk_name;
$dir_name =~ s/[^A-Za-z0-9\-\.]/_/g;
$dir_name = $ARGV[0].'-'.$dir_name;

mkdir $dir_name
   || die "Can not create directory $dir_name.";
unlink( "$disk_name/directory.txt" );
tee STDOUT, '>>', "$dir_name/directory.txt";

# NOTE: Automatic size based mode selection is unreliable!!!
#
if( length($ARGV[1]) == 0 ) {
   print "\nNOTE: Automatic size based mode selection is unreliable!!!";
   if( $total_space <= 83*2*9*256 ) {
      print "\nFM mode selected.\n";
      $sec_per_tr = 9;
   } else {
      print "\nMFM mode selected.\n";
      $sec_per_tr = 16;
   }
} else {
   if( $ARGV[1] eq "MFM" ) {
      print "\nMFM mode selected.\n";
      $sec_per_tr = 16;
   }
   elsif( $ARGV[1] eq "FM" ) {
      print "\nFM mode selected.\n";
      $sec_per_tr = 9;
   } else {
      die "ERROR: Acceptable dencity is MFM or FM.";
   }
}

# Global first free sector
#
$global_free_sector = $first_free_track * $sec_per_tr + $first_free_sector;

print "\nDisk image file name: $ARGV[0]\n\n";
print "Disk Name: $disk_name\n";
if( $write_protection == 0 ) {
   print "Write Protection Flag is OFF\n";
} else {
   print "Write Protection Flag is ON\n";
}
if( $hidden_dir == 0 ) {
   print "Hide Directory Flag is OFF\n";
} else {
   print "Hide Directory Flag is ON\n";
}
print "\n";

$used_space = $first_free_track * 256 * $sec_per_tr + ($first_free_sector - 1) * 256;
$free_space =  $total_space - $used_space;
print "Total : $total_space bytes\nUsed  : $used_space bytes\nFree  : $free_space bytes";

print " ERROR: Lost 81 & 82 tracks?" if $free_space < 0;
print "\n";

# Read directory
#
$i = 0;
while( $i < $max_files && sysread(IN, $buff, 17) && $buff ne "\0"x17 ) {
   ( $directory[$i]{'file_name'},
     $directory[$i]{'file_extension'},
     $directory[$i]{'file_status'},
     $directory[$i]{'first_sector'},
     $directory[$i]{'first_track'},
     $directory[$i]{'start_address'},
     $directory[$i]{'sector_length'} )  = unpack ( 'A8 A3 C C C v C', $buff);

  $directory[$i]{'attributes'} = '';
  $directory[$i]{'attributes'} .= "DELETED " if substr( $directory[$i]{'file_name'},0,1 ) eq "\001";
  $directory[$i]{'attributes'} .= "PROTECTED " if $directory[$i]{'file_status'} == 0x01;
  $directory[$i]{'attributes'} .= "PROTECTED & HIDDEN " if $directory[$i]{'file_status'} == 0x02;

  # Global start sector
  #
  $start_sector = $directory[$i]{'first_track'} * $sec_per_tr + $directory[$i]{'first_sector'};

  $directory[$i]{'attributes'} .= "LOCATED IN FREE SPACE" if $start_sector >= $global_free_sector;

  $i++;
  if( ($i-14)%15 == 0 ) {
     sysread(IN, $buff, 1);

     # Next sector
     #
     $cur_sector++;
  }
}

$file_number = $i;

# Go to the beginning of next sector
# (at the moment one empty directory record of 17 bytes has ben read)
#
if( ($i+1-14)%15 == 0 ) {
   sysread(IN, $buff, 1);
   $cur_sector++;
}
if( ($i+1-14)%15 != 0 ) {
   sysread(IN, $buff, (15-($i+1-14)%15) *17 + 1 );
   $cur_sector++;
}

print "$file_number files\n\n";

# Print directory
#
print " ##\tName\tExt\tStatus\tStart\tBytes\tAttributes\n\n";
for( $i=0; $i < $file_number; $i++ ) {
   printf "%3u\t%-8s.%3s\t0x%02X\t0x%04X\t%5u\t%s\n",
      $i+1,
      $directory[$i]{'file_name'},
      $directory[$i]{'file_extension'},
      $directory[$i]{'file_status'},
      $directory[$i]{'start_address'},
      $directory[$i]{'sector_length'} * 256,
      $directory[$i]{'attributes'};
}
print "\n";

# Extract all COM files to LVT
#

for( $i=0; $i < $file_number; $i++ ) {
      # Create LVT file
      #
      $directory[$i]{'file_name'} =~ s/[^A-Za-z0-9\-\.]/_/g;
      $directory[$i]{'file_extension'} =~ s/[^A-Za-z0-9\-\.]/_/g;
      if( $directory[$i]{'file_extension'} ne "EXE" && $directory[$i]{'file_extension'} ne "COM" ) {
         $file_name = $directory[$i]{'file_name'}.'_'.$directory[$i]{'file_extension'}.'.lvt';
      } elsif ( $directory[$i]{'file_extension'} eq "COM" ) {
         $file_name = $directory[$i]{'file_name'}.'.lvt';
      } else {
         $directory[$i]{'file_name'} =~ /^(.*)\D(\d*)(\s*)$/;
         $overlay_index = $2 - 1;
         $overlay_index = 0 if $overlay_index < 0;
         $file_name = $directory[$i]{'file_name'}.'.lv'.$overlay_index;
      }
      printf "\nExtracting\t%-8s.%3s\t", $directory[$i]{'file_name'}, $directory[$i]{'file_extension'};
      print "as $file_name\t";

      # Global start sector
      #
      $start_sector = $directory[$i]{'first_track'} * $sec_per_tr + $directory[$i]{'first_sector'} - 1;

      # Common error: Chameleon disks have 83 tracks,
      # some images contain only 80 tracks :(
      #
      if ( $start_sector * 256 > $total_space
       || ( $start_sector + $directory[$i]{'sector_length'} ) * 256 > $total_space ) {
         print "\tERROR: Track not foud! DSK file is too small?";
      } else {
         print "\tWARNING: Free space reached!" if $global_free_sector == $cur_sector;

         if( $start_sector < $cur_sector ) {
            print "ERROR: Impossible to extract deleted file in free space.";
         } else {
            unlink( $file_name );
            open(OUT, ">$dir_name/$file_name")
               || die "Can not create output file $file_name";

            # Create LVT header
            #
            print OUT "LVOV/2.0/";
            if( $directory[$i]{'file_extension'} eq "BAS" ) {
               print OUT pack( 'C', 0xD3 );
            } else {
               print OUT pack( 'C', 0xD0 ); 
            }
            print OUT pack( 'A6', $directory[$i]{'file_name'} );

            $end_address = $directory[$i]{'start_address'} + $directory[$i]{'sector_length'} * 256 - 1;

            if( $directory[$i]{'file_extension'} ne "BAS" ) {
               print OUT pack( "v", $directory[$i]{'start_address'} );
               print OUT pack( "v", $end_address );
               print OUT pack( "v", $directory[$i]{'start_address'} );
            }

            # Advance to start sector
            #
            if( $cur_sector != $start_sector ) {
               sysread(IN, $buff, ($start_sector-$cur_sector)*256);
               $cur_sector = $start_sector;
            }

            # Extract and dump data
            #
            sysread(IN, $buff, $directory[$i]{'sector_length'}*256 );
            $cur_sector += $directory[$i]{'sector_length'};

	    if( $directory[$i]{'file_extension'} eq "BAS" ) {
            	print OUT substr( $buff, 0, $directory[$i]{'start_address'} );
	    } else {
		print OUT $buff;
	    }
            close(OUT);
         }
      }
}
print "\n";

close(IN);
