#!/usr/bin/perl
#
# Print Chameleon DOS directory and convert LVT to file on disk.
# MFM disks of 16 sectors of 256 bytes per track and
# FM disks of 9 sectors of 256 bytes per track are supported.
# 3 type of LVT files processed automatically:
# - loading address = start address (Direct)
# - loading address != start address && loading address > 2 (+JMP command)
# - loading address != start address && loading address < 2 (+Relocating Loader Code) 
# BASIC files are supported.
#
# (c) 2014 PortaOne, Inc.
# kapitan@portaone.com
#
# Usage:
# lviv-lvt-to-dsk.pl <lvt file> <in dsk file> [target name]
#
use File::Tee qw(tee);
use POSIX qw(ceil);

$| = 1;

# Chameleon maximum number of files for FM and MFM disks
#
$max_files = 134;

open(LVT, "<$ARGV[0]")
   || die "File $ARGV[0] not found.";

print "\nLVT file $ARGV[0]\n";

if( defined $ARGV[2] ) {
   $ARGV[2] =~ /(.*)[.]([^.]*)$/;
   $target_name = $1;
   $target_extension = $2;
   die "Traget file name must be 1 to 8 charachters long with 3 charachter extension."
      if length($target_name) < 1 || length($target_name) > 8 || length($target_extension) != 3;
}

# Read LVT header
#
$lvt_size = -s $ARGV[0];

# Signature
#
sysread(LVT, $buff, 9);
$lvt_size -= 9;
die "ERROR: $ARGV[0] not LVT file or type not supported." if $buff ne 'LVOV/2.0/';

# File name and type
#
sysread(LVT, $buff, 7);
$lvt_size -= 7;
( $lvt_type,
  $lvt_name )  = unpack ( 'C A6', $buff);

print "Name: $lvt_name\n";
printf "Type: 0x%02X\n", $lvt_type;

if( !defined $target_name ) {
   die "Misisng file name in LVT. Please specify target file name." if length($lvt_name) == 0;
   $target_name = $lvt_name;
}

if( $lvt_type == 0xD0 ) {
   print "Binary Mode.\n\n";
   $target_extension = 'COM' if !defined $target_extension;

   # 3x 16 bit addresses for binary file
   #
   sysread(LVT, $buff, 6);
   ( $loading_address,
     $end_address,
     $run_address )  = unpack ( 'v v v', $buff);
   $lvt_size -= 6;

   printf "Loading address: 0x%04X\n", $loading_address;
   printf "End     address: 0x%04X\n", $end_address;
   printf "Run     address: 0x%04X\n", $run_address;
   print "Payload size: $lvt_size bytes.\n";

   die "ERROR: LVT file size vs payload size declared in header missmatch."
      if $lvt_size != $end_address-$loading_address+1;

   # Read LVT payload
   #
   sysread(LVT, $lvt_payload, $lvt_size);
   close(LVT);

   # Include JMP $run_address at $loading_address-3 if needed
   #
   if( $loading_address != $run_address ) {
      print "\nApplying Loading address = Run address patch.\n";
      if( $loading_address > 2 ) {
         $loading_address -= 3;
         $lvt_payload = pack( "C v", 0xC3, $run_address).$lvt_payload; # JMP $run_address
         $lvt_size += 3;
      } else {
         print "Loading address <2 !!! Using advanced patch.\n";
         $lvt_payload = pack( "C v", 0xC3, $lvt_size+3).$lvt_payload; # JMP $lvt_size+3

         $lvt_payload .= pack( "C v", 0x01, $lvt_size);        # lxi  b,PROG_LEN   ; BC - # of bytes to copy
         $lvt_payload .= pack( "C v", 0x11, 3);                # lxi  d,3          ; DE - Source Address
         $lvt_payload .= pack( "C v", 0x21, $loading_address); # lxi  h,PROG_LOAD  ; HL - Target Address 

         $lvt_payload .= pack( "C v", 0xCD, 0xE11F);           # call E11F         ; SYS ROM MEMCOPY
         $lvt_payload .= pack( "C v", 0xC3, $run_address);     # jmp  $run_address ;Start Program
     
         $loading_address = 0; 
         $lvt_size += 3+15;
      }
   }

} elsif( $lvt_type == 0xD3 ) {
   print "BASIC Mode.\n\n";
   print "Payload size: $lvt_size bytes.\n";

   $loading_address = $lvt_size;
   $target_extension = 'BAS' if !defined $target_extension;

   # Read LVT payload
   #
   sysread(LVT, $lvt_payload, $lvt_size);
   close(LVT);

} else {
   printf "LVT type 0x%02X is not supported.", $lvt_type;
   die;
}

# Adjusting payload size to the nearest full sector
#
$lvt_sec_size = ceil( $lvt_size/256 );
$lvt_payload = $lvt_payload . chr(0)x($lvt_sec_size*256-$lvt_size);
$lvt_size = $lvt_sec_size*256;

open(IN, "<$ARGV[1]")
   || die "File $ARGV[1] not found.";

$total_space = -s $ARGV[1];

# FM or MFM image ?
#
if( $total_space <= 83*2*9*256 ) {
   $sec_per_tr = 9;
} else {
   $sec_per_tr = 16;
}

unlink( "$ARGV[1].tmp" );
open(OUT, ">$ARGV[1].tmp")
   || die "Can not create output file $ARGV[2]";

# Global sector number cursor
#
$cur_sector = 1;

# Read disk attributes
#
sysread(IN, $buff, 17);
( $disk_name,
  $first_free_sector,
  $first_free_track,
  $unknown_1,
  $hidden_dir,
  $write_protection,
  $unknown_2 )  = unpack ( 'A11 C C C C C C', $buff);

print "\nDisk image file name: $ARGV[1]\n";
print "Disk Name: $disk_name\n";
if( $write_protection == 0 ) {
   print "Write Protection Flag is OFF\n";
} else {
   print "Write Protection Flag is ON\n";
}
if( $hidden_dir == 0 ) {
   print "Hide Directory Flag is OFF\n";
} else {
   print "Hide Directory Flag is ON\n";
}
print "\n";

$used_space = $first_free_track * 256 * $sec_per_tr + ($first_free_sector - 1) * 256;
$free_space =  $total_space - $used_space;
print "Total : $total_space bytes\nUsed  : $used_space bytes\nFree  : $free_space bytes\n\n";

die "ERROR: Not enough space on DSK." if $free_space < $lvt_size;

# Global first free sector
#
$global_free_sector = $first_free_track * $sec_per_tr + $first_free_sector;

# New first free sector and track numbers
#
$new_global_free_sector = $global_free_sector + $lvt_sec_size;
$new_first_free_track = int($new_global_free_sector/$sec_per_tr);
$new_first_free_sector = $new_global_free_sector - $new_first_free_track * $sec_per_tr + 1;

# Write new disk attributes
#
print OUT pack( 'A11 C C C C C C',
  $disk_name,
  $new_first_free_sector,
  $new_first_free_track,
  $unknown_1,
  $hidden_dir,
  $write_protection,
  $unknown_2 );

# Read-Write directory
#
$i = 0;

while( $i < $max_files && sysread(IN, $buff, 17) && $buff ne "\0"x17 ) {
   ( $directory[$i]{'file_name'},
     $directory[$i]{'file_extension'},
     $directory[$i]{'file_status'},
     $directory[$i]{'first_sector'},
     $directory[$i]{'first_track'},
     $directory[$i]{'start_address'},
     $directory[$i]{'sector_length'} )  = unpack ( 'A8 A3 C C C v C', $buff);

   # Check for name conflict
   #
   $name_conflict = 1
      if $directory[$i]{'file_name'} eq $target_name
      && $directory[$i]{'file_extension'} eq $target_extension;

   print OUT pack( 'A8 A3 C C C v C',
     $directory[$i]{'file_name'},
     $directory[$i]{'file_extension'},
     $directory[$i]{'file_status'},
     $directory[$i]{'first_sector'},
     $directory[$i]{'first_track'},
     $directory[$i]{'start_address'},
     $directory[$i]{'sector_length'} );

  $i++;
  if( ($i+1)%15 == 0 ) {
     sysread(IN, $buff, 1);
     print OUT $buff;

     # Next sector
     #
     $cur_sector++;
  }
}
$file_number = $i;

if( defined $name_conflict || $file_number == $max_files ) {
   print "Directory\n";
   $new_file_number = $file_number;
   $lvt_size = 0;
} else {

$new_cur_sector = $cur_sector;

# Create new file entry
#
$new_cur_sector = $cur_sector;
$new_file_number = $file_number+1;

$directory[$i]{'file_name'} = $target_name;
$directory[$i]{'file_extension'} = $target_extension;
$directory[$i]{'file_status'} = 0;
$directory[$i]{'first_sector'} = $first_free_sector;
$directory[$i]{'first_track'} = $first_free_track;
$directory[$i]{'start_address'} = $loading_address;
$directory[$i]{'sector_length'} = $lvt_sec_size;

print OUT pack( 'A8 A3 C C C v C',
 $directory[$i]{'file_name'},
 $directory[$i]{'file_extension'},
 $directory[$i]{'file_status'},
 $directory[$i]{'first_sector'},
 $directory[$i]{'first_track'},
 $directory[$i]{'start_address'},
 $directory[$i]{'sector_length'} );

if( ($i+2)%15 == 0 ) {
   sysread(IN, $buff, 1);
   print OUT $buff;

   # Next sector
   #
   $cur_sector++;
  } else {

   # Go to the end of current sector in input and output DSK
   #
   sysread(IN, $buff, (15-($i+2)%15) *17 + 1 );
   print OUT $buff;
   $cur_sector++;
   $new_cur_sector++;
}

print "New Directory\n";
}

print "$new_file_number files\n\n";

# Print new directory
#
print " ##\tName\tExt\tStatus\tStart\tBytes\n\n";
for( $i=0; $i < $new_file_number; $i++ ) {
   printf "%3u\t%-8s.%3s\t0x%02X\t0x%04X\t%5u\n",
      $i+1,
      $directory[$i]{'file_name'},
      $directory[$i]{'file_extension'},
      $directory[$i]{'file_status'},
      $directory[$i]{'start_address'},
      $directory[$i]{'sector_length'} * 256;
}
print "\n";

# Copy all sectors up to first free sector
#
if( $cur_sector != $global_free_sector ) {
   sysread(IN, $buff, ($global_free_sector-$cur_sector)*256);
   print OUT $buff;
   $cur_sector = $global_free_sector;
}

if( $lvt_size !=0 ) {
   # Copy LVT payload
   #
   print OUT $lvt_payload;
} elsif( $file_number == $max_files ) {
   print "ERROR: Maximum number of $file_number files is reached.\n";
} else {
   print "ERROR: File $target_name.$target_extension anready exists.\n";
}

# Pad remaining free space in new DSK
#
print OUT "\0"x($free_space-$lvt_size);

close(IN);
close(OUT);

unlink( $ARGV[1] );
rename( "$ARGV[1].tmp", $ARGV[1] );
