// ====================================================================
//                Radio-86RK FPGA REPLICA
//
//            Copyright (C) 2011 Dmitry Tselikov
//
// This core is distributed under modified BSD license. 
// For complete licensing information see LICENSE.TXT.
// -------------------------------------------------------------------- 
//
// An open implementation of Radio-86RK home computer
//
// Author: Dmitry Tselikov   http://bashkiria-2m.narod.ru/
// 
// Minor changes for adaptation to SDRAM: Ivan Gorodetsky, 2014
// Modified to Mikro80: Ivan Gorodetsky, 2015
//
// Design File: mikro80.v
//
// Top level design file.

`define WITH_LEDs

module mikro80(
	input			clk50mhz,

	
	input [3:0] 	KEY,
	output [9:0] 	LEDr,
	output [7:0] 	LEDg,
	input [9:0] 	SW, 

	output [6:0] 	HEX0,
	output [6:0] 	HEX1,
	output [6:0] 	HEX2,
	output [6:0] 	HEX3,

//	inout	[15:0]	DRAM_DQ,				//	SDRAM Data bus 16 Bits
//	output	[11:0]	DRAM_ADDR,				//	SDRAM Address bus 12 Bits
//	output			DRAM_LDQM,				//	SDRAM Low-byte Data Mask 
//	output			DRAM_UDQM,				//	SDRAM High-byte Data Mask
//	output			DRAM_WE_N,				//	SDRAM Write Enable
//	output			DRAM_CAS_N,				//	SDRAM Column Address Strobe
//	output			DRAM_RAS_N,				//	SDRAM Row Address Strobe
//	output			DRAM_CS_N,				//	SDRAM Chip Select
//	output			DRAM_BA_0,				//	SDRAM Bank Address 0
//	output			DRAM_BA_1,				//	SDRAM Bank Address 0
//	output			DRAM_CLK,				//	SDRAM Clock
//	output			DRAM_CKE,				//	SDRAM Clock Enable
	
	output [15:0] ram_addr,//iaddr(addrbus[15:0]),
    output [7:0] idata,//(cpu_o),
    output rd,//(cpu_rd&~io_read),
    output we_n,//(cpu_wr_n|io_write),
    input [7:0] odata,//(dramout)

	output 			VGA_HS,
	output 			VGA_VS,
	output	[3:0] 	VGA_R,
	output	[3:0] 	VGA_G,
	output	[3:0] 	VGA_B,

	inout			I2C_SDAT,				//	I2C Data
	output			I2C_SCLK,				//	I2C Clock

	inout			AUD_BCLK,
	output			AUD_DACDAT,
	output			AUD_DACLRCK,
	output			AUD_XCK,
	output			AUD_ADCLRCK,			//	Audio CODEC ADC LR Clock
	input			AUD_ADCDAT,				//	Audio CODEC ADC Data

	input			PS2_CLK,
	input			PS2_DAT,

	////////////////////	USB JTAG link	////////////////////////////
	input  			TDI,					// CPLD -> FPGA (data in)
	input  			TCK,					// CPLD -> FPGA (clk)
	input  			TCS,					// CPLD -> FPGA (CS)
	output 			TDO,					// FPGA -> CPLD (data out)

	input			SD_DAT,					//	SD Card Data 			(MISO)
	output			SD_DAT3,				//	SD Card Data 3 			(CSn)
	output			SD_CMD,					//	SD Card Command Signal	(MOSI)
	output			SD_CLK,					//	SD Card Clock			(SCK)

	output			UART_TXD,
	input			UART_RXD,

	input tape_in,
	output tape_out,
	output beep_out,
	
	output [12:0] 	GPIO_0,
	output [35:0]	GPIO_1 );

//`default_nettype none 	
	
assign GPIO_0 = 0;
assign GPIO_1 = 0;
assign TDO = 0;
assign UART_TXD = 0;

////////////////////   RESET   ////////////////////
reg[3:0] reset_cnt;
reg reset_n;
wire reset = ~reset_n;

always @(posedge clk50mhz) begin
	if (KEY[0] && reset_cnt==4'd14)
		reset_n <= 1'b1;
	else begin
		reset_n <= 1'b0;
		reset_cnt <= reset_cnt+4'd1;
	end
end

////////////////////   LEDs   ////////////////////

`ifdef WITH_LEDs

assign LEDr[0] = cpu_sync;
assign LEDr[1] = cpu_rd;
assign LEDr[2] = ~cpu_wr_n;
assign LEDr[9:3] = 0;
assign LEDg = cpu_rd ? cpu_i : cpu_o;

seg7_lut4 seg0(HEX0,HEX1,HEX2,HEX3, addrbus);

`else

assign LEDr = 0;
assign LEDg = 0;
assign HEX0 = 7'h7F;
assign HEX1 = 7'h7F;
assign HEX2 = 7'h7F;
assign HEX3 = 7'h7F;

`endif


wire[7:0] mem_o = dramout[7:0];

wire[7:0] rom_o,rom_sd;
wire[15:0] addrbus;

reg[7:0] address_bus_r;
wire io_read,io_write;
assign io_write = status_word[4];
assign io_read  = status_word[6];  

monitorw(.address(addrbus[10:0]), .clock(clk50mhz), .q(rom_o));
xsdw(.address(addrbus[10:0]), .clock(clk50mhz), .q(rom_sd));

////////////////////   CPU   ////////////////////
wire[7:0] cpu_o;
wire cpu_sync;
wire cpu_rd;
wire cpu_wr_n;
wire cpu_int;
wire cpu_inta_n;
wire inte;
reg[7:0] cpu_i;

////////////////////   MEM   ////////////////////
assign DRAM_CLK=clk50mhz;				//	SDRAM Clock
assign DRAM_CKE=1;				//	SDRAM Clock Enable
wire[15:0] dramout;
//SDRAM_Controller ramd(
//	.clk50mhz(clk50mhz),				//  Clock 50MHz
//	.reset(reset),					//  System reset
//	.DRAM_DQ(DRAM_DQ),				//	SDRAM Data bus 16 Bits
//	.DRAM_ADDR(DRAM_ADDR),			//	SDRAM Address bus 12 Bits
//	.DRAM_LDQM(DRAM_LDQM),				//	SDRAM Low-byte Data Mask 
//	.DRAM_UDQM(DRAM_UDQM),				//	SDRAM High-byte Data Mask
//	.DRAM_WE_N(DRAM_WE_N),				//	SDRAM Write Enable
//	.DRAM_CAS_N(DRAM_CAS_N),				//	SDRAM Column Address Strobe
//	.DRAM_RAS_N(DRAM_RAS_N),				//	SDRAM Row Address Strobe
//	.DRAM_CS_N(DRAM_CS_N),				//	SDRAM Chip Select
//	.DRAM_BA_0(DRAM_BA_0),				//	SDRAM Bank Address 0
//	.DRAM_BA_1(DRAM_BA_1),				//	SDRAM Bank Address 1
//	.iaddr(addrbus[15:0]),
//	.idata(cpu_o),
//	.rd(cpu_rd&~io_read),
//	.we_n(cpu_wr_n|io_write),
//	.odata(dramout)
//);

	assign ram_addr = (addrbus[15:0]);
	assign idata = (cpu_o);
	assign rd = (cpu_rd&~io_read);
	assign we_n = (cpu_wr_n|io_write);
	assign odata = (dramout);

always @(posedge clk50mhz)
	casex ({addrbus[15:11],(address_bus_r[7:2]==6'b000001)&io_read&cpu_rd,
				(address_bus_r==8'h71)&io_read&cpu_rd,
				(address_bus_r==8'h01)&io_read&cpu_rd})
	8'b0xxxx000: cpu_i = mem_o;
	8'b10xxx000: cpu_i = mem_o;
	8'b1100x000: cpu_i = mem_o;
	8'b11010000: cpu_i = mem_o;
	8'b11011000: cpu_i = rom_sd;
	8'b11100000: cpu_i = mem_o;
	8'b11101000: cpu_i = mem_o;
	8'b11110000: cpu_i = mem_o;
	8'b11111000: cpu_i = rom_o;
	8'bxxxxx100: cpu_i = ppa1_o;
	8'bxxxxx010: cpu_i = sd_o;
	8'bxxxxx001: cpu_i = {7'h7f,tape_in};
//	8'bxxxxx001: cpu_i = {7'h7f,tapein};
	default:cpu_i = 8'hFF;
	endcase
	
wire ppa1_we_n = address_bus_r[7:2]!=6'b000001|~io_write|cpu_wr_n;


reg[4:0] cpu_cnt;
reg cpu_ce2;
wire cpu_ce = cpu_ce2;
always @(posedge clk50mhz) begin
	if(reset) begin cpu_cnt<=0; cpu_ce2<=0; end
	else
	if(cpu_cnt<=24) begin cpu_cnt <= cpu_cnt + 1; cpu_ce2<=0; end
	else begin cpu_cnt<=0; cpu_ce2<=1; end
end

k580wm80a CPU(.clk(clk50mhz), .ce(cpu_ce), .reset(reset),
	.idata(cpu_i), .addr(addrbus), .sync(cpu_sync), .rd(cpu_rd), .wr_n(cpu_wr_n),
	.intr(cpu_int), .inta_n(cpu_inta_n), .odata(cpu_o), .inte_o(inte));

////////////////////   VIDEO   ////////////////////
wire[7:0] vid_char;
wire[10:0] chadr;
wire vid_rvv;

chramw (
	.clock(clk50mhz),
	.data(cpu_o),
	.rdaddress(chadr),
	.wraddress(addrbus[10:0]),
	.wren((~cpu_wr_n)&(~io_write)&addrbus[15:11]==5'b11101),
	.q(vid_char)
	);

curramw (
	.clock(clk50mhz),
	.data(cpu_o[7]),
	.rdaddress(chadr+1),
	.wraddress(addrbus[10:0]),
	.wren((~cpu_wr_n)&(~io_write)&addrbus[15:11]==5'b11100),
	.q(vid_rvv)
	);


rk_video vid(.clk50mhz(clk50mhz), .hr(VGA_HS), .vr(VGA_VS),
	.r(VGA_R), .g(VGA_G), .b(VGA_B), .ichar(vid_char),
	.rvv(vid_rvv),
	.chadr(chadr)
	);

////////////////////   KBD   ////////////////////
wire[6:0] kbd_o;
wire[2:0] kbd_shift;

////////////////////   SYS PPA   ////////////////////
wire[7:0] ppa1_o;
wire[7:0] ppa1_a;
wire[7:0] ppa1_b;
wire[7:0] ppa1_c;

////////////////////   KBD   ////////////////////

rk_kbd kbd(.clk(clk50mhz), .reset(reset), .ps2_clk(PS2_CLK), .ps2_dat(PS2_DAT),
	.addr(~ppa1_a),.odata(kbd_o), .shift(kbd_shift));

////////////////////   SYS PPA   ////////////////////

k580ww55 ppa1(.clk(clk50mhz), .reset(reset), .addr(~address_bus_r[1:0]), .we_n(ppa1_we_n),
	.idata(cpu_o), .odata(ppa1_o),
	.ipa(ppa1_a),
	.opa(ppa1_a),
	.ipb({ppa1_b[7],~kbd_o}), .opb(ppa1_b), .ipc({ppa1_c[7:3],~kbd_shift}), .opc(ppa1_c));

////////////////////   SOUND   ////////////////////
reg tapein;
reg tapeout;
reg[15:0] linein;
reg[15:0] adcbuf;
reg[13:0] cnt18;
reg[4:0] cntbit;
reg bitclk;

wire[15:0] pulses = {3'b0,tapeout,12'b0};
wire[5:0] line6bit = {~linein[15],linein[14:10]};

wire[13:0] cnt18next = cnt18+14'd755; // 755/2048*50MHz ~ 18.432MHz

assign AUD_XCK = cnt18[10];
assign AUD_BCLK = reset ? 1'bZ : bitclk;
assign AUD_DACLRCK = reset ? 1'b0 : cntbit[4];
assign AUD_ADCLRCK = reset ? 1'b0 : cntbit[4];
assign AUD_DACDAT = reset ? 1'b0 : pulses[~cntbit[3:0]];

always @(posedge clk50mhz) begin
	if (cnt18next[13:11]==3'd6) begin
		cnt18 <= {3'd0, cnt18next[10:0]};
		bitclk <= ~bitclk;
		if (bitclk) begin // negedge bitclk
			if (cntbit==4'd0) linein <= adcbuf;
			adcbuf[~cntbit[3:0]] <= AUD_ADCDAT;
			cntbit <= cntbit+5'b1;
		end
	end else
		cnt18 <= cnt18next;
	if (line6bit < 31) tapein <= 1'b0;
	if (line6bit > 32) tapein <= 1'b1;
end

//I2C_AV_Config sndcfg(.iCLK(clk50mhz), .iRST_N(reset_n), .I2C_SCLK(I2C_SCLK), .I2C_SDAT(I2C_SDAT));

assign tape_out = tapeout;

always @(posedge clk50mhz or posedge reset)
	if (reset)
		tapeout <= 1'b0;
	else
		if ((address_bus_r==8'h01)&io_write&~cpu_wr_n) tapeout <= cpu_o[0];


////////////////////   SD CARD   ////////////////////
reg[7:0] status_word;
always @(posedge clk50mhz) begin
	if (cpu_ce) begin
		if (cpu_sync) begin
			status_word <= cpu_o;
		end 
		address_bus_r <= addrbus[7:0];
	end
end

reg sdcs;
reg sdclk;
reg sdcmd;
reg[6:0] sddata;
wire[7:0] sd_o = {sddata, SD_DAT};

assign SD_DAT3 = ~sdcs;
assign SD_CMD = sdcmd;
assign SD_CLK = sdclk;

always @(posedge clk50mhz or posedge reset) begin
	if (reset) begin
		sdcs <= 1'b0;
		sdclk <= 1'b0;
		sdcmd <= 1'h1;
	end else begin
		if ((address_bus_r==8'h70)&io_write&~cpu_wr_n) sdcs <= cpu_o[0];
		if ((address_bus_r==8'h71)&io_write&~cpu_wr_n) begin
			if (sdclk) sddata <= {sddata[5:0],SD_DAT};
			sdcmd <= cpu_o[7];
			sdclk <= 1'b0;
		end
		if (cpu_rd) sdclk <= 1'b1;
	end
end 

endmodule
