// ====================================================================
//                Bashkiria-2M FPGA REPLICA
//
//            Copyright (C) 2010 Dmitry Tselikov
//
// This core is distributed under modified BSD license. 
// For complete licensing information see LICENSE.TXT.
// -------------------------------------------------------------------- 
//
// An open implementation of Bashkiria-2M home computer
//
// Author: Dmitry Tselikov   http://bashkiria-2m.narod.ru/
// 
// Design File: b2m_de1.v
//
// Top-level design file of Bashkiria-2M replica.

`define WITH_LEDs

module b2m_de1(
	input			clk50mhz,
	input [3:0] 	KEY,
	output [9:0] 	LEDr,
	output [7:0] 	LEDg,
	input [9:0] 	SW, 

	output [6:0] 	HEX0,
	output [6:0] 	HEX1,
	output [6:0] 	HEX2,
	output [6:0] 	HEX3,

	inout	[15:0]	SRAM_DQ,				//	SRAM Data bus 16 Bits
	output	[17:0]	SRAM_ADDR,				//	SRAM Address bus 18 Bits
	output			SRAM_UB_N,				//	SRAM High-byte Data Mask 
	output			SRAM_LB_N,				//	SRAM Low-byte Data Mask 
	output			SRAM_WE_N,				//	SRAM Write Enable
	output			SRAM_CE_N,				//	SRAM Chip Enable
	output			SRAM_OE_N,				//	SRAM Output Enable

	output 			VGA_HS,
	output 			VGA_VS,
	output	[3:0] 	VGA_R,
	output	[3:0] 	VGA_G,
	output	[3:0] 	VGA_B,

	inout			I2C_SDAT,				//	I2C Data
	output			I2C_SCLK,				//	I2C Clock

	inout			AUD_BCLK,
	output			AUD_DACDAT,
	output			AUD_DACLRCK,
	output			AUD_XCK,
	output			AUD_ADCLRCK,			//	Audio CODEC ADC LR Clock
	input			AUD_ADCDAT,				//	Audio CODEC ADC Data

	input			PS2_CLK,
	input			PS2_DAT,

	////////////////////	USB JTAG link	////////////////////////////
	input  			TDI,					// CPLD -> FPGA (data in)
	input  			TCK,					// CPLD -> FPGA (clk)
	input  			TCS,					// CPLD -> FPGA (CS)
	output 			TDO,					// FPGA -> CPLD (data out)

	input			SD_DAT,					//	SD Card Data 			(MISO)
	output			SD_DAT3,				//	SD Card Data 3 			(CSn)
	output			SD_CMD,					//	SD Card Command Signal	(MOSI)
	output			SD_CLK,					//	SD Card Clock			(SCK)

	output			UART_TXD,
	input			UART_RXD,

	input tape_in,
    output tape_out,
    output beep_out,

	output [12:0] 	GPIO_0,
	output [35:0]	GPIO_1 );

assign GPIO_0 = 0;
assign GPIO_1 = 0;
assign TDO = 0;
assign UART_TXD = 0;

////////////////////   RESET   ////////////////////
reg[3:0] reset_cnt;
reg reset_n;
wire reset = ~reset_n;

always @(posedge clk50mhz) begin
	if (KEY[0] && reset_cnt==4'd14)
		reset_n <= 1'b1;
	else begin
		reset_n <= 1'b0;
		reset_cnt <= reset_cnt+4'd1;
	end
end

////////////////////   CPU   ////////////////////
wire[15:0] addrbus;
wire[7:0] cpu_o;
wire cpu_sync;
wire cpu_rd;
wire cpu_wr_n;
wire cpu_int;
wire cpu_inta_n;
reg[7:0] sysctl;
reg[7:0] cpu_i;
reg[7:0] port_o;

////////////////////   PIT   ////////////////////
wire[7:0] pit_o;
wire pit_out0;
wire pit_out1;
wire pit_out2;

////////////////////   SOUND   ////////////////////
reg tapein;
reg[15:0] linein;
reg[15:0] adcbuf;
reg[13:0] cnt18;
reg[4:0] cntbit;
reg bitclk;

////////////////////   LEDs   ////////////////////

`ifdef WITH_LEDs

assign LEDr[0] = cpu_sync;
assign LEDr[1] = cpu_rd;
assign LEDr[2] = ~cpu_wr_n;
assign LEDr[9:3] = 0;
assign LEDg = cpu_rd ? cpu_i : cpu_o;

seg7_lut4 seg0(HEX0,HEX1,HEX2,HEX3, addrbus);

`else

assign LEDr = 0;
assign LEDg = 0;
assign HEX0 = 7'h7F;
assign HEX1 = 7'h7F;
assign HEX2 = 7'h7F;
assign HEX3 = 7'h7F;

`endif

////////////////////   STEP & GO   ////////////////////
reg		stepkey;
reg		onestep;

always @(posedge clk50mhz) begin
	stepkey <= KEY[1];
	onestep <= stepkey & ~KEY[1];
end

////////////////////   MEM   ////////////////////
wire mapvpage = ^mmap[2:1] && addrbus[15:11]>=5'b00101 && addrbus[15:12]<4'b0111;
wire rom_oe = mmap==3'h7 || (mmap!=3'h6 && &addrbus[15:13]);
wire kbd_oe = mapvpage && addrbus[15:12]==4'b0010;
wire sram_msb = npage[0];
wire[2:0] mmap = ppa1_c[2:0];
wire[2:0] npage = mapvpage ? {1'b0,mmap[1:0]} : {1'b1,addrbus[15:14]};
wire[7:0] mem_o = sram_msb ? SRAM_DQ[15:8] : SRAM_DQ[7:0];
wire[7:0] rom_o;

assign SRAM_DQ[7:0]  = SRAM_WE_N| sram_msb ? 8'bZZZZZZZZ : cpu_o;
assign SRAM_DQ[15:8] = SRAM_WE_N|~sram_msb ? 8'bZZZZZZZZ : cpu_o;
assign SRAM_ADDR = vid_rd ? {3'b000,~ppa1_c[7],vid_addr[13:8],vid_addr[7:0]+ppa1_b} : {2'b00,npage[2:1],addrbus[13:0]};
assign SRAM_UB_N = vid_rd ? 1'b0 : ~sram_msb;
assign SRAM_LB_N = vid_rd ? 1'b0 : sram_msb;
assign SRAM_WE_N = vid_rd ? 1'b1 : cpu_wr_n|sysctl[4]|(|vid_state);
assign SRAM_OE_N = ~(vid_rd|cpu_rd);
assign SRAM_CE_N = 0;

bios rom(.address(addrbus[12:0]), .clock(clk50mhz), .q(rom_o));

////////////////////   CPU   ////////////////////

always @(posedge clk50mhz)
	casex (addrbus[4:0])
	5'b000xx: port_o <= pit_o;
	5'b010xx: port_o <= ppa1_o;
	5'b1010x: port_o <= pic_o;
	5'b11011: port_o <= sd_o;
	default: port_o <= 0;
	endcase

wire port_wr_n = cpu_wr_n|~sysctl[4];
wire port_rd = cpu_rd&sysctl[6];
wire pit_we_n =  addrbus[4:2]!=3'b000|port_wr_n;
wire ppa1_we_n = addrbus[4:2]!=3'b010|port_wr_n;
wire pal_we_n  = addrbus[4:2]!=3'b100|port_wr_n;
wire pic_we_n  = addrbus[4:1]!=4'b1010|port_wr_n;
wire sio_we_n  = addrbus[4:1]!=4'b1100|port_wr_n;
wire pit_rd = addrbus[4:2]==3'b000&port_rd;

always @(*)
	casex ({~cpu_inta_n,sysctl[6],rom_oe,kbd_oe})
	4'b1xxx: cpu_i = pic_o;
	4'b01xx: cpu_i = port_o;
	4'b001x: cpu_i = rom_o;
	4'b0001: cpu_i = kbd_o;
	4'b0000: cpu_i = mem_o;
	endcase

reg sound_on;
reg cpu_flag;
reg[10:0] cpu_cnt;

wire cpu_ce = onestep | (cpu_ce2 & ~SW[9]) | (SW[8] && vid_state==0);
wire cpu_ce2 = (cpu_flag^cpu_cnt[10]) && vid_state==0;

always @(posedge clk50mhz) begin
	if (cpu_sync) sysctl <= cpu_o;
	if (addrbus[0]&~sio_we_n) sound_on <= ~cpu_o[5];
	cpu_cnt <= cpu_cnt + 11'd41;
	cpu_flag <= cpu_flag^cpu_ce2;
end

k580wm80a CPU(.clk(clk50mhz), .ce(cpu_ce), .reset(reset),
	.idata(cpu_i), .addr(addrbus), .sync(cpu_sync), .rd(cpu_rd), .wr_n(cpu_wr_n),
	.intr(cpu_int), .inta_n(cpu_inta_n), .odata(cpu_o));

////////////////////   VIDEO   ////////////////////
wire vid_irq;
wire vid_drq;
wire[13:0] vid_addr;
reg[15:0] vid_data;
reg[1:0] vid_state;
reg vid_exdrq;

wire vid_rd = vid_state==2'b10;

always @(posedge clk50mhz) begin
	vid_exdrq <= vid_drq;
	case (vid_state)
	2'b00: vid_state <= vid_drq && ~vid_exdrq ? 2'b01 : 2'b00;
	2'b01: vid_state <= 2'b10;
	2'b10: vid_state <= 2'b11;
	2'b11: vid_state <= 2'b00;
	endcase
	if(vid_rd) vid_data <= SRAM_DQ;
end

b2m_video video(.clk50mhz(clk50mhz), .hr(VGA_HS), .vr(VGA_VS), .vid_irq(vid_irq),
	.r(VGA_R), .g(VGA_G), .b(VGA_B), .drq(vid_drq), .addr(vid_addr), .idata(vid_data),
	.pal_idx(addrbus[1:0]), .pal_data(cpu_o), .pal_we_n(pal_we_n), .color_mode(SW[0]), .mode(SW[1]));

////////////////////   KBD   ////////////////////
wire[7:0] kbd_o;

b2m_kbd kbd(.clk(clk50mhz), .reset(reset), .ps2_clk(PS2_CLK), .ps2_dat(PS2_DAT),
	.addr(addrbus[8:0]), .odata(kbd_o));

////////////////////   SYS PPA   ////////////////////
wire[7:0] ppa1_o;
wire[7:0] ppa1_a;
wire[7:0] ppa1_b;
wire[7:0] ppa1_c;

k580ww55 ppa1(.clk(clk50mhz), .reset(reset), .addr(addrbus[1:0]), .we_n(ppa1_we_n),
	.idata(cpu_o), .odata(ppa1_o), .ipa(ppa1_a), .opa(ppa1_a),
	.ipb(ppa1_b), .opb(ppa1_b), .ipc(ppa1_c), .opc(ppa1_c));

////////////////////   PIC   ////////////////////
wire[7:0] pic_o;

k580wn59 pic(.clk(clk50mhz), .reset(reset), .addr(addrbus[0]), .we_n(pic_we_n),
	.idata(cpu_o), .odata(pic_o), .intr(cpu_int), .inta_n(cpu_inta_n),
	.irq({3'b0,tape_in,2'b0,pit_out0,vid_irq}));
//	.irq({3'b0,tapein,2'b0,pit_out0,vid_irq}));
	
////////////////////   PIT   ////////////////////

k580wi53 pit(.clk(clk50mhz), .c0(pit_out2), .c1(cpu_ce), .c2(cpu_ce),
	.g0(1'b1), .g1(sound_on), .g2(1'b1), .out0(pit_out0), .out1(pit_out1), .out2(pit_out2),
	.addr(addrbus[1:0]), .rd(pit_rd), .we_n(pit_we_n), .idata(cpu_o), .odata(pit_o));

////////////////////   SOUND   ////////////////////
assign tape_out = ppa1_c;
assign beep_out = pit_out1;

wire[15:0] pulses = {3'b0,pit_out1,ppa1_c[6],11'b0};
wire[5:0] line6bit = {~linein[15],linein[14:10]};

wire[13:0] cnt18next = cnt18+14'd755; // 755/2048*50MHz ~ 18.432MHz

assign AUD_XCK = cnt18[10];
assign AUD_BCLK = reset ? 1'bZ : bitclk;
assign AUD_DACLRCK = reset ? 1'b0 : cntbit[4];
assign AUD_ADCLRCK = reset ? 1'b0 : cntbit[4];
assign AUD_DACDAT = reset ? 1'b0 : pulses[~cntbit[3:0]];

always @(posedge clk50mhz) begin
	if (cnt18next[13:11]==3'd6) begin
		cnt18 <= {3'd0, cnt18next[10:0]};
		bitclk <= ~bitclk;
		if (bitclk) begin // negedge bitclk
			if (cntbit==4'd0) linein <= adcbuf;
			adcbuf[~cntbit[3:0]] <= AUD_ADCDAT;
			cntbit <= cntbit+5'b1;
		end
	end else
		cnt18 <= cnt18next;
	if (line6bit < 31) tapein <= 1'b0;
	if (line6bit > 32) tapein <= 1'b1;
end

I2C_AV_Config sndcfg(.iCLK(clk50mhz), .iRST_N(reset_n), .I2C_SCLK(I2C_SCLK), .I2C_SDAT(I2C_SDAT));

////////////////////   SD CARD   ////////////////////
reg sdcs;
reg sdclk;
reg sdcmd;
reg[6:0] sddata;
wire[7:0] sd_o = {sddata, SD_DAT};

assign SD_DAT3 = ~sdcs;
assign SD_CMD = sdcmd;
assign SD_CLK = sdclk;

always @(posedge clk50mhz or posedge reset) begin
	if (reset) begin
		sdcs <= 1'b0;
		sdclk <= 1'b0;
		sdcmd <= 1'h1;
	end else begin
		if (addrbus[4:0]==5'h1A && ~port_wr_n) sdcs <= cpu_o[0];
		if (addrbus[4:0]==5'h1B && ~port_wr_n) begin
			if (sdclk) sddata <= {sddata[5:0],SD_DAT};
			sdcmd <= cpu_o[7];
			sdclk <= 1'b0;
		end
		if (cpu_rd) sdclk <= 1'b1;
	end
end

endmodule
